// Flatshaded cube in 256-color mode on the 32X
// Mic, 2008

#include <stdlib.h>
#include "32x.h"

#define MAX_POLYS 32

// Convert between integer and fixed point
#define INT2FIX(a) (((int)(a))<<7)
#define FIX2INT(a) (((int)(a))>>7)

/* Create a 5:5:5 RGB color with the MSB set */
#define COLOR(r,g,b)    (((r)&0x1F)|((g)&0x1F)<<5|((b)&0x1F)<<10|0x8000)

typedef struct
{
	int x,y,z;
} point3d;

typedef struct
{
	int p0,p1,p2,p3;
} quad;


const char sintb[256] =
{0,3,6,9,12,15,18,21,24,27,30,33,36,39,42,45,48,51,54,57,59,62,65,67,
70,73,75,78,80,82,85,87,89,91,94,96,98,100,102,103,105,107,108,110,
112,113,114,116,117,118,119,120,121,122,123,123,124,125,125,126,126,
126,126,126,126,126,126,126,126,126,125,125,124,123,123,122,121,120,
119,118,117,116,114,113,112,110,108,107,105,103,102,100,98,96,94,91,
89,87,85,82,80,78,75,73,70,67,65,62,59,57,54,51,48,45,42,39,36,33,30,
27,24,21,18,15,12,9,6,3,0,-4,-7,-10,-13,-16,-19,-22,-25,-28,-31,-34,
-37,-40,-43,-46,-49,-52,-55,-58,-60,-63,-66,-68,-71,-74,-76,-79,-81,
-83,-86,-88,-90,-92,-95,-97,-99,-101,-103,-104,-106,-108,-109,-111,
-113,-114,-115,-117,-118,-119,-120,-121,-122,-123,-124,-124,-125,-126,
-126,-127,-127,-127,-127,-127,-127,-127,-127,-127,-127,-127,-126,-126,
-125,-124,-124,-123,-122,-121,-120,-119,-118,-117,-115,-114,-113,-111,
-109,-108,-106,-104,-103,-101,-99,-97,-95,-92,-90,-88,-86,-83,-81,-79,
-76,-74,-71,-68,-66,-63,-60,-58,-55,-52,-49,-46,-43,-40,-37,-34,-31,
-28,-25,-22,-19,-16,-13,-10,-7,-4};


int 		theta = 0;


/*  Points defining the cube

       4----5
      /|   /|
     0----1 |
     | 7--|-6
     |/   |/
     3----2

*/
int 		points[8][3] = {{-1,-1,-1},{1,-1,-1},{1,1,-1},{-1,1,-1},
 						  {-1,-1,1},{1,-1,1},{1,1,1},{-1,1,1}};

// Faces defining the cube.
int 		faces[6][4] = {{0,1,2,3},{5,4,7,6},{4,5,1,0},
    		               {1,5,6,2},{6,7,3,2},{4,0,3,7}};

point3d		*model=(point3d*)points,rotated[8],projected[8],camera;
quad 		*cube=(quad*)faces;
int 		faceOrder[MAX_POLYS];
int 		avgZ[MAX_POLYS];



void rotate(point3d *in, point3d *out, int angle, int n)
{
	int i;
	int xr,yr,zr,temp;
	int can,san;

	san = sintb[angle & 0xff];
	can = sintb[(angle + 0x40) & 0xff];

	for (i=0; i<n; i++)
	{
		// About X
		out[i].y = ((in[i].y * can) - (in[i].z * san)); out[i].y=FIX2INT(out[i].y);
		out[i].z = ((in[i].y * san) + (in[i].z * can)); out[i].z=FIX2INT(out[i].z);

		// About Y
		out[i].x = ((in[i].x * can) - (out[i].z * san)); out[i].x=FIX2INT(out[i].x);
		out[i].z = ((in[i].x * san) + (out[i].z * can)); out[i].z=FIX2INT(out[i].z);

		// About Z
		temp = out[i].x;
		out[i].x = ((out[i].x * can) - (out[i].y * san)); out[i].x=FIX2INT(out[i].x);
		out[i].y = ((temp * san) + (out[i].y * can)); out[i].y=FIX2INT(out[i].y);
	}

}


void project(point3d *in, point3d *out, int n)
{
	int i;

	for (i=0; i<n; i++)
	{
		out[i].x = camera.x + FIX2INT(((in[i].x * camera.z) / (camera.z - FIX2INT(in[i].z))));
		out[i].y = camera.y + FIX2INT(((in[i].y * camera.z) / (camera.z - FIX2INT(in[i].z))));
		out[i].z = in[i].z;
	}
}


void sort_quads(quad *f, point3d *p, int *order, int n)
{
	int i,j,temp,gap,last;
	int avgZi,avgZj;
	int orderi,orderj;

	// Initialize arrays
	for (i=0; i<n; i++)
	{
		avgZ[i] = p[f[i].p0].z +
                  p[f[i].p1].z +
                  p[f[i].p2].z +
                  p[f[i].p3].z;
		order[i] = i;
	}

	// Do a shellsort on the faces
	last = n - 1;
	gap = (n / 3) + 1;
	while (1)
	{
		for (i=gap; i<=last; i++)
		{
			avgZi = avgZ[i]; orderi = order[i];
			j = i - gap;
			while (1)
			{
				avgZj = avgZ[j]; orderj = order[j];
				if (avgZj < avgZi)
				{
					j += gap;
					break;
				}
				avgZ[j + gap] = avgZj; order[j + gap] = orderj;
				if (j <= gap)
				{
					break;
				}
				j -= gap;
			}
			avgZ[j] = avgZi; order[j] = orderi;
		}
		if (gap == 1)
		{
			break;
		}
		gap = (gap / 3) + 1;
	}
}


// Draw one section of a polygon
void draw_section(int *x1, int y1, int *x2, int y2, int dxdy1, int dxdy2, uint8 color)
{
	int i,y;
	long x1_,x2_;
	vu8 *dest,*last_dest;
	long clipped;

	if (y1 >= 320) return;
	if (y2 < 0) return;


  	if (y1 < 0)
  	{
    	clipped = -y1;
    	*x1 += (dxdy1 * clipped);
    	*x2 += (dxdy2 * clipped);
    	y1 = 0;
  	}

  	if (y2 >= 224) y2 = 223;
  	dest = (vu8*)&MARS_FRAMEBUFFER;
  	dest += y1 * 320;
  	dest += 0x200;

  	for (y=y1; y<=y2; y++)
  	{
    	if (*x1 <= *x2)
    	{
      		x1_ = *x1;
      		x2_ = *x2;
    	}
    	else
    	{
      		x1_ = *x2;
      		x2_ = *x1;
    	}
    	x1_ = FIX2INT(x1_);
    	x2_ = FIX2INT(x2_);

    	if ((x1_<320) && (x2_ >= 0))
    	{
      		last_dest = dest;
      		dest += x1_;

			for (i=x1_; i<=x2_; i++)
				*(dest++) = color;
			dest = last_dest;
		}
		if (y<y2)
		{
			dest += 320;
			*x1 += dxdy1;
			*x2 += dxdy2;
		}
	}
}


// Draw a flatshaded triangle
void flat_tri(int x1, int y1, int x2, int y2, int x3, int y3, uint8 color)
{
	int dx1,dx2,dx3;
	int dy1,dy2,dy3;
	int dxdy1,dxdy2,dxdy3;
	int x1_,x2_;
	int tx,ty;

  	if (y2 < y1) { tx=x1; ty=y1; x1=x2; y1=y2; x2=tx; y2=ty; }
  	if (y3 < y1) { tx=x1; ty=y1; x1=x3; y1=y3; x3=tx; y3=ty; }
  	if (y3 < y2) { tx=x2; ty=y2; x2=x3; y2=y3; x3=tx; y3=ty; }

	dx1 = (x2 - x1);
	dx2 = (x3 - x2);
	dx3 = (x3 - x1);

	dy1 = (y2 - y1);
	dy2 = (y3 - y2);
	dy3 = (y3 - y1);
	if (dy1 == 0) dy1++;
	if (dy2 == 0) dy2++;
	if (dy3 == 0) dy3++;

	dxdy1 = INT2FIX(dx1)/dy1;
	dxdy2 = INT2FIX(dx2)/dy2;
	dxdy3 = INT2FIX(dx3)/dy3;

	x1_ = INT2FIX(x1);
	x2_ = x1_;
 	draw_section(&x1_, y1, &x2_, y2, dxdy1, dxdy3, color);
 	x1_ = INT2FIX(x2);
 	draw_section(&x1_, y2, &x2_, y3, dxdy2, dxdy3, color);
}



int main()
{
	uint16 currentFB=0;
	uint16 lineOffs;
	uint16 p;
	vu16 *frameBuffer16 = &MARS_FRAMEBUFFER;
	vu16 *palette = &MARS_CRAM;
	int fbOffs,ltOffs;
	int i,j;
	uint8 shade;

	// Wait for the SH2 to gain access to the VDP
	while ((MARS_SYS_INTMSK & MARS_SH2_ACCESS_VDP) == 0) {}

	// Set 8-bit paletted mode, 224 lines
	MARS_VDP_DISPMODE = MARS_224_LINES | MARS_VDP_MODE_256;

	MARS_VDP_FBCTL = currentFB;

	// Set up the palette
	for (i=0; i<32; i++)
	{
		palette[i] = COLOR(i>>1, i, i);
		palette[i+32] = COLOR(i, 0, 0);
		palette[i+64] = COLOR(31, i, 0);
	}

	for (i=0; i<8; i++)
	{
		model[i].x = INT2FIX(model[i].x * 35);
		model[i].y = INT2FIX(model[i].y * 35);
		model[i].z = INT2FIX(model[i].z * 35);
	}

	camera.x = 160; camera.y = 112; camera.z = -200;

    for(;;)
    {
		rotate(model, rotated, theta++, 8);
		project(rotated, projected, 8);
		sort_quads(cube, rotated, faceOrder, 6);

		// Flip the framebuffer selection bit and wait for it to take effect
		MARS_VDP_FBCTL = currentFB ^ 1;
		while ((MARS_VDP_FBCTL & MARS_VDP_FS) == currentFB) {}
		currentFB ^= 1;

		// Clear the screen
		fbOffs = 0x100;
		ltOffs = 0;
		for (i=0; i<224; i++)
		{
			frameBuffer16[ltOffs++] = fbOffs;
			p = (32 + (i>>2)) | ((32 + (i>>2))<<8);
			for (j=0; j<160; j++)
			{
				frameBuffer16[fbOffs++] = p;
			}
		}

		// Draw all 6 faces of the cube
 		for (i=0; i<6; i++)
 		{
			j = faceOrder[5 - i];
			shade = (((-FIX2INT(avgZ[5 - i]) / 4) + 35) * 31) / 70;		// Calculate shade, 0..31

			flat_tri(projected[cube[j].p0].x, projected[cube[j].p0].y,
			         projected[cube[j].p1].x, projected[cube[j].p1].y,
			         projected[cube[j].p2].x, projected[cube[j].p2].y,
			         shade);
			flat_tri(projected[cube[j].p0].x, projected[cube[j].p0].y,
			         projected[cube[j].p2].x, projected[cube[j].p2].y,
			         projected[cube[j].p3].x, projected[cube[j].p3].y,
			         shade);
		}
	}
}
