
// epimgconv: Enterprise 128 image converter utility
// Copyright (C) 2008-2016 Istvan Varga <istvanv@users.sourceforge.net>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// The Enterprise 128 program files generated by this utility are not covered
// by the GNU General Public License, and can be used, modified, and
// distributed without any restrictions.

#include "epimgconv.hpp"
#include "imageconv.hpp"
#include "pixel256.hpp"

#include <vector>

namespace Ep128ImgConv {

  void ImageConv_Pixel256::initializePalettes()
  {
    for (int i = 0; i < 256; i++)
      convertEPColorToYUV(i, paletteY[i], paletteU[i], paletteV[i]);
  }

  void ImageConv_Pixel256::pixelStoreCallback(void *userData, int xc, int yc,
                                              float y, float u, float v)
  {
    ImageConv_Pixel256&  this_ =
        *(reinterpret_cast<ImageConv_Pixel256 *>(userData));
    xc = xc >> 3;
    yc = yc >> 1;
    if (xc < 0 || xc >= this_.width || yc < 0 || yc >= this_.height)
      return;
    limitYUVColor(y, u, v);
    this_.inputImage.y(xc, yc) += (y * 0.0625f);
    this_.inputImage.u(xc, yc) += (u * 0.0625f);
    this_.inputImage.v(xc, yc) += (v * 0.0625f);
  }

  void ImageConv_Pixel256::pixelStoreCallbackI(void *userData, int xc, int yc,
                                               float y, float u, float v)
  {
    ImageConv_Pixel256&  this_ =
        *(reinterpret_cast<ImageConv_Pixel256 *>(userData));
    xc = xc >> 3;
    if (xc < 0 || xc >= this_.width || yc < 0 || yc >= this_.height)
      return;
    limitYUVColor(y, u, v);
    this_.inputImage.y(xc, yc) += (y * 0.125f);
    this_.inputImage.u(xc, yc) += (u * 0.125f);
    this_.inputImage.v(xc, yc) += (v * 0.125f);
  }

  ImageConv_Pixel256::ImageConv_Pixel256()
    : ImageConverter(),
      width(1),
      height(1),
      inputImage(1, 1),
      ditherErrorImage(1, 1),
      convertedImage(1, 1),
      borderColor(0x00),
      ditherType(1),
      ditherDiffusion(0.95f)
  {
    initializePalettes();
  }

  ImageConv_Pixel256::~ImageConv_Pixel256()
  {
  }

  bool ImageConv_Pixel256::processImage(
      ImageData& imgData, const char *infileName,
      YUVImageConverter& imgConv, const ImageConvConfig& config)
  {
    width = config.width << 1;
    height = ((imgData[5] & 0x80) == 0 ? config.height : (config.height << 1));
    borderColor = config.borderColor & 0xFF;
    float   borderY = 0.0f;
    float   borderU = 0.0f;
    float   borderV = 0.0f;
    convertEPColorToYUV(borderColor, borderY, borderU, borderV);
    inputImage.setBorderColor(borderY, borderU, borderV);
    ditherType = config.ditherType;
    limitValue(ditherType, 0, 5);
    ditherDiffusion = float(config.ditherDiffusion);

    inputImage.resize(width, height);
    ditherErrorImage.resize(width, height);
    convertedImage.resize(width, height);
    inputImage.clear();
    ditherErrorImage.clear();
    convertedImage.clear();

    initializePalettes();

    if (!(imgData[5] & 0x80))
      imgConv.setPixelStoreCallback(&pixelStoreCallback, (void *) this);
    else
      imgConv.setPixelStoreCallback(&pixelStoreCallbackI, (void *) this);
    if (!imgConv.convertImageFile(infileName))
      return false;

    progressMessage("Converting image");
    setProgressPercentage(0);
    unsigned char palette[256];
    for (int i = 0; i < 256; i++)
      palette[i] = (unsigned char) i;
    for (int yc = 0; yc < height; yc++) {
      ditherLine(convertedImage, inputImage, ditherErrorImage, yc,
                 ditherType, ditherDiffusion,
                 float(config.colorErrorScale), &(palette[0]), 256,
                 paletteY, paletteU, paletteV);
    }
    imgData.setBorderColor(borderColor);
    for (int yc = 0; yc < height; yc++) {
      for (int xc = 0; xc < width; xc++)
        imgData.setPixel(xc, yc, convertedImage[yc][xc]);
    }
    setProgressPercentage(100);
    progressMessage("");
    return true;
  }

}       // namespace Ep128ImgConv

