;▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓
;▓▓                                                                                  ▓▓░░
;▓▓        Floating-point Instruction Bytecode Generator & Recompiler, v1.0.0        ▓▓░░
;▓▓                               (for fasm 1.73.11+)                                ▓▓░░
;▓▓                                                                                  ▓▓░░
;▓▓▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▀▄▓▓░░
;▓▓                                                                                  ▓▓░░
;▓▓  Copyright (c) 2021 Eugene Krasnikov / Евгений Красников (aka Jin X)             ▓▓░░
;▓▓                                                                                  ▓▓░░
;▓▓  Permission is hereby granted, free of charge, to any person obtaining a copy    ▓▓░░
;▓▓  of this software and associated documentation files (the "Software"), to deal   ▓▓░░
;▓▓  in the Software without restriction, including without limitation the rights    ▓▓░░
;▓▓  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell       ▓▓░░
;▓▓  copies of the Software, and to permit persons to whom the Software is           ▓▓░░
;▓▓  furnished to do so, subject to the following conditions:                        ▓▓░░
;▓▓                                                                                  ▓▓░░
;▓▓  The above copyright notice and this permission notice shall be included in all  ▓▓░░
;▓▓  copies or substantial portions of the Software.                                 ▓▓░░
;▓▓                                                                                  ▓▓░░
;▓▓  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR      ▓▓░░
;▓▓  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,        ▓▓░░
;▓▓  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE     ▓▓░░
;▓▓  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER          ▓▓░░
;▓▓  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,   ▓▓░░
;▓▓  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE   ▓▓░░
;▓▓  SOFTWARE.                                                                       ▓▓░░
;▓▓                                                                                  ▓▓░░
;▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓░░
;  ░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░


; WHAT IS THIS?
; =============
;
; Floating-point Instruction Bytecode Generator & Recompiler, v1.0.0.
;
; In fact, this is a packer of x86 FPU instructions with ≈ 3-4x efficiency (not taking into account recompiler code and constants).
; All you need is just define some symbolic constants, include this file in your fasm project, use a few macros and perform a call to recompiled function.
; Macros allow you:
; - to check module version compatibility;
; - to declare set of bytecode instructions (used FPU instructions and their aliases);
; - to insert code of bytecode recompiler;
; - to generate bytecode (replacing native x86 FPU code);
; - to define constants/variables and their aliases (names);
;
;   >>> FEATURES <<<
;
; • Definition of FPU operations in easy to read format of Reverse Polish Notation (RPN).
; • Flexibly customizable recompiler which creates native x86 functions from bytecode (that is, fast for execution).
; • Native code copier (so you can mix FPU instructions with native x86 code inside bytecode block).
; • Declaration of up to 11-13 bytecode instructions associated with 2-byte FPU (...or not only FPU?...) instructions (this number is sufficient for most purposes).
; • Definition of up to 17 2-byte integer and floating-point constants.
; • Definition of up to 16 4-byte floating-point variables.
; • Declaration of unlimited subfunctions with custom names inside bytecode block.
; • Possibility to override some predefined bytecode instructions associated with native x86 instructions.
; • The size of recompiler is ≈ 50..74 bytes (for 1..13 types of bytecode instructions, with no support of native code copier, variables and subfunctions)
;   to ≈ 65..85 / 76..96 bytes (for 1..11 types of bytecode instructions, with max functionality) including writting of input stream offset to register.
;   This size can vary depending on used options of macro 'fpb_recompile!'.
;
;   >>> TERMS <<<
;
; I will use the following terms in this file:
; • bytecode (bytecode block) - sequence of bytecode instructions (block of code, function);
; • bytecode instruction - one element of bytecode block associated with native instruction;
; • native instruction (code) - instruction (set of instructions) encoded in ready-to-execute format of x86 CPU;
; • opcode - numeric representation of instruction;
; • subfunction - part of bytecode ending with return ('ret') bytecode instruction;
; • recompiler - native code that translates bytecode into native code (ready-to-execute function);
; • native code copier - block of recompiler that copies native x86 code from bytecode block into destination function;
; • nibble - half of a byte;
; • alias - symbolic name (consisting of letter, digits and other characters);
; • symbolic constant - fasm constant (defined with 'define', 'equ' directives or '=' sign);
; • constants and variables - constants and variables defined with macros 'fpb_flt', 'fpb_int' and 'fpb_var';
; • constant / variable array - set of constants / variables, also offset of the 1st constant / variable;
; • int and float (value, variable, etc) - integer and floating-point (value, variable, etc);
; • offset - address offset within a single segment;
; • pointer - pair of segment and offset.
;
;   >>> WHAT DOES THE CODE LOOK LIKE? <<<
;
; * Let's first consider the simplest variant...
;
; define fpb_copy_support 0			; disable native code copier support
; define fpb_var_support 0			; disable variable support
;
; include 'fpb.inc'				; include fpb generator & recompiler
; fpb_check_version 1,0,0			; check fpb module version compatibility
;
; ; Declare fpb bytecode instructions
; declare_fpb	*, fmulp
; declare_fpb	sincos, fsincos
; ; . . .
;		mov	di,$200			; destination offset for recompilation
;		fpb_recompile! fpb_nibbles, bh=0 ; insert recompiler code here (doesn't require 'call')
; ; . . .
;		mov	bx,consts		; constant array (doesn't require offset correction because fpb_data_offset is not defined)
;		call	$200			; call recompiled function
; ; . . .
;	fpb_nibbles:
;	  fpb_start	consts			; start of bytecode generation
;		fpb	rad * sincos		; convert degrees to radiants and calc sin & cos
;	  fpb_end				; end of bytecode generation
;
;	consts = $-2				; don't define this symbolic constant as label here, otherwise an error can be generated (-2 is required because the 1st constant is float, not integer, see below)
;		fpb_flt	rad, 0.0174533		; define 2-byte float constant with name 'rad' and value pi/180
;
; * Not so hard, really? :))
; * And now let's consider more complex case...
;
; define fpb_copy_support 1			; enable native code copier support
; define fpb_var_support 1			; enable variable support
; define fpb_before_ret salc			; one-byte native instruction before 'ret' in recompiled function
; fpb_data_offset =	vars-consts		; offset of constants and variables
;
; include 'fpb.inc'				; include fpb generator & recompiler
; fpb_check_version 1,0,0			; check fpb module version compatibility
;
; ; Declare fpb bytecode instructions
; declare_fpb	+, faddp
; declare_fpb	^2, fmul st0,st0
; declare_fpb	0, fldz
; declare_fpb	sin, fsin
; ; . . .
;		mov	di,bp			; destination offset for recompilation
;		fpb_recompile! fpb_nibbles, bh=0, ax=0 ; insert recompiler code here (doesn't require 'call')
; ; . . .
;		mov	bx,consts-fpb_data_offset ; constant and variable arrays (with offset correction)
;		call	bp			; call recompiled function
; ; . . .
;	fpb_nibbles:
;	  fpb_start	consts			; start of bytecode generation
;		fpb	x pi + sin ^2 0		; list of bytecode aliases (sequence of RPN-operations in fact)
;	    fpb_copy_start			; start of native code block
;		fcomp
;		fstsw	ax
;		sahf
;		jnb	@F
;		lea	ax,[bp+fpb_offset_subfn_lt0] ; offset of subfunction 'lt0'
;		jmp	ax
;	@@:
;	    fpb_copy_end			; end of native code block
;		fpb	=res			; store value to variable 'res'
;	    fpb_subfn	lt0			; declare new subfunction (and its offset as 'fpb_offset_subfn_lt0' relative to start of native code block (bp))
;		fpb	2 + =res
;	  fpb_end				; end of bytecode generation
;
;	consts = $				; don't define this symbolic constant (and 'vars' too) as label here, otherwise an error can be generated
;		fpb_int	2			; define 2-byte integer constant with name and value '2'
;		fpb_flt	pi, 3.1415927		; define 2-byte float constant 'pi' with specified value
;	vars = $
;		fpb_var	x, 0.0			; declare 4-byte float variable 'x' with initial value 0.0
;		fpb_var	res			; declare 4-byte float variable 'res' without initial value
;
;   >>> BYTECODE FORMAT <<<
;
; Bytecode is just a sequence of bytecode instructions (encoded as a sequence of opcodes with their operands). Each opcode is encoded as nibble (the 1st nibble is encoded in higher
; 4 bits, the 2nd one is encoded in lower 4 bits). So bytecode is nibblecode in fact :).
; Number of bytecode instructions declared by macro 'declare_fpb' is stored to internal symbolic constant '@$fpb_gen_bytecode_ins'. Let's denote this value as BIC (bytecode instruction count).
; * Thus opcodes from 0 to BIC-1 encodes native instructions assocoated with declared bytecode instructions.
; * Opcode BIC encodes bytecode instructions 'copy' and 'ret'. If native code copier is not supported then this definitely means 'ret' instruction. Otherwise the next nibble X contains
;   operand. If X = 0 then opcode BIC treats as 'ret' bytecode instruction. If X > 0 then it treats as 'copy' bytecode instruction and X means the number of native bytes that must be copied
;   (actually the number of bytes to copy = X + copymin-1, see below).
; * Opcodes BIC+1 and BIC+2 encode bytecode instructions 'ldc' and 'ldic'. The next nibble X contains the number of constant in constant array which must be loaded into FPU stack.
;   Address of this constant is calculated as (reg + fpb_data_offset + X*2).
; * If variable support is enabled then opcodes BIC+3 and BIC+4 encode bytecode instructions 'ldv' and 'stv'. The next nibble X contains the number of variable in variable array which must be
;   loaded/store into/from FPU stack. Address of this variable is calculated as (reg + fpb_data_offset + X*4).
;
;   >>> ABOUT ERROR MESSAGES <<<
;
; Framed error messages generated by this module contain macro name in square brackets where this error occured.
; In rare cases you can see strange error message (this is a feature of fasm, I don't know how to fix it yet). I recommend you to look at details of fasm error message carefully in this case.
;
;
; USAGE DETAILS
; =============
;
;   >>> PREPARATION <<<
;
; First define symbolic constant 'fpb_copy_support' (BEFORE including this module):
; 1 - enable native code copier;
; 0 - disable native code copier [default value (if not defined)].
;
; Next define symbolic constant 'fpb_var_support' (BEFORE including this module):
; 1 - enable variable support (and 'ldv', 'stv' bytecode instructions) [default value (if not defined)];
; 0 - disable variable support.
;
; Also define symbolic constant 'fpb_data_offset' (BEFORE including this module and ONLY if variable support is enabled) [default is 0 (if not defined)].
; Just write 'fpb_data_offset = vars-consts', where 'vars' is offset of variable array, 'consts' is offset of constant array.
; Actually you shouldn't do this if you don't see an error message that some variable is out of reach (you will see this when
;   [number of constants + (number of variables) * 2] is more than 32).
;
; Then include this module (include 'fpb.inc').
; After that I recommend to add 'fpb_check_version' macro with specification of major, middle, minor version numbers of module you are using (see example above).
;
; Declare bytecode instructions using macro 'declare_fpb'. The 1st parameter is bytecode instruction alias to use later as argument of macro 'fpb'.
; The 2nd parameter is corresponding x86 instruction (it can contain commas). You can omit it if it matches the alias (e.g. 'declare_fpb faddp').
; Maximal number of bytecode instruction types is 11 if variables are supported or 13 if not.
; E.g.:
; declare_fpb	+, faddp
; declare_fpb	-, fsubp
; declare_fpb	*, fmulp
; declare_fpb	/, fdivp
; declare_fpb	x2, fadd st0,st0
; declare_fpb	^2, fmul st0,st0
; declare_fpb	abs, fabs
; declare_fpb	sqrt, fsqrt
; declare_fpb	sin, fsin
; declare_fpb	cos, fcos
; declare_fpb	arctg, fpatan
;
; Actually, you can use any instruction, not only FPU (e.g. 'mov ax,dx' or 'call cx') but any native instruction associated with bytecode instruction must be exactly 2 byte long!
; Note that you can use aliases that contain only literal. Literal can consist of letters, digits, pseudographics and characters '_', '.', '@', '$', '%', '^', '!', '?',
; e.g. 'sqrt', 'log2', 'fp_atan', '0.314e1', '5x', '^2', '.' and even '¼', '√', '²', '@$%', 'wat!?', 'плюс', '__', or strings in single or double quotes: '"hi <man>"')
; or a SINGLE special character (only one of the following: '+', '-', '*', '/', '=', '(', ')', '[', ']', ':', '~', '<,>' (comma must be enclosed with angle brackets here);
; you can't use characters '\', '|', '&', '#', '<', '>', '{', '}', '`', "'", '"', ';', spaces and tabs unless inside strings.
; So you can't use aliases like '-1.5', '1e+6', 'a/b', '**', ';)', 'hello world' because they mix literals and special characters (e.g. '-' and '1.5' or 'a', '/' and 'b'),
; consist of multiple special characters (e.g. '**') or contain illegal characters (e.g. ';' or space). Don't worry, you will see an error message in this case ;)
;
; There're 6 predefined bytecode instructions:
; - ret (usage: 'fpb ret') - end of subfunction / code block (you must not use it, macros 'fpb_end' and 'fpb_subfn' do it themselves);
; - copy (usage: 'fpb copy,n') - copy n following native code bytes ('copy,0' is equivalent to 'ret') [available only if native code copier support is enabled];
; - ldc (usage: 'fpb ldc,n') - load 32-bit float constant from [reg + fpb_data_offset + n*2] into FPU stack;
; - ldic (usage: 'fpb ldic,n') - load 16-bit integer constant from [reg + fpb_data_offset + n*2] into FPU stack;
; - ldv (usage: 'fpb ldv,n') - load 32-bit float variable from [reg + fpb_data_offset*2 + n*2] into FPU stack;
; - stv (usage: 'fpb stv,n') - store and pop 32-bit float variable to [reg + fpb_data_offset*2 + n*2] from FPU stack;
; Register 'reg' here is 'fnreg' register, specified for macro 'fpb_recompile!' (see below) [default is 'bx'].
; Values of 'n' for all predefined bytecode instructions must be in range from 0 to 15 inclusively.
; ATTENTION! I strongly recommend to use 'fpb_copy_start' / 'fpb_copy_end' instead of 'copy' and constant/variable aliases instead of 'ldc', 'ldic', 'ldv', 'stv' (see below).
;
; You can define symbolic constants 'fpb_opcode_BCI' (before using macro 'fpb_recompile!'), where BCI is 'ldc', 'ldic', 'ldv', 'stv' to override native instructions for these bytecode instructions.
; Note that opcode must be 3 bytes long, where the last byte is 8-bit offset of indirect addressing with offset. Default definitions (when constants are not defined) are:
; - ldc: fld dword [reg+1]
; - ldic: fild word [reg+1]
; - ldv: fld dword [reg+1]
; - stv: fstp dword [reg+1]
; E.g. if you don't want to pop value from FPU stack after storing variable, you can write 'define fpb_opcode_stv fst dword [reg+1]'.
; Remember that only 2 first bytes of each opcode will be stored and an offset of indirect addressing will be added after that (fpb_data_offset + X*2 for 'ldc' and 'ldic';
; fpb_data_offset*2 + X*4 for 'ldv' and 'stv', where X is bytecode instruction operand (e.g. X = 3 for 'fpb ldc, 3')).
; I strongly recommend to use 'reg' name inside definition to specify addressing register.
;
; Native instruction of 'ret' bytecode instruction is predefined as 'nop' + 'ret'.
; You can replace this 'nop' with useful instruction by defining constant 'fpb_before_ret' (e.g. 'define fpb_before_ret salc').
;
;   >>> USING GENERAL FPB MACROS <<<
;
; Macro 'fpb_recompile!' inserts recompiler code (this code is placed 'as is' and doesn't require 'call' or something else). Offset of bytecode block (input stream) is specified as
; the 1st parameters 'bytecode'. Other parameters can specify different options. Read description of this macro for more information (below).
; Before using 'fpb_recompile!' you must set register pair es:di to compiled function pointer (output stream);
; Bytecode block (input stream) must be located at address dseg:bytecode, where 'dseg' is value of corresponding option [default is ds], 'bytecode' is the 1st parameter of macro;
;
; To use recompiled function just set required register values and perform near call. Generally you need to set 'fnreg' register (see description of macro 'fpb_recompile'; default is bx)
; to offset of constant array minus fpb_data_offset (mov bx,consts-fpb_data_offset).
;
; Macros 'fpb_start', 'fpb_end', 'fpb', 'fpb_subfn', 'fpb_copy_start' and 'fpb_copy_end' are made to generate floating-point bytecode.
; Macro 'fpb_start' starts bytecode generation and requires a parameter 'data' which is offset of constant array. The name of main (zero) subfunction can be specified as the 2nd parameter.
; Macro 'fpb_end' ends bytecode generation. You can't use any other bytecode generation macros after 'fpb_end' except for 'fpb_start'.
; Macro 'fpb_subfn' declares new subfunction. It can get a subfunction name as parameter. A symbolic constant 'fpb_offset_subfn#' will be defined when using macros 'fpb_start' and
;   'fpb_subfn' (where # is a serial number, starting from 0). Also symbolic constant 'fpb_offset_subfn_NAME' will be defined if subfunction name is specified.
; Macro 'fpb_copy_start' and 'fpb_copy_end' declares a block of native code. You can place any native x86 instructions between them. The lengths of this code must be in range from
;   copymin to copymin+14 (see option 'copymin' of macro 'fpb_recompile!' below; default size range is 1..15). However, you can use multiple blocks of native code one after the other.
;   These 2 macros are allowed only if native code copier is enabled (fpb_copy_support <> 0).
;
; Macro 'fpb' generates bytecode instructions. It gets list of bytecode instruction, constants and variables aliases separated by space.
;   During recompilation instruction aliases declared by macro 'declare_fpb' are translated into native instructions specified on declaration.
;   Constant and variables aliases means pushing corresponding value into FPU stack and translated into native instructions associated with 'ldc', 'ldic' and 'ldv' bytecode instructions
;     (see information about symbolic constants 'fpb_opcode_BCI' above).
;   Equal sign followed by variable alias (e.g. '=var') means storing top FPU stack value into variable (and popping it from stack, at least if 'stv' bytecode instruction is not overridden)
;     and translated into native instruction associated with 'stv' bytecode instruction.
;   E.g. let's assume that we declared bytecode instructions '/', '*', defined constants 'pi', '180' and variables 'x', 'y'. Then macros 'fpb x pi * 180 / =y' will be equivalent to
;     the following expression in high-level language: 'y = x * pi / 180'.
;
; Macro 'fpb_end' also defines 5 extra symbolic constants:
; - fpb_recompiled_code_size - the size of native code after recompilation (including copied native code);
; - fpb_subfn_count - number of subfunctions (including the main);
; - fpb_bytecode_count - number of bytecode instructions (bytecode instruction with operand counts as one instruction);
; - fpb_nibble_count - number of bytecode nibbles (excluding copied native code);
; - fpb_copied_code_size - size of copied native code.
; So you can use these constants in your code.
;
; To define fpb constants use macros 'fpb_flt' (for floating-point values) and 'fpb_int' (for integer values).
; To define variables use macro 'fpb_var'. You can define up to 16 (or 17, see below) constants and variables. Variables MUST be placed AFTER constants!
; E.g.:
; fpb_int 10
; fpb_flt 2.0
; fpb_flt pi, 3.14159275359
; fpb_var temp
;
; The 1st parameter is alias, the 2nd one is value. If value is absent then value = alias for constants and undefined ('?') for variables, so this is equivalent to:
; fpb_int 10, 10
; fpb_flt 2.0, 2.0
; fpb_flt pi, 3.14159275359
; fpb_var temp, ?
;
; It's better to define integer constants before floating-point ones, this way you can define up to 17 (instead of 16) constants (not variables).
; I recommend to define constant and variable positions as constants ('consts = $', 'vars = $') instead of label ('consts:', 'vars:'), otherwise an error can be generated.
; IMPORTANT! If the 1st constant is a floating-point value then offset of constant array must be 2 bytes less than offset of the 1st macro 'fpb_flt' ('consts = $-2', see example above).
; Constants must be aligned by word (16 bits) relative to constant array, variables must be aligned by dword (32 bits) relative to constant array + fpb_data_offset.
; If you'll see the message about wrong alignment of variable then just add a word ('rw 1') before variable array.
; Please don't forget to add fractional part ('.') for float values (e.g. 'fpb_var x, 10' will define wrong initial value for variable 'x', you should use '10.0').
; Note that float value is stored as 16-bit value. How is it possible? It's simple. Only higher word of 32-bit single precision float is stored. This accuracy is enough for most cases.
; So the real offset of float constant is 2 byte less than offset of point where macros 'fpb_flt' is used. If you need higher accuracy then define it as variable with initialized value.
;
; When calling recompiled function, access to constants and variables depends on 'fpb_data_offset' symbolic constant value:
; - offset of constant array is calculated as 'reg + fpb_data_offset';
; - offset of variable array is calculated as 'reg + fpb_data_offset*2';
; where 'reg' is 'fnreg' resigter (see description of macro 'fpb_recompile!' below).
; So as I wrote above you must set 'reg' to constant array offset - fpb_data_offset (e.g. mov bx,consts-fpb_data_offset)
; And accordingly, you should set 'fpb_data_offset = vars-consts', where 'vars' is offset of variable array, 'consts' is offset of constant array.
; Value of 'fpb_data_offset' must be even number in range from -64 to 32 (value 0 saves 2 bytes of code size). [default is 0 (if not defined)].
; I remind you that this symbolic constant must be defined BEFORE including this file.
;
;   >>> NOTES <<<
;
; I recommend you to examine the comments before each macros below for more understanding. Especially the comments of macro 'fpb_recompile!'.
;
; See 'fpb_demo.fasm' and HerbaKaif + HK256 intros as examples and enjoy! ;)
; https://www.pouet.net/prod.php?which=88492
;
; Short statistics of HerbaKaif intro:
; - bytecode contains 244 nibbles (= 122 bytes = 169 opcodes; average length of 1 bytecode instruction ≈ 1.44 nibbles ≈ 0.72 bytes);
; - size of recompiled function = 413 bytes (compression ratio ≈ 3.39x; average length of 1 unpacked native x86 instruction ≈ 2.44 bytes);
; - size of recompiler = 74 bytes (including writting of input stream offset to register, although this is partly part of another code);
; - compression ratio including bytecode recompiler = 413/(122+74) ≈ 2.11x.
;
;
; CONTACTS
; ========
;
; Feel free to contact the author (bug and inaccuracy reports, suggestions for ideas and optimizations):
; * via Telegram: @jinxonik (preferably)
; * by e-mail: jin_x@list.ru
;
;
; ┌─── ───────── ───────────────────────── ────────── -··
; │ ┌─── ───────── ───────────────────── ────────── ────┐
; │ │ ┌─── ───────── ───────────────── ▄▄▄▄▄▄▄▄▄─ ────┐ │
; │ │ │ ┌─── ───────── ───────────── ▄▀      ▄▀ ────┐ │ │
; │ │ │ │ ┌─── ▄▄▄▄▄▄▄▄▄ ────────  ▄▀      ▄▀ ────┐ │ │ │
; │ │ │ │ │     ▀▄      ▀▄       ▄▀      ▄▀     : │ │ │ │
; │ │ │ │ │       ▀       ▀▄   ▄▀      ▄▀       │ │ │ │ │
; │ │ │ │ │  ████████████▀  ▀▄▀ ▄██▀ ▄▀   ▄███  │ │ │ │ │
; │ │ │ │ │         ▄██▀      ▄██▀ ▄▀   ▄██▀██  │ │ │ │ │
; │ │ │ │ │       ▄██▀  ▀▄  ▄██▀ ▄▀   ▄██▀  ██  │ │ │ │ │
; │ │ │ │ │     ▄██▀   ▄▀ ▄██▀    ▀ ▄██▀    ██  │ │ │ │ │
; │ │ │ │ │   ▄██▀   ▄▀ ▄██▀ ▄    ▄██▀      ██  │ │ │ │ │
; │ │ │ │ │  ▀▀▀   ▄▀  ▀▀▀ ▄▀ ▀▄ ▀▀▀  ▀▄    ▀▀  │ │ │ │ │
; │ │ │ │ :      ▄▀      ▄▀     ▀▄      ▀▄      │ │ │ │ │
; │ │ │ └───── ▄▀      ▄▀ ─────── ▀▀▀▀▀▀▀▀▀ ────┘ │ │ │ │
; │ │ └───── ▄▀      ▄▀ ─────────── ───────── ────┘ │ │ │
; │ └───── ─▀▀▀▀▀▀▀▀▀ ─────────────── ───────── ────┘ │ │
; └───── ────────── ─────────────────── ───────── ────┘ │
; ··-— ────────── ─────────────────────── ───────── ────┘
;
; ====================[ Greetings to... ]====================
; HellMood^DESiRE  TomCat^Abaddon  superogue^Marquee Design
; Řrřola  Digimind  g0blinish  frag^fsqrt  Baudsurfer^RSi
; sensenstahl  Kuemmel  iONic^Astroidea  Georgy Lomsadze
; wbcbz7^sibCrew  DArt^Fenomen  Adam Bazaroff^Excess Team
; Dolphin Soft  bfox^insiders  TmK^deMarche  Manwe^SandS
; baze^3SC  ryg^Farbrausch  provod^jetlag … all sizecoders!
; ===========================================================
;
; .------------< Welcome to sizecoding chats!!! >-----------.
; | https://discord.gg/NeCSgBTZmh • https://t.me/sizecoders |
; '---------------------------------------------------------'


;-----------------------------------------------------------------------------------------------------------------------
;-- MODULE VERSION -----------------------------------------------------------------------------------------------------
;-----------------------------------------------------------------------------------------------------------------------

; Module version
fpb_version_str equ 1.0.0
fpb_version_hex equ $10000
fpb_version_major equ 1
fpb_version_middle equ 0
fpb_version_minor equ 0

; Check module version and show error in case of incompatibility
macro	fpb_check_version	major, middle, minor
{
  @error_msg_if major <> fpb_version_major, "Incompatible MAJOR version of FPB module! This module can't be used with this code [fpb_check_version]!"
  @error_msg_if middle <> fpb_version_middle, "Incompatible MIDDLE version of FPB module! You can try to use it by commenting out call of 'fpb_check_version' macro [fpb_check_version]!"
  if minor > fpb_version_minor
    display "Minor version of FPB module is less than required. Some features may be missing in this module!",10
  else if minor < fpb_version_minor
    display "Minor version of FPB module is greater than required. This shouldn't be a problem ;)",10
  end if ; minor <> fpb_version_minor
} ; fpb_check_version


;-----------------------------------------------------------------------------------------------------------------------
;-- MACROS FOR DECLARATION OF BYTECODE INSTRUCTIONS AND DEFINITION OF CONSTANTS AND VARIABLES --------------------------
;-----------------------------------------------------------------------------------------------------------------------

; Declare bytecode instruction (alias and corresponding 'native' x86 instruction)
; 'alias' will be used as native instruction if 'native' is not specified
; 'alias' can contain special characters (see above), 'native' can contain commas.
macro	declare_fpb	alias*, native&
{
  local max, max_digit, parts
  if fpb_var_support
    max = 11
  else ; ~ fpb_var_support
    max = 13
  end if ; fpb_var_support
  max_digit = (max mod 10+'0')
  @error_msg_if @$fpb_gen_bytecode_ins >= max, "Too many FPB bytecode instructions (more than 1" <max_digit> ") when declaring '" `alias "' [declare_fpb]!"

  parts = 0
  irps a, alias \{
    parts = parts + 1
  \}
  @error_msg_if parts <> 1, "Alias '" `alias "' contains unacceptable chars [declare_fpb]!"

  @$fpb_check_alias <alias>, "declare_fpb", 0

  rept 1 n:@$fpb_gen_bytecode_ins, next:@$fpb_gen_bytecode_ins+1 \{
    @$fpb_alias\#n equ alias
    @$fpb_opcode\#n equ native
    match , native \\{
      @$fpb_opcode\#n equ alias			; if native is not specified
    \\}
    @$fpb_gen_bytecode_ins equ next
  \} ; rept
} ; declare_fpb

; Define floating-point constant (use '?' as 'value' to leave value undefined)
; Alias can start with a digit if desired
; 'alias' will be used as value if 'value' is not specified
macro	fpb_flt	alias*, value
{
  local val
  @$fpb_check_alias <alias>, "fpb_flt", 1
  @fpb_flt_#alias = $-2
  val equ value
  match , val \{
    val equ alias				; if value is not specified
  \} ; match
  df2 val
} ; fpb_flt

; Define integer constant (use '?' as 'value' to leave value undefined)
; Alias can start with a digit if desired
; 'alias' will be used as value if 'value' is not specified
macro	fpb_int		alias*, value
{
  local val
  @$fpb_check_alias <alias>, "fpb_int", 1
  @fpb_int_#alias = $
  val equ value
  match , val \{
    val equ alias				; if value is not specified
  \} ; match
  dw val
} ; fpb_int

; Define floating-point variable (use '?' or omit 'value' to leave value undefined)
; Alias can start with a digit if desired
macro	fpb_var		alias*, value
{
  local val
  @$fpb_check_alias <alias>, "fpb_var", 2
  @fpb_var_#alias = $
  val equ value
  match , val \{
    val equ ?					; if value is not specified
  \} ; match
  dd val
} ; fpb_var


;-----------------------------------------------------------------------------------------------------------------------
;-- MACROS FOR ENCODING FLOATING-POINT OPERATIONS AS BYTECODE OPCODES --------------------------------------------------
;-----------------------------------------------------------------------------------------------------------------------

; Start bytecode generator
; 'data' is offset of constant array
; offset of variable array (if variables are supported) = data + fpb_data_offset
macro	fpb_start	data*, name
{
  @error_msg_if @$fpb_started, "FPB generator is already started [fpb_start]!"

  @$fpb_started equ 1
  @$fpb_const_offset equ data
  @$fpb_nibble = 0
  @$fpb_nibble_shift = 4
  @$fpb_copy_start = -1
  @$fpb_recompiled_code_size = 0
  @$fpb_subfn_count = 1
  @$fpb_bytecode_count = 0
  @$fpb_nibble_count = 0
  @$fpb_copied_code_size = 0
  @$next_subfunction name
} ; fpb_start

; Start new subfunction
macro	fpb_subfn	name
{
  @fpb ret
  @$fpb_subfn_count = @$fpb_subfn_count + 1
  @$next_subfunction name
} ; fpb_subfn

; Stop bytecode generator
macro	fpb_end
{
  @$fpb_check_started "fpb_end"
  @$fpb_check_copy_stopped "fpb_end"

  @fpb ret
  fpb_recompiled_code_size = @$fpb_recompiled_code_size
  fpb_subfn_count = @$fpb_subfn_count
  fpb_bytecode_count = @$fpb_bytecode_count
  fpb_nibble_count = @$fpb_nibble_count
  fpb_copied_code_size = @$fpb_copied_code_size

  @$fpb_started equ 0
} ; fpb_end

; Translate list of bytecode instruction, constant and variable aliases (separated by space) into bytecode opcodes (and place it at point of macro usage)
macro	fpb	list*&
{
  local state, code

  state equ 0					; 0 - no bytecode (last time), 1 - bytecode (copy/ldc/ldic/ldv/stv), 2 - colon after bytecode, 3 - equals sign (=), x - processed (placed)
  code equ
  irps el, list \{
    match =2, state \\{				; operand after bytecode alias and colon
      @fpb code, <el>
      state equ x
    \\}

    match =3, state \\{				; operand after equals sign
      @fpb <el>, store
      state equ x
    \\}

    match =1 =:, state el \\{			; command after bytecode alias
      state equ 2
    \\}
    match =1, state \\{				; not colon after bytecode alias
      @error_msg "No colon between bytecode copy/ldc/ldic/ldv/stv and operand [fpb]!"
    \\}

    match =0 =copy, state el \\{
      state equ 1
      code equ el
    \\}
    match =0 =ldc, state el \\{
      state equ 1
      code equ el
    \\}
    match =0 =ldic, state el \\{
      state equ 1
      code equ el
    \\}
    match =0 =ldv, state el \\{
      state equ 1
      code equ el
    \\}
    match =0 =stv, state el \\{
      state equ 1
      code equ el
    \\}
    match =0 ==, state el \\{
      state equ 3
      code equ el
    \\}

    match =0, state \\{				; bytecode <> copy/ldc/ldic/ldv/stv
      @fpb <el>
    \\}

    match =x, state \\{				; bytecode is placed
      state equ 0
    \\}
  \}

  @error_msg_if ~ state eq 0, "Last bytecode has no required operand [fpb]!"
}

; Declare start of native code block (if native code copier support is enabled)
macro	fpb_copy_start
{
  @$fpb_check_copy_support "fpb_copy_start"
  @$fpb_check_started "fpb_copy_start"
  @$fpb_check_copy_stopped "fpb_copy_start", 1

  @$fpb_add_nibble @$fpb_gen_bytecode_ins	; native code copier bytecode instruction
  @$fpb_add_nibble 0				; temporary operand (native code size)
  @$fpb_copy_start = $

  if <fpb_before_ret> eq <nop>
    @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 1
  end if ; fpb_before_ret
  @$fpb_bytecode_count = @$fpb_bytecode_count + 1
} ; fpb_copy_start

; Declare end of native code block
macro	fpb_copy_end
{
  local code_size, size_nibble
  @$fpb_check_copy_support "fpb_copy_end"
  @$fpb_check_started "fpb_copy_end"
  @error_msg_if @$fpb_copy_start = -1, "FPB native code block is not started yet [fpb_copy_end]!"
  code_size = $-@$fpb_copy_start
  size_nibble = code_size-@$fpb_copy_len_inc
  @error_msg_if size_nibble < 1, "Native code size is too short (less than value 'copymin') [fpb_copy_end]!"
  @error_msg_if size_nibble > 15, "Native code size is too long (more than value 'copymin+14') [fpb_copy_end]!"

  if @$fpb_nibble_shift = 0			; last bytecode was written to higher nibble of a new byte
    @$fpb_nibble = size_nibble shl 4
    store byte size_nibble at @$fpb_last_offset	; write native code size (operand)
    @$large_get_nibble_code_neg = 1		; use large '.$get_nibble' code if bytecode offset > $8000
  else ; @$fpb_nibble_shift <> 0		; last bytecode was written to lower nibble of previously written byte
    load @$fpb_nibble byte from @$fpb_last_offset
    @$fpb_nibble = @$fpb_nibble or size_nibble
    store byte @$fpb_nibble at @$fpb_last_offset ; native code size (operand)
    @$fpb_nibble = 0
    @$large_get_nibble_code_pos = 1		; use large '.$get_nibble' code if bytecode offset < $8000
  end if ; @$fpb_nibble_shift...
  @$fpb_copy_start = -1

  @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + code_size
  @$fpb_copied_code_size = @$fpb_copied_code_size + code_size
} ; fpb_copy_end

; Define 16-bit float number (store higher word of 32-bit float number)
macro	df2	n*
{
	@bytes	2, 2, dd n
} ; df2


;-----------------------------------------------------------------------------------------------------------------------
;-- MACROS FOR BYTECODE RECOMPILATION ----------------------------------------------------------------------------------
;-----------------------------------------------------------------------------------------------------------------------

; Recompile bytecode (place recompiler in the current code position)
;
; 'bytecode' is offset of bytecode block (input stream).
;
; 'opt' is option list (via comma):
;   nosi - don't initialize register 'si' (use this if you've already initialized register 'si' before using macro 'fpb_recompile!';
;     note that its value must be set to negative offset of bytecode block in some cases!)
;   bh=0 - specifies that value of bh = 0, this saves a byte of code if 'rcaddr' is 'bx'.
;   ah=# / al=# / ax=# (where # is a particular number) - specifies current value of ah, al or ax (the best if 3rd bit of ah=0: ah and 8=0; slightly worse if al<$80),
;     this can save 1-2 bytes in some cases.
;   rcaddr=reg[=0] - register for addressing inside recompiler (must be differrent from 'subfn' reg, see below). Must be 'bx' or 'bp' [default is 'bx'].
;     If 'reg' is 'bx' then '=0' can be specified if bh=0, this saves a byte of code (the same as 'bh=0').
;   fnaddr=reg - register(s) for addressing inside recompiled functions (can match 'rcaddr' reg). Must be 'bx', 'bp', 'si', 'di', 'bx+si', 'bx+di', 'bp+si' or 'bp+di'
;     [default is 'bx'].
;   subfn=reg[=n] / subfn=n - use and initialize specified register as subfunction counter (must be 'dx', 'dl', 'dh', 'bx', 'bl', 'bh' or 'bp' and can't match 'rcaddr' even partially
;     (e.g. can't be set to 'bl' if 'rcaddr=bx' is specified [default is 'dx'])). If value is not sepcified then it is initialized with value of symbolic constant 'fpb_subfn_count'.
;     Register will be not initialized if n = 0 (use this if you've already done it). Register is not used if n = 1 (or if 'n' is not specified and fpb_subfn_count = 1).
;     You can specify both 'reg' and 'n' values or only one of them, e.g. 'subfn=dl', 'subfn=bx=3', 'subfn=5'.
;   dseg=seg - segment if bytecode block (input stream); it will be used when accessing native opcodes [default is 'ds'] (use another only if ds <> segment of bytecode block).
;   cseg=seg - segment of native opcodes and code segment [default is 'ds' if 'rcaddr' is 'bx' and 'ss' if 'rcaddr' is 'bp'] (use another only if default segment <> cs).
;   copymin=n - specify minimal size of native code that can be copied inside fpb_copy_start / fpb_copy_end block (default is 1).
;
; Input:
;   'dseg':'bytecode' = bytecode pointer (input stream), where 'bytecode' is the 1st parameter of macro, 'dseg' is value of corresponding option;
;   es:di = compiled function pointer (output stream);
;   if 'subfn' options with zero value if specified: 'subfn' reg [default is dx] = number of subfunctions to recompile;
;   cs = 'cseg'.
;
; Return:
;   al = 0 if fpb_copy_support, else al = 1;
;   cx = 0;
;   di = end of recompiled function;
;   'rcaddr' reg [default is bx] = number of defined bytecode instructions * 2 (@$fpb_gen_bytecode_ins * 2);
;   'subfn' reg [default is dx] = 0 (if number of recompiled subfunctions <> 1);
;   ah, si are undefined;
;   flag cf = 1.
macro	fpb_recompile!	bytecode*, [opt]
{
common
  @soft_error_msg_if ~ defined fpb_subfn_count, "Bytecode block is not defined [fpb_recompile!]"

  local large_code, bytecode_ofs, set_si, rcaddr_reg, bh_zero, ah_val, al_val, fnaddr_reg, subfn_reg, subfn_n, data_seg, code_seg, copy_min, error
  set_si = 1
  rcaddr_reg equ bx
  bh_zero = 0
  ah_val = $FF
  al_val = $FF
  fnaddr_reg equ bx
  subfn_reg equ dx
  subfn_n = fpb_subfn_count			; 0 - use 'subfn_reg' but doesn't initialize it
  data_seg equ ds
  code_seg equ
  copy_min = 1

forward
  if opt eq nosi
    set_si = 0

  else if opt eq bh=0
    bh_zero = 1

  else if opt eq ah | opt eq al | opt eq ax
    @error_msg "No value for option 'ah', 'al' or 'ax' is specified [fpb_recompile!]!"

  else if opt eq rcaddr
    @error_msg "No value for option 'rcaddr' is specified [fpb_recompile!]!"

  else if opt eq fnaddr
    @error_msg "No value for option 'fnaddr' is specified [fpb_recompile!]!"

  else if opt eq subfn
    @error_msg "No value for option 'subfn' is specified [fpb_recompile!]!"

  else if opt eq dseg
    @error_msg "No value for option 'dseg' is specified [fpb_recompile!]!"

  else if opt eq cseg
    @error_msg "No value for option 'cseg' is specified [fpb_recompile!]!"

  else if opt eq copymin
    @error_msg "No value for option 'copymin' is specified [fpb_recompile!]!"

  else if ~ opt eq
    error equ 1
    match =ah==val, opt \{
      ah_val equ (val) and $FF
      error equ 0
    \} ; ah

    match =al==val, opt \{
      al_val equ (val) and $FF
      error equ 0
    \} ; al

    match =ax==val, opt \{
      ah_val equ (val) shr 8 and $FF
      al_val equ (val) and $FF
      error equ 0
    \} ; ax

    match =rcaddr==reg_n, opt \{
      match reg==n, reg_n \\{
        rcaddr_reg equ reg
        match =bx, reg \\\{
          bh_zero = 1
        \\\}
        @error_msg_if n <> 0, "Only value '0' is allowed for register in option 'rcaddr' [fpb_recompile!]!"
        error equ 0
      \\}
      match =1, error \\{
        rcaddr_reg equ reg_n
      \\}
      @error_msg_if <~ rcaddr_reg in <bx,bp>>, "Only registers 'bx' and 'bp' for option 'rcaddr' are allowed [fpb_recompile!]!"
      error equ 0
    \} ; rcaddr

    match =fnaddr==val, opt \{
      fnaddr_reg equ val
      @error_msg_if <~ fnaddr_reg in <bx,bp,si,di,bx+si,bx+di,bp+si,bp+di>>, "Only values 'bx', 'bp', 'si', 'di', 'bx+si', 'bx+di', 'bp+si' and 'bp+di' of option 'fnaddr' are allowed [fpb_recompile!]!"
      error equ 0
    \} ; fnaddr

    match =subfn==reg_n, opt \{
      match reg==n, reg_n \\{
        subfn_reg equ reg
        subfn_n = n
        @error_msg_if subfn_n < 0, "Value of option 'subfn' must be positive or zero [fpb_recompile!]!"
        error equ 0
      \\}
      match =1, error \\{
        subfn_n = fpb_subfn_count
        match =dx, reg_n \\\{
          subfn_reg equ reg_n
          error equ 0
        \\\}
        match =dl, reg_n \\\{
          subfn_reg equ reg_n
          error equ 0
        \\\}
        match =dh, reg_n \\\{
          subfn_reg equ reg_n
          error equ 0
        \\\}
        match =bx, reg_n \\\{
          subfn_reg equ reg_n
          error equ 0
        \\\}
        match =bl, reg_n \\\{
          subfn_reg equ reg_n
          error equ 0
        \\\}
        match =bh, reg_n \\\{
          subfn_reg equ reg_n
          error equ 0
        \\\}
        match =bp, reg_n \\\{
          subfn_reg equ reg_n
          error equ 0
        \\\}
        match =1, error \\\{
          subfn_n = reg_n
          @error_msg_if subfn_n < 0, "Value of option 'subfn' must be positive or zero [fpb_recompile!]!"
        \\\}
      \\} ; match =1, error
      @error_msg_if <~ subfn_reg in <dx,dl,dh,bx,bl,bh,bp>>, "Only registers 'dx', 'dl', 'dh', 'bx', 'bl', 'bh' and 'bp' for option 'subfn' are allowed [fpb_recompile!]!"
      error equ 0
    \} ; subfn

    match =dseg==val, opt \{
      data_seg equ val
      @error_msg_if <~ data_seg in <cs,ds,es,ss,fs,gs>>, "Only values 'cs', 'ds', 'es', 'ss', 'fs' and 'gs' of option 'dseg' are allowed [fpb_recompile!]!"
      error equ 0
    \} ; dseg

    match =cseg==val, opt \{
      code_seg equ val
      @error_msg_if <~ code_seg in <cs,ds,es,ss,fs,gs>>, "Only values 'cs', 'ds', 'es', 'ss', 'fs' and 'gs' of option 'cseg' are allowed [fpb_recompile!]!"
      error equ 0
    \} ; cseg

    match =copymin==n, opt \{
      @$fpb_check_copy_support "fpb_recompile!"
      @error_msg_if n <= 0, "Value of option 'copymin' must be positive [fpb_recompile!]!"
      copy_min = n
      error equ 0
    \} ; copymin

    @error_msg_if error, "Wrong option '" opt "' is specified [fpb_recompile!]!"
  end if ; opt

common
  @$fpb_copy_len_inc = (copy_min)-1

  @error_msg_if <rcaddr_reg eq subfn_reg | (rcaddr_reg in <bl,bh> & subfn_reg eq bx)>, "Register in 'rcaddr' option can't match register in 'subfn' option (even partially) [fpb_recompile!]!"
  match , code_seg \{
    match =bx, rcaddr_reg \\{ code_seg equ ds \\}
    match =bp, rcaddr_reg \\{ code_seg equ ss \\}
  \}
  
  bytecode_ofs = bytecode and $FFFF
  if (fpb_copy_support & defined @$large_get_nibble_code_pos & @$large_get_nibble_code_pos & bytecode_ofs < $8000) | \
     (fpb_copy_support & defined @$large_get_nibble_code_neg & @$large_get_nibble_code_neg & bytecode_ofs > $8000) | \
     bytecode_ofs = 0 | bytecode_ofs = $8000
    large_code = 1
  else
    large_code = 0
  end if ; (fpb_copy_support...)

  if set_si
    if large_code
      if bytecode_ofs = 0
  		xor	si,si
      else ; bytecode_ofs <> 0
  		mov	si,bytecode_ofs		; bytecode data offset
      end if ; bytecode_ofs...
    else ; ~ large_code
  		mov	si,-bytecode_ofs	; negative bytecode data offset (for not large '.$get_nibble' code)
    end if ; large_code
  end if ; set_si

  if subfn_n > 1
		mov	subfn_reg,subfn_n	; number of subfunctions
  end if ; subfn_n > 1

  if large_code
    if ah_val and 8 <> 0
      if al_val < $80
		cbw				; ah = 0 (reset nibble flag, see below)
      else ; al_val >= $80
		xor	ax,ax
      end if
    end if ; ah_val and 8 <> 0
  end if ; large_code

	.$next_bytecode:
		xor	cx,cx			; cx = 0 for new bytecode, cx <> 0 for operand X

		; Read nibble (higher at first, lower next time) from input stream to al (or from ah every 2nd time)
		; Value of ah must be saved in code below!
	.$get_nibble:
  if large_code
		shr	ax,12			; al = nibble, cf = 1 if next nibble was already in ah (nnnn|1000)
		jc	short @F
		mov	ah,$80
    if ~ data_seg eq ds
		data_seg
    end if ; ~ data_seg eq ds
		lodsb				; ax = $80xx
		ror	ax,4			; al = 1st nibble (0000|nnnn), ah = 2nd nibble and flag (nnnn|1000)
	@@:
  else ; ~ large_code... (this code is 2 bytes shorter but can't be used in some cases)
		neg	si			; sf = 0 if next nibble is already in ah
    if bytecode_ofs < $8000
		js	short @F
    else ; bytecode_ofs > $8000
		jns	short @F
    end if ; bytecode_ofs
    if ~ data_seg eq ds
		data_seg
    end if ; ~ data_seg eq ds
		lodsb				; load next byte
		aam	16			; ah = 1st nibble, al = 2nd nibble
	@@:	xchg	al,ah			; al = result
  end if ; fpb_copy_support...

		; Check for step 2 and process operand
		jcxz	short .$bytecode	; new bytecode is loaded (not operand)

  if fpb_copy_support
		shr	cl,1			; 0..2 (0,1,1,2,2 for bytecode opcodes from @$fpb_gen_bytecode_ins to @$fpb_gen_bytecode_ins+4)
		jz	short .$copy_native	; cl = 0 (@$fpb_gen_bytecode_ins)
  end if ; fpb_copy_support

  if (fpb_data_offset)/2 = 1
		inc	ax
  else if (fpb_data_offset)/2 = -1
		dec	ax
  else if (fpb_data_offset)/2
		add	al,(fpb_data_offset)/2	; adjust data offset
  end if ; (fpb_data_offset)...
		shl	al,cl			; scale X (2x or 4x)
		stosb				; output scaled operand X
		jmp	short .$next_bytecode

		; Opcode table
		@fpb_opcodes fnaddr_reg

	.$bytecode:
		; Find and copy x86 opcode
  if bh_zero & rcaddr_reg eq bx
		mov	bl,al
  else ; bh_zero...
		movzx	rcaddr_reg,al
  end if ; bh_zero...
		shl	rcaddr_reg,1		; opcode offset

		push	ax
		mov	ax,[code_seg:rcaddr_reg+.$fpb_opcodes]
		stosw				; output opcode
		pop	ax

		; Load/store/copy bytecode instructions
		sub	al,@$fpb_gen_bytecode_ins-1
		jbe	short .$next_bytecode	; no operand required

		mov	cl,al			; 1..5 (bytecode instructions from @$fpb_gen_bytecode_ins to @$fpb_gen_bytecode_ins+4)
  if fpb_copy_support
		jmp	short .$get_nibble

		; Copy stream
	.$copy_native:
		mov	cl,al
		jcxz	short .$done		; end of data (X = 0)
    if @$fpb_copy_len_inc = 1 | @$fpb_copy_len_inc = 2
      times (copy_min-1) inc cx
    else if @$fpb_copy_len_inc >= 3 & @$fpb_copy_len_inc <= 240
		add	cl,@$fpb_copy_len_inc
    else if copy_min > 240
		add	cx,@$fpb_copy_len_inc
    end if ; copy_min...
		dec	di
    if ~ <fpb_before_ret> eq <nop>
		dec	di
    end if ; ...fpb_before_ret...
		rep
    if ~ data_seg eq ds
		data_seg
    end if ; ~ data_seg eq ds
		movsb				; copy native code from input to output stream
		jmp	short .$next_bytecode
  else  ; ~ fpb_copy_support
		shr	cl,1			; 0..2 (0,1,1,2,2 for bytecode instructions from @$fpb_gen_bytecode_ins+1 to @$fpb_gen_bytecode_ins+4)
		jnz	short .$get_nibble	; cl = 1..2, else exit (if cl = 0, @$fpb_gen_bytecode_ins)
  end if ; fpb_copy_support

	.$done:
  if subfn_n <> 1
		dec	subfn_reg		; decrease subfunction counter
		jnz	.$next_bytecode		; next subfunction
  end if ; subfn_n <> 1
} ; fpb_recompile!


;-----------------------------------------------------------------------------------------------------------------------
;-- DEFINITIONS OF DEFAULT VALUES --------------------------------------------------------------------------------------
;-----------------------------------------------------------------------------------------------------------------------

; Default byte before 'ret' native instruction in macro '@fpb_opcodes' (first byte of 'ret' bytecode instruction)
match	=fpb_before_ret, fpb_before_ret {
  fpb_before_ret equ nop
}

; Default opcode for 'ldc' bytecode instruction
match	=fpb_opcode_ldc, fpb_opcode_ldc {
  define fpb_opcode_ldc fld dword [reg+1]
}

; Default opcode for 'ldic' bytecode instruction
match	=fpb_opcode_ldic, fpb_opcode_ldic {
  define fpb_opcode_ldic fild word [reg+1]
}

; Default opcode for 'ldv' bytecode instruction
match	=fpb_opcode_ldv, fpb_opcode_ldv {
  define fpb_opcode_ldv fld dword [reg+1]
}

; Default opcode for 'stv' bytecode instruction
match	=fpb_opcode_stv, fpb_opcode_stv {
  define fpb_opcode_stv fstp dword [reg+1]
}

; Default value of data offset relative to 'fnreg'
if ~ definite fpb_data_offset
  fpb_data_offset = 0
end if ; ~ definite fpb_data_offset

; Default value of native code copier support definition
if ~ definite fpb_copy_support
  fpb_copy_support = 0
end if ; ~ definite fpb_copy_support

; Default value of variable support definition
if ~ definite fpb_var_support
  fpb_var_support = 1
end if ; ~ definite fpb_var_support


;-----------------------------------------------------------------------------------------------------------------------
;-- MACROS AND DEFINITIONS FOR INTERNAL USE ----------------------------------------------------------------------------
;-----------------------------------------------------------------------------------------------------------------------

; Define and check important constants
@$fpb_started equ 0				; fpb is not started
@$fpb_gen_bytecode_ins equ 0			; number of general bytecode instructions
@$fpb_next_subfn equ 0				; number of next subfunction

; Place one fpb opcode
; 'alias' is bytecode instruction, constant or variable alias
; 'op' is operand for 'ret', 'copy', 'ldc', 'ldic', 'ldv', 'stv' bytecode instruction or command for variable (empty to load, 'store' to store)
macro	@fpb	alias*, op&
{
  @$fpb_check_started "fpb/@fpb"
  @$fpb_check_copy_stopped "fpb/@fpb"

  local ofs
  if <alias> eq <@$fpb_alias0>
    @$fpb_add_nibble 0
    @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 2
    @$fpb_bytecode_count = @$fpb_bytecode_count + 1
  else if <alias> eq <@$fpb_alias1>
    @$fpb_add_nibble 1
    @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 2
    @$fpb_bytecode_count = @$fpb_bytecode_count + 1
  else if <alias> eq <@$fpb_alias2>
    @$fpb_add_nibble 2
    @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 2
    @$fpb_bytecode_count = @$fpb_bytecode_count + 1
  else if <alias> eq <@$fpb_alias3>
    @$fpb_add_nibble 3
    @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 2
    @$fpb_bytecode_count = @$fpb_bytecode_count + 1
  else if <alias> eq <@$fpb_alias4>
    @$fpb_add_nibble 4
    @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 2
    @$fpb_bytecode_count = @$fpb_bytecode_count + 1
  else if <alias> eq <@$fpb_alias5>
    @$fpb_add_nibble 5
    @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 2
    @$fpb_bytecode_count = @$fpb_bytecode_count + 1
  else if <alias> eq <@$fpb_alias6>
    @$fpb_add_nibble 6
    @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 2
    @$fpb_bytecode_count = @$fpb_bytecode_count + 1
  else if <alias> eq <@$fpb_alias7>
    @$fpb_add_nibble 7
    @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 2
    @$fpb_bytecode_count = @$fpb_bytecode_count + 1
  else if <alias> eq <@$fpb_alias8>
    @$fpb_add_nibble 8
    @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 2
    @$fpb_bytecode_count = @$fpb_bytecode_count + 1
  else if <alias> eq <@$fpb_alias9>
    @$fpb_add_nibble 9
    @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 2
    @$fpb_bytecode_count = @$fpb_bytecode_count + 1
  else if <alias> eq <@$fpb_alias10>
    @$fpb_add_nibble 10
    @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 2
    @$fpb_bytecode_count = @$fpb_bytecode_count + 1
  else if <alias> eq <@$fpb_alias11>
    @$fpb_add_nibble 11
    @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 2
    @$fpb_bytecode_count = @$fpb_bytecode_count + 1
  else if <alias> eq <@$fpb_alias12>
    @$fpb_add_nibble 12
    @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 2
    @$fpb_bytecode_count = @$fpb_bytecode_count + 1
  else if defined @fpb_flt_#alias		; ldc (load float const)
    ofs = @fpb_flt_#alias - @$fpb_const_offset
    @soft_error_msg_if ofs and 1 <> 0, "Floating-point constant '" `alias "' is not aligned [fbp/@fpb]!"
    ofs = ofs / 2
    @soft_error_msg_if ofs < 0 | ofs > 15, "Floating-point constant '" `alias "' is out of reach [fbp/@fpb]!"
    @error_msg_if op eq store, "Value can't be stored in floating-point constant '" `alias "' [fbp/@fpb]!"
    @error_msg_if ~ op eq, "Operand (command) are not allowed for floating-point constant '" `alias "' [fbp/@fpb]!"
    @$fpb_add_nibble @$fpb_gen_bytecode_ins+1
    @$fpb_add_nibble ofs
    @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 3
    @$fpb_bytecode_count = @$fpb_bytecode_count + 1
  else if defined @fpb_int_#alias		; int (load int const)
    ofs = @fpb_int_#alias - @$fpb_const_offset
    @soft_error_msg_if ofs and 1 <> 0, "Integer constant '" `alias "' is not aligned [fbp/@fpb]!"
    ofs = ofs / 2
    @soft_error_msg_if ofs < 0 | ofs > 15, "Integer constant '" `alias "' is out of reach [fbp/@fpb]!"
    @error_msg_if op eq store, "Value can't be stored in integer constant '" `alias "' [fbp/@fpb]!"
    @error_msg_if ~ op eq, "Operand (command) are not allowed for integer constant '" `alias "' [fbp/@fpb]!"
    @$fpb_add_nibble @$fpb_gen_bytecode_ins+2
    @$fpb_add_nibble ofs
    @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 3
    @$fpb_bytecode_count = @$fpb_bytecode_count + 1
  else if defined @fpb_var_#alias		; ldv, stv (load/store var)
    @$fpb_check_var_support "fpb/@fpb"
    ofs = @fpb_var_#alias - (@$fpb_const_offset + fpb_data_offset)
    @soft_error_msg_if ofs and 3 <> 0, "Variable '" `alias "' is not aligned [fbp/@fpb]!"
    ofs = ofs / 4
    @soft_error_msg_if ofs < 0 | ofs > 15, "Variable '" `alias "' is out of reach [fbp/@fpb]!"
    if op in <, load>
      @$fpb_add_nibble @$fpb_gen_bytecode_ins+3	; ldv (load var)
    else if op eq store
      @$fpb_add_nibble @$fpb_gen_bytecode_ins+4	; stv (store var)
    else
      @error_msg "Wrong command '" `op "' for variable '" `alias "' [fpb/@fpb]!"
    end if ; <op> eq...
    @$fpb_add_nibble ofs
    @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 3
    @$fpb_bytecode_count = @$fpb_bytecode_count + 1
  else if <alias> eq <ret>			; end of subfunction
    @$fpb_add_nibble @$fpb_gen_bytecode_ins
    if fpb_copy_support
      @$fpb_add_nibble 0
    end if ; fpb_copy_support
    @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 2
    @$fpb_bytecode_count = @$fpb_bytecode_count + 1
  else if <alias> eq <copy>			; copy native code
    @$fpb_check_copy_support "fpb/@fpb"
    @$fpb_add_nibble @$fpb_gen_bytecode_ins
    @$fpb_add_nibble op
    if (op) = 0
      @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 2
    else ; (op) <> 0
      if <fpb_before_ret> eq <nop>
        @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 1
      end if ; fpb_before_ret
      @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + (op) + @$fpb_copy_len_inc
    end if ; op
    @$fpb_bytecode_count = @$fpb_bytecode_count + 1
    @$fpb_copied_code_size = @$fpb_copied_code_size + (op) + @$fpb_copy_len_inc
  else if <alias> eq <ldc>			; load float const
    @$fpb_add_nibble @$fpb_gen_bytecode_ins+1
    @$fpb_add_nibble op
    @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 3
    @$fpb_bytecode_count = @$fpb_bytecode_count + 1
  else if <alias> eq <ldic>			; load int const
    @$fpb_add_nibble @$fpb_gen_bytecode_ins+2
    @$fpb_add_nibble op
    @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 3
    @$fpb_bytecode_count = @$fpb_bytecode_count + 1
  else if <alias> eq <ldv>			; load var
    @$fpb_check_var_support "fpb/@fpb"
    @$fpb_add_nibble @$fpb_gen_bytecode_ins+3
    @$fpb_add_nibble op
    @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 3
    @$fpb_bytecode_count = @$fpb_bytecode_count + 1
  else if <alias> eq <stv>			; store var
    @$fpb_check_var_support "fpb/@fpb"
    @$fpb_add_nibble @$fpb_gen_bytecode_ins+4
    @$fpb_add_nibble op
    @$fpb_recompiled_code_size = @$fpb_recompiled_code_size + 3
    @$fpb_bytecode_count = @$fpb_bytecode_count + 1
  else
    @soft_error_msg "Bytecode, constant or variable with alias '" `alias "' is not defined yet [fpb/@fpb]!"
  end if ; alias...
} ; @fpb

; Place a nibble (doesn't increase value of @$fpb_recompiled_code_size and @$fpb_bytecode_count but increases @$fpb_nibble_count!!!)
macro	@$fpb_add_nibble n
{
  @$fpb_check_started "fpb/@fpb/@$fpb_add_nibble"
  @error_msg_if n eq, "Value is absent [fpb/@fpb/@$fpb_add_nibble]!"
  @error_msg_if (n) < 0 | (n) > 15, "Specified value is out of range 0..15 [fpb/@fpb/@$fpb_add_nibble]!"

  @$fpb_nibble = @$fpb_nibble or ((n) shl @$fpb_nibble_shift)
  if @$fpb_nibble_shift <> 0			; 1st nibble of a byte
    @$fpb_last_offset = $
    db @$fpb_nibble
  else ; @$fpb_nibble_shift = 0			; 2nd nibble of a byte
    store byte @$fpb_nibble at @$fpb_last_offset
    @$fpb_nibble = 0
  end if ; @$fpb_nibble_shift...
  @$fpb_nibble_shift = @$fpb_nibble_shift xor 4	; next nibble

  @$fpb_nibble_count = @$fpb_nibble_count + 1
} ; @$fpb_add_nibble

; Check if fpb generator is started
; 'name' is a name of calling macros
macro	@$fpb_check_started name*
{
  @error_msg_if ~ @$fpb_started, "FPB generator is not started yet [" name "]!"
} ; @$fpb_check_started

; Check if native code copier is supported
; 'name' is a name of calling macros
macro	@$fpb_check_copy_support name*
{
  @error_msg_if ~ (fpb_copy_support), "Support of FPB native code copier is disabled [" name "]!"
} ; @$fpb_check_copy_support

; Check if variables are supported
; 'name' is a name of calling macros
macro	@$fpb_check_var_support name*
{
  @error_msg_if ~ (fpb_var_support), "Support of FPB variables is disabled [" name "]!"
} ; @$fpb_check_var_support

; Check if native code block is not started yet
; 'name' is a name of calling macros
; 'already' = 1 to show 'already started' message instead of 'is not stopped'
macro	@$fpb_check_copy_stopped name*, already=0
{
  @error_msg_if @$fpb_copy_start <> -1 & already, "FPB native code block is already started [" name "]!"
  @error_msg_if @$fpb_copy_start <> -1 & ~ already, "FPB native code block is not stopped yet [" name "]!"
} ; @$fpb_check_copy_stopped

; Check 'alias' declaration
; 'name' is a name of calling macros
; 'type' = 0 for bytecode instruction alias, 1 for constant alias, 2 for variable alias
macro	@$fpb_check_alias alias*, name*, type*
{
  if <alias> in <@$fpb_alias0, @$fpb_alias1, @$fpb_alias2, @$fpb_alias3, @$fpb_alias4, @$fpb_alias5, @$fpb_alias6, \
                 @$fpb_alias7, @$fpb_alias8, @$fpb_alias9, @$fpb_alias10, @$fpb_alias11, @$fpb_alias12>
    if type = 0
      @display_error_once "Bytecode instruction alias '" `alias "' is already declared [" name "]!"
    else if type = 1
      @display_error_once "Constant alias ('" `alias "') must not be the same as bytecode alias [" name "]!"
    else if type = 2
      @display_error_once "Variable alias ('" `alias "') must not be the same as bytecode alias [" name "]!"
    end if ; type...
  end if ; alias...
} ; @$fpb_check_alias

; Declare offset of the next subfunction
macro	@$next_subfunction	name
{
  rept 1 n:@$fpb_next_subfn, next:@$fpb_next_subfn+1 \{
    fpb_offset_subfn\#n = @$fpb_recompiled_code_size
    if ~ name eq
      fpb_offset_subfn_\#name = @$fpb_recompiled_code_size
    end if
    @$fpb_next_subfn equ next
  \}
} ; @$next_function

; Place 'count' bytes of 'instr' from 'start' position (0-based)
macro	@bytes	start*, count*, instr*&
{
  local x, .code
  virtual at 0
	.code::
		instr
  end virtual
  repeat count
    load x byte from .code:%+start-1
		db	x
  end repeat
} ; @bytes

; Show error message only for the 1st time
macro	@display_error_once text*&
{
  if ~ definite @$compile_error | @$compile_error = 0
    local strlen, value
    virtual at 0
      value equ 0
      irps s,text \{
        match =>, s \\{ value equ x \\}
        match =1, value \\{ db s \\}
        match =<, s \\{ value equ 1 \\}
        match =0, value \\{ db \\`s \\}
        match =x, value \\{ value equ 0 \\}
      \}
      strlen = $
    end virtual

    repeat strlen
      display '-'
    end repeat
      display 10
      value equ 0
      irps s,text \{
        match =>, s \\{ value equ x \\}
        match =1, value \\{ display s \\}
        match =<, s \\{ value equ 1 \\}
        match =0, value \\{ display \\`s \\}
        match =x, value \\{ value equ 0 \\}
      \}
      display 10
    repeat strlen
      display '-'
    end repeat
    @$compile_error = 1
  end if
;  \} ; match
} ; @display_error_once

; Show error message with hard stop
macro	@error_msg	text*&
{
  @display_error_once text
  err
} ; @error_msg

; Conditional error message with hard stop
macro	@error_msg_if	cond*, text*&
{
  if cond
    @error_msg text
  end if ; cond
} ; @error_msg_if

; Show error message with soft stop
macro	@soft_error_msg	text*&
{
  @display_error_once text
  assert 0
} ; @soft_error_msg

; Conditional error message with soft stop
macro	@soft_error_msg_if cond*, text*&
{
  if cond
    @soft_error_msg text
  end if ; cond
} ; @soft_error_msg_if

; Check value of 'fpb_data_offset'
@soft_error_msg_if (fpb_data_offset) and 1 | (fpb_data_offset) < -64 | (fpb_data_offset) > 32, "Value of 'fpb_data_offset' must be even and in range -64..32!"

; Opcode table
macro	@fpb_opcodes	fnaddr_reg
{
  @error_msg_if ~ @$fpb_gen_bytecode_ins, "FPB bytecode instructions are not declared!"

  local opcode
  macro @$replace val
  \{
    local is_set
    opcode equ
    match val, val \\{
      irps v, val \\\{
        is_set equ 0
        match =reg, v \\\\{
          opcode equ opcode fnaddr_reg
          is_set equ 1
        \\\\}
        match =0, is_set \\\\{
          opcode equ opcode v
        \\\\}
      \\\}
    \\}
  \}

	.$fpb_opcodes:
	@@:	@$fpb_opcode0
    @error_msg_if $-@B <> 2, "Wrong x86 opcode 0 size!"
  if @$fpb_gen_bytecode_ins > 1
	@@:	@$fpb_opcode1
    @error_msg_if $-@B <> 2, "Wrong x86 opcode 1 size!"
  end if ; @$fpb_gen_bytecode_ins...
  if @$fpb_gen_bytecode_ins > 2
	@@:	@$fpb_opcode2
    @error_msg_if $-@B <> 2, "Wrong x86 opcode 2 size!"
  end if ; @$fpb_gen_bytecode_ins...
  if @$fpb_gen_bytecode_ins > 3
	@@:	@$fpb_opcode3
    @error_msg_if $-@B <> 2, "Wrong x86 opcode 3 size!"
  end if ; @$fpb_gen_bytecode_ins...
  if @$fpb_gen_bytecode_ins > 4
	@@:	@$fpb_opcode4
    @error_msg_if $-@B <> 2, "Wrong x86 opcode 4 size!"
  end if ; @$fpb_gen_bytecode_ins...
  if @$fpb_gen_bytecode_ins > 5
	@@:	@$fpb_opcode5
    @error_msg_if $-@B <> 2, "Wrong x86 opcode 5 size!"
  end if ; @$fpb_gen_bytecode_ins..
  if @$fpb_gen_bytecode_ins > 6
	@@:	@$fpb_opcode6
    @error_msg_if $-@B <> 2, "Wrong x86 opcode 6 size!"
  end if ; @$fpb_gen_bytecode_ins...
  if @$fpb_gen_bytecode_ins > 7
	@@:	@$fpb_opcode7
    @error_msg_if $-@B <> 2, "Wrong x86 opcode 7 size!"
  end if ; @$fpb_gen_bytecode_ins...
  if @$fpb_gen_bytecode_ins > 8
	@@:	@$fpb_opcode8
    @error_msg_if $-@B <> 2, "Wrong x86 opcode 8 size!"
  end if ; @$fpb_gen_bytecode_ins...
  if @$fpb_gen_bytecode_ins > 9
	@@:	@$fpb_opcode9
    @error_msg_if $-@B <> 2, "Wrong x86 opcode 9 size!"
  end if ; @$fpb_gen_bytecode_ins...
  if @$fpb_gen_bytecode_ins > 10
	@@:	@$fpb_opcode10
    @error_msg_if $-@B <> 2, "Wrong x86 opcode 10 size!"
  end if ; @$fpb_gen_bytecode_ins...
  if @$fpb_gen_bytecode_ins > 11
	@@:	@$fpb_opcode11
    @error_msg_if $-@B <> 2, "Wrong x86 opcode 11 size!"
  end if ; @$fpb_gen_bytecode_ins...
  if @$fpb_gen_bytecode_ins > 12
	@@:	@$fpb_opcode12
    @error_msg_if $-@B <> 2, "Wrong x86 opcode 12 size!"
  end if ; @$fpb_gen_bytecode_ins...

	@@:	@$replace fpb_before_ret
		opcode				; @$fpb_gen_bytecode_ins - end of data marker if fpb_copy_support = 0; otherwise...
    @error_msg_if $-@B <> 1, "Wrong size of 'fpb_before_ret' opcode!"
		ret				; ...copy X bytes from input to output stream (insert native code; X = next nibble), end of data marker if X = 0
		@$replace fpb_opcode_ldc
		@bytes	0, 2, opcode		; @$fpb_gen_bytecode_ins+1 - load FLOAT dword CONST number X (X = next nibble) [reg + fpb_data_offset + X*2]
		@$replace fpb_opcode_ldic
		@bytes	0, 2, opcode		; @$fpb_gen_bytecode_ins+2 - load INT word CONST number X (X = next nibble) [reg + fpb_data_offset + X*2]
  if fpb_var_support
		@$replace fpb_opcode_ldv
		@bytes	0, 2, opcode		; @$fpb_gen_bytecode_ins+3 - load FLOAT dword VAR number X (X = next nibble) [reg + fpb_data_offset*2 + X*4]
		@$replace fpb_opcode_stv
		@bytes	0, 2, opcode		; @$fpb_gen_bytecode_ins+4 - store FLOAT dword VAR number X (X = next nibble) [reg + fpb_data_offset*2 + X*4]
  end if ; fpb_var_support

  purge @$replace
} ; @fpb_opcodes
