; VGM player for the NES
; /Mic, 2011
;
; Supports SN76489 (VGM command 0x50) and 2A03 (VGM command 0xB4).
;
; Assemble with NESASM


 .inesprg    16			; Sixteen 16k PRG-ROM banks
 .ineschr    1			; CHR-RAM
 .inesmir    1			; Vertical mirroring
 .inesmap    24			; Mapper 24 (VRC6)


; Sample period (Ps) should be:
;   38 cycles average for PAL  (~43753 Hz)
;   41 cycles average for NTSC (~43653 Hz)
;

; 16K chunks of VGM data are switched in at $8000-$BFFF
; PAL/NTSC-specific code and data at $C000-$DFFF
; Common code and data at $E000-$FFFF


; VGM commands
VGM_BANK_SWITCH	= $4B	; (added by me. not in the VGM spec)
VGM_GG_STEREO	= $4F
VGM_PSG_WRITE	= $50
VGM_LONG_DELAY	= $61
VGM_NTSC_DELAY	= $62
VGM_PAL_DELAY	= $63
VGM_SONG_END	= $66
VGM_DATA_BLOCK	= $67
VGM_SHORT_DELAY	= $70
VGM_2A03_WRITE	= $B4	; (added in VGM spec 1.60)


; Zeropage variables
CYCCTR		= $08
VGM_PTR 	= $10
DELAY 		= $12
TONE0_LATCH = $14
TONE1_LATCH	= $16
TONE2_LATCH	= $18
NOISE_LATCH	= $1A
VOL0_LATCH  = $1C
VOL1_LATCH	= $1E
VOL2_LATCH	= $20
VOL3_LATCH	= $22
PERH0		= $24
PERH1		= $26
PERH2		= $28
PERH3		= $2A
TEMP		= $2C
TEMP2		= $2E
SONG_BANK	= $30
LATCHED_REG = $32
FREQ_PTR	= $34
DUMMY		= $38

ISPAL		= $200

play = $C000


 .bank 0
 .org  $8000
 
 .include	"io.inc"
 .include	"macros.inc"

; 11.015625 cycles average
decw .macro
	lda <\1   ; 3
	bne	.x\@  ; 3|2
	dec	<\1+1 ; 5
	.x\@:     
    dec <\1   ; 5
 .endm


; 5.015625 cycles average
incptr .macro
	iny       ; 2
	bne	.x\@  ; 3|2
	inc	<\1+1 ; 5
	.x\@:
 .endm


;-----------------------------
;         Playback code
;-----------------------------

 .bank 2
 .org $C000

; NTSC Playback loop	
play_NTSC:
	lda		[VGM_PTR],y		; 5
	incptr	VGM_PTR			; 5
	cmp		#VGM_SHORT_DELAY ; 2
	bcc		.not_short_wait	; 3|2

;------------------------------------
; Delay for n/44100 s  (n = [1..16])
; 14.015625 cycles will have been
; spent when we enter this routine.
;

	cmp		#VGM_2A03_WRITE	; 2
	bne		.short_wait		; 3|2
	lda		[VGM_PTR],y		; 5
	incptr	VGM_PTR			; 5
	tax						; 2
	lda		[VGM_PTR],y		; 5
	incptr	VGM_PTR			; 5
	sta		$4000,x			; 4
	jmp		play			; 3
.short_wait:

;	clc						; 2
	and		#$0F			; 2
	adc		<DELAY			; 3
	sta		<DELAY			; 3
	bne		.low_byte_nz	; 3|2
; 26 cycles spent
	lda		<DELAY+1		; 3
	adc		#0				; 2
	sta		<DELAY+1		; 3
 	;sta		<DUMMY			; 3
	jmp		play			; 3
.low_byte_nz:
; 27 cycles spent
	lda		<DELAY+1		; 3
	adc		#0				; 2
	sta		<DELAY+1		; 3
 	sta		<DUMMY			; 3
	bmi		play			; 3|2
; 39 cycles spent
.short_wait_loop:
	inc		<DUMMY			; 5
 	lda		<DUMMY			; 3
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	dec		<DELAY			; 5
	bne		.short_wait_loop	; 3|2.  41 cycles/loop
	beq		play			; 3
.not_short_wait:	

; 15 cycles spent	
	cmp		#VGM_PSG_WRITE	; 2
	bne		.check_63
	jmp		psg_param		; 3|2
.check_63:
; 19
	cmp		#VGM_PAL_DELAY	; 2. wait one pal frame (1/50 s)
	bne		.check_62		; 3|2
	jmp		wait_frame_pal	; 3

.check_62:
; 24
	cmp		#VGM_NTSC_DELAY	; 2. wait one ntsc frame (1/60 s)
	bne		.check_61		; 3|2	
	jmp		wait_frame_ntsc	; 3

.check_61:
; 29
	cmp		#VGM_LONG_DELAY	; wait xxyy samples
	bne		.check_66
	jmp		long_wait

.check_66:	
; 34
	cmp		#VGM_SONG_END	; loop
	bne		.check_4B
	jmp		loop_song

.check_4B:
; 39
	cmp		#VGM_BANK_SWITCH ; bank switch
	bne		.check_4F
	jmp		bank_switch
	
.check_4F
	cmp		#VGM_GG_STEREO	; set gamegear stereo parameter
	bne		.check_67
	jmp		gg_stereo_param

.check_67:
	cmp		#VGM_DATA_BLOCK
; TODO: something

	; All other commands are unhandled and assumed to be 3 bytes long
	incptr	VGM_PTR         ; 5
	incptr	VGM_PTR         ; 5
	jmp		play	       ; 3
	

 .org $C400
; Wait nnnn samples (nnnn = 0..65535)
long_wait:
	lda		[VGM_PTR],y		; 5
	incptr	VGM_PTR			; 5
	clc						; 2
	adc		<DELAY			; 3
	sta		<DELAY			; 3
	lda		[VGM_PTR],y		; 5
	incptr	VGM_PTR			; 5
	adc		<DELAY+1		; 3
	sta		<DELAY+1		; 3
	bpl		long_wait_loop	; 3|2
	jmp		play			; 3
long_wait_loop:
; 37
	beq		long_wait_loop_2 ; 3|2
; Wait 256 VGM frames (256/44100 s)
	ldx		#0				; 2
short_wait_loop2:
	inc		<DUMMY			; 5
 	sta		<DUMMY			; 3
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	lda		<VGM_PTR		; 3
	dex						; 2
	bne		short_wait_loop2 ; 3|2
	dec		<DELAY+1		; 5
	jmp		long_wait_loop	; 3
long_wait_loop_2:
; 39
	decw	DELAY			; 11
	ldx		<DELAY+1
	bpl		.hi_ok			
	jmp		play
.hi_ok:
	ldx		<DELAY			; 3
; 58
	cpx 	#0				; 2
	bne		.short_wait_3	; 3|2
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	jmp		play			; 3
.short_wait_3:
	inc		<DUMMY			; 5  
	lda		<VGM_PTR		; 3
.short_wait_l3:
	inc		<DUMMY			; 5  
 	sta		<DUMMY			; 3
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	dec		<DELAY			; 5
	bne		.short_wait_l3	; 3|2
	jmp		play			; 3
	
	
 .org $C800
pernoise_table:
 .dw $07E,$0FE,$1FE 

 
 .org $D000
tone_freq_l:
	.db $FF,$FF,$00,$01,$02,$03,$04,$05,$06,$07,$08,$09,$0A,$0B,$0C,$0D
	.db $0E,$0F,$10,$11,$12,$13,$14,$15,$16,$17,$18,$19,$1A,$1B,$1C,$1D
	.db $1E,$1F,$20,$21,$22,$23,$24,$25,$26,$27,$28,$29,$2A,$2B,$2C,$2D
	.db $2E,$2F,$30,$31,$32,$33,$34,$35,$36,$37,$38,$39,$3A,$3B,$3C,$3D
	.db $3E,$3F,$40,$41,$42,$43,$44,$45,$46,$47,$48,$49,$4A,$4B,$4C,$4D
	.db $4E,$4F,$50,$51,$52,$53,$54,$55,$56,$57,$58,$59,$5A,$5B,$5C,$5D
	.db $5E,$5F,$60,$61,$62,$63,$64,$65,$66,$67,$68,$69,$6A,$6B,$6C,$6D
	.db $6E,$6F,$70,$71,$72,$73,$74,$75,$76,$77,$78,$79,$7A,$7B,$7C,$7D
	.db $7E,$7F,$80,$81,$82,$83,$84,$85,$86,$87,$88,$89,$8A,$8B,$8C,$8D
	.db $8E,$8F,$90,$91,$92,$93,$94,$95,$96,$97,$98,$99,$9A,$9B,$9C,$9D
	.db $9E,$9F,$A0,$A1,$A2,$A3,$A4,$A5,$A6,$A7,$A8,$A9,$AA,$AB,$AC,$AD
	.db $AE,$AF,$B0,$B1,$B2,$B3,$B4,$B5,$B6,$B7,$B8,$B9,$BA,$BB,$BC,$BD
	.db $BE,$BF,$C0,$C1,$C2,$C3,$C4,$C5,$C6,$C7,$C8,$C9,$CA,$CB,$CC,$CD
	.db $CE,$CF,$D0,$D1,$D2,$D3,$D4,$D5,$D6,$D7,$D8,$D9,$DA,$DB,$DC,$DD
	.db $DE,$DF,$E0,$E1,$E2,$E3,$E4,$E5,$E6,$E7,$E8,$E9,$EA,$EB,$EC,$ED
	.db $EE,$EF,$F0,$F1,$F2,$F3,$F4,$F5,$F6,$F7,$F8,$F9,$FA,$FB,$FC,$FD
	.db $FE,$FF,$00,$01,$02,$03,$04,$05,$06,$07,$08,$09,$0A,$0B,$0C,$0D
	.db $0E,$0F,$10,$11,$12,$13,$14,$15,$16,$17,$18,$19,$1A,$1B,$1C,$1D
	.db $1E,$1F,$20,$21,$22,$23,$24,$25,$26,$27,$28,$29,$2A,$2B,$2C,$2D
	.db $2E,$2F,$30,$31,$32,$33,$34,$35,$36,$37,$38,$39,$3A,$3B,$3C,$3D
	.db $3E,$3F,$40,$41,$42,$43,$44,$45,$46,$47,$48,$49,$4A,$4B,$4C,$4D
	.db $4E,$4F,$50,$51,$52,$53,$54,$55,$56,$57,$58,$59,$5A,$5B,$5C,$5D
	.db $5E,$5F,$60,$61,$62,$63,$64,$65,$66,$67,$68,$69,$6A,$6B,$6C,$6D
	.db $6E,$6F,$70,$71,$72,$73,$74,$75,$76,$77,$78,$79,$7A,$7B,$7C,$7D
	.db $7E,$7F,$80,$81,$82,$83,$84,$85,$86,$87,$88,$89,$8A,$8B,$8C,$8D
	.db $8E,$8F,$90,$91,$92,$93,$94,$95,$96,$97,$98,$99,$9A,$9B,$9C,$9D
	.db $9E,$9F,$A0,$A1,$A2,$A3,$A4,$A5,$A6,$A7,$A8,$A9,$AA,$AB,$AC,$AD
	.db $AE,$AF,$B0,$B1,$B2,$B3,$B4,$B5,$B6,$B7,$B8,$B9,$BA,$BB,$BC,$BD
	.db $BE,$BF,$C0,$C1,$C2,$C3,$C4,$C5,$C6,$C7,$C8,$C9,$CA,$CB,$CC,$CD
	.db $CE,$CF,$D0,$D1,$D2,$D3,$D4,$D5,$D6,$D7,$D8,$D9,$DA,$DB,$DC,$DD
	.db $DE,$DF,$E0,$E1,$E2,$E3,$E4,$E5,$E6,$E7,$E8,$E9,$EA,$EB,$EC,$ED
	.db $EE,$EF,$F0,$F1,$F2,$F3,$F4,$F5,$F6,$F7,$F8,$F9,$FA,$FB,$FC,$FD
	.db $FE,$FF,$00,$01,$02,$03,$04,$05,$06,$07,$08,$09,$0A,$0B,$0C,$0D
	.db $0E,$0F,$10,$11,$12,$13,$14,$15,$16,$17,$18,$19,$1A,$1B,$1C,$1D
	.db $1E,$1F,$20,$21,$22,$23,$24,$25,$26,$27,$28,$29,$2A,$2B,$2C,$2D
	.db $2E,$2F,$30,$31,$32,$33,$34,$35,$36,$37,$38,$39,$3A,$3B,$3C,$3D
	.db $3E,$3F,$40,$41,$42,$43,$44,$45,$46,$47,$48,$49,$4A,$4B,$4C,$4D
	.db $4E,$4F,$50,$51,$52,$53,$54,$55,$56,$57,$58,$59,$5A,$5B,$5C,$5D
	.db $5E,$5F,$60,$61,$62,$63,$64,$65,$66,$67,$68,$69,$6A,$6B,$6C,$6D
	.db $6E,$6F,$70,$71,$72,$73,$74,$75,$76,$77,$78,$79,$7A,$7B,$7C,$7D
	.db $7E,$7F,$80,$81,$82,$83,$84,$85,$86,$87,$88,$89,$8A,$8B,$8C,$8D
	.db $8E,$8F,$90,$91,$92,$93,$94,$95,$96,$97,$98,$99,$9A,$9B,$9C,$9D
	.db $9E,$9F,$A0,$A1,$A2,$A3,$A4,$A5,$A6,$A7,$A8,$A9,$AA,$AB,$AC,$AD
	.db $AE,$AF,$B0,$B1,$B2,$B3,$B4,$B5,$B6,$B7,$B8,$B9,$BA,$BB,$BC,$BD
	.db $BE,$BF,$C0,$C1,$C2,$C3,$C4,$C5,$C6,$C7,$C8,$C9,$CA,$CB,$CC,$CD
	.db $CE,$CF,$D0,$D1,$D2,$D3,$D4,$D5,$D6,$D7,$D8,$D9,$DA,$DB,$DC,$DD
	.db $DE,$DF,$E0,$E1,$E2,$E3,$E4,$E5,$E6,$E7,$E8,$E9,$EA,$EB,$EC,$ED
	.db $EE,$EF,$F0,$F1,$F2,$F3,$F4,$F5,$F6,$F7,$F8,$F9,$FA,$FB,$FC,$FD
	.db $FE,$FF,$00,$01,$02,$03,$04,$05,$06,$07,$08,$09,$0A,$0B,$0C,$0D
	.db $0E,$0F,$10,$11,$12,$13,$14,$15,$16,$17,$18,$19,$1A,$1B,$1C,$1D
	.db $1E,$1F,$20,$21,$22,$23,$24,$25,$26,$27,$28,$29,$2A,$2B,$2C,$2D
	.db $2E,$2F,$30,$31,$32,$33,$34,$35,$36,$37,$38,$39,$3A,$3B,$3C,$3D
	.db $3E,$3F,$40,$41,$42,$43,$44,$45,$46,$47,$48,$49,$4A,$4B,$4C,$4D
	.db $4E,$4F,$50,$51,$52,$53,$54,$55,$56,$57,$58,$59,$5A,$5B,$5C,$5D
	.db $5E,$5F,$60,$61,$62,$63,$64,$65,$66,$67,$68,$69,$6A,$6B,$6C,$6D
	.db $6E,$6F,$70,$71,$72,$73,$74,$75,$76,$77,$78,$79,$7A,$7B,$7C,$7D
	.db $7E,$7F,$80,$81,$82,$83,$84,$85,$86,$87,$88,$89,$8A,$8B,$8C,$8D
	.db $8E,$8F,$90,$91,$92,$93,$94,$95,$96,$97,$98,$99,$9A,$9B,$9C,$9D
	.db $9E,$9F,$A0,$A1,$A2,$A3,$A4,$A5,$A6,$A7,$A8,$A9,$AA,$AB,$AC,$AD
	.db $AE,$AF,$B0,$B1,$B2,$B3,$B4,$B5,$B6,$B7,$B8,$B9,$BA,$BB,$BC,$BD
	.db $BE,$BF,$C0,$C1,$C2,$C3,$C4,$C5,$C6,$C7,$C8,$C9,$CA,$CB,$CC,$CD
	.db $CE,$CF,$D0,$D1,$D2,$D3,$D4,$D5,$D6,$D7,$D8,$D9,$DA,$DB,$DC,$DD
	.db $DE,$DF,$E0,$E1,$E2,$E3,$E4,$E5,$E6,$E7,$E8,$E9,$EA,$EB,$EC,$ED
	.db $EE,$EF,$F0,$F1,$F2,$F3,$F4,$F5,$F6,$F7,$F8,$F9,$FA,$FB,$FC,$FD
tone_freq_h:
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
tone2_noise_freq_l:
	.db $0E,$0E,$1E,$2E,$3E,$4E,$5E,$6E,$7E,$8E,$9E,$AE,$BE,$CE,$DE,$EE
	.db $FE,$0E,$1E,$2E,$3E,$4E,$5E,$6E,$7E,$8E,$9E,$AE,$BE,$CE,$DE,$EE
	.db $FE,$0E,$1E,$2E,$3E,$4E,$5E,$6E,$7E,$8E,$9E,$AE,$BE,$CE,$DE,$EE
	.db $FE,$0E,$1E,$2E,$3E,$4E,$5E,$6E,$7E,$8E,$9E,$AE,$BE,$CE,$DE,$EE
	.db $FE,$0E,$1E,$2E,$3E,$4E,$5E,$6E,$7E,$8E,$9E,$AE,$BE,$CE,$DE,$EE
	.db $FE,$0E,$1E,$2E,$3E,$4E,$5E,$6E,$7E,$8E,$9E,$AE,$BE,$CE,$DE,$EE
	.db $FE,$0E,$1E,$2E,$3E,$4E,$5E,$6E,$7E,$8E,$9E,$AE,$BE,$CE,$DE,$EE
	.db $FE,$0E,$1E,$2E,$3E,$4E,$5E,$6E,$7E,$8E,$9E,$AE,$BE,$CE,$DE,$EE
	.db $FE,$0E,$1E,$2E,$3E,$4E,$5E,$6E,$7E,$8E,$9E,$AE,$BE,$CE,$DE,$EE
	.db $FE,$0E,$1E,$2E,$3E,$4E,$5E,$6E,$7E,$8E,$9E,$AE,$BE,$CE,$DE,$EE
	.db $FE,$0E,$1E,$2E,$3E,$4E,$5E,$6E,$7E,$8E,$9E,$AE,$BE,$CE,$DE,$EE
	.db $FE,$0E,$1E,$2E,$3E,$4E,$5E,$6E,$7E,$8E,$9E,$AE,$BE,$CE,$DE,$EE
	.db $FE,$0E,$1E,$2E,$3E,$4E,$5E,$6E,$7E,$8E,$9E,$AE,$BE,$CE,$DE,$EE
	.db $FE,$0E,$1E,$2E,$3E,$4E,$5E,$6E,$7E,$8E,$9E,$AE,$BE,$CE,$DE,$EE
	.db $FE,$0E,$1E,$2E,$3E,$4E,$5E,$6E,$7E,$8E,$9E,$AE,$BE,$CE,$DE,$EE
	.db $FE,$0E,$1E,$2E,$3E,$4E,$5E,$6E,$7E,$8E,$9E,$AE,$BE,$CE,$DE,$EE
	.db $FE,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
tone2_noise_freq_h:
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$04,$04,$04,$04,$04,$04,$04,$04,$04,$04,$04,$04,$04,$04,$04
	.db $04,$05,$05,$05,$05,$05,$05,$05,$05,$05,$05,$05,$05,$05,$05,$05
	.db $05,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06
	.db $06,$07,$07,$07,$07,$07,$07,$07,$07,$07,$07,$07,$07,$07,$07,$07
	.db $07,$08,$08,$08,$08,$08,$08,$08,$08,$08,$08,$08,$08,$08,$08,$08
	.db $08,$09,$09,$09,$09,$09,$09,$09,$09,$09,$09,$09,$09,$09,$09,$09
	.db $09,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A
	.db $0A,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B
	.db $0B,$0C,$0C,$0C,$0C,$0C,$0C,$0C,$0C,$0C,$0C,$0C,$0C,$0C,$0C,$0C
	.db $0C,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D
	.db $0D,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E
	.db $0E,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	
	
;**************************************************************************************
;**************************************************************************************
;**************************************************************************************
;**************************************************************************************

 .bank 3
 .org $C000			


; PAL Playback loop	
play_PAL:
	lda		[VGM_PTR],y		; 5
	incptr	VGM_PTR			; 5
	cmp		#VGM_SHORT_DELAY ; 2
	bcc		.not_short_wait	; 3|2

;------------------------------------
; Delay for n/44100 s  (n = [1..16])
; 14.015625 cycles will have been
; spent when we enter this routine.
;

	cmp		#VGM_2A03_WRITE	; 2
	bne		.short_wait		; 3|2
	lda		[VGM_PTR],y		; 5
	incptr	VGM_PTR			; 5
	tax						; 2
	lda		[VGM_PTR],y		; 5
	incptr	VGM_PTR			; 5
	sta		$4000,x			; 4
	jmp		play			; 3
.short_wait:
	;clc						; 2
	and		#$0F			; 2
	adc		<DELAY			; 3
	sta		<DELAY			; 3
	bne		.low_byte_nz	; 3|2
; 26 cycles spent
	lda		<DELAY+1		; 3
	adc		#0				; 2
	sta		<DELAY+1		; 3
	jmp		play			; 3
.low_byte_nz:
; 27 cycles spent
	lda		<DELAY+1		; 3
	adc		#0				; 2
	sta		<DELAY+1		; 3
	bmi		play			; 3|2
; 37 cycles spent
.short_wait_loop:
	inc		<DUMMY			; 5
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	dec		<DELAY			; 5
	bne		.short_wait_loop	; 3|2.  38 cycles/loop
	beq		play			; 3
.not_short_wait:	

; 15 cycles spent	
	cmp		#VGM_PSG_WRITE	; 2
	bne		.check_63
	jmp		psg_param		; 3|2
.check_63:
; 19
	cmp		#VGM_PAL_DELAY	; 2. wait one pal frame (1/50 s)
	bne		.check_62		; 3|2
	jmp		wait_frame_pal	; 3

.check_62:
; 24
	cmp		#VGM_NTSC_DELAY	; 2. wait one ntsc frame (1/60 s)
	bne		.check_61		; 3|2	
	jmp		wait_frame_ntsc	; 3

.check_61:
; 29
	cmp		#VGM_LONG_DELAY	; wait xxyy samples
	bne		.check_66
	jmp		long_wait

.check_66:	
; 34
	cmp		#VGM_SONG_END	; loop
	bne		.check_4B
	jmp		loop_song

.check_4B:
; 39
	cmp		#VGM_BANK_SWITCH ; bank switch
	bne		.check_4F
	jmp		bank_switch
	
.check_4F
	cmp		#VGM_GG_STEREO	; set gamegear stereo parameter
	bne		.check_67
	jmp		gg_stereo_param

.check_67:
	cmp		#VGM_DATA_BLOCK
; TODO: something

;foo: jmp foo
	; All other commands are unhandled and assumed to be 3 bytes long
	incptr	VGM_PTR         ; 5
	incptr	VGM_PTR         ; 5
	jmp		play            ; 3



 .org $C400
; Wait nnnn samples (nnnn = 0..65535)
;long_wait:
	lda		[VGM_PTR],y		; 5
	incptr	VGM_PTR			; 5
	clc						; 2
	adc		<DELAY			; 3
	sta		<DELAY			; 3
	lda		[VGM_PTR],y		; 5
	incptr	VGM_PTR			; 5
	adc		<DELAY+1		; 3
	sta		<DELAY+1		; 3
	bpl		.long_wait_loop	; 3|2
	jmp		play			; 3
.long_wait_loop:
; 37
	beq		.long_wait_loop_2 ; 3|2
; Wait 256 VGM frames (256/44100 s)
	ldx		#0				; 2
.short_wait_loop2:
	inc		<DUMMY			; 5
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	lda		<VGM_PTR		; 3
	dex						; 2
	bne		.short_wait_loop2 ; 3|2
	dec		<DELAY+1		; 5
	jmp		.long_wait_loop	; 3
.long_wait_loop_2:
; 39
	decw	DELAY			; 11
	ldx		<DELAY+1
	bpl		.hi_ok			
	jmp		play
.hi_ok:
	ldx		<DELAY			; 3
; 58
	cpx 	#0				; 2
	bne		.short_wait_3	; 3|2
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	jmp		play			; 3
.short_wait_3:
	inc		<DUMMY			; 5  
	lda		<VGM_PTR		; 3
.short_wait_l3:
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	inc		<DUMMY			; 5  
	dec		<DELAY			; 5
	bne		.short_wait_l3	; 3|2
	jmp		play			; 3
	
 .org $C800
; pernoise_table:
 .dw $075,$0EC,$1DA 


 .org $D000
;tone_freq_l:
	.db $00,$00,$00,$01,$02,$03,$04,$05,$06,$07,$08,$09,$0A,$0B,$0C,$0C
	.db $0D,$0E,$0F,$10,$11,$12,$13,$14,$15,$16,$17,$18,$19,$19,$1A,$1B
	.db $1C,$1D,$1E,$1F,$20,$21,$22,$23,$24,$25,$26,$26,$27,$28,$29,$2A
	.db $2B,$2C,$2D,$2E,$2F,$30,$31,$32,$33,$33,$34,$35,$36,$37,$38,$39
	.db $3A,$3B,$3C,$3D,$3E,$3F,$40,$40,$41,$42,$43,$44,$45,$46,$47,$48
	.db $49,$4A,$4B,$4C,$4D,$4D,$4E,$4F,$50,$51,$52,$53,$54,$55,$56,$57
	.db $58,$59,$5A,$5A,$5B,$5C,$5D,$5E,$5F,$60,$61,$62,$63,$64,$65,$66
	.db $67,$67,$68,$69,$6A,$6B,$6C,$6D,$6E,$6F,$70,$71,$72,$73,$74,$74
	.db $75,$76,$77,$78,$79,$7A,$7B,$7C,$7D,$7E,$7F,$80,$81,$81,$82,$83
	.db $84,$85,$86,$87,$88,$89,$8A,$8B,$8C,$8D,$8E,$8E,$8F,$90,$91,$92
	.db $93,$94,$95,$96,$97,$98,$99,$9A,$9B,$9B,$9C,$9D,$9E,$9F,$A0,$A1
	.db $A2,$A3,$A4,$A5,$A6,$A7,$A8,$A8,$A9,$AA,$AB,$AC,$AD,$AE,$AF,$B0
	.db $B1,$B2,$B3,$B4,$B5,$B6,$B6,$B7,$B8,$B9,$BA,$BB,$BC,$BD,$BE,$BF
	.db $C0,$C1,$C2,$C3,$C3,$C4,$C5,$C6,$C7,$C8,$C9,$CA,$CB,$CC,$CD,$CE
	.db $CF,$D0,$D0,$D1,$D2,$D3,$D4,$D5,$D6,$D7,$D8,$D9,$DA,$DB,$DC,$DD
	.db $DD,$DE,$DF,$E0,$E1,$E2,$E3,$E4,$E5,$E6,$E7,$E8,$E9,$EA,$EA,$EB
	.db $EC,$ED,$EE,$EF,$F0,$F1,$F2,$F3,$F4,$F5,$F6,$F7,$F7,$F8,$F9,$FA
	.db $FB,$FC,$FD,$FE,$FF,$00,$01,$02,$03,$04,$04,$05,$06,$07,$08,$09
	.db $0A,$0B,$0C,$0D,$0E,$0F,$10,$11,$11,$12,$13,$14,$15,$16,$17,$18
	.db $19,$1A,$1B,$1C,$1D,$1E,$1E,$1F,$20,$21,$22,$23,$24,$25,$26,$27
	.db $28,$29,$2A,$2B,$2B,$2C,$2D,$2E,$2F,$30,$31,$32,$33,$34,$35,$36
	.db $37,$38,$38,$39,$3A,$3B,$3C,$3D,$3E,$3F,$40,$41,$42,$43,$44,$45
	.db $45,$46,$47,$48,$49,$4A,$4B,$4C,$4D,$4E,$4F,$50,$51,$52,$52,$53
	.db $54,$55,$56,$57,$58,$59,$5A,$5B,$5C,$5D,$5E,$5F,$60,$60,$61,$62
	.db $63,$64,$65,$66,$67,$68,$69,$6A,$6B,$6C,$6D,$6D,$6E,$6F,$70,$71
	.db $72,$73,$74,$75,$76,$77,$78,$79,$7A,$7A,$7B,$7C,$7D,$7E,$7F,$80
	.db $81,$82,$83,$84,$85,$86,$87,$87,$88,$89,$8A,$8B,$8C,$8D,$8E,$8F
	.db $90,$91,$92,$93,$94,$94,$95,$96,$97,$98,$99,$9A,$9B,$9C,$9D,$9E
	.db $9F,$A0,$A1,$A1,$A2,$A3,$A4,$A5,$A6,$A7,$A8,$A9,$AA,$AB,$AC,$AD
	.db $AE,$AE,$AF,$B0,$B1,$B2,$B3,$B4,$B5,$B6,$B7,$B8,$B9,$BA,$BB,$BB
	.db $BC,$BD,$BE,$BF,$C0,$C1,$C2,$C3,$C4,$C5,$C6,$C7,$C8,$C8,$C9,$CA
	.db $CB,$CC,$CD,$CE,$CF,$D0,$D1,$D2,$D3,$D4,$D5,$D5,$D6,$D7,$D8,$D9
	.db $DA,$DB,$DC,$DD,$DE,$DF,$E0,$E1,$E2,$E2,$E3,$E4,$E5,$E6,$E7,$E8
	.db $E9,$EA,$EB,$EC,$ED,$EE,$EF,$EF,$F0,$F1,$F2,$F3,$F4,$F5,$F6,$F7
	.db $F8,$F9,$FA,$FB,$FC,$FC,$FD,$FE,$FF,$00,$01,$02,$03,$04,$05,$06
	.db $07,$08,$09,$09,$0A,$0B,$0C,$0D,$0E,$0F,$10,$11,$12,$13,$14,$15
	.db $16,$17,$17,$18,$19,$1A,$1B,$1C,$1D,$1E,$1F,$20,$21,$22,$23,$24
	.db $24,$25,$26,$27,$28,$29,$2A,$2B,$2C,$2D,$2E,$2F,$30,$31,$31,$32
	.db $33,$34,$35,$36,$37,$38,$39,$3A,$3B,$3C,$3D,$3E,$3E,$3F,$40,$41
	.db $42,$43,$44,$45,$46,$47,$48,$49,$4A,$4B,$4B,$4C,$4D,$4E,$4F,$50
	.db $51,$52,$53,$54,$55,$56,$57,$58,$58,$59,$5A,$5B,$5C,$5D,$5E,$5F
	.db $60,$61,$62,$63,$64,$65,$65,$66,$67,$68,$69,$6A,$6B,$6C,$6D,$6E
	.db $6F,$70,$71,$72,$72,$73,$74,$75,$76,$77,$78,$79,$7A,$7B,$7C,$7D
	.db $7E,$7F,$7F,$80,$81,$82,$83,$84,$85,$86,$87,$88,$89,$8A,$8B,$8C
	.db $8C,$8D,$8E,$8F,$90,$91,$92,$93,$94,$95,$96,$97,$98,$99,$99,$9A
	.db $9B,$9C,$9D,$9E,$9F,$A0,$A1,$A2,$A3,$A4,$A5,$A6,$A6,$A7,$A8,$A9
	.db $AA,$AB,$AC,$AD,$AE,$AF,$B0,$B1,$B2,$B3,$B3,$B4,$B5,$B6,$B7,$B8
	.db $B9,$BA,$BB,$BC,$BD,$BE,$BF,$C0,$C1,$C1,$C2,$C3,$C4,$C5,$C6,$C7
	.db $C8,$C9,$CA,$CB,$CC,$CD,$CE,$CE,$CF,$D0,$D1,$D2,$D3,$D4,$D5,$D6
	.db $D7,$D8,$D9,$DA,$DB,$DB,$DC,$DD,$DE,$DF,$E0,$E1,$E2,$E3,$E4,$E5
	.db $E6,$E7,$E8,$E8,$E9,$EA,$EB,$EC,$ED,$EE,$EF,$F0,$F1,$F2,$F3,$F4
	.db $F5,$F5,$F6,$F7,$F8,$F9,$FA,$FB,$FC,$FD,$FE,$FF,$00,$01,$02,$02
	.db $03,$04,$05,$06,$07,$08,$09,$0A,$0B,$0C,$0D,$0E,$0F,$0F,$10,$11
	.db $12,$13,$14,$15,$16,$17,$18,$19,$1A,$1B,$1C,$1C,$1D,$1E,$1F,$20
	.db $21,$22,$23,$24,$25,$26,$27,$28,$29,$29,$2A,$2B,$2C,$2D,$2E,$2F
	.db $30,$31,$32,$33,$34,$35,$36,$36,$37,$38,$39,$3A,$3B,$3C,$3D,$3E
	.db $3F,$40,$41,$42,$43,$43,$44,$45,$46,$47,$48,$49,$4A,$4B,$4C,$4D
	.db $4E,$4F,$50,$50,$51,$52,$53,$54,$55,$56,$57,$58,$59,$5A,$5B,$5C
	.db $5D,$5D,$5E,$5F,$60,$61,$62,$63,$64,$65,$66,$67,$68,$69,$6A,$6A
	.db $6B,$6C,$6D,$6E,$6F,$70,$71,$72,$73,$74,$75,$76,$77,$78,$78,$79
	.db $7A,$7B,$7C,$7D,$7E,$7F,$80,$81,$82,$83,$84,$85,$85,$86,$87,$88
	.db $89,$8A,$8B,$8C,$8D,$8E,$8F,$90,$91,$92,$92,$93,$94,$95,$96,$97
	.db $98,$99,$9A,$9B,$9C,$9D,$9E,$9F,$9F,$A0,$A1,$A2,$A3,$A4,$A5,$A6
	.db $A7,$A8,$A9,$AA,$AB,$AC,$AC,$AD,$AE,$AF,$B0,$B1,$B2,$B3,$B4,$B5
;tone_freq_h:
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$01,$01,$01,$01,$01,$01,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03

;tone2_noise_freq_l:
	.db $0D,$0D,$1C,$2B,$3A,$49,$58,$67,$75,$84,$93,$A2,$B1,$C0,$CF,$DD
	.db $EC,$FB,$0A,$19,$28,$37,$45,$54,$63,$72,$81,$90,$9F,$AE,$BC,$CB
	.db $DA,$E9,$F8,$07,$16,$24,$33,$42,$51,$60,$6F,$7E,$8C,$9B,$AA,$B9
	.db $C8,$D7,$E6,$F5,$03,$12,$21,$30,$3F,$4E,$5D,$6B,$7A,$89,$98,$A7
	.db $B6,$C5,$D3,$E2,$F1,$00,$0F,$1E,$2D,$3C,$4A,$59,$68,$77,$86,$95
	.db $A4,$B2,$C1,$D0,$DF,$EE,$FD,$0C,$1A,$29,$38,$47,$56,$65,$74,$83
	.db $91,$A0,$AF,$BE,$CD,$DC,$EB,$F9,$08,$17,$26,$35,$44,$53,$61,$70
	.db $7F,$8E,$9D,$AC,$BB,$C9,$D8,$E7,$F6,$05,$14,$23,$32,$40,$4F,$5E
	.db $6D,$7C,$8B,$9A,$A8,$B7,$C6,$D5,$E4,$F3,$02,$10,$1F,$2E,$3D,$4C
	.db $5B,$6A,$79,$87,$96,$A5,$B4,$C3,$D2,$E1,$EF,$FE,$0D,$1C,$2B,$3A
	.db $49,$57,$66,$75,$84,$93,$A2,$B1,$C0,$CE,$DD,$EC,$FB,$0A,$19,$28
	.db $36,$45,$54,$63,$72,$81,$90,$9E,$AD,$BC,$CB,$DA,$E9,$F8,$07,$15
	.db $24,$33,$42,$51,$60,$6F,$7D,$8C,$9B,$AA,$B9,$C8,$D7,$E5,$F4,$03
	.db $12,$21,$30,$3F,$4D,$5C,$6B,$7A,$89,$98,$A7,$B6,$C4,$D3,$E2,$F1
	.db $00,$0F,$1E,$2C,$3B,$4A,$59,$68,$77,$86,$94,$A3,$B2,$C1,$D0,$DF
	.db $EE,$FD,$0B,$1A,$29,$38,$47,$56,$65,$73,$82,$91,$A0,$AF,$BE,$CD
	.db $DB,$EA,$F9,$08,$17,$26,$35,$44,$52,$61,$70,$7F,$8E,$9D,$AC,$BA
	.db $C9,$D8,$E7,$F6,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
	.db $FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF
;tone2_noise_freq_h:
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.db $01,$01,$01,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02,$02
	.db $02,$02,$02,$02,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03,$03
	.db $03,$03,$03,$03,$03,$04,$04,$04,$04,$04,$04,$04,$04,$04,$04,$04
	.db $04,$04,$04,$04,$04,$04,$04,$05,$05,$05,$05,$05,$05,$05,$05,$05
	.db $05,$05,$05,$05,$05,$05,$05,$05,$06,$06,$06,$06,$06,$06,$06,$06
	.db $06,$06,$06,$06,$06,$06,$06,$06,$06,$07,$07,$07,$07,$07,$07,$07
	.db $07,$07,$07,$07,$07,$07,$07,$07,$07,$07,$08,$08,$08,$08,$08,$08
	.db $08,$08,$08,$08,$08,$08,$08,$08,$08,$08,$08,$08,$09,$09,$09,$09
	.db $09,$09,$09,$09,$09,$09,$09,$09,$09,$09,$09,$09,$09,$0A,$0A,$0A
	.db $0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0B,$0B
	.db $0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0B,$0C
	.db $0C,$0C,$0C,$0C,$0C,$0C,$0C,$0C,$0C,$0C,$0C,$0C,$0C,$0C,$0C,$0C
	.db $0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D
	.db $0D,$0D,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E
	.db $0E,$0E,$0E,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F


	
 .bank 4
 .db 0

 .bank 6
 .db 0

 .bank 8
 .db 0

 .bank 10
 .db 0

 .bank 12
 .db 0
 
 .bank 31
 .org $E000
 
 reset:
 	cld
 	sei
 	ldx 	#$00
 	stx 	REG_PPUCNT1	; No NMI
 	stx 	REG_PPUCNT2	; Disable screen
 	inx
 waitvb:	
 	lda 	REG_PPUSTAT
 	bpl 	waitvb	; Wait a few frames
 	dex	
 	bpl 	waitvb
 	txs		; Set up stack pointer
 
 redo:	
 	lda 	#0
 	sta 	<CYCCTR	
 	sta 	<CYCCTR+1
 detect_mode:
 	inc 	<CYCCTR
 	bne 	no_cyc
 	inc 	<CYCCTR+1
 no_cyc:
 	lda 	REG_PPUSTAT
 	bpl 	detect_mode
 	lda 	<CYCCTR+1
 	cmp 	#9
 	bpl 	redo
 	cmp 	#8
 	bmi 	ntsc
 	lda 	<CYCCTR
 	cmp 	#$32
 	bmi 	ntsc
 	lda		#1
 	sta		ISPAL
 	jmp		mode_detection_done
 ntsc:
 	lda		#0
 	sta		ISPAL
 mode_detection_done:	
 	
 	lda		#0
 	sta		REG_VRC6_MIRRORING
 	sta		REG_VRC6_CHR0
 	lda		#1
 	sta		REG_VRC6_CHR1
 	lda		#2
 	sta		REG_VRC6_CHR2
 	
 	set_vram_adr $3F00
 	memcpy_fixed_dest REG_VRAMIO,palette,32
 
 	set_vram_adr $2000
 	ldy 	#$1E			
 	lda 	#0
 clear_nt:
 	ldx 	#$20				; 32 tiles per row
 clear_row:
 	sta 	REG_VRAMIO			; Write to VRAM, address is auto-incremented
 	dex
 	bne 	clear_row
 	dey
 	bne 	clear_nt
 
 	set_vram_adr $210C
 	lda		#(szMessage0%256)
 	sta		<$40
 	lda		#(szMessage0/256)
 	sta		<$41
 	lda		#64
 	sta		<$42
 	jsr		puts
 	
 	set_vram_adr $2140
 	lda		#(szMessage%256)
 	sta		<$40
 	lda		#(szMessage/256)
 	sta		<$41
 	lda		#-32
 	sta		<$42
 	jsr		puts
 
 	set_vram_adr $2200
 	lda		#(szMessage2%256)
 	sta		<$40
 	lda		#(szMessage2/256)
 	sta		<$41
 	jsr		puts
 
 	set_vram_adr $22C0
 	lda		#(szMessage3%256)
 	sta		<$40
 	lda		#(szMessage3/256)
 	sta		<$41
 	jsr		puts
 	
 	set_vram_adr $23C0
 	memset_fixed_dest $2007,#$00,$40
 	
 	lda 	#(PPUCNT2_E_BG | PPUCNT2_NOCLIP_BG | PPUCNT2_COLOR)
 	sta 	REG_PPUCNT2  
 	lda 	#(PPUCNT1_BG_0000 | PPUCNT1_INC_1 | PPUCNT1_NT_2000)
 	sta 	REG_PPUCNT1
 	set_vram_adr $2000
 	
     lda     #(APU_STATUS_SQR1_ENA | APU_STATUS_SQR2_ENA | APU_STATUS_NOISE_ENA)
     sta     REG_APU_STATUS
 
     lda     #0
     sta     REG_APU_TRI_LINC
     sta     REG_APU_SQR1_CTL
     sta     REG_APU_SQR2_CTL
     sta     REG_APU_NOISE_CTL
     sta	 REG_APU_DMC_CTL
     sta     REG_VRC6_SAW_RATE
     sta     REG_VRC6_SAW_PERL
     sta     REG_VRC6_SAW_PERH
     sta     REG_VRC6_SQR1_CTL
     sta     REG_VRC6_SQR2_CTL
     sta	 REG_VRC6_IRQCTL
 
     ldx     #0
     lda     #0
 .clear_zp:
	sta     <$00,x
    inx
    bne     .clear_zp    
 
 	lda		#$FF
 	sta		<PERH0
 	sta		<PERH1
 	sta		<PERH2
 	
 	lda		$8034
    sta     <VGM_PTR
    lda     $8035
    sta     <VGM_PTR+1

	lda		ISPAL
	clc
	adc		#2
	sta		REG_VRC6_PRG_8K
	
    lda     #2
    sta     SONG_BANK
    sta     REG_VRC6_PRG_16K
 
	ldy		#0
	jmp		play



;-----------------------------------------------
; A value is being written to the SN76489 port
; Check if it's a 'latch' or 'data' byte and 
; handle accordingly.
; 
; 20 cycles spent when we get here
;-----------------------------------------------
psg_param:
    lda     [VGM_PTR],y             ; 5
    bmi     latch_data              ; 3|2
    incptr  VGM_PTR                 ; 5
    ldx     <LATCHED_REG            ; 3
    cpx     #6                      ; 2
    bcs     volume_or_noise_data    ; 3|2
; 39
	; Set high 6 bits of tone register
    sta     <TEMP                   ; 3
    asl     a                       ; 2. 
    asl     a                       ; 2. 
    asl     a                       ; 2. 
    asl     a                       ; 2. 
    sta     <TEMP2                  ; 3
    lda     <TONE0_LATCH,x          ; 4
    and     #$0F                    ; 2
    ora     <TEMP2                  ; 3
    sta     <TONE0_LATCH,x          ; 4
    lda     <TEMP                   ; 3
    lsr		a						; 2
    lsr		a						; 2
    lsr		a						; 2
    lsr		a						; 2
    and     #3                      ; 2
    sta     <TONE0_LATCH+1,x        ; 4    
	jmp		tone_reg_updated        ; 3 (86 total)
volume_or_noise_data:
; 40 cycles spent
    and     #$0F                    ; 2
    sta     <TONE0_LATCH,x          ; 4
check_vol_noise:
    txa                             ; 2
    and     #8                      ; 2
    bne     volume_reg_updated      ; 3|2 (53 total if taken)
	inc		<DUMMY					; 5  
    nop                             ; 2
    nop								; 2
    nop								; 2
	jmp		noise_reg_updated       ; 3 (66 total)
latch_data:
; 28 cycles spent
	incptr	VGM_PTR					; 5
    sta     <TEMP                   ; 3.  Lcctdddd
    lsr     a                       ; 2.  0Lcctddd
    lsr     a                       ; 2.  00Lcctdd    
    lsr     a                       ; 2.  000Lcctd    
    lsr     a                       ; 2.  0000Lcct
    lsr     a                       ; 2.  00000Lcc
    and     #3                      ; 2.  000000cc
    bcc     .not_vol                ; 3|2
    ora     #4                      ; 2
.not_vol:
    asl     a                       ; 2.  0000tcc0
    sta     <LATCHED_REG            ; 3
; 56 cycles spent    
    tax                             ; 2
    lda     <TONE0_LATCH,x          ; 4
    and     #$F0                    ; 2
    sta     <TEMP2                  ; 3
    lda     <TEMP                   ; 3
    and     #$0F                    ; 2
    ora     <TEMP2                  ; 3
    sta     <TONE0_LATCH,x          ; 4
    cpx     #6                      ; 2
    bcs     check_vol_noise2        ; 3|2 (84 total if taken)
    jmp     tone_reg_updated        ; 3 (86 total)    
check_vol_noise2:
    decw    DELAY                   ; 11
    txa                             ; 2
    and     #8                      ; 2
    bne     volume_reg_updated      ; 3|2 (102-Ps total if taken)
	jmp		noise_reg_updated       ; 3 (104-Ps total)


;-----------------------------------------------
; An SN76489 volume register has been updated
; Convert to 2A03/VRC6 volume and set the
; correct register.
; 
; Either 53 or 102-Ps cycles spent when we get here
;-----------------------------------------------
volume_reg_updated:
    decw    DELAY                   ; 11
    decw    DELAY                   ; 11
    decw    DELAY                   ; 11
    nop								; 2
    cpx     #$0E                    ; 2
    beq     handle_noise_volume     ; 3|2
    cpx     #$0A                    ; 2
    beq     handle_tone1_volume     ; 3|2
    cpx     #$0C                    ; 2
    beq     handle_tone2_volume     ; 3|2
; 100 | 149
    ldx     <VOL0_LATCH             ; 3
    lda     volume_table,x          ; 4    
    sta     REG_APU_SQR1_CTL        ; 4
    jmp     play                    ; 3 (114-3Ps or 163-4Ps total)
handle_noise_volume:
; 93 | 142
    ldx     <VOL3_LATCH             ; 3
    lda     volume_table,x    		; 4    
    sta     REG_APU_NOISE_CTL       ; 4
    and		#$0F					; 2
    sta		REG_VRC6_SQR2_CTL		; 4
    jmp     play                    ; 3 (113-3Ps or 162-4Ps total)
handle_tone1_volume:
; 97 | 146
    ldx     <VOL1_LATCH             ; 3
    lda     volume_table,x          ; 4    
    sta     REG_APU_SQR2_CTL        ; 4
    lda     <VGM_PTR                ; 3. dummy read
    jmp     play                    ; 3 (114-3Ps or 163-4Ps total)
handle_tone2_volume:
; 101 | 150
    ldx     <VOL2_LATCH          	; 3
    lda     vrc6_volume_table,x   	; 4    
    sta     REG_VRC6_SQR1_CTL    	; 4
    jmp     play                 	; 3 (115-Ps or 164-4Ps total)




;-----------------------------------------------
; An SN76489 tone register has been updated
; 
; 86 cycles spent when we get here
;-----------------------------------------------
tone_reg_updated:
	decw	DELAY				; 11
	decw	DELAY				; 11
	decw	DELAY				; 11
	decw	DELAY				; 11
	decw	DELAY				; 11
  	decw	DELAY				; 11
  	inc 	<DUMMY				; 5
  	inc 	<DUMMY				; 5
  	nop							; 2
    cpx     #4                  ; 2
    beq     tone2_updated       ; 3|2
    cpx     #2                  ; 2
    beq     tone1_updated       ; 3|2
; 149
	sty 	<TEMP				; 3
    ldy     <TONE0_LATCH        ; 3
    lda     <TONE0_LATCH+1      ; 3
    ora		#$D0				; 2
    sta		<FREQ_PTR+1			; 3
    lda		[FREQ_PTR],y		; 5
	sta		REG_APU_SQR1_PERL	; 4
	clc							; 2
	lda		<FREQ_PTR+1			; 3
	adc		#4					; 2
; 179
	sta		<FREQ_PTR+1			; 3
	lda		[FREQ_PTR],y		; 5
	ldy		<TEMP				; 3
    cmp     <PERH0              ; 3
    beq     .perh0_same         ; 3|2
    sta     <PERH0              ; 3
    sta     REG_APU_SQR1_PERH   ; 4
    jmp     play                ; 3 (228-6Ps total)
.perh0_same:
; 196
	nop							; 2
	nop							; 2
	nop							; 2
    jmp     play                ; 3 (228-6Ps total)
tone2_updated:
; 146
	sty 	<TEMP				; 3
    ldy     <TONE2_LATCH        ; 3
    lda     <TONE2_LATCH+1      ; 3
    ora		#$D0				; 2
    sta		<FREQ_PTR+1			; 3
; 160
    lda		[FREQ_PTR],y		; 5
	sta		REG_VRC6_SQR1_PERL	; 4
	clc							; 2
	lda		<FREQ_PTR+1			; 3
	adc		#4					; 2
	sta		<FREQ_PTR+1			; 3	
	lda		[FREQ_PTR],y		; 5
	ldy		<TEMP				; 3
	ora     #$80                ; 2
    cmp     <PERH2              ; 3
; 192
    beq     .perh2_same         ; 3|2
    sta     <PERH2              ; 3
    sta     REG_VRC6_SQR1_PERH 	; 4
    jmp     play          		; 3 (227-6Ps total)
.perh2_same:
	nop							; 2
	nop							; 2
	nop							; 2
    jmp     play                ; 3 (227-6Ps total)
tone1_updated:
; 150
	sty 	<TEMP				; 3
    ldy     <TONE1_LATCH        ; 3
    lda     <TONE1_LATCH+1      ; 3
    ora		#$D0				; 2
    sta		<FREQ_PTR+1			; 3
    lda		[FREQ_PTR],y		; 5	
	sta		REG_APU_SQR2_PERL	; 4	
	clc							; 2
; 175
	lda		<FREQ_PTR+1			; 3
	adc		#4					; 2
	sta		<FREQ_PTR+1			; 3
	lda		[FREQ_PTR],y		; 5
	ldy		<TEMP           	; 2
; 190
    cmp     <PERH1              ; 3
    beq     .perh1_same         ; 3|2
    sta     <PERH1              ; 3
    sta     REG_APU_SQR2_PERH   ; 4
    jmp     play                ; 3 (228-6Ps total)
.perh1_same:
	nop							; 2
	nop							; 2
	nop							; 2
    jmp     play                ; 3 (228-6Ps total)


;---------------------------------------------------
; The SN76489 noise register has been updated
; 
; Either 66 or 104-Ps cycles spent when we get here
;---------------------------------------------------
noise_reg_updated:
    decw    DELAY               ; 11
    decw    DELAY               ; 11
    decw    DELAY               ; 11
    decw    DELAY               ; 11

    lda     #4                  ; 2
    bit     <NOISE_LATCH        ; 3
    beq     periodic_noise      ; 3|2
; 117 | 155
	bit		<PERH3				; 3
    bvs     .noise_on           ; 3|2
    lda		#0					; 2
    sta		REG_VRC6_SQR2_PERH	; 4
	lda     #(APU_STATUS_SQR1_ENA | APU_STATUS_SQR2_ENA | APU_STATUS_NOISE_ENA) ; 2
	sta		REG_APU_STATUS		; 4
    lda     #$F8                ; 2
    sta     <PERH3              ; 3
    sta     REG_APU_NOISE_LENC  ; 4
.noise_on:
; 123 | 161
    lda     <NOISE_LATCH        ; 3
    and     #3                  ; 2
    cmp		#3					; 2
    beq		white_noise_t2		; 3|2
; 133 | 171
	asl		a					; 2
    clc                         ; 2
    adc     #9                  ; 2
    sta     REG_APU_NOISE_PER   ; 4
    nop
    nop
    jmp     play               	; 3 (150-4Ps or 188-5Ps) 
white_noise_t2:
; 134 | 172
	ldx		<TONE2_LATCH		; 3
	lda		<TONE2_LATCH+1		; 3
	bne		.wn_max				; 3|2
	lda		tone2_white_noise,x	; 4
	sta		REG_APU_NOISE_PER	; 4
	jmp		play				; 3 (153-4Ps or 191-5Ps)
.wn_max:
	lda		#$0F				; 2
	sta		REG_APU_NOISE_PER	; 4
	jmp		play				; 3 (152-4Ps or 190-5Ps)
periodic_noise:
; 118 | 156
    lda     <NOISE_LATCH        ; 3
    and     #3                  ; 2
    cmp		#3					; 2
    beq		periodic_noise_t2	; 3|2
; 127 | 165
    asl		a					; 2
    tax							; 2
	lda     #(APU_STATUS_SQR1_ENA | APU_STATUS_SQR2_ENA) ; 2
	sta		REG_APU_STATUS		; 4
	lda		pernoise_table,x	; 4
	sta		REG_VRC6_SQR2_PERL	; 4
	clc							; 2
	lda		pernoise_table+1,x	; 4
    ora		#$80				; 2
    cmp     <PERH3              ; 3
    beq     .perh3_same3        ; 3|2
    sta     <PERH3              ; 3
    sta     REG_VRC6_SQR2_PERH 	; 4
    jmp     play          		; 3 (168-4Ps or 206-5Ps)
.perh3_same3:
	jmp		play				; 3 (162-4Ps or 200-5Ps)
periodic_noise_t2:
; 128 | 166
	decw	DELAY				; 11
	lda     #(APU_STATUS_SQR1_ENA | APU_STATUS_SQR2_ENA) ; 4
	sta		REG_APU_STATUS		; 4
	sty 	<TEMP				; 3
    ldy     <TONE2_LATCH        ; 3
    lda     <TONE2_LATCH+1      ; 3
    ora		#$D8				; 2
    sta		<FREQ_PTR+1			; 3
; 161 | 199
    lda		[FREQ_PTR],y		; 5
	sta		REG_VRC6_SQR2_PERL	; 4
	clc							; 2
	lda		<FREQ_PTR+1			; 3
	adc		#4					; 2
	sta		<FREQ_PTR+1			; 3
	lda		[FREQ_PTR],y		; 5
	ldy		<TEMP				; 3
    ora		#$80				; 2
    cmp     <PERH3              ; 3
    beq     .perh3_same2        ; 3|2
    sta     <PERH3              ; 3
    sta     REG_VRC6_SQR2_PERH 	; 4
    jmp     play          		; 3
.perh3_same2:
	jmp		play				; 3 (199-5Ps or 237-6Ps)
	

; Wait 735 samples
wait_frame_ntsc:
	lda		<DELAY		    ; 3
	clc					    ; 2
	adc		#$DF		    ; 2
	sta		<DELAY		    ; 3
	lda		<DELAY+1	    ; 3
	adc		#2			    ; 2
	sta		<DELAY+1	    ; 3
	bmi		.no_wait		; 3|2
	jmp		long_wait_loop	; 3
.no_wait:
	jmp		play		    ; 3
	
	
; Wait 882 samples
wait_frame_pal:
	lda		<DELAY		    ; 3
	clc					    ; 2
	adc		#$72		    ; 2
	sta		<DELAY		    ; 3
	lda		<DELAY+1	    ; 3
	adc		#3			    ; 2
	sta		<DELAY+1	    ; 3
	bmi		.no_wait		; 3|2
	jmp		long_wait_loop	; 3
.no_wait:
	jmp		play		    ; 3

	
; Switch to the next bank	
bank_switch:
	inc		<SONG_BANK		; 5
	lda		<SONG_BANK		; 3
	sta		REG_VRC6_PRG_16K ; 4
	lda		#$00			; 2
	sta		<VGM_PTR		; 3
	lda		#$80			; 2
	sta		<VGM_PTR+1		; 3
	ldy		#0				; 2
	nop
	nop
	jmp		play			; 3 (27)
	

; Restart from the song's loop point	
loop_song:
	decw	DELAY			; 11
	lda		#2				; 2
	sta		<SONG_BANK		; 3
	sta		REG_VRC6_PRG_16K ; 4
	lda		$8038			; 4
	sta		<VGM_PTR		; 3
	lda		$8039			; 4
	sta		<VGM_PTR+1		; 3
	ldy		#0				; 2
	jmp		play			; 3 (39)
	
	
gg_stereo_param:
	incptr	VGM_PTR         ; 5
	jmp		play            ; 3



puts:
	ldy 	#0
.puts_loop:
	lda 	[$40],y
	beq 	.puts_done
	clc
	adc		<$42
	sta 	REG_VRAMIO
	iny
	jmp 	.puts_loop
.puts_done:
	rts


nmi:
	rti
irq:
	rti
	

volume_table:
 .db $BF,$BD,$BA,$B8,$B6,$B5,$B4,$B3
 .db $B3,$B2,$B2,$B2,$B1,$B1,$B1,$B0
; .db $BF,$BE,$BD,$BC,$BB,$BA,$B9,$B8,$B7,$B6,$B5,$B4,$B3,$B2,$B1,$B0

 
vrc6_volume_table:
 .db $7F,$7D,$7A,$78,$76,$75,$74,$73
 .db $73,$72,$72,$71,$71,$71,$71,$70
; .db $7F,$7E,$7D,$7C,$7B,$7A,$79,$78,$77,$76,$75,$74,$73,$72,$71,$70

 palette:
 	.db 2,32,32,32 ;36,2,32		; 
 	.db 2,14,32,38		; 
 	.db 2,3,32,10		; 
 	.db 2,3,32,9		; 
 
 	.db 2,32,32,32 ;2,49,33,32		; 
 	.db 2,19,35,48		; 
 	.db 2,19,19,35		; 
 	.db 2,51,52,51		; 
 
 
 szMessage0: .db "Mic 2011",0
 szMessage:
	.db " A fatal exception 0E has occur "
	.db " red at 000:C000 in VxD VGM(01) "
	.db " + 00004000. The current applic "
	.db " ation will be playing music ma "
	.db " de for a competing console.    ",0
 szMessage2:
	.db " * Do not press any key, as it  "
	.db "   will not do anything at all. "
	.db " * Press CTRL+ALT+DEL to resta  "
	.db "   rt your computer. It won't   "
	.db "   help one bit.                ",0
 szMessage3:
	.db "     Do nothing at all to       "
	.db "     continue listening.        ",0

 .org $FD00
tone2_white_noise:	
	.db $02,$02,$03,$04,$04,$05,$05,$06,$06,$07,$07,$07,$08,$08,$08,$09
	.db $09,$09,$09,$09,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0A,$0B,$0B,$0B,$0B
	.db $0B,$0B,$0B,$0B,$0B,$0B,$0B,$0C,$0C,$0C,$0C,$0C,$0C,$0C,$0C,$0C
	.db $0C,$0C,$0C,$0C,$0C,$0C,$0C,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D
	.db $0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D,$0D
	.db $0D,$0D,$0D,$0D,$0D,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E
	.db $0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E
	.db $0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E
	.db $0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E
	.db $0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E
	.db $0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0E,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F
	.db $0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F,$0F

	
 .org  $fffa
	.dw   nmi
	.dw   reset
	.dw   irq


 .bank 32
 	; Font
	.incbin "adore64.pat"
	.incbin "adore642.pat"	
 
 
