;==============================================================
; WLA-DX banking setup
;==============================================================
.memorymap
defaultslot 0
slotsize $7ff0 ; unpaged area
slot 0 $0000
slotsize $0010 ; header, given a slot to make sure numbering is correct
slot 1 $7ff0
slotsize $4000 ; paged area
slot 2 $8000
slotsize $2000 ; RAM
slot 3 $c000
.endme

.rombankmap
bankstotal 3
banksize $7ff0
banks 1
banksize $0010
banks 1
banksize $4000
banks 1
.endro

.struct Time
Frames    db
Seconds   db
Minutes   db
Hours     db
Days      db ; Enough? 256 days is a lot...
.endst

.ramsection "Memory" slot 3
; aligned stuff first, keep it all to suitable boundaries
SpriteTable                          dsb 256
MusicEngineRAM                       dsb 256*2 ; the upper 256b is not used but a few of them must stay zero for it to not go strange
Port3EValue                          db
VBlank                               dw
Width                                db
Height                               db
Top                                  db ; - x,y of top-left of grid on screen
Left                                 db ; /
CursorScreenX                        db
CursorScreenY                        db
CursorBoardX                         db
CursorBoardY                         db
NumSprites                           db
CursorSpriteTableStart               dw
LevelNumber                          dw
CurrentlyPressedButtons              db
JustPressedButtons                   db
CurrentGameTime                      instanceof Time
TimeDisplay                          dsb 12*2 ; 12 digits (255d23:59:59) x 2 rows (too big?)
Solution                             dsb 25*20+8 ; max size (+ a bit for unpacking overestimation, only needs 4 actually)
Puzzle                               dsb 25*20+8 ; the one we're working on
PuzzleLength                         dw ; = width * height
RepeatCounter                        db
Paused                               db ; set/unset by NMI
PaletteFadeControl                   db
PaletteSize                          db
CurrentPalette                       dsb 32
TargetPalette                        dsb 32
PaletteFadeFrameCounter              db
TileMapCopy                          dsb 32*28*2 ; for when I want to mess with the whole screen but be able to get it all back
FramesPerSecond                      db
VScroll                              db
ChangedPixelXY                       dw
ChangedPixelState                    db
CursorColourIndex                    db
CursorColourTimer                    db
LevelNumberDisplay                   dsb 6 ; max 6 digits
CompletedCountDisplay                dsb 6 ; ditto
LevelDone                            db
LastChangedFrom                      db
LastChangedTo                        db
RandomNumberGeneratorWord            dw
.ends

.ramsection "Save RAM" slot 2
Marker                               dsb 16
TotalGameTime                        instanceof Time
CompletedCount                       dw
MaxLevel                             dw
LastTrackNumber                      db
CompletedLevels                      dsb 500 ; how many?
.ends

.struct Tiles
; Tile indices for gameplay
; using a struct as an enum here
Background     db
Blank          db
Filled         db
Dotted         db
ShadowTR       db
ShadowR        db
ShadowBL       db
ShadowB        db
ShadowBR       db
SixthColumn    dsb 3
SixthRow       dsb 3
SixthBoth      dsb 3
HintNumbers    dsb 26
Cursors        dsb 12
Digits         dsb 18 ; number may change if graphics change
BigText        dsb 8 ; text is exported from BMP2TIle so it's not needed anyway
Done           dsb 2
.endst

.macro CallHL
  rst $10
.endm

.macro AddHLA
  rst $18
.endm

.define FastOtirBlockSize 256
.macro FastOtir args count
.ifgr count FastOtirBlockSize
  call outiblock
  FastOtir count-FastOtirBlockSize
.else
  call outiblock+(FastOtirBlockSize-count)*2
.endif
.endm

.macro ldde args dval,eval
  ld de,((dval&$ff)<<8)|(eval&$ff)
.endm

.bank 0 slot 0

.include "Useful functions.inc"
.include "Phantasy Star decompressors.inc"
.sdsctag 1.01,"Picross",Comments,"Maxim"

.org 0
; standard startup
.section "Startup" force
di
im 1
ld sp, $dff0
jp main
.ends

.org $10 ; rst $10 = CallHL
.section "rst $10" force
CallHL:
  jp (hl)
DoNothing:
  ret ; could save a whole byte by reusing another innocent ret
.ends

.org $18 ; rst $18 = AddHLA
.section "rst $18" force
AddHLA:
  ; somewhat like add hl,a
  ; but it kills a
  ; may not even be needed, if things are nicely aligned...
  ; only works for positive (or unsigned) a
  add a,l
  ld l,a
  ret nc
  inc h
  ret
.ends

; space for more..?

.org $38
.section "VBlank handler" force
  ex af,af'
  exx
  push ix
  push iy
    in a,(VDPStatus) ; satisfy interrupt
    ; assume no H-ints
    ld hl,(VBlank)
    CallHL
    call GetRandomNumber ; randomise
  pop iy
  pop ix
  exx
  ex af,af'
  ei
  reti
.ends

.org $66
.section "Pause handler" force
NMI:
  ; debug
  /*
  ld hl,TileMapAddress
  call VRAMToHL
  ld bc,32*28*2
-:xor a
  out (VDPData),a
  dec bc
  ld a,b
  or c
  jr nz,-
  ld hl,(LevelNumber)
  inc hl
  ld (LevelNumber),hl
  call LoadLevel
  call InitPuzzle
  call DrawGrid
  call DrawTips

  retn
*/

  push af
    ld a,(Paused)
    xor 1
    ld (Paused),a
  pop af
  retn
.ends

.section "main" free
main:
  call StopSound

  call InitialiseSRAM

  call DefaultInitialiseVDP
  call ClearVRAM
  ; clear RAM
  ld hl,$c001
  ld de,$c002
  ld bc,$dff0-$c002
  ld (hl),0
  ldir

  ; blacken the palette
  ld hl,PaletteAddress
  call VRAMToHL
  ld hl,$c001 ; good as anywhere
  ld c,VDPData
  FastOtir 32

  call IsPAL
  or a
  ld a,60
  jr z,+
  ld a,50
+:ld (FramesPerSecond),a

  call Nintendont

  jp TitleScreen
.ends

.section "Game main loop" free
GameMainLoop:
  di

  call ClearVRAM

  ld hl,GameTiles
  ld de,$4000 ; tile 0
  call LoadTiles4BitRLENoDI

  ld hl,CloudTiles
  ld de,$4000|(256*32)
  call LoadTiles4BitRLENoDI

  ld hl,GameTilePalette
  ld de,TargetPalette
  ld bc,16
  ldir
  ld hl,GameTilePalette
  ld bc,16
  ldir

  xor a
  ld (NumSprites),a

  ld hl,(LevelNumber)
  call LoadLevel
  call InitPuzzle
  call DrawGrid
  call DrawTips
  call DrawRandomClouds
  call InitialiseCursor
  ; blank the time counter
  ld hl,CurrentGameTime
  ld de,CurrentGameTime+1
  ld (hl),0
  ld bc,_sizeof_Time-1 ; yay for WLA DX structs
  ldir

  call LookupAndPlayTrack

  call TurnOnScreen

  ei

ContinueGameMainLoop: ; where to enter after unpausing
  ld hl,GameVBlank
  ld (VBlank),hl

  call FadeInFullPalette

  ; in case of furious pausing
  xor a
  ld (Paused),a
-:halt ; be nice

  ld a,(Paused)
  or a
  jp nz,GamePausedMainLoop

  ; do any non-VBlank work here
  call IsBoardCorrect
  jr c,BoardFinished

  jr - ; loop forever

BoardFinished:
  ; give it a chance to update
  halt

  ld hl,GamePauseVBlank
  ld (VBlank),hl

  call UpdateSRAMStats

  call StopSound

  halt ; wait for VBlank to try to do this stuff inside it
  call NoSprites

  call BlankDots

  ld hl,BigText_Complete
  ldde 6,1
  call DrawBigText

  ld a,$8d ; completed music
  ld (TrackNumber),a

  ; wait for button press
-:halt
  ld a,(JustPressedButtons)
  and P11|P12
  jr z,-

  call FadeOutSound
  call FadeOutFullPalette
  call TurnOffScreen
  call StopSound
  di
  ld hl,DoNothing
  ld (VBlank),a

  ; increment the level number
  ld hl,(LevelNumber)
  inc hl
  ld (LevelNumber),hl

  jp TitleScreen

.ends

.section "Game paused main loop" free
GamePausedMainLoop:
  ld hl,GamePauseVBlank
  ld (VBlank),hl

  call PauseMusic

  ; kill sprites
  halt
  ld hl,SpriteTableAddress
  call VRAMToHL
  ld a,208
  out (VDPData),a

  call BackupTilemap

  ldde 12,7
  ld hl,BigText_BGM
  call DrawBigText

  ldde 9,13
  ld hl,BigText_Retire
  call DrawBigText

  ; in case of furious pausing
  ld a,1
  ld (Paused),a
-:ld a,(Paused)
  or a
  jr z,_ExitPauseState

  ; do work here
  
  ld a,(JustPressedButtons)
  and P1U
  call nz,_ChangeBGM
  ld a,(JustPressedButtons)
  and P1D
  jr nz,_Retire

  halt ; be nice
  jr - ; loop forever

_ChangeBGM:
  call NextTrack
  call PlayMusic
  ret

_Retire:
  call RestoreTilemap

  ldde 9,1
  ld hl,BigText_Retire
  call DrawBigText

  ld a,$8e
  ld (TrackNumber),a
  call PlayMusic

  ; wait a bit
  ld b,60*2
-:halt
  djnz -

  call FadeOutFullPalette
  call TurnOffScreen

  jp TitleScreen

_ExitPauseState:
  call RestoreTilemap
  call PlayMusic
  jp ContinueGameMainLoop ; will fade palette back in

.ends

.section "Title screen" free
TitleScreen:
  di

  ; load graphics
  ld hl,GameTilePalette
  ld de,TargetPalette
  ld bc,16
  ldir
  ld hl,GameTilePalette
  ld bc,16
  ldir

  ld hl,GameTiles
  ld de,$4000 ; tile 0
  call LoadTiles4BitRLENoDI

  ld hl,TitleScreenTiles
  ld de,$4000 + 256 * 32 ; tile $100
  call LoadTiles4BitRLENoDI

  ld hl,TitleScreenTilemap
  ld de,TileMapAddress
  call LoadTilemapToVRAMNoDI

  ; draw the all-time clock (one-time since it's not increasing)
  call SRAMOn
  ld ix,TotalGameTime
  ld (ix+Time.Frames),1 ; need to kludge it to work
  call UpdateClock
  ldde 11,19
  call UpdateClockDisplay ; one-time
  call SRAMOffAndRet

  ld hl,TitleScreenVBlank
  ld (VBlank),hl

  ei

  call TurnOnScreen

  call FadeInFullPalette

  ld a,$90 ; short music
  ld (TrackNumber),a

-:halt
  ; do work here

  ; handle inputs
  ld a,(JustPressedButtons)
  and P1L
  call nz,_DecrementLevel
  ld a,(JustPressedButtons)
  and P1R
  call nz,_IncrementLevel
  ld a,(JustPressedButtons)
  and P11
  jr nz,_StartGame

  ld a,(CurrentlyPressedButtons)
  and P12
  call nz,_IncrementIfDone

  jr -

_IncrementIfDone:
  ld a,(LevelDone)
  or a
  ret z
  jr _IncrementLevel

_DecrementLevel:
  ld hl,(LevelNumber)
  dec hl
  ; need to manually check for carry
  ld a,h
  and l ; will be $ff only if hl=$ffff
  inc a
  ret z
  jr +

_IncrementLevel:
  ld hl,(LevelNumber)
  call GetMaxLevel
  dec de
  sbc hl,de
  ret nc    ; if this is true then level = max - 1
  adc hl,de ; else, carry is set so this will restore and also increment
  ; fall through

+:ld (LevelNumber),hl
  ret

_StartGame:
  ld a,$b5 ; sound to start
  ld (TrackNumber),a
  halt

  call FadeOutSound
  call FadeOutFullPalette
  call TurnOffScreen
  call StopSound
  halt
  ld hl,DoNothing
  ld (VBlank),hl

  jp GameMainLoop


TitleScreenVBlank:
  call UpdatePalette
  call UpdateLevelNumbers
  ; All these are OK in the display period
  call GetInputs
  call HandlePaletteFade
  call UpdateSound
  call UpdateNumberDisplayData
  ret
.ends

.section "Level number drawing" free
.define LevelChars 3 ; 3 digits
UpdateLevelNumbers:
  ; level number
  ldde 11,13
  call SetTilemapWriteAddressXYinDE
  ld hl,LevelNumberDisplay
  call _DoRow
  inc de
  call SetTilemapWriteAddressXYinDE
  call _DoRow

  ; draw the "done"
  ; skip 1 tile first
  xor a
  out (VDPData),a
  out (VDPData),a
  
  ld a,(LevelDone)
  or a
  jr nz,_leveldone

  xor a
  out (VDPData),a
  out (VDPData),a
  out (VDPData),a
  out (VDPData),a
  jr +

_leveldone:
  ld a,Tiles.Done
  out (VDPData),a
  xor a
  out (VDPData),a
  ld a,Tiles.Done+1
  out (VDPData),a
  xor a
  out (VDPData),a



+:; completed count
  ldde 11,16
  call SetTilemapWriteAddressXYinDE
  ld hl,CompletedCountDisplay
  call _DoRow
  inc de
  call SetTilemapWriteAddressXYinDE

  ; fall through for last row

_DoRow:
  ; copy to tilemap
+:ld b,LevelChars
-:ld a,(hl)
  inc hl
  out (VDPData),a
  xor a
  out (VDPData),a
  djnz -
  ret

UpdateNumberDisplayData:
  ld hl,(LevelNumber)
  inc hl ; make it 1-based
  ld iy,LevelNumberDisplay

  call +

  call GetCompletedCount; in hl
  ld iy,CompletedCountDisplay

  ; fall through and ret

+:
  ; will be at least one digit
  xor a
  ld (iy+1),a
  ld (iy+2),a
  ld (iy+3),a
  ld (iy+4),a
  ld (iy+5),a

  ; get hundreds (assume no thousands)
  ld de,-100
  ld a,-1
-:inc a
  add hl,de
  jr c,- ; will carry every time?
  sbc hl,de ; carry is zero
  ; a = hundreds
  ; hl = hl mod 100
  ld d,a ; remember to see if I need a zero tens digit
  or a
  call nz,_DoDigit

  ; get tens (8-bit time)
  ld a,l
  ld c,-1
-:inc c
  add a,-10
  jr c,- ; will carry every time?
  add a,10
  ld b,a ; backup 1s
  ld a,c
  or d ; see if either non-zero or there was a hundreds digit
  or a
  ld a,c
  call nz,_DoDigit

  ld a,b ; always draw 1s
  call _DoDigit

  ; find whether or not his level has been "done"
  call IsLevelDone
  jr nz,_done
_notdone:
  xor a
  jr +
_done:
  ld a,1
+:ld (LevelDone),a
  ret

_DoDigit:
  ; iy = RAM location
  ; a = number
  push hl
    ld hl,DigitTiles
    add a,a
    AddHLA
    ld a,(hl)
    ld (iy+0),a
    inc hl
    ld a,(hl)
    ld (iy+LevelChars),a
  pop hl
  inc iy
  ret

.ends

.section "Screen control" free
TurnOffScreen:
  ld a,%10100100
    ;    |||| |`- Zoomed sprites -> 16x16 pixels
    ;    |||| `-- Doubled sprites -> 2 tiles per sprite, 8x16
    ;    |||`---- 30 row/240 line mode
    ;    ||`----- 28 row/224 line mode
    ;    |`------ Enable VBlank interrupts
    ;    `------- Enable display
  jr +
TurnOnScreen:
  ; turn on screen
  ld a,%11100100
    ;    |||| |`- Zoomed sprites -> 16x16 pixels
    ;    |||| `-- Doubled sprites -> 2 tiles per sprite, 8x16
    ;    |||`---- 30 row/240 line mode
    ;    ||`----- 28 row/224 line mode
    ;    |`------ Enable VBlank interrupts
    ;    `------- Enable display
+:out (VDPStatus),a
  ld a,$81
  out (VDPStatus),a
  ret
.ends

.section "VBlank handlers" free
GameVBlank:
  call UpdatePalette
  call UpdateSpriteTable
  ldde 1,1
  call UpdateClockDisplaySmall
  call UpdateChangedPixel
  ; All these are OK in the display period
  call GetInputs
  call UpdateTimer
  ld ix,CurrentGameTime
  call UpdateClock
  call HandleGameInputs
  call HandlePaletteFade
  call UpdateSound
  call CycleCursorColour
  ret

GamePauseVBlank:
  call UpdatePalette
  ; these are OK in the display period
  call GetInputs
  call HandlePaletteFade
  call UpdateSound
  ret
.ends

.section "Tilemap backup/restore" free
; screen should be off in here!
BackupTilemap:
  di
  ; set VDP for reading, not writing
  ld a,TileMapAddress & $ff
  out (VDPAddress),a
  ld a,(TileMapAddress & $3f00) >> 8
  out (VDPAddress),a

  ld hl,TileMapCopy
  ld c,VDPData
  ld b,0
  call WaitForVBlank
  inir ; read 256*7 times
  inir ; seems there's room for 512, not 768 in the VBlank
  call WaitForVBlank
  inir
  inir
  call WaitForVBlank
  inir
  inir
  call WaitForVBlank
  inir
  ei
  ret

WaitForVBlank:
-:in a,(VDPStatus)
  and %10000000
  jr z,-
  ret

RestoreTilemap:
  di
  ld hl,TileMapAddress
  call VRAMToHL
  ld hl,TileMapCopy
  ld c,VDPData
  call WaitForVBlank
  FastOtir 512
  call WaitForVBlank
  FastOtir 512
  call WaitForVBlank
  FastOtir 512
  call WaitForVBlank
  FastOtir 256
  ei
  ret
.ends

.section "Cursor palette cycling" free
.define CursorColour 27
.define FramesPerColour 3
CycleCursorColour:
  ld hl,CursorColourTimer
  dec (hl)
  ret p
  ld (hl),FramesPerColour

  ld a,(CursorColourIndex)
  inc a
  cp CursorColoursEnd-CursorColours
  jr nz,+
  xor a
+:ld (CursorColourIndex),a
  ld hl,CursorColours
  AddHLA
  ld a,(hl)
  ld hl,CurrentPalette+CursorColour
  ld (hl),a
  ret

CursorColours:
.db cl130
.db cl030
.db cl031
.db cl032
.db cl033
.db cl032
.db cl031
.db cl030
CursorColoursEnd:
.ends

.section "Palette fading" free
FadeInFullPalette:
  ld hl,$2089
  ld (PaletteFadeControl),hl ; PaletteFadeControl = fade in/counter=9; PaletteSize=32
  jr _DoFade

FadeOutFullPalette:
  ld hl,$2009
  ld (PaletteFadeControl),hl ; PaletteFadeControl = fade out/counter=9; PaletteSize=32
  ; fall through

_DoFade:
  halt
  ld a,(PaletteFadeControl)       ; wait for palette to fade out
  or a
  jp nz,_DoFade
  ret

HandlePaletteFade:
; stolen from Phantasy Star
; must run every VBlank
; Main function body only runs every 4 calls (using PaletteFadeFrameCounter as a counter)
; Checks PaletteFadeControl - bit 7 = fade in, rest = counter
; PaletteSize tells it how many palette entries to fade
; TargetPalette and ActualPalette are referred to
    ld hl,PaletteFadeFrameCounter ; Decrement PaletteFadeFrameCounter
    dec (hl)
    ret p              ; return if >=0
    ld (hl),3          ; otherwise set to 3 and continue (so only do this part every 4 calls)
    ld hl,PaletteFadeControl ; Check PaletteFadeControl
    ld a,(hl)
    bit 7,a            ; if bit 7 is set
    jp nz,_FadeIn      ; then fade in
    or a               ; If PaletteFadeControl==0
    ret z              ; then return
    dec (hl)           ; Otherwise, decrement PaletteFadeControl
    inc hl
    ld b,(hl)          ; PaletteSize
    ld hl,CurrentPalette
  -:call _FadeOut      ; process PaletteSize bytes from ActualPalette
    inc hl
    djnz -
    ret

_FadeOut:
    ld a,(hl)
    or a
    ret z              ; zero = black = no fade to do

    and %11<<0         ; check red
    jr z,+
    dec (hl)           ; If non-zero, decrement
    ret

  +:ld a,(hl)
    and %11<<2         ; check green
    jr z,+
    ld a,(hl)
    sub 1<<2           ; If non-zero, decrement
    ld (hl),a
    ret

  +:ld a,(hl)
    and %11<<4         ; check blue
    ret z
    sub 1<<4            ; If non-zero, decrement
    ld (hl),a
    ret

_FadeIn:
    cp $80             ; Is only bit 7 set?
    jr nz,+            ; If not, handle that
    ld (hl),$00        ; Otherwise, zero it (PaletteFadeControl)
    ret
  +:dec (hl)           ; Decrement it (PaletteFadeControl)
    inc hl
    ld b,(hl)          ; PaletteSize
    ld hl,TargetPalette
    ld de,CurrentPalette
  -:call _FadePaletteEntry ; fade PaletteSize bytes from ActualPalette
    inc hl
    inc de
    djnz -
    ret

_FadePaletteEntry:
    ld a,(de)          ; If (de)==(hl) then leave it
    cp (hl)
    ret z
    add a,%00010000    ; increment blue
    cp (hl)
    jr z,+
    jr nc,++           ; if it's too far then try green
  +:ld (de),a          ; else save that
    ret
 ++:ld a,(de)
    add a,%00000100    ; increment green
    cp (hl)
    jr z,+
    jr nc,++           ; if it's too far then try red
  +:ld (de),a          ; else save that
    ret
 ++:ex de,hl
    inc (hl)           ; increment red
    ex de,hl
    ret
.ends

.section "Update palette" free
UpdatePalette:
  ; Copy palette from RAM to VRAM
  ld hl,PaletteAddress
  call VRAMToHL
  ld hl,CurrentPalette
  ld c,VDPData
  FastOtir 32
  ret
.ends

.section "In-game inputs handler" free
.define RepeatInitialDelay 60/3 ; 0.33s
.define RepeatDelay 60/10       ; 10 per second

HandleGameInputs:
  ld hl,JustPressedButtons
  ld a,(hl)
  or a
  jr z,_NoNewButtons
  and P1U|P1D|P1L|P1R
  jp z,_NoDirection

_DirectionPressed:
  ; new direction pressed, init repeat counter
  ld a,RepeatInitialDelay
  ld (RepeatCounter),a
  ; and actually move
  call _ProcessMovement

_NoDirection:
  ld hl,JustPressedButtons
  ld a,(hl)
  and P11
  jr nz,_Button1
  ld a,(hl)
  and P12
  ret z ; strange...
_Button2:
  call GetBoardPixelXY
  ; store it
  ld (LastChangedFrom),a
  cp Tiles.Blank
  jr nz,_UnsetPixel
_DotPixel:
  ld a,$a5 ; short low beep
  ld (TrackNumber),a
  ld a,Tiles.Dotted
  jr +
_Button1:
  ; button 1 = set (if blank), unset (if dotted or set)
  ; look at what's at x,y
  call GetBoardPixelXY
  ; store it
  ld (LastChangedFrom),a
  cp Tiles.Blank
  jr nz,_UnsetPixel
_SetPixel:
  ld a,$a2 ; high beep
  ld (TrackNumber),a
  ld a,Tiles.Filled
  jr +
_UnsetPixel:
  ld a,$bb ; bang
  ld (TrackNumber),a
  ld a,Tiles.Blank
+:ld (LastChangedTo),a
  call SetBoardPixelXY
  ret

_NoNewButtons:
  ld hl,CurrentlyPressedButtons
  ld a,(hl)
  and P1U|P1D|P1L|P1R
  ret z ; don't care about held buttons without a direction

  ; I expect there to only be 1 direction pressed
  ; so I'll do nothing if that's not the case
  ld c,a
  ld b,4
  xor a
  ld d,a
-:srl c
  adc a,d
  djnz -
  dec a
  ret nz

  ; Next, decrement the counter
  ld hl,RepeatCounter
  dec (hl)
  ret nz ; and do nothing until it's zero

  ld (hl),RepeatDelay ; repopulate it with the new delay
  ; and treat the direction as a new movement
  ld hl,CurrentlyPressedButtons
  ; fall through to process it

_ProcessMovement:
  ld a,(hl)
  and P1U
  jr nz,_MoveUp
  ld a,(hl)
  and P1D
  jr nz,_MoveDown
  ld a,(hl)
  and P1L
  jr nz,_MoveLeft
  ld a,(hl)
  and P1R
  jr nz,_MoveRight
  ; don't care about buttons here
  ret

_MoveUp:
  ld a,(CursorBoardY)
  dec a
  ret m ; do nothing at top
  ld (CursorBoardY),a
  ld a,(CursorScreenY)
  sub 8
-:ld (CursorScreenY),a
  jr _end

_MoveDown:
  ld a,(CursorBoardY)
  ld hl,Height
  inc a
  cp (hl)
  ret z ; do nothing at bottom
  ld (CursorBoardY),a
  ld a,(CursorScreenY)
  add a,8
  jr -

_MoveLeft:
  ld a,(CursorBoardX)
  dec a
  ret m ; do nothing at top
  ld (CursorBoardX),a
  ld a,(CursorScreenX)
  sub 8
  jr +

_MoveRight:
  ld a,(CursorBoardX)
  ld hl,Width
  inc a
  cp (hl)
  ret z ; do nothing at bottom
  ld (CursorBoardX),a
  ld a,(CursorScreenX)
  add a,8
+:ld (CursorScreenX),a

_end:
  call UpdateCursorInSpriteTable

  ld a,(CurrentlyPressedButtons)
  ; expect only one of them
  and P11|P12
  ret z
  cp %11
  ret z

  call GetBoardPixelXY
  ; if (tile==Tiles.Blank)
  ; {
  ;   if (LastChangedFrom==Tiles.Blank)
  ;     tile=LastChangedTo;
  ; }
  ; else if (LastChangedTo=Tiles.Blank)
  ; {
  ;   tile=Tiles.Blank
  ; }
  cp Tiles.Blank
  jr nz,_NotBlank
  ld hl,LastChangedFrom
  cp (hl)
  ret nz
  ; change to LastChangedTo
  ld a,(LastChangedTo)
  cp Tiles.Filled
  jp z,_SetPixel
  jp _DotPixel

_NotBlank:
  ld a,(LastChangedTo)
  cp Tiles.Blank
  jp z,_UnsetPixel

  ret

.ends

.section "Update a single on-screen pixel" free
UpdateChangedPixel:
  ld de,(ChangedPixelXY)
  ld a,d
  or e
  ret z ; do nothing if x,y=0,0
  call SetTilemapWriteAddressXYinDE
  ld a,(ChangedPixelState)
  out (VDPData),a
  push af
    xor a
    out (VDPData),a
  pop af
  ld de,0
  ld (ChangedPixelXY),de
  ret
.ends

.section "Board pixel getters and setters" free
GetBoardPixelXY:
  ; get the board pixel at CursorBoardX, CursorBoardY
  ld a,(CursorBoardX)
  ld d,a
  ld a,(CursorBoardY)
  ld e,a
  ld hl,Puzzle
  ; fall through

;GetPixelDEinHL:
  call _GetPixelAddress
  ld a,(hl)
  ; clean up specialised pixels
;SixthColumn    dsb 3
;SixthRow       dsb 3
;SixthBoth      dsb 3
  cp Tiles.SixthBoth
  jr nc,_both
  cp Tiles.SixthRow
  jr nc,_row
  cp Tiles.SixthColumn
  jr nc,_col
  ret
_both:
  sub Tiles.SixthBoth-Tiles.Blank
  ret
_row:
  sub Tiles.SixthRow-Tiles.Blank
  ret
_col:
  sub Tiles.SixthColumn-Tiles.Blank
  ret

_GetPixelAddress:
  ; hl += d+e*(Width)
  ld a,(Width)
  ld b,0
  ld c,a
  inc e ; pre-increment so I cen test for zero to handle already-zero cases
-:dec e ; destructive!
  jr z,+
  add hl,bc
  jr -
+:ld c,d
  add hl,bc
  ret

SetBoardPixelXY:
; a = tile to change to
; X,Y are (CursorBoardX), (CursorBoardY)
  push af
    ; get the board pixel at CursorBoardX, CursorBoardY into de
    ld a,(CursorBoardX)
    ld d,a
    ld a,(CursorBoardY)
    ld e,a

    ld b,d ; copy into bc
    ld c,e
    call GetTileOffsetForXYinBC ; returns in a
    ld b,a
  pop af
  ld hl,Puzzle
  ; fall through

;SetPixelDEinHL:
  push de
  push bc
  push af
    call _GetPixelAddress
  pop af
  pop bc
  pop de
  ld (hl),a

  add a,b ; add offset onto tile here so it's only offset for display
  ld (ChangedPixelState),a
  ; need to offset x,y by left,top
  ld a,(Left)
  add a,d
  ld d,a
  ld a,(Top)
  add a,e
  ld e,a
  ld (ChangedPixelXY),de

  ret

.ends

.section "InitPuzzle" free
; initialise the Puzzle memory region as blank
InitPuzzle:
  ld bc,25*20
  ld hl,Puzzle
  ld de,Puzzle+1
  ld (hl),Tiles.Blank
  ldir
  ret
.ends

.section "Update clock" free
.define ClockChars 12
UpdateClockDisplay:
; draws clock to name table from RAM copy
; de = x,y

  call SetTilemapWriteAddressXYinDE
  ld hl,TimeDisplay
  call + ; do top row

  inc de ; easy to move down by 1
  call SetTilemapWriteAddressXYinDE
  ld hl,TimeDisplay+ClockChars
+:; fall through for second time
  ld b,ClockChars
-:ld a,(hl)
  out (VDPData),a
  xor a
  out (VDPData),a
  inc hl
  djnz -
  ret

UpdateClockDisplaySmall:
; draws clock to name table from RAM copy
; only draws 5 chars (mm:ss)
; de = x,y

  call SetTilemapWriteAddressXYinDE
  ld hl,TimeDisplay
  call + ; do top row

  inc e ; easy to move down by 1
  call SetTilemapWriteAddressXYinDE
  ld hl,TimeDisplay+ClockChars
+:; fall through for second time
  ld b,5
-:ld a,(hl)
  out (VDPData),a
  xor a
  out (VDPData),a
  inc hl
  djnz -
  ret

UpdateClock:
; draws clock name table data to RAM copy
; ix = Time struct to use
;  ld ix,CurrentGameTime
  ; do nothing if frames!=1
  ld a,(ix+Time.Frames)
  dec a
  ret nz

  ; blank out the RAM
  ld hl,TimeDisplay
  ld bc,ClockChars*2-2
  ld de,TimeDisplay+1
  ld (hl),0
  ldir

  ld iy,TimeDisplay ; iy points to the current RAM location
  ld d,0

  ; any days?
  ld a,(ix+Time.Days)
  or a
  jr z,_Hours
  cp 10
  jr c,_1DigitDays
  cp 100
  jr c,_2DigitDays
  ; fall through for 3 digits

_3DigitDays:
  ld b,0
-:sub 100
  jr c,+
  inc b
  jr -
+:add a,100
  ; now a = days mod 100, b = days div 100
  ld c,a
  ld a,b
  call _DoDigit
  ld a,c

_2DigitDays:
  ld b,0
-:sub 10
  jr c,+
  inc b
  jr -
+:add a,10
  ; now a = units, b = tens
  ld c,a
  ld a,b
  call _DoDigit
  ld a,c

_1DigitDays:
  call _DoDigit
  ld a,11 ; 'd'
  call _DoDigit

_Hours:
  ; from here on, all numbers are in BCD
  ld a,(ix+Time.Hours)
  cp $10
  jr nc,_2DigitHours ; both digits if hours > 10

;  ld a,d
;  or a
;  ld a,(ix+Time.Hours)
;  jr nz,_2DigitHours ; want to draw both digits if days > 0 ; no I don't

  jr _1DigitHours

_2DigitHours:
  ; get the tens
  ld b,a
  srl a
  srl a
  srl a
  srl a
  call _DoDigit
  ld a,b
_1DigitHours:
  and %00001111
  or a
  jr nz,_DrawHoursDigit ; draw hours units if non-zero, of course

  ld b,a
  ld a,d
  or a                  ; but also if we drew some days
  ld a,b
  jr nz,_DrawHoursDigit

  jr _Minutes

_DrawHoursDigit:
  call _DoDigit
  ld a,10 ; ':'
  call _DoDigit

_Minutes:
  ld a,(ix+Time.Minutes)
  cp $10
  jr nc,_2DigitMinutes ; draw 2 digits if we have that many
  ; but also draw 2 if we had hours or days
  ld a,d
  or a
  ld a,(ix+Time.Minutes)
  jr nz,_2DigitMinutes
  jr _1DigitMinutes

_2DigitMinutes:
  ; get the tens
  ld b,a
  srl a
  srl a
  srl a
  srl a
  call _DoDigit
  ld a,b
_1DigitMinutes:
  and %00001111
  call _DoDigit
  ld a,10 ; ':'
  call _DoDigit
  ld a,(ix+Time.Seconds)
  ; get the tens
  ld b,a
  srl a
  srl a
  srl a
  srl a
  call _DoDigit
  ld a,b
  and %00001111
  call _DoDigit

  ret

_DoDigit:
  ; iy = RAM location
  ; a = number
  push af
    ld hl,DigitTiles
    add a,a
    AddHLA
    ld a,(hl)
    ld (iy+0),a
    inc hl
    ld a,(hl)
    ld (iy+ClockChars),a
  pop af
  inc iy
  inc d ; Increment digits counter
  ret

DigitTiles:
.db Tiles.Digits+0, Tiles.Digits+9  ; 0
.db Tiles.Digits+1, Tiles.Digits+10 ; 1
.db Tiles.Digits+2, Tiles.Digits+11 ; 2
.db Tiles.Digits+2, Tiles.Digits+12 ; 3
.db Tiles.Digits+3, Tiles.Digits+13 ; 4
.db Tiles.Digits+4, Tiles.Digits+12 ; 5
.db Tiles.Digits+4, Tiles.Digits+14 ; 6
.db Tiles.Digits+5, Tiles.Digits+15 ; 7
.db Tiles.Digits+6, Tiles.Digits+14 ; 8
.db Tiles.Digits+6, Tiles.Digits+12 ; 9
.db Tiles.Digits+7, Tiles.Digits+16 ; :
.db Tiles.Digits+8, Tiles.Digits+14 ; d
.ends

.section "Update timer" free
UpdateTimer:
  ld a,(Paused)
  ret nz

  call SRAMOn
  ld ix,TotalGameTime
  call +
  call SRAMOffAndRet

  ld ix,CurrentGameTime
  ; fall through and ret
+:
  ld a,(ix+Time.Frames)
  inc a
  ld hl,FramesPerSecond
  cp (hl)
  ld (ix+Time.Frames),a
  ret nz
  xor a
  ld (ix+Time.Frames),a

  ld a,(ix+Time.Seconds)
  inc a
  daa
  cp $60
  ld (ix+Time.Seconds),a
  ret nz
  xor a
  ld (ix+Time.Seconds),a

  ld a,(ix+Time.Minutes)
  inc a
  daa
  cp $60
  ld (ix+Time.Minutes),a
  ret nz
  xor a
  ld (ix+Time.Minutes),a

  ld a,(ix+Time.Hours)
  inc a
  daa
  cp $24
  ld (ix+Time.Hours),a
  ret nz
  xor a
  ld (ix+Time.Hours),a

  ld a,(ix+Time.Days)
  inc a
  jr z,_overflow
  ld (ix+Time.Days),a
  ret

_overflow:
  ; could do something cute here
  ; eg. "You play this game too much"
  ; peg it at the maximum time
  ; days wasn't saved
  ld a,$23
  ld (ix+Time.Hours),a
  ld a,$59
  ld (ix+Time.Minutes),a
  ld (ix+Time.Seconds),a
  ret
.ends

.section "Get Inputs" free
GetInputs:
  in a,(IOPort1)
  cpl
  and P1U|P1D|P1L|P1R|P11|P12
  ld b,a             ; b = all buttons pressed
  ld hl,CurrentlyPressedButtons
  xor (hl)           ; xor with what was pressed already
  ld (hl),b
  and b              ; a = all buttons pressed since last time
  ld (JustPressedButtons),a
  in a,(IOPort2)
  cpl
  and ResetButton
  jp nz,0 ; always reset
  ret
.ends

.section "Update sprite table" free
UpdateSpriteTable:
  ; terminate table
  ld hl,SpriteTable
  ld a,(NumSprites)
  AddHLA
  ld (hl),208

  ld hl,SpriteTableAddress
  call VRAMToHL
  ld hl,SpriteTable
  ld c,VDPData
  FastOtir 256
  ret
.ends

.section "OUTI block" free
outiblock:
.rept FastOtirBlockSize
  outi
.endr
  ret
.ends

.section "Level loader" free
LoadLevel:
; params: hl = number
  add hl,hl
  ld de,Levels
  add hl,de
  ld a,(hl)
  inc hl
  ld h,(hl)
  ld l,a

  ; read in width and height
  ld a,(hl)
  ld (Width),a
  ld b,a
  inc hl
  ld a,(hl)
  ld (Height),a
  ld e,a

  push hl
    ; calculate amount to be read
    ; = ceil(b * e / 8)
    ld hl,0
    ld d,0
-:  add hl,de
    djnz - ; now hl  = a * b
    ld (PuzzleLength),hl
    ; rotate right by 3 to divide by 8
    ; only need to carry 1 bit over from h
    rr h
    rr l
    srl l
    srl l
    ld a,l ; save it in a for now
    inc a  ; add one for a ceil()-like  effect (it will be 1 too many sometimes, doesn't matter)

    ; zero the solution
    ld bc,25*20
    ld hl,Solution
    ld de,Solution+1
    ld (hl),0
    ldir
  pop hl

  ; now read data in and write it out to RAM unpacked
  ld de,Solution
  ld c,a ; packed byte count
--:
  inc hl
  ld b,8 ; bits to unpack
  ld a,(hl)
-:
  sla a
  jr nc,+
  ; output a 1
  ex de,hl
  ld (hl),1
  ex de,hl
+:inc de
  djnz - ; repeat for 8 bits
  ; repeat until we've unpacked enough
  dec c
  jr nz,--

  ret
.ends

.section "Draw tips" free
DrawTips:
; no params
.define HintTilesStart Tiles.HintNumbers ; index of 0 number tile

  ld hl,Solution
  ld a,(Height);
  ld ixh,a
--:
  ld a,(Width)
  ld b,a ; how many pixels to look over
  ld c,0 ; counter
  ld d,0 ; last seen (init to 0)
  ld e,0 ; how many I've pushed
  ; scan a row for 1s and count them, push the length the stack for each run found
-:ld a,(hl)
  or a
  jr z,_zero
_one:
  ; is it a new run?
  cp d
  jr z,+
  ld c,0 ; new run -> init counter to 0
  ld d,a ; remember for next time
+:inc c ; add one to counter
  jr _done
_zero:
  cp d
  jr z,+
  ; first 0 after some 1s
  ld d,a ; remember for next time
  ld a,c
  push af ; record count on stack
  inc e
+:; nothing to do here: 0 after 0

_done:
  inc hl ; increment pointer
  djnz -

  ; finished row, see if it finished on a black (-> not recorded yet)
  ld a,d
  or a
  jr z,+
  ; 1 -> record count
  ld a,c
  push af
  inc e

+:
  ; now I know there are e hints and they are all on the stack
  ; e -> b
  ld b,e
  ; so let's draw them
  ld a,(Left)
  ld d,a
  ld a,(Top)
  ld e,a
  ; de = xy

  ; move left a bit
  dec d

  ; select our row
  ; current row = height - ixh
  ld a,ixh
  ld c,a
  ld a,(Height)
  sub c
  ; add that on to the Y coordinate
  add a,e
  ld e,a

  ; loop over hints
  ; counter already in b
  ; need to handle no-hints
  ld a,b
  or a
  jr nz,+
  ; push a zero
  push af
  inc b
+:
-:
  call SetTilemapWriteAddressXYinDE
  pop af ; get hint from stack
  add a,HintTilesStart
  out (VDPData),a ; write number to screen
  xor a
  out (VDPData),a
  ; move left by 1
  dec d
  djnz -

  ; repeat for each row
  dec ixh
  jr nz,--

  ; now do the columns - a bit trickier ------------------------------------------------------

  ld hl,Solution
  ld a,(Width)
  ld ixh,a
--:
  ld a,(Height)
  ld b,a ; how many pixels to look over
  ld c,0 ; counter
  ld d,0 ; last seen (init to 0)
  ld e,0 ; how many I've pushed
  ; scan a column for 1s and count them, push the length the stack for each run found
-:ld a,(hl)
  or a
  jr z,_zero2
_one2:
  ; is it a new run?
  cp d
  jr z,+
  ld c,0 ; new run -> init counter to 0
  ld d,a ; remember for next time
+:inc c ; add one to counter
  jr _done2
_zero2:
  cp d
  jr z,+
  ; first 0 after some 1s
  ld d,a ; remember for next time
  ld a,c
  push af ; record count on stack
  inc e
+:; nothing to do here: 0 after 0

_done2:
  ; increment pointer
  ld a,(Width)
  AddHLA
  djnz -

  ; finished row, see if it finished on a black (-> not recorded yet)
  ld a,d
  or a
  jr z,+
  ; 1 -> record count
  ld a,c
  push af
  inc e

+:
  ; now I know there are e hints and they are all on the stack
  ; e -> b
  ld b,e
  ; so let's draw them
  ld a,(Left)
  ld d,a
  ld a,(Top)
  ld e,a
  ; de = xy

  ; move up a bit
  dec e

  ; select our column
  ; current col = width - ixh
  ld a,ixh
  ld c,a
  ld a,(Width)
  sub c
  ; add that on to the X coordinate
  add a,d
  ld d,a

  ; loop over hints
  ; counter already in b
  ; need to handle no-hints
  ld a,b
  or a
  jr nz,+
  ; push a zero
  push af
  inc b
+:
-:
  call SetTilemapWriteAddressXYinDE
  pop af ; get hint from stack
  add a,HintTilesStart
  out (VDPData),a ; write number to screen
  xor a
  out (VDPData),a
  ; move up by 1
  dec e
  djnz -

  ; repeat for each column
  ; need to fix up hl here
  push de
    ld hl,Solution
    ; current column number = width - ixh
    ; want hl = Solution + column number + 1
    ld e,ixh
    ld a,(Width)
    sub e
    inc a
    ld e,a
    ld d,0
    add hl,de
  pop de
  dec ixh
  jr nz,--

  ret
.ends

.section "Draw grid" free

DrawGrid:
; draw an empty grid to the screen
  ; calculate how big it needs to be
  ; depends on the tips length?
  call GetTipsSize ; returns in b,c

  ; X padding = (31 - width - 1 - tips)/2
  ; grid X = padding + tips + 1 (to avoid the left column)
  ;        = (30 - width + tips)/2 + 1
  ld a,(Width)
  ld d,a ; width
  ld a,30
  sub d
  add a,b ; tips
  srl a
  inc a
  ; phew!
  ld (Left),a

  ; Y padding = (24 - height - 1 - tips) / 2
  ; grid Y = padding + tips
  ;        = (23 - height + tips) / 2
  ld a,(Height)
  ld d,a ; height
  ld a,24
  sub d
  add a,c ; tips
  srl a
  ; and another one! wow
  ld (Top),a

  ; put x,y in d,e
  ld e,a
  ld a,(Left)
  ld d,a

  ld a,(Height)
  ld c,a
--:
  call SetTilemapWriteAddressXYinDE
  ; write (width) tiles
  ld a,(Width)
  ld b,a
-:call GetTileOffsetForXYinBC
  add a,Tiles.Blank
  out (VDPData),a
  xor a
  out (VDPData),a
  djnz -
  ; end of row, draw the right shadow
  ld a,Tiles.ShadowR
  out (VDPData),a
  xor a
  out (VDPData),a
  ; add 1 row
  inc e
  dec c
  jr nz,--

  ; draw the bottom shadow and the corners
  ld a,(Left)
  ld d,a
  ld a,(Top)
  ld e,a
  ld a,(Height)
  add a,e
  ld e,a
  call SetTilemapWriteAddressXYinDE
  ld a,Tiles.ShadowBL
  out (VDPData),a
  xor a
  out (VDPData),a

  ld a,(Width)
  dec a
  ld b,a
-:ld a,Tiles.ShadowB
  out (VDPData),a
  xor a
  out (VDPData),a
  djnz -

  ld a,Tiles.ShadowBR
  out (VDPData),a
  xor a
  out (VDPData),a

  ld a,(Left)
  ld d,a
  ld a,(Width)
  add a,d
  ld d,a
  ld a,(Top)
  ld e,a
  call SetTilemapWriteAddressXYinDE
  ld a,Tiles.ShadowTR
  out (VDPData),a
  xor a
  out (VDPData),a

  ret

GetTileOffsetForXYinBC:
; bc = x,y
; both %5=0 for special tile, except when ==width or height
; result = tile offset from regular tiles, ie. 0 if not one of the special ones
  push de
  push hl
    ld de,0 ; flags for special x,y
    ld a,b
    or a
    jr z,_specialx
-:  sub 5
    jr z,_specialx
    jr c,_nospecialx
    jr -
_specialx:
    ld d,%01
    ; fall through
_nospecialx:
    ld a,c
    or c
    jr z,_specialy
  -:sub 5
    jr z,_specialy
    jr c,_nospecialy
    jr -
_specialy:
    ld e,%10
    ; fall through
_nospecialy:
    ld a,d
    or e
  pop hl
  pop de
  cp %01
  jr z,_x
  cp %10
  jr z,_y
  cp %11
  jr z,_xy
  xor a ; default
  ret
_x:
  ld a,Tiles.SixthColumn-Tiles.Blank
  ret
_y:
  ld a,Tiles.SixthRow-Tiles.Blank
  ret
_xy:
  ld a,Tiles.SixthBoth-Tiles.Blank
  ret

SetTilemapWriteAddressXYinDE:
; sets the VRAM write address to tilemap location d,e
; clobbers: af
  push de
  push hl
    ld h,0
    ld l,e
    add hl,hl
    add hl,hl
    add hl,hl
    add hl,hl
    add hl,hl
    ld e,d
    ld d,0
    add hl,de
    add hl,hl
    ld de,TileMapAddress ; pre-ORed to be a write address
    add hl,de

    ld a,l
    out (VDPAddress),a
    ld a,h
    out (VDPAddress),a

  pop hl
  pop de
  ret

GetTipsSize:
  ; parses the solution and finds the longest number of tips needed in each direction
  ; returns horizontal in b, vertical in c
  ld hl,Solution
  ld a,(Height);
  ld ixh,a
  ld e,0 ; maximum seen so far
--:
  ld a,(Width)
  ld b,a ; how many pixels to look over
  ld c,0 ; counter
  ld d,0; last seen (init to 0)
  ; scan a row for the number of blocks of 1s = the number of white->black transitions
-:ld a,(hl)
  or a
  jr z,+ ; can't be anything if it's a zero
  cp d
  jr z,+
  inc c ; current is a 1 and old is a 0
+:ld d,a ; remember last seen
  inc hl
  djnz -
  ; c is now the count for this row
  ; see if it's the longest so far
  ld a,e
  cp c
  jr nc,+
  ; it is! remember this one then
  ld e,c
+:; repeat for each row
  dec ixh
  jr nz,--
  ; e is now the highest count seen
  ld b,a ; return it in b
  push bc
    ; now loop over y (a bit tougher!)
    ld hl,Solution
    ld a,(Width);
    ld e,a ; columns to loop over (ran out of normal registers)
    ld ixh,0 ; maximum seen so far
  --:
    ld a,(Height)
    ld b,a ; how many pixels to look over
    ld c,0 ; counter
    ld d,0; last seen (init to 0)
    ; scan a column for the number of blocks of 1s = the number of white->black transitions
  -:ld a,(hl)
    or a
    jr z,+ ; can't be anything if it's a zero
    cp d
    jr z,+
    inc c ; current is a 1 and old is a 0
  +:ld d,a ; remember last seen
    push de
      ; horribly unoptimised here
      ld d,0
      ld a,(Width)
      ld e,a
      add hl,de
    pop de
    djnz -
    ; c is now the count for this row
    ; see if it's the longest so far
    ld a,ixh
    cp c
    jr nc,+
    ; it is! remember this one then
    ld ixh,c
  +:; repeat for each row
    ; need to reset hl here
    ld a,(Width)
    sub e
    inc a
    ld hl,Solution
    AddHLA
    dec e
    jr nz,--
    ; ixh is now the highest count seen
    ld a,ixh
  pop bc
  ld c,a
  ret
.ends

.section "Cursor handling" free
.define CursorOffsetX 3
.define CursorOffsetY 4
InitialiseCursor:
  xor a
  ld (CursorBoardX),a
  ld (CursorBoardY),a
  ld a,(Top)
  sla a
  sla a
  sla a
  sub CursorOffsetY
  ld (CursorScreenY),a
  ld a,(Left)
  sla a
  sla a
  sla a
  sub CursorOffsetX
  ld (CursorScreenX),a
  ; fall through

AddCursorToSpriteTable:
  ; set it up in the sprite table
  ld a,(NumSprites)
  ld hl,SpriteTable
  AddHLA
  ld (CursorSpriteTableStart),hl
  ld a,(NumSprites)
  add a,12
  ld (NumSprites),a
  ; fall through

UpdateCursorInSpriteTable:
  ld ix,(CursorSpriteTableStart)
  ; Y coordinates:
  ;            +8+9
  ;            +a+b top
  ;
  ;
  ;  +4+5      +0+1
  ;  +6+7      +2+3 main
  ;  left
  ld a,(CursorScreenY)
  ; main
  ld (ix+0),a
  ld (ix+1),a
  ; left
  ld (ix+4),a
  ld (ix+5),a
  add a,8
  ; main
  ld (ix+2),a
  ld (ix+3),a
  ; left
  ld (ix+6),a
  ld (ix+7),a
  ; top
  ld a,(Top)
  sla a
  sla a
  sla a
  sub CursorOffsetY+10
  ld (ix+8),a
  ld (ix+9),a
  add a,8
  ld (ix+10),a
  ld (ix+11),a


  ; X coordinates:
  ;              +16 +18
  ;              +20 +22 top
  ;
  ;
  ;  + 8 +10     + 0 + 2
  ;  +12 +14     + 4 + 6 main
  ;  left
  ld bc,128
  add ix,bc
  ld a,(CursorScreenX)
  ; main
  ld (ix+0),a
  ld (ix+4),a
  ; top
  ld (ix+16),a
  ld (ix+20),a
  add a,8
  ; main
  ld (ix+2),a
  ld (ix+6),a
  ; top
  ld (ix+18),a
  ld (ix+22),a
  ; left
  ld a,(Left)
  sla a
  sla a
  sla a
  sub CursorOffsetY+9
  ld (ix+8),a
  ld (ix+12),a
  add a,8
  ld (ix+10),a
  ld (ix+14),a

  ; Tiles:
  ;              +17 +19
  ;              +21 +23 top
  ;
  ;
  ;  + 9 +11     + 1 + 3
  ;  +13 +15     + 5 + 7 main
  ;  left
  ; easy!
  ld a,Tiles.Cursors
  ld b,12
  dec ix
-:inc ix
  inc ix
  ld (ix+0),a
  inc a
  djnz -
  ret
.ends

.section "Music engine helpers" free
UpdateSound:
  ld a,(Frame2Paging)
  push af
    ld a,(SRAMPaging)
    push af
      ld a,SRAMPagingOff
      ld (SRAMPaging),a
      ld a,:MusicEngine
      ld (Frame2Paging),a
      call MusicEngine
    pop af
    ld (SRAMPaging),a
  pop af
  ld (Frame2Paging),a
  ret

StopSound:
  ld a,$d1
  ld (TrackNumber),a
  ret

FadeOutSound: ; and stop?
  ld a,$d0
  ld (TrackNumber),a
  ret

PauseMusic:
  ld a,$80
  jr +
PlayMusic:
  xor a
+:ld (PlayControl),a
  ret
  
LookupAndPlayTrack:
  call SRAMOn
  ld a,(LastTrackNumber)
-:ld hl,Tracks
  AddHLA
  ld a,(hl)
  ld (TrackNumber),a
  jp SRAMOffAndRet

NextTrack:
  call SRAMOn
  ld a,(LastTrackNumber)
  inc a
  cp TracksEnd-Tracks
  jr nz,+
  xor a
+:ld (LastTrackNumber),a
  jr -

Tracks:
; looping tracks, longest first
.db $97,$82,$95,$81,$88,$93,$8f,$87,$91,$92,$8c,$8b,$8a,$89,$83,$94,$86,$85,$84
TracksEnd:

.ends

.section "Graphics" free
GameTiles:
.incbin "gametiles.pscompr"
GameTilePalette:
.db $00 $15 $2A $3F $03 $07 $0B $24 $34 $10 $20
CloudTiles:
.incbin "Clouds.pscompr"
MediumCloud:
.db 8,5
.dw $0000,$0101,$0102,$0103,$0104,$0105,$0106,$0107
.dw $0108,$0109,$010A,$010A,$010A,$010A,$010B,$010C
.dw $010D,$010E,$010A,$010A,$010A,$010F,$0110,$0111
.dw $0112,$0113,$0114,$0115,$0116,$0117,$0118,$0119
.dw $0000,$011A,$011B,$011C,$011D,$011E,$011F,$0000
SmallCloud:
.db 6,4
.dw $0120,$0121,$0122,$0123,$0124,$0000
.dw $0125,$030B,$010A,$010A,$0126,$0127
.dw $0128,$0129,$012A,$012B,$012C,$012D
.dw $012E,$012F,$0130,$0131,$0132,$0133
BigCloud:
.db 15,7
.dw $0000,$0000,$0000,$0134,$0135,$0136,$0137,$0138,$0139,$013A,$013B,$013C,$0000,$0000,$0000
.dw $013D,$013E,$013F,$0140,$010A,$010A,$010A,$010A,$010A,$010B,$0141,$0142,$0143,$0144,$0000
.dw $0145,$010A,$010A,$010A,$010A,$010A,$010A,$010A,$010A,$010A,$010A,$010A,$050F,$0146,$0147
.dw $0148,$010A,$010A,$010A,$010A,$010A,$010A,$010A,$010A,$010A,$010A,$010A,$010A,$0149,$014A
.dw $014B,$014C,$014D,$014E,$070B,$010A,$010A,$010A,$010A,$010A,$010A,$014F,$0150,$0151,$0152
.dw $0153,$0154,$0155,$0156,$0157,$0158,$0159,$015A,$015B,$015C,$015D,$015E,$015F,$0160,$0161
.dw $0000,$0162,$0163,$0164,$0165,$0166,$0167,$0168,$0169,$016A,$016B,$016C,$016D,$0000,$0000

TitleScreenTiles:
.incbin "titlescreen.pscompr"
TitleScreenTilemap:
.incbin "titlescreen tilemap.pscompr"
.ends

.section "Nintendon't" free
Nintendont:
  ; screen is already off, palette is black
  di

  call TurnOffScreen

  ld hl,_Tiles
  ld de,$4000 ; tile 0
  call LoadTiles4BitRLENoDI

  ld hl,_Tilemap
  ld de,TileMapAddress
  call LoadTilemapToVRAMNoDI

  ld hl,_Palette
  ld de,TargetPalette
  ld bc,32
  ldir

  ld hl,NintendontVBlank
  ld (VBlank),hl

  ld a,110
  ld (VScroll),a

  ei

  call TurnOnScreen

  call FadeInFullPalette

  ; scroll down
-:ld a,(VScroll)
  dec a
  ld (VScroll),a
  or a
  jr z,+
  halt
  jr -
+:
  ; play a sound
  ld a,$A1 ; nasty sound
  ld (TrackNumber),a

  ; pause a short time
  ld b,60
-:halt
  djnz -

  call FadeOutFullPalette

  call TurnOffScreen

  ld hl,DoNothing
  ld (VBlank),hl

  ret

NintendontVBlank:
  call UpdatePalette

  ld a,(VScroll)
  out (VDPStatus),a
  ld a,$89 ; VScroll
  out (VDPStatus),a

  ; ---- VRAM accessing finished ----
  call HandlePaletteFade
  call UpdateSound
  ret

_Tiles:
.incbin "Nintendon't.pscompr"
_Tilemap:
.incbin "Nintendon't tilemap.pscompr"
_Palette:
.db $04 $2F $2F $2F $2F $2F $2F $2F $2F $2F $2F $2F $2F $2F $2F $2F
.db $2F $2F $2F $2F $2F $2F $2F $2F $2F $2F $2F $2F $2F $2F $2F $2F
.ends

.section "Board testing" free
IsBoardCorrect:
  ; return carry if puzzle has been completed
  ld hl,Solution
  ld de,Puzzle
  ld bc,(PuzzleLength)
-:ld a,(hl)
  or a
  jr z,_zero
_one:
  ; if it'a a 1 then we want a Tiles.Filled
  ld a,(de)
  cp Tiles.Filled
  jr nz,_failed
  jr _loop

_zero:
  ; if it's a 0 then we want a Tiles.Blank or Tiles.Dotted
  ld a,(de)
  cp Tiles.Blank
  jr z,_loop       ; correct, loop
  cp Tiles.Dotted
  jr nz,_failed

  ; fall through

_loop:
  inc hl
  inc de
  dec bc
  ld a,b
  or c
  jr nz,-
  ; fall through for success
_success:
  scf
  ret

_failed:
  xor a ; clear carry flag, also clear a in case I want to look at that instead
  ret
.ends

.section "Big text drawing" free
DrawBigText:
; hl = tilemap data, preceded by width, height bytes
; d,e = x,y
  ld a,(hl)
  ld b,a ; width
  inc hl
  ld a,(hl)
  ld c,a ; height
  inc hl

  call WaitForVBlank
--:
  ld a,c
  and 1
  call nz,WaitForVBlank ; draw 2 lines per frame
  push bc
  push de
-:call SetTilemapWriteAddressXYinDE
  push de
    ld a,(hl)
    inc hl
    ld d,a
    ld e,(hl)
    inc hl
    ; de = tilemap data
    or e
    jr z,_loopend

    ; draw tile
    ld a,d
    out (VDPData),a
    ld a,e
    out (VDPData),a

_loopend:
  pop de
  inc d
  djnz -
  pop de
  pop bc

  ; move to next row
  inc e
  dec c
  jr nz,--
  ret

/* Old version:
  call SetTilemapWriteAddressXYinDE
  ld a,(hl)
  ld b,a ; width
  inc hl
  ld a,(hl)
  ld c,a ; height
  inc hl

--:
  push bc
-:ld a,(hl)
  inc hl
  ld d,a
  ld e,(hl)
  inc hl
  ; de = word
  or e
  jr nz,+
  ; need to skip 2 bytes of VRAM
  in a,(VDPData)
  in a,(VDPData)
  jr _loopend

+:; draw tile
  ld a,d
  out (VDPData),a
  ld a,e
  out (VDPData),a

_loopend:
  djnz -
  pop bc
  ; move to next row
  ; read 32-b bytes
  ld a,32
  sub b
  push bc
    ld b,a
  -:in a,(VDPData)
    in a,(VDPData)
    djnz -
  pop bc
  dec c
  jr nz,--
  ret
*/

BigText_Complete:
.db 21,5 ; w,h
.dw $0000,$0000,$0000,$0000,$0000,$0000,$0000,$0000,$0000,$0000,$0000,$0000,$004A,$0000,$0000,$0000,$004A,$0000,$0000,$0000,$0000
.dw $004B,$004C,$004D,$004B,$004C,$024B,$004B,$004E,$024B,$004B,$004C,$024B,$004F,$004B,$004C,$024B,$0050,$004D,$004B,$004C,$024B
.dw $004F,$0000,$0000,$004F,$0000,$004F,$004F,$004F,$004F,$004F,$0000,$004F,$004F,$0050,$004C,$064B,$004F,$0000,$0050,$004C,$064B
.dw $044B,$004C,$004D,$044B,$004C,$064B,$044A,$044A,$044A,$0050,$004C,$064B,$044A,$044B,$004C,$004D,$044B,$004D,$044B,$004C,$004D
.dw $0000,$0000,$0000,$0000,$0000,$0000,$0000,$0000,$0000,$044A,$0000,$0000,$0000,$0000,$0000,$0000,$0000,$0000,$0000,$0000,$0000

BigText_Retire:
.db 13,4
.dw $0000,$0000,$0000,$0000,$0000,$004A,$0000,$0051,$0000,$0000,$0000,$0000,$0000
.dw $004B,$004D,$004B,$004C,$024B,$0050,$004D,$004A,$004B,$004D,$004B,$004C,$024B
.dw $004F,$0000,$0050,$004C,$064B,$004F,$0000,$004F,$004F,$0000,$0050,$004C,$064B
.dw $044A,$0000,$044B,$004C,$004D,$044B,$004D,$044A,$044A,$0000,$044B,$004C,$004D

BigText_BGM:
.db 9,5
.dw $004A,$0000,$0000,$0000,$0000,$0000,$0000,$0000,$0000
.dw $0050,$004C,$024B,$004B,$004C,$024B,$004B,$004E,$024B
.dw $004F,$0000,$004F,$004F,$0000,$004F,$004F,$004F,$004F
.dw $044B,$004C,$064B,$044B,$004C,$0250,$044A,$044A,$044A
.dw $0000,$0000,$0000,$024D,$004C,$064B,$0000,$0000,$0000

.ends

.section "Level data" free
.incdir "boards"
.include "levels.inc"
.incdir ""
.ends

.section "SRAM handlers" free
GetCompletedCount:
  call SRAMOn
  ld hl,(CompletedCount)
  jp SRAMOffAndRet

GetMaxLevel:
  call SRAMOn
  ld de,(MaxLevel)
  jp SRAMOffAndRet

UpdateSRAMStats:
  ; sets the completed bit
  ; if it was previously zero, increments the completed count
  call SRAMOn

  call _GetLevelFlag
  jp nz,SRAMOffAndRet ; nothing to do if 1
  ld a,1
  ld (hl),a ; set completed flag
  ld hl,CompletedCount
  inc (hl) ; increment completed count

  ; increase MaxLevel by 10 whenever Completed is within 5
  ld hl,(MaxLevel)
  ld de,(CompletedCount)
  or a ; reset carry
  sbc hl,de ; now hl = completed (eg. 5) - max (eg. 10)
  ld a,h    ; h should be zero then, sanity check
  or a
  jr nz,SRAMOffAndRet
  ld a,l    ; check the value
  cp 5
  jr nc,SRAMOffAndRet
  ld hl,(MaxLevel) ; actually do it
  ld de,10
  add hl,de
  ; check it does not exceed NumberOfLevels
  ld de,NumberOfLevels
  or a ; clear carry
  sbc hl,de
  jr c,+ ; no carry -> max >= number, carry -> max < number
  ld hl,NumberOfLevels
  jr ++
+:add hl,de ; restore it
++:
  ld (MaxLevel),hl

  jr SRAMOffAndRet

_GetLevelFlag:
  ld de,(LevelNumber)
  ld hl,CompletedLevels
  add hl,de
  ld a,(hl) ; get bit
  or a
  ret

IsLevelDone:
  call SRAMOn
  call _GetLevelFlag
  jr SRAMOffAndRet


InitialiseSRAM:
  call SRAMOn
  ; check for marker

  ld hl,ROMMarker
  ld de,Marker
  ld b,16
-:ld a,(de)
  cp (hl)
  jr nz,_failed
  inc hl
  inc de
  djnz -
  jr SRAMOffAndRet

_failed:
  ; initialise SRAM
  ; blank it all first
  xor a
  ld bc,8*1024
  ld hl,$8000
  ld de,$8001
  ld (hl),a
  ldir

  ; write in the marker
  ld hl,ROMMarker
  ld de,Marker
  ld bc,16
  ldir
  ; init counters
  ld hl,0
  ld (CompletedCount),hl
  ld hl,10
  ld (MaxLevel),hl

SRAMOffAndRet:
  ld a,SRAMPagingOff
  ld (SRAMPaging),a
  ret

SRAMOn:
  ld a,SRAMPagingOn
  ld (SRAMPaging),a
  ret

ROMMarker:
;    0123465789abcdef
.db "Picross by Maxim"
.ends

.section "Draw random clouds" free
; going to be tricky...
DrawRandomClouds:
  call BackupTilemap ; easier to deal with it in RAM

  ; draw over the timer space because it has the capability to grow (unlikely, but...)
  ld hl,TileMapCopy+(32*1+1)*2
  ld de,TileMapCopy+(32*1+1)*2+1
  ld (hl),1
  ld bc,5*2-1
  ldir
  ld hl,TileMapCopy+(32*2+1)*2
  ld de,TileMapCopy+(32*2+1)*2+1
  ld (hl),1
  ld bc,5*2-1
  ldir
  ; the timer drawing code will not let these ever be seen

  ld bc,0 ; b = clouds drawn, c = attempts
_cloudloop:
  push bc

    ; choose a random cloud
  -:call GetRandomNumber
    and %11 ; limit to 0..3
    cp 2+1    ; highest I can actually accept, +1
    jr nc,- ; loop until it fits

    ; look it up
    ld hl,Clouds
    add a,a
    AddHLA
    ld a,(hl)
    inc hl
    ld h,(hl)
    ld l,a
    ; now hl points to the data
    ; try not to mess with it
    push hl

      push hl
      pop ix

      ld b,(ix+0) ; width
      ld c,(ix+1) ; height
      inc ix
      inc ix

      ; choose a random x,y position
      ; x = -(width-1)..32+(width-1)
      ;   = 1-width..31+width
      ;   = 0..30+2*width - width - 1
      ld a,30
      add a,b
      add a,b
      ld d,a
    -:call GetRandomNumber
      and %111111 ; limit to 0..63
      cp d
      jr nc,-
      sub b
      dec a
      ld d,a

      ; y = 0..24-height (no clouds allowed to be cut off vertically)
      ld a,24
      sub c
      inc a
      ld e,a
    -:call GetRandomNumber
      and %11111 ; limit to 0..31
      cp e
      jr nc,- ; loop until it's small enough
      ld e,a

      ; now d,e is an x,y position on the tilemap
      ; d could be negative! I haven't coded for that yet
      ; b,c are still width,height

      push de
  --:   ; for each row
        push bc
        push de
          call GetTilemapCopyAddressXYinDE
        pop de
        push de
  -:      ; for each column
          call _IsOnscreenX
          jr nz,+      ; ignore off-screen tiles
          ld a,(ix+0) ; read the tile
          or (ix+1)   ; see if it's zero
          jr z,+      ; if so, nothing to check
          ; else, see if there's anything in the tilemap there
          ld a,(hl)
          inc hl
          or (hl)     ; is it zero?
          jr nz,_collision
          dec hl
  +:      inc ix      ; move src on by 2 bytes
          inc ix
          inc hl      ; move dest on too
          inc hl
          inc d       ; x++
          djnz -      ; repeat b times (1 row)
        pop de
        pop bc

        inc e         ; y++

        dec c
        jr nz,--      ; repeat for all rows
      pop de
    pop hl

    ; must have been successful
_foundaplace:

    ; occlusion test passed, draw to tilemap in RAM
    ; hl = data
    ; de = x,y
    ld b,(hl) ; width
    inc hl
    ld c,(hl) ; height
    inc hl
    push hl
    pop ix

  --: ; for each row
    push bc
    push de
      call GetTilemapCopyAddressXYinDE
    pop de
    push de
  -:  call _IsOnscreenX
      jr nz,+
      ; for each column
      ld a,(ix+0) ; read the tile
      or (ix+1)   ; see if it's zero
      jr z,+      ; if so, nothing to draw
      ; else, copy to the tilemap
      ld a,(ix+0)
      ld (hl),a
      ld a,(ix+1)
      inc hl
      ld (hl),a
      dec hl
  +:  inc ix      ; move src on by 2 bytes
      inc ix
      inc hl      ; move dest on too
      inc hl
      inc d       ; x++
      djnz -      ; repeat b times (1 row)
    pop de
    pop bc
    inc e         ; y++
    dec c
    jr nz,--      ; repeat for all rows

  pop bc

  inc b ; drew one, no failure

  ; fall through
_cloudloopend:

  ld a,c ; how many failures?
  cp 100 ; too many - give up
  jr nc,+

  ld a,b ; how many have we drawn?
  cp 8 ; I want 8
  jp nz,_cloudloop ; keep trying if I haven't got them

+:call RestoreTilemap

  ret

_collision:
  ; clean up the stack from the jump point
        pop de
        pop bc
      pop de
    pop hl
  pop bc

  inc c ; a failure

  jr _cloudloopend

_IsOnscreenX:
  ; returns z flag set is d>=0 and d<=32
  ld a,d
  or a
  jp m,_failed
  cp 32
  jr nc,_failed
  ; passed
_passed:
  xor a ; set zero flag
  ret
_failed:
  xor a
  inc a ; clear zero flag
  ret

GetTilemapCopyAddressXYinDE:
  ; need to handle negative x
  ld a,d
  or a
  jp p,+ ; positive -> as normal
  ld d,0 ; go to 0,y
  call + ; get the address
  ; then subtract the original -x, *2
  add a,a ; now a = -2x, eg. $fe for -1
  add a,l ; add l to make l-2n
  ld l,a
  ret c   ; it will carry if the result is positive
  dec h
  ret

+:ld h,0
  ld l,e
  add hl,hl ; first get e*32
  add hl,hl
  add hl,hl
  add hl,hl
  add hl,hl
  ld e,d    ; add d
  ld d,0
  add hl,de
  add hl,hl ; double
  ld de,TileMapCopy
  add hl,de ; add to base address
  ret

Clouds:
.dw BigCloud,MediumCloud,SmallCloud

GetRandomNumber:
  ; Uses a 16-bit RAM variable called RandomNumberGeneratorWord
  ; Returns an 8-bit pseudo-random number in a
  push hl
    ld hl,(RandomNumberGeneratorWord)
    ld a,h         ; get high byte
    rrca           ; rotate right by 2
    rrca
    xor h          ; xor with original
    rrca           ; rotate right by 1
    xor l          ; xor with low byte
    rrca           ; rotate right by 4
    rrca
    rrca
    rrca
    xor l          ; xor again
    rra            ; rotate right by 1 through carry
    adc hl,hl      ; add RandomNumberGeneratorWord to itself
    jr nz,+
    ld hl,$733c    ; if last xor resulted in zero then re-seed random number generator
+:  ld a,r         ; r = refresh register = semi-random number
    xor l          ; xor with l which is fairly random
    ld (RandomNumberGeneratorWord),hl
  pop hl
  ret                ; return random number in a
.ends

.section "Blank Dots" free
BlankDots:
  ; examine tilemap for Dotted tiles, turn them to Blank
  call BackupTilemap

  ld ix,TileMapCopy
  ld bc,32*24
-:ld a,(ix+0) ; check low byte
  cp Tiles.Dotted
  jr z,+
  cp Tiles.Dotted-Tiles.Blank+Tiles.SixthRow
  jr z,+
  cp Tiles.Dotted-Tiles.Blank+Tiles.SixthColumn
  jr z,+
  cp Tiles.Dotted-Tiles.Blank+Tiles.SixthBoth
  jr z,+
  jr ++
+:ld a,(ix+1) ; check high byte is zero
  or a
  jr nz,++
  ld a,Tiles.Blank
  ld (ix+0),a
++:
  inc ix
  inc ix
  dec bc
  ld a,b
  or c
  jr nz,-

  call RestoreTilemap
  ret
.ends

.section "SDSC tag comments" free
Comments:
.db "SMS Power! 10th Birthday Coding Competition entry.",13
.db "Heavily based on Picross DS.",13
.db "Thanks to Martin Konrad for his help entering the level data.",0
.ends

.bank 2 slot 2
.org 0
MusicEngine:
.incbin "Patched music engine.bin"
.define TrackNumber $c108 ; write following bytes
.define FirstMusicTrack $81
.define LastMusicTrack $97 ; !
.define FirstSFX $98
.define LastSFX $c1 ; again, !
.define PlayControl $c10a ; write $80 to pause, 0 to play

