#include <stdbool.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include "SMSlib.h"

#include "patterns.h"

typedef struct Sprite_t {
    uint8_t x;
    uint8_t y;
    uint8_t id;
    uint8_t sprite;
} Sprite;

typedef struct Animal_t {
    uint8_t type;
    uint8_t x;
    uint8_t y;
    int8_t partial_x; /* Stored as 1/16 pixels */
    int8_t partial_y;
    int8_t velocity_x;
    int8_t velocity_y;
    uint8_t id;
    uint8_t sprite;
    bool sleep;
    bool alive;
} Animal;

/* TODO: The code ran faster when this was just an aray of uint8_t. If we need more speed,
 *       remove these structs and manually index the array instead */
typedef struct Segment_H_t {
    uint8_t y;
    uint8_t x_from;
    uint8_t x_to;
} Segment_H;

typedef struct Segment_V_t {
    uint8_t x;
    uint8_t y_from;
    uint8_t y_to;
} Segment_V;


/****************************
 *        Constants         *
 ****************************/
#define PLAYER_SPEED 1

#define DIRECTION_NONE  0
#define DIRECTION_UP    1
#define DIRECTION_DOWN  2
#define DIRECTION_LEFT  3
#define DIRECTION_RIGHT 4


/****************************
 *    External Functions    *
 ****************************/

/* bitmap2.c */
#define SET_PIXEL_FLUSH 0xff
void bitmap2_init (void);
void bitmap2_clear (void);
void bitmap2_set_palette (uint8_t *palette);
void bitmap2_set_pixel (uint8_t x, uint8_t y, uint8_t value);
void bitmap2_set_line (uint8_t x, uint8_t y, uint16_t value);

/****************************
 *       Global State       *
 ****************************/

uint32_t score = 0;
Animal animal [32];

/* A list of completed fence segments. */
Segment_H completed_fence_h_list [128];
Segment_V completed_fence_v_list [128];

uint8_t fence_direction;
uint8_t fence_start_x;
uint8_t fence_start_y;
uint8_t fence_last_x;
uint8_t fence_last_y;

/* Store the current-paddock polygon as a set of vertical fence segments. */
Segment_V polygon [32];

/****************************
 *           Code           *
 ****************************/

/* TODO: Is this slow enough to make a stored index worth it? */
void log_segment_h (int y, int x_from, int x_to)
{
    int i = 0;

    /* Make sure we store the lower value first */
    if (x_from > x_to)
    {
        int t = x_to;
        x_to = x_from;
        x_from = t;
    }

    /* Find an empty slot */
    for (i = 0; completed_fence_h_list[i].y != 0; i++);

    completed_fence_h_list[i].y      = y;
    completed_fence_h_list[i].x_from = x_from;
    completed_fence_h_list[i].x_to   = x_to;
}

void log_segment_v (int x, int y_from, int y_to)
{
    int i = 0;

    /* Make sure we store the lower value first */
    if (y_from > y_to)
    {
        int t = y_to;
        y_to = y_from;
        y_from = t;
    }

    /* Find an empty slot */
    for (i = 0; completed_fence_v_list[i].x != 0; i++);

    completed_fence_v_list[i].x      = x;
    completed_fence_v_list[i].y_from = y_from;
    completed_fence_v_list[i].y_to   = y_to;
}

void fence_segment_complete ()
{
    if (fence_direction == DIRECTION_UP ||
        fence_direction == DIRECTION_DOWN)
    {
        log_segment_v (fence_last_x, fence_start_y, fence_last_y);
    }
    if (fence_direction == DIRECTION_LEFT ||
        fence_direction == DIRECTION_RIGHT)
    {
        log_segment_h (fence_last_y, fence_start_x, fence_last_x);
    }

    fence_direction = DIRECTION_NONE;
}

void fence_from (uint8_t from_x, uint8_t from_y)
{
    fence_direction = DIRECTION_NONE;
    fence_last_x = fence_start_x = from_x;
    fence_last_y = fence_start_y = from_y;
}

/* Ideas on fence-completion:
 *  -> When a fence segment completes, run a fill-algorithm on the completed fence
 *     data, starting from either side of the fence completion point.
 *  -> If the filling area contains only wah or only sneps, consider this part solved.
 *     -> Sneps and wah sleep.
 *     -> Area is shaded.
 *     -> Some kind of marking to prevent new fences being constructed?
 *
 *  -> If the area contains a mix of wah and sneps, nothing new happens.
 */

/* Idea: A limited amount of fence that the user has to solve the level */

/* Draw a segment of fence.
 * Only one axis should be changed per call. */
/* TODO: Disallow a new fence connecting to itself */
void fence_to (uint8_t to_x, uint8_t to_y)
{
    /* Early return if we've not moved since the last call */
    if (to_x == fence_last_x &&
        to_y == fence_last_y)
    {
        return;
    }

    if (to_y == fence_last_y) /* Draw horizontal fence */
    {
        /* Log the previous segment and update the start coordinate for this segment */
        if (fence_direction != DIRECTION_LEFT &&
            fence_direction != DIRECTION_RIGHT)
        {
            fence_segment_complete ();
            fence_start_x = fence_last_x;
        }

        if (to_x > fence_last_x)
        {
            fence_direction = DIRECTION_RIGHT;
            for (int x = fence_last_x; x <= to_x; x++)
            {
                bool post = ! ((x ^ fence_start_x) & 1);

                bitmap2_set_pixel (x, fence_last_y, 2 /* White */);
                bitmap2_set_pixel (x, fence_last_y - 2, 2 /* White */);

                if (((x + fence_start_x) & 0x03) == 0)
                {
                    bitmap2_set_pixel (x, fence_last_y - 1, 2 /* White */);
                    bitmap2_set_pixel (x, fence_last_y - 3, 2 /* White */);
                }
            }
        }
        else
        {
            fence_direction = DIRECTION_LEFT;
            for (int x = fence_last_x; x >= to_x; x--)
            {
                bool post = ! ((x ^ fence_start_x) & 1);

                bitmap2_set_pixel (x, fence_last_y, 2 /* White */);
                bitmap2_set_pixel (x, fence_last_y - 2, 2 /* White */);

                if (((x + fence_start_x) & 0x03) == 0)
                {
                    bitmap2_set_pixel (x, fence_last_y - 1, 2 /* White */);
                    bitmap2_set_pixel (x, fence_last_y - 3, 2 /* White */);
                }
            }
        }
    }
    else /* Draw vertical fence */
    {
        /* Log the previous segment and update the start coordinate for this segment */
        if (fence_direction != DIRECTION_UP &&
            fence_direction != DIRECTION_DOWN)
        {
            fence_segment_complete ();
            fence_start_y = fence_last_y;
        }

        /* To give the height, we draw an extra three pixels. Is there a better way? */
        if (to_y > fence_last_y)
        {
            fence_direction = DIRECTION_DOWN;
            for (int y = fence_last_y - 3; y <= to_y; y++)
            {
                bitmap2_set_pixel (fence_last_x, y, 2 /* White */);
            }
        }
        else
        {
            fence_direction = DIRECTION_UP;
            for (int y = fence_last_y; y >= to_y - 3; y--)
            {
                bitmap2_set_pixel (fence_last_x, y, 2 /* White */);
            }
        }
    }

    fence_last_x = to_x;
    fence_last_y = to_y;
}

/* Note: Due to the linear search, if the player makes a lot of small fences,
 *       they will move more slowly over the later fences... */
/* Check if a coordinate contains fence */
bool is_fence (uint8_t x, uint8_t y)
{
    for (int i = 0; completed_fence_h_list[i]; i++)
    {
        if (y == completed_fence_h_list[i].y &&
            x >= completed_fence_h_list[i].x_from &&
            x <= completed_fence_h_list[i].x_to)
        {
            return true;
        }
    }

    for (int i = 0; completed_fence_v_list[i]; i++)
    {
        if (x == completed_fence_v_list[i].x &&
            y >= completed_fence_v_list[i].y_from &&
            y <= completed_fence_v_list[i].y_to)
        {
            return true;
        }
    }

    return false;
}

/* Note: We only generate one polygon when the fence self-terminates.
 *       -> As this is kinda-cheating if we don't want to allow animals
 *          hitting an incomplete fence, perhaps just prohibit self-terminating
 *          fences.
 */

/* Idea: A dotted-line fence for work-in-progress, which only becomes a
 *       real fence once it completes.
 */

/* Completing a fence segment:
 *
 * 1. For each the left and the right fork of the completion point:
 *    follow the fence to construct the polygon we have created.
 *
 * 2. Use a ray-casting algorithm to determine if the patch has a mix of sneps and wah.
 *
 * 3. If there is no mix, again use a ray-casting algorithm, but this time fill the area
 *    with a darker colour to mark the area as complete.
 */
int polygon_seg_count = 0;

void polygon_log (int x, int y_from, int y_to)
{
    int i = 0;

    /* Make sure we store the lower value first */
    if (y_from > y_to)
    {
        int t = y_to;
        y_to = y_from;
        y_from = t;
    }

    /* Find an empty slot */
    for (i = 0; polygon[i].x != 0; i++);

    polygon[i].x      = x;
    polygon[i].y_from = y_from;
    polygon[i].y_to   = y_to;

    polygon_seg_count = i + 1;
}

int polygon_comparitor (const void *s1, const void *s2)
{
    const Segment_V *seg1 = s1;
    const Segment_V *seg2 = s2;

    if (seg1->x < seg2->x)
    {
        return -1;
    }
    if (seg1->x == seg2->x)
    {
        return 0;
    }
    else
    {
        return 1;
    }
}

void polygon_process ()
{
    uint8_t snep_count = 0;
    uint8_t wah_count = 0;

    /* bounding box */
    int x_min = 255, x_max = 0;
    int y_min = 255, y_max = 0;

    /* First up, sort the polygon list left-to-right */
    qsort (polygon, polygon_seg_count, sizeof (Segment_V), polygon_comparitor);

    /* Count the animals */
    /* TODO: Corner case: How do we treat a snep who is on the same line as a fence corner?
     *       Maybe we need to count the one extreme of fences, but not the other, so that
     *       a U would be ignored and an S would be counted once.  */
    for (uint8_t i = 0; animal[i].x; i++)
    {
        bool inside = false;

        for (uint8_t j = 0; polygon[j].x != 0; j++)
        {
#if 0
            /* XXX: Holy shit this is a buggy compiler. The line below shouldn't have
             *      any effect, but manages to trick the compiler into working… */
            for (int y = polygon[j].y_from; y < polygon[j].y_to; y++);
#endif

            if ((polygon[j].x      <= animal[i].x) &&
                (polygon[j].y_from <= animal[i].y) &&
                (polygon[j].y_to   >= animal[i].y))
            {
                inside = !inside;
            }
        }

        if (inside && animal[i].type == 's')
        {
            snep_count++;
        }

        if (inside && animal[i].type == 'w' && animal[i].alive)
        {
            wah_count++;
        }
    }

    /* If there are no sneps around, the wah can sleep */
    if (snep_count == 0 && wah_count > 0)
    {
        /* Ten points for saving a lone wah */
        /* Twenty points after the first as a breeding-bonus */
        score += 10 + 20 * (wah_count - 1);
    }

    /* Check if this paddock contains a mix of sneps and wah */
    if (snep_count != 0 && wah_count != 0)
    {
        /* There is a mixture, so no going to sleep */
        return;
    }

    /* Send the animals to sleep */
    for (uint8_t i = 0; animal[i].x; i++)
    {
        bool inside = false;

        for (uint8_t j = 0; polygon[j].x != 0; j++)
        {
            if ((polygon[j].x      <= animal[i].x) &&
                (polygon[j].y_from <= animal[i].y) &&
                (polygon[j].y_to   >= animal[i].y))
            {
                inside = !inside;
            }
        }

        if (inside)
        {
            animal[i].sleep = true;
        }
    }


    /* Fill */
    /* TODO: It would be nice not to cover up the fence pixels */
    for (uint8_t p = 0; polygon[p].x != 0; p++)
    {
        if (polygon[p].x < x_min)
        {
            x_min = polygon[p].x;
        }
        if (polygon[p].x > x_max)
        {
            x_max = polygon[p].x;
        }
        if (polygon[p].y_from < y_min)
        {
            y_min = polygon[p].y_from;
        }
        if (polygon[p].y_to > y_max)
        {
            y_max = polygon[p].y_to;
        }
    }

    for (uint8_t y = y_min; y < y_max; y++)
    {
        bool inside = false;

        bool h_fence = false;
        bool was_top = false;

        uint8_t p_start = 0;

        bool mark_valid = false;
        uint8_t mark_from = 0;

        for (uint16_t x = x_min; x <= x_max; x++)
        {
            bool fence = h_fence;
            for (uint8_t p = p_start; polygon[p].x != 0; p++)
            {
                if (polygon[p].x > x)
                {
                    /* The segments are sorted, so as soon as we see one to
                     * our right we know the rest are all to our right and
                     * can break early. */
                    /* To get here, we must have already processed all of
                     * the segments to our left, so set the starting point
                     * for the next pixel to this segment */
                    p_start = p;
                    break;
                }

                /* Corner case */
                if (polygon[p].x == x && (polygon[p].y_from == y || polygon [p].y_to == y))
                {
                    if (h_fence)
                    {
                        if (was_top == (polygon[p].y_from == y))
                        {
                            /* Treat a U-bend as nothing */
                        }
                        else
                        {
                            /* Treat an S-bend as a single fence-crossing */
                            inside = !inside;
                        }
                        h_fence = false;
                        fence = true;
                        break;
                    }
                    else
                    {
                        was_top = polygon[p].y_from == y;
                        h_fence = true;
                        fence = true;
                        break;
                    }
                }

                /* Regular fence */
                if ((polygon[p].x == x) &&
                    (polygon[p].y_from < y) &&
                    (polygon[p].y_to   > y))
                {
                    fence = true;
                    inside = !inside;
                    break;
                }
            }

            /* Mark the start of a line to fill */
            if (inside && !fence)
            {
                if (!mark_valid)
                {
                    mark_from = x;
                    mark_valid = true;
                }
            }

            /* Fill from the mark until the current x position */
            if (mark_valid && !inside)
            {
                for (uint8_t mark_x = mark_from; mark_x < x; mark_x++)
                {
                    if ((!(mark_x & 0x07)) && (x > mark_x + 7))
                    {
                        /* We can fill an eight-pixel line */
                        bitmap2_set_line (mark_x, y, 0x0000 /* dark green */);
                        mark_x += 7;
                    }
                    else
                    {
                        /* Fill a single pixel */
                        bitmap2_set_pixel (mark_x, y, 0 /* dark green */);
                    }
                }
                mark_valid = false;
            }
        }
    }
}

/* TEMP - For debugging */
uint8_t global_trace_colour = 0;

/* TODO: Make this faster, there is a bit of a delay when completing a fence */
void generate_polygon (uint8_t start_x, uint8_t start_y, uint8_t start_direction, bool clockwise)
{
    uint8_t x = start_x;
    uint8_t y = start_y;
    uint8_t direction = start_direction;

    uint8_t segment_x_start = start_x;
    uint8_t segment_y_start = start_y;

    /* 1. Empty the polygon fence list */
    memset (polygon, 0, sizeof (polygon));

    /* 2. Follow the fence, generating the polygon fence list */
    do
    {
        switch (direction)
        {
            case DIRECTION_UP:
                if (clockwise)
                {
                    /* Try right, up, then left */
                    if      (is_fence (x + 1, y    )) { x += 1; direction = DIRECTION_RIGHT; }
                    else if (is_fence (x,     y - 1)) { y -= 1;                              }
                    else if (is_fence (x - 1, y    )) { x -= 1; direction = DIRECTION_LEFT;  }
                }
                else
                {
                    /* Try left, up, then right */
                    if      (is_fence (x - 1, y    )) { x -= 1; direction = DIRECTION_LEFT;  }
                    else if (is_fence (x,     y - 1)) { y -= 1;                              }
                    else if (is_fence (x + 1, y    )) { x += 1; direction = DIRECTION_RIGHT; }
                }
                break;
            case DIRECTION_DOWN:
                if (clockwise)
                {
                    /* Try left, down, then right */
                    if      (is_fence (x - 1, y    )) { x -= 1; direction = DIRECTION_LEFT;  }
                    else if (is_fence (x,     y + 1)) { y += 1;                              }
                    else if (is_fence (x + 1, y    )) { x += 1; direction = DIRECTION_RIGHT; }
                }
                else
                {
                    /* Try right, down, then left */
                    if      (is_fence (x + 1, y    )) { x += 1; direction = DIRECTION_RIGHT; }
                    else if (is_fence (x,     y + 1)) { y += 1;                              }
                    else if (is_fence (x - 1, y    )) { x -= 1; direction = DIRECTION_LEFT;  }
                }
                break;
            case DIRECTION_LEFT:
                if (clockwise)
                {
                    /* Try up, left, then down */
                    if      (is_fence (x,     y - 1)) { y -= 1; direction = DIRECTION_UP;   }
                    else if (is_fence (x - 1, y    )) { x -= 1;                             }
                    else if (is_fence (x,     y + 1)) { y += 1; direction = DIRECTION_DOWN; }
                }
                else
                {
                    /* Try down, left, then up */
                    if      (is_fence (x,     y + 1)) { y += 1; direction = DIRECTION_DOWN; }
                    else if (is_fence (x - 1, y    )) { x -= 1;                             }
                    else if (is_fence (x,     y - 1)) { y -= 1; direction = DIRECTION_UP;   }
                }
                break;
            case DIRECTION_RIGHT:
                if (clockwise)
                {
                    /* Try down, right, then up */
                    if      (is_fence (x,     y + 1)) { y += 1; direction = DIRECTION_DOWN; }
                    else if (is_fence (x + 1, y    )) { x += 1;                             }
                    else if (is_fence (x,     y - 1)) { y -= 1; direction = DIRECTION_UP;   }
                }
                else
                {
                    /* Try up, right, then down */
                    if      (is_fence (x,     y - 1)) { y -= 1; direction = DIRECTION_UP;   }
                    else if (is_fence (x + 1, y    )) { x += 1;                             }
                    else if (is_fence (x,     y + 1)) { y += 1; direction = DIRECTION_DOWN; }
                }
                break;
        }

        /* Check if we've hit a corner */
        if (direction != start_direction)
        {
            /* We've just completed a vertical segment */
            if (start_direction == DIRECTION_UP || start_direction == DIRECTION_DOWN)
            {
                polygon_log (segment_x_start, segment_y_start, y);
                segment_y_start = y;
            }

            /* We've just completed a horizontal segment */
            if (start_direction == DIRECTION_LEFT || start_direction == DIRECTION_RIGHT)
            {
                /* Start logging the new vertical segment */
                segment_x_start = x;

            }
            start_direction = direction;
        }
    }
    while (!(x == start_x && y == start_y));

    if (direction == DIRECTION_UP || direction == DIRECTION_DOWN)
    {
        polygon_log (segment_x_start, segment_y_start, y);
    }
}

/* Note: What about the messy edge case of two adjacent fences? They close a gap without completing...
 *       Should a fence auto-complete once it's within a pixel of another fence? */

/* TODO: Relies on fence_direction being accurate, which is not currently true, as
 *       it is cleared when fence_segment_complete is called. */

/* TODO: Is this needed? Or can we just jump straight into generate_polygon instead? */
void fence_complete (uint8_t x, uint8_t y)
{
    switch (fence_direction)
    {
        case DIRECTION_UP:
            generate_polygon (x, y, DIRECTION_LEFT,  false);    polygon_process ();
            generate_polygon (x, y, DIRECTION_RIGHT, true);     polygon_process ();
            break;
        case DIRECTION_DOWN:
            generate_polygon (x, y, DIRECTION_LEFT,  true);     polygon_process ();
            generate_polygon (x, y, DIRECTION_RIGHT, false);    polygon_process ();
            break;
        case DIRECTION_LEFT:
            generate_polygon (x, y, DIRECTION_UP,   true);      polygon_process ();
            generate_polygon (x, y, DIRECTION_DOWN, false);     polygon_process ();
            break;
        case DIRECTION_RIGHT:
            generate_polygon (x, y, DIRECTION_UP,   false);     polygon_process ();
            generate_polygon (x, y, DIRECTION_DOWN, true);      polygon_process ();
            break;
    }
}

/* TODO: Lives */
/* TODO: Cycle between four palettes, as four seasons */
/* TODO: A "loading" screen during bitmap resets? */
int game (int round)
{
    /* State */
    Sprite player = { .x = 128, .y = 189 };
    bool building = false;
    bool wah_are_alive = true;
    bool wah_are_afraid = true;

    SMS_initSprites ();
    player.id = SMS_addSprite (player.x - 3, player.y - 4, PATTERN_FARMER_RIGHT);

    /* Clear the fence list */
    memset (completed_fence_h_list, 0, sizeof (completed_fence_h_list));
    memset (completed_fence_v_list, 0, sizeof (completed_fence_v_list));

    for (int i = 0; i < (2 + round * 2); i++)
    {
        uint8_t pattern;
        uint8_t speed = 7 + round;

        if (i == 32)
        {
            break; /* Limit of 32 animals on screen at once */
        }

        if (speed > 16)
        {
            speed = 16;
        }

        /* half sneps, half wah */
        animal[i].type = (i & 1) ? 's' : 'w';
        animal[i].alive = true;

        /* Random starting positions */
        animal[i].x = 10 + rand () % (256 - 20);
        animal[i].y = 10 + rand () % (192 - 20);
        animal[i].partial_x = 0;
        animal[i].partial_y = 0;

        /* Randomize velocity */
        animal[i].velocity_x = rand () % speed;
        animal[i].velocity_y = speed - animal[i].velocity_x;
        if (rand () & 1) { animal[i].velocity_x = -animal[i].velocity_x; }
        if (rand () & 1) { animal[i].velocity_y = -animal[i].velocity_y; }

        if (animal[i].type == 's')
        {
            pattern = animal[i].velocity_x < 0 ? PATTERN_SNEP_LEFT : PATTERN_SNEP_RIGHT;
        }
        else {
            pattern = animal[i].velocity_x < 0 ? PATTERN_WAH_LEFT : PATTERN_WAH_RIGHT;
        }
        animal[i].id = SMS_addSprite (animal[i].x, animal[i].y, pattern);
        animal[i].sleep = false;
    }
    SMS_finalizeSprites ();
    SMS_copySpritestoSAT();

    /* Grass texture */
    /* Idea: What if we make the light-green spots more sparse? */
    for (int y = 7; y < 189; y++)
    {
        /* for (int x = 3; x < 253; x++) */
        for (int x = 4; x < 252; x++)
        {
            if ((x % 8) == 0 && x + 8 < 252)
            {
                bitmap2_set_line (x, y, (y & 1) ? 0x00aa : 0x0055);
                x += 7;
                continue;
            }

            else if ((x ^ y) & 1)
            {
                bitmap2_set_pixel (x, y, 1 /* Light green */);
            }
        }
    }

    /* Border fence */
    fence_from (  3,   6); /* Start at the top-left corner */
    fence_to   (252,   6); /* Top */
    fence_to   (252, 189); /* Right */
    fence_to   (  3, 189); /* Bottom */
    fence_to   (  3,   6); /* Left */
    fence_segment_complete ();

    while (wah_are_alive && wah_are_afraid)
    {
        Sprite player_previous;

        /* Input */
        unsigned int pressed = SMS_getKeysStatus ();

        /* Logic */
        if (is_fence (player.x, player.y) &&
            !(pressed & PORT_A_KEY_1))
        {
            building = false;
        }
        else
        {
            if (!building)
            {
                fence_from (player.x, player.y);
            }
            building = true;
        }

        /* Store previous position */
        player_previous.x = player.x;
        player_previous.y = player.y;


        /* Check if the player is moving */
        if ((pressed & PORT_A_KEY_UP && player.y > 6) &&
            (is_fence (player.x, player.y - 1) || (building && fence_direction != DIRECTION_DOWN)))
        {
            player.y -= PLAYER_SPEED;
        }
        else if ((pressed & PORT_A_KEY_DOWN && player.y < 189) &&
                 (is_fence (player.x, player.y + 1) || (building && fence_direction != DIRECTION_UP)))
        {
            player.y += PLAYER_SPEED;
        }
        else if ((pressed & PORT_A_KEY_LEFT && player.x > 3) &&
                 (is_fence (player.x - 1, player.y) || (building && fence_direction != DIRECTION_RIGHT)))
        {
            player.x -= PLAYER_SPEED;
            SMS_updateSpriteImage (player.id, PATTERN_FARMER_LEFT);
        }
        else if ((pressed & PORT_A_KEY_RIGHT && player.x < 252) &&
                (is_fence (player.x + 1, player.y) || (building && fence_direction != DIRECTION_LEFT)))
        {
            player.x += PLAYER_SPEED;
            SMS_updateSpriteImage (player.id, PATTERN_FARMER_RIGHT);
        }

        /* Prevent building over pre-existing fence */
        if (is_fence (player_previous.x, player_previous.y) && is_fence (player.x, player.y))
        {
                building = false;
        }

        if (building)
        {
            fence_to (player.x, player.y);
            if (is_fence (player.x, player.y))
            {
                uint8_t direction_backup = fence_direction;
                fence_segment_complete ();
                fence_direction = direction_backup;

                global_trace_colour += 1;
                global_trace_colour %= 3;

                fence_complete (player.x, player.y);
            }
        }

        /* Update animal positions */
        for (uint8_t i = 0; animal[i].x; i++)
        {
            if (animal[i].sleep || !animal[i].alive)
            {
                continue;
            }

            /* Update x */
            if (is_fence (animal[i].x + (animal[i].partial_x + animal[i].velocity_x) / 16 + (animal[i].velocity_x > 0 ? 8 : 0), animal[i].y))
            {
                uint8_t pattern;
                animal[i].velocity_x = -animal[i].velocity_x;

                /* Update sprite image */
                if (animal[i].type == 's')
                {
                    pattern = animal[i].velocity_x < 0 ? PATTERN_SNEP_LEFT : PATTERN_SNEP_RIGHT;
                }
                else {
                    pattern = animal[i].velocity_x < 0 ? PATTERN_WAH_LEFT : PATTERN_WAH_RIGHT;
                }
                SMS_updateSpriteImage (animal[i].id, pattern);
            }
            animal[i].partial_x += animal[i].velocity_x;
            if (animal[i].partial_x >= 16)
            {
                animal[i].partial_x -= 16;
                animal[i].x += 1;
            }
            else if (animal[i].partial_x <= -16)
            {
                animal[i].partial_x += 16;
                animal[i].x -= 1;
            }

            /* Update y */
            if (is_fence (animal[i].x, animal[i].y + (animal[i].partial_y + animal[i].velocity_y) / 16))
            {
                animal[i].velocity_y = -animal[i].velocity_y;
            }
            animal[i].partial_y += animal[i].velocity_y;
            if (animal[i].partial_y >= 16)
            {
                animal[i].partial_y -= 16;
                animal[i].y += 1;
            }
            else if (animal[i].partial_y <= -16)
            {
                animal[i].partial_y += 16;
                animal[i].y -= 1;
            }

            /* If we are a snep, check if we can eat a wah */
            if (animal[i].type == 's')
            {
                for (uint8_t w = 0; animal[w].x; w++)
                {
                    if (animal[w].type == 'w' &&
                        animal[w].sleep == false &&
                        animal[w].alive == true)
                    {
                        /* Calculate absolute delta in each x and y */
                        int16_t x_delta = animal[i].x - animal[w].x;
                        int16_t y_delta = animal[i].y - animal[w].y;
                        if (x_delta < 0) { x_delta = -x_delta; }
                        if (y_delta < 0) { y_delta = -y_delta; }

                        /* Check if we are clone enough to munch */
                        if (x_delta < 8 && y_delta < 8)
                        {
                            animal[w].sleep = true;
                            animal[w].alive = false;
                            SMS_updateSpriteImage (animal[w].id, PATTERN_WAH_DEAD);
                        }
                    }
                }
            }
        }

        /* Check for the lose-condition */
        wah_are_alive = false;
        for (uint8_t i = 0; animal[i].x; i++)
        {
            if (animal[i].type == 'w' && animal[i].alive)
            {
                wah_are_alive = true;
            }
        }
        if (!wah_are_alive)
        {
            break;
        }

        /* Check for the win-condition */
        wah_are_afraid = false;
        for (uint8_t i = 0; animal[i].x; i++)
        {
            if (animal[i].type == 'w' && !animal[i].sleep)
            {
                wah_are_afraid = true;
            }
        }
        if (!wah_are_afraid)
        {
            break;
        }


        /* Render */
        SMS_waitForVBlank ();
        SMS_updateSpritePosition (player.id, player.x - 3, player.y - 7);
        for (int i = 0; animal [i].x != 0; i++)
        {
            if (animal[i].type == 's')
            {
                SMS_updateSpritePosition (animal[i].id, animal[i].x, animal[i].y - 6);
            }
            else
            {
                SMS_updateSpritePosition (animal[i].id, animal[i].x, animal[i].y - 5);
            }
        }
        SMS_copySpritestoSAT ();
        bitmap2_set_pixel (0, 0, SET_PIXEL_FLUSH);
    }

    /* Remove sprites */
    SMS_initSprites ();
    SMS_finalizeSprites ();
    SMS_copySpritestoSAT();

    if (wah_are_alive)
    {
        return 1; /* Next round */
    }
    else
    {
        return 0; /* Game over */
    }
}

void show_splash (void)
{
    const char *line_1 = "          WAH  MUNCHERS         ";
    const char *line_2 = "          BY JOPPY FURR         ";
    const char *line_3 = "        EARLY RELEASE FOR       ";
    const char *line_4 = "  SMS POWER CODING COMPETITION  ";
    const char *line_5 = "              2018              ";
    const char *line_6 = "                              K2"; /* K becomes R */

    bool waiting = true;
    uint16_t frame = 0;

    /* Show text */
    for (int i = 0; i < 32; i++)
    {
        if (line_1[i] != ' ') { uint16_t tile_index = PATTERN_FONT_A + line_1[i] - 'A'; SMS_loadTileMapArea (i,  2, &tile_index, 1, 1); }
        if (line_2[i] != ' ') { uint16_t tile_index = PATTERN_FONT_A + line_2[i] - 'A'; SMS_loadTileMapArea (i,  4, &tile_index, 1, 1); }
        if (line_3[i] != ' ') { uint16_t tile_index = PATTERN_FONT_A + line_3[i] - 'A'; SMS_loadTileMapArea (i, 18, &tile_index, 1, 1); }
        if (line_4[i] != ' ') { uint16_t tile_index = PATTERN_FONT_A + line_4[i] - 'A'; SMS_loadTileMapArea (i, 20, &tile_index, 1, 1); }
        if (line_5[i] != ' ') { uint16_t tile_index = PATTERN_FONT_0 + line_5[i] - '0'; SMS_loadTileMapArea (i, 22, &tile_index, 1, 1); }
        if (line_6[i] != ' ') { uint16_t tile_index = PATTERN_FONT_0 + line_6[i] - '0'; SMS_loadTileMapArea (i, 23, &tile_index, 1, 1); }
    }

    while (waiting)
    {
        /* Input */
        frame++;

        /* Render */
        SMS_waitForVBlank ();

        /* Show for five seconds */
        if (frame > 60 * 5)
        {
            waiting = false;
        }
    }
}

/* TODO: It would be nice to have the game title be drawn in fence */
void show_title (void)
{
    const char *wah_munchers = "          WAH  MUNCHERS         ";
    const char *press_start =  "           PRESS START          ";
    bool waiting = true;
    uint8_t frame = 0;

    /* Show "wah munchers" */
    for (int i = 0; i < 32; i++)
    {
        if (wah_munchers[i] != ' ')
        {
            uint16_t tile_index = PATTERN_FONT_A + wah_munchers[i] - 'A';
            SMS_loadTileMapArea (i,  8, &tile_index, 1, 1);
        }
    }

    while (waiting)
    {
        /* Input */
        unsigned int pressed = SMS_getKeysStatus ();
        frame++;

        /* Logic */
        if ( (pressed & PORT_A_KEY_1) ||
             (pressed & PORT_A_KEY_2) )
        {
            waiting = false;
            break;
        }

        /* Render */
        SMS_waitForVBlank ();

        if (frame == 30)
        {
            /* Show "press start"  */
            for (int i = 0; i < 32; i++)
            {
                if (press_start[i] != ' ') { uint16_t tile_index = PATTERN_FONT_A + press_start[i] - 'A'; SMS_loadTileMapArea (i,  16, &tile_index, 1, 1); }
            }
        }
        else if (frame == 60)
        {
            /* Hide "press start */
            for (int i = 0; i < 32; i++) { uint16_t tile_index = PATTERN_BLANK; SMS_loadTileMapArea (i,  16, &tile_index, 1, 1); }
            frame = 0;
        }
    }
}

void show_score (void)
{
    const char *line_1 =    "           GAME OVER            ";
    const char *line_2 =    "          SCORE ------          ";
    bool waiting = true;
    uint8_t frame = 0;

    /* Show "SCORE" string */
    for (int i = 0; i < 32; i++)
    {
        if (line_2[i] != ' ') { uint16_t tile_index = PATTERN_FONT_A + line_2[i] - 'A'; SMS_loadTileMapArea (i,  18, &tile_index, 1, 1); }
    }

    /* Show up to six digits of value */
    for (int i = 6; i > 0; i--)
    {
        uint16_t tile_index = PATTERN_FONT_0 + score % 10;
        SMS_loadTileMapArea (16 + i,  18, &tile_index, 1, 1);
        score /= 10;

        /* No need to show leading zeros */
        if (score == 0)
        {
            break;
        }
    }
    score = 0;

    while (waiting)
    {
        /* Input */
        unsigned int pressed = SMS_getKeysStatus ();
        frame++;

        /* Logic */
        if ( (pressed & PORT_A_KEY_1) ||
             (pressed & PORT_A_KEY_2) )
        {
            waiting = false;
            break;
        }

        /* Render */
        SMS_waitForVBlank ();

        if (frame == 30)
        {
            /* Show "GAME OVER" string */
            for (int i = 0; i < 32; i++)
            {
                if (line_1[i] != ' ') { uint16_t tile_index = PATTERN_FONT_A + line_1[i] - 'A'; SMS_loadTileMapArea (i,  5, &tile_index, 1, 1); }
            }
        }
        else if (frame == 60)
        {
            /* Hide "GAME OVER" string */
            for (int i = 0; i < 32; i++) { uint16_t tile_index = PATTERN_BLANK; SMS_loadTileMapArea (i,  5, &tile_index, 1, 1); }
            frame = 0;
        }
    }
}

void main (void)
{
    int round = 0;

    /* 2-bit bitmap palette */
    uint8_t palette[4] = { 0x04, /* Dark Green  */
                           0x08, /* Light Green */
                           0x3f, /* White       */
                           0x00  /* Unused      */ };
    bitmap2_set_palette (palette);

    /* Using three of the four 2-bit colours, there is room for seven sprite colours */
    SMS_setSpritePaletteColor (0x03, 0x00); /* Black      */
    SMS_setSpritePaletteColor (0x07, 0x34); /* Blue       */
    SMS_setSpritePaletteColor (0x0b, 0x2b); /* Tan        */
    SMS_setSpritePaletteColor (0x0c, 0x2a); /* Light grey */
    SMS_setSpritePaletteColor (0x0d, 0x15); /* Dark grey  */
    SMS_setSpritePaletteColor (0x0e, 0x0b); /* Light Wah  */
    SMS_setSpritePaletteColor (0x0f, 0x07); /* Dark Wah   */

    /* Load sprites */
    SMS_loadTiles (patterns, 0, sizeof (patterns));

    /* Set up the playfield bitmap. */
    bitmap2_init ();

    /* Set up sprites */
    SMS_useFirstHalfTilesforSprites (true);

    /* Enable VDP output and get started */
    SMS_displayOn ();

    /* Show a splash screen for the coding-competition release */
    show_splash ();
    bitmap2_init ();

    while (true)
    {
        show_title ();
        bitmap2_init (); /* The title uses font tiles, so re-init the 2-bit bitmap */
        round = 0;
        while (game (round++))
        {
            bitmap2_clear ();
        }
        bitmap2_clear ();
        show_score ();
        bitmap2_init (); /* The score screen uses font tiles, so re-init the 2-bit bitmap */
    }
}
