//#include <config.h>
#include <stdlib.h>

/******************************************************************************
 * texjpgtinyread.h - JPEG Tiny Texture Read                                  *
 ******************************************************************************
 * Project      : jfyeNG                                                      *
 * Author       : Arnaud Storq (norecess@planet-d.net)                        *
 * Based on work of Jeff Frohwein (www.devrs.com/gba), IJG (www.ijg.org) and  *
 *                Dmitry Brant (www.dmitrybrant.com)                          *
 ******************************************************************************/

/*---------------------------------------------------------------------------- TYPEDEF */
typedef char                INT8;
typedef unsigned char       UINT8;
typedef short               INT16;
typedef unsigned short      UINT16;
typedef int                 INT32;
typedef unsigned int        UINT32;

/*---------------------------------------------------------------------------- DEFINES */
#define NULL 0

/*---------------------------------------------------------------------------- LOCAL MACRO */
#define RGB32(r, g, b)									\
(                                                       \
    (UINT32) ((255<<24)+(b<<16)+(g<<8)+r)               \
)

/*---------------------------------------------------------------------------- JPEG CODE */
UINT8 *jpgdata = NULL;
UINT8 *vram = NULL;

void JPGToRGB (INT16 y0, INT16 cb0, INT16 cr0, INT16 *r0, INT16 *g0, INT16 *b0);
INT16 JPGReceiveBits (UINT16 cat);

UINT8 JPGZig1 [64] = 
{
   0,0,1,2,1,0,0,1,2,3,4,3,2,1,0,0,1,2,3,4,5,6,5,4,3,2,1,0,0,1,2,3,
   4,5,6,7,7,6,5,4,3,2,1,2,3,4,5,6,7,7,6,5,4,3,4,5,6,7,7,6,5,6,7,7
};

UINT8 JPGZig2 [64] = 
{
   0,1,0,0,1,2,3,2,1,0,0,1,2,3,4,5,4,3,2,1,0,0,1,2,3,4,5,6,7,6,5,4,
   3,2,1,0,1,2,3,4,5,6,7,7,6,5,4,3,2,3,4,5,6,7,7,6,5,4,5,6,7,7,6,7
};

UINT16 aanscales[64] = {
   /* precomputed values scaled up by 14 bits */
   16384, 22725, 21407, 19266, 16384, 12873,  8867,  4520,
   22725, 31521, 29692, 26722, 22725, 17855, 12299,  6270,
   21407, 29692, 27969, 25172, 21407, 16819, 11585,  5906,
   19266, 26722, 25172, 22654, 19266, 15137, 10426,  5315,
   16384, 22725, 21407, 19266, 16384, 12873,  8867,  4520,
   12873, 17855, 16819, 15137, 12873, 10114,  6967,  3552,
    8867, 12299, 11585, 10426,  8867,  6967,  4799,  2446,
    4520,  6270,  5906,  5315,  4520,  3552,  2446,  1247
   };

#define FIX_1_082392200  277     /* FIX(1.082392200) */
#define FIX_1_414213562  362     /* FIX(1.414213562) */
#define FIX_1_847759065  473     /* FIX(1.847759065) */
#define FIX_2_613125930  669     /* FIX(2.613125930) */

struct JpegType {                      // some type definitions (for coherence)
   UINT16 Rows;                           // image height
   UINT16 Cols;                           // image width
   UINT16 SamplesY;                       // sampling ratios
   UINT16 SamplesCbCr;
   UINT16 QuantTableY;                    // quantization table numbers
   UINT16 QuantTableCbCr;
   UINT16 HuffDCTableY;                   // huffman table numbers
   UINT16 HuffDCTableCbCr;
   UINT16 HuffACTableY;
   UINT16 HuffACTableCbCr;
   UINT16 NumComp;                        // number of components
   };

struct JPGHuffmanEntry {                   // a type for huffman tables
   UINT16 Index;
   INT16 Code;
   UINT16 Length;
   };/*__attribute__ ((packed));*/

UINT32 findex;
UINT16 DCTables;
UINT16 ACTables;
UINT16 QTables;
UINT8 curByte;
UINT8 curBits;
UINT8 EOI;
struct JPGHuffmanEntry HuffmanDC0[256];
struct JPGHuffmanEntry HuffmanDC1[256];
struct JPGHuffmanEntry HuffmanAC0[256];
struct JPGHuffmanEntry HuffmanAC1[256];
//UINT16 ZigIndex;
UINT16 QuantTable[2][8][8];    // 2 quantization tables (Y, CbCr)
struct JpegType Image;
//UINT16 flen;
//INT16 i;




void gfxPixel (INT16 x, INT16 y, UINT32 pixel, UINT32 vram)
{	
	UINT32 *dst = (UINT32 *) vram;
	
	dst[(y*Image.Cols)+x] = pixel;
}



UINT32 JPGpower2 (UINT16 pwr)
   {
   UINT8 i;
   UINT16 total = 1;

   for (i=0; i<pwr; i++)
      total <<= 1;
   return (total);
   }

UINT8 JPGGetByte (void)
   {
   return (jpgdata[findex++]);
   }

UINT16 JPGGetWord (void)
   {
   UINT16 a = (UINT16) (JPGGetByte()<<8);
   a = (UINT16) (a+JPGGetByte());
   return a;
}

UINT8 JPGNextBit (void)
   {
   UINT8 NextBit;

   curBits >>= 1;
   NextBit = (UINT8) ((curByte >> 7) & 1);
   curByte <<= 1;
   if (curBits == 0)
      {
      curBits = 128;
      curByte = JPGGetByte();
      if (curByte == 255)
         if (JPGGetByte() == 0xD9)
            {
            EOI = 1;
            return(0);
            }
      }
   return(NextBit);
   }

INT16 JPGDecode(struct JPGHuffmanEntry inArray[256])
   {
   UINT16 n1; UINT16 n2; UINT16 i; UINT16 l;
   INT32 CurVal;
   INT16 MatchFound;

   if (JPGGetByte() == 255)
      {
      n1 = JPGGetByte();
      findex -= 2;
      if ((n1 >= 0xd0) && (n1 <= 0xd7))
         {
         n2 = (UINT16) (curBits - 1);
         if ((curByte & n2) == n2)     // if the remaining bits are 1
            {
            EOI = 1;
            return(0);
            }
         }
      }
   else
      findex--;

   CurVal = 0;
   MatchFound = -1;
   for (l=1; l<16+1; l++)    // cycle through 16 possible Huffman lengths
      {
      CurVal = (CurVal << 1) + JPGNextBit();
      if (EOI) return(0);
      for (i=0; i<256; i++)              // look for a match in the Huffman table
         {
         if (inArray[i].Length > l) break;
         if (inArray[i].Length == l)
            if (inArray[i].Index == CurVal)
               {
               MatchFound = i;
               break;
               }
         }

      if (MatchFound > -1) break;
      }

   if (MatchFound == -1)
      return(-1);

   return(inArray[MatchFound].Code);  // return the appropriate code
   }

void jpeg_idct_ifast (INT16 inarray[8][8], INT16 outarray[8][8], UINT16 QuantNum)
  {
  // The following variables only need 16bits precision
  // but the resulting code is smaller if you use INT32.
  INT32 tmp0, tmp1, tmp2, tmp3, tmp4, tmp5, tmp6, tmp7;
  INT32 tmp10, tmp11, tmp12, tmp13;
  INT32 z5, z10, z11, z12, z13;

  UINT32 ctr;
  INT16 warray[8][8];

  /* Pass 1: process columns from input, store into work array. */

  for (ctr = 0; ctr < 8; ctr++)
    {
    /* Due to quantization, we will usually find that many of the input
     * coefficients are zero, especially the AC terms.  We can exploit this
     * by short-circuiting the IDCT calculation for any column in which all
     * the AC terms are zero.  In that case each output is equal to the
     * DC coefficient (with scale factor as needed).
     * With typical images and quantization tables, half or more of the
     * column DCT calculations can be simplified this way.
     */

    if (inarray[1][ctr] == 0 && inarray[2][ctr] == 0 &&
        inarray[3][ctr] == 0 && inarray[4][ctr] == 0 &&
        inarray[5][ctr] == 0 && inarray[6][ctr] == 0 &&
        inarray[7][ctr] == 0)
      {
      /* AC terms all zero */
      INT16 dcval = (INT16) (inarray[0][ctr] * QuantTable[QuantNum][0][ctr]);

      warray[0][ctr] = dcval;
      warray[1][ctr] = dcval;
      warray[2][ctr] = dcval;
      warray[3][ctr] = dcval;
      warray[4][ctr] = dcval;
      warray[5][ctr] = dcval;
      warray[6][ctr] = dcval;
      warray[7][ctr] = dcval;

      continue;
      }

    /* Even part */

    tmp0 = inarray[0][ctr] * QuantTable[QuantNum][0][ctr];
    tmp1 = inarray[2][ctr] * QuantTable[QuantNum][2][ctr];
    tmp2 = inarray[4][ctr] * QuantTable[QuantNum][4][ctr];
    tmp3 = inarray[6][ctr] * QuantTable[QuantNum][6][ctr];

    tmp10 = tmp0 + tmp2;	/* phase 3 */
    tmp11 = tmp0 - tmp2;

    tmp13 = tmp1 + tmp3;	/* phases 5-3 */
    tmp12 = (((tmp1 - tmp3)*( FIX_1_414213562)) >> 8) - tmp13; /* 2*c4 */

    tmp0 = tmp10 + tmp13;	/* phase 2 */
    tmp3 = tmp10 - tmp13;
    tmp1 = tmp11 + tmp12;
    tmp2 = tmp11 - tmp12;

    /* Odd part */

    tmp4 = inarray[1][ctr] * QuantTable[QuantNum][1][ctr];
    tmp5 = inarray[3][ctr] * QuantTable[QuantNum][3][ctr];
    tmp6 = inarray[5][ctr] * QuantTable[QuantNum][5][ctr];
    tmp7 = inarray[7][ctr] * QuantTable[QuantNum][7][ctr];

    z13 = tmp6 + tmp5;		/* phase 6 */
    z10 = tmp6 - tmp5;
    z11 = tmp4 + tmp7;
    z12 = tmp4 - tmp7;

    tmp7 = z11 + z13;		/* phase 5 */
    tmp11 = ((z11 - z13) * FIX_1_414213562) >> 8; /* 2*c4 */

    z5 = ((z10 + z12) * FIX_1_847759065) >> 8; /* 2*c2 */
    tmp10 = ((z12 * FIX_1_082392200) >> 8) - z5; /* 2*(c2-c6) */
    tmp12 = ((z10 * - FIX_2_613125930) >> 8) + z5; /* -2*(c2+c6) */

    tmp6 = tmp12 - tmp7;	/* phase 2 */
    tmp5 = tmp11 - tmp6;
    tmp4 = tmp10 + tmp5;

    warray[0][ctr] = (INT16) (tmp0 + tmp7);
    warray[7][ctr] = (INT16) (tmp0 - tmp7);
    warray[1][ctr] = (INT16) (tmp1 + tmp6);
    warray[6][ctr] = (INT16) (tmp1 - tmp6);
    warray[2][ctr] = (INT16) (tmp2 + tmp5);
    warray[5][ctr] = (INT16) (tmp2 - tmp5);
    warray[4][ctr] = (INT16) (tmp3 + tmp4);
    warray[3][ctr] = (INT16) (tmp3 - tmp4);
    }

  /* Pass 2: process rows from work array, store into output array. */
  /* Note that we must descale the results by a factor of 8 == 2**3, */
  /* and also undo the PASS1_BITS scaling. */

  for (ctr = 0; ctr < 8; ctr++)
    {
    /* Rows of zeroes can be exploited in the same way as we did with columns.
     * However, the column calculation has created many nonzero AC terms, so
     * the simplification applies less often (typically 5% to 10% of the time).
     * On machines with very fast multiplication, it's possible that the
     * test takes more time than it's worth.  In that case this section
     * may be commented out.
     */

    if (warray[ctr][1] == 0 && warray[ctr][2] == 0 && warray[ctr][3] == 0 && warray[ctr][4] == 0 &&
        warray[ctr][5] == 0 && warray[ctr][6] == 0 && warray[ctr][7] == 0)
      {
      /* AC terms all zero */
      INT16 dcval = (INT16) ((warray[ctr][0] >> 5)+128);
      if (dcval<0) dcval = 0;
      if (dcval>255) dcval = 255;

      outarray[ctr][0] = dcval;
      outarray[ctr][1] = dcval;
      outarray[ctr][2] = dcval;
      outarray[ctr][3] = dcval;
      outarray[ctr][4] = dcval;
      outarray[ctr][5] = dcval;
      outarray[ctr][6] = dcval;
      outarray[ctr][7] = dcval;
      continue;
      }

    /* Even part */

    tmp10 = warray[ctr][0] + warray[ctr][4];
    tmp11 = warray[ctr][0] - warray[ctr][4];

    tmp13 = warray[ctr][2] + warray[ctr][6];
    tmp12 = (((warray[ctr][2] - warray[ctr][6]) * FIX_1_414213562) >> 8) - tmp13;

    tmp0 = tmp10 + tmp13;
    tmp3 = tmp10 - tmp13;
    tmp1 = tmp11 + tmp12;
    tmp2 = tmp11 - tmp12;

    /* Odd part */

    z13 = warray[ctr][5] + warray[ctr][3];
    z10 = warray[ctr][5] - warray[ctr][3];
    z11 = warray[ctr][1] + warray[ctr][7];
    z12 = warray[ctr][1] - warray[ctr][7];

    tmp7 = z11 + z13;		/* phase 5 */
    tmp11 = ((z11 - z13) * FIX_1_414213562) >> 8; /* 2*c4 */

    z5 = ((z10 + z12) * FIX_1_847759065) >> 8; /* 2*c2 */
    tmp10 = ((z12 * FIX_1_082392200) >> 8) - z5; /* 2*(c2-c6) */
    tmp12 = ((z10 * - FIX_2_613125930) >> 8) + z5; /* -2*(c2+c6) */

    tmp6 = tmp12 - tmp7;	/* phase 2 */
    tmp5 = tmp11 - tmp6;
    tmp4 = tmp10 + tmp5;

    /* Final output stage: scale down by a factor of 8 and range-limit */

    outarray[ctr][0] = (INT16) (((tmp0 + tmp7) >> 5)+128);
    if ((outarray[ctr][0])<0)  outarray[ctr][0] = 0;
    if ((outarray[ctr][0])>255) outarray[ctr][0] = 255;

    outarray[ctr][7] = (INT16) (((tmp0 - tmp7) >> 5)+128);
    if ((outarray[ctr][7])<0)  outarray[ctr][7] = 0;
    if ((outarray[ctr][7])>255) outarray[ctr][7] = 255;

    outarray[ctr][1] = (INT16) (((tmp1 + tmp6) >> 5)+128);
    if ((outarray[ctr][1])<0)  outarray[ctr][1] = 0;
    if ((outarray[ctr][1])>255) outarray[ctr][1] = 255;

    outarray[ctr][6] = (INT16) (((tmp1 - tmp6) >> 5)+128);
    if ((outarray[ctr][6])<0)  outarray[ctr][6] = 0;
    if ((outarray[ctr][6])>255) outarray[ctr][6] = 255;

    outarray[ctr][2] = (INT16) (((tmp2 + tmp5) >> 5)+128);
    if ((outarray[ctr][2])<0)  outarray[ctr][2] = 0;
    if ((outarray[ctr][2])>255) outarray[ctr][2] = 255;

    outarray[ctr][5] = (INT16) (((tmp2 - tmp5) >> 5)+128);
    if ((outarray[ctr][5])<0)  outarray[ctr][5] = 0;
    if ((outarray[ctr][5])>255) outarray[ctr][5] = 255;

    outarray[ctr][4] = (INT16) (((tmp3 + tmp4) >> 5)+128);
    if ((outarray[ctr][4])<0)  outarray[ctr][4] = 0;
    if ((outarray[ctr][4])>255) outarray[ctr][4] = 255;

    outarray[ctr][3] = (INT16) (((tmp3 - tmp4) >> 5)+128);
    if ((outarray[ctr][3])<0)  outarray[ctr][3] = 0;
    if ((outarray[ctr][3])>255) outarray[ctr][3] = 255;

    }
  }

void JPGGetBlock (INT16 vector[8][8], UINT16 HuffDCNum, UINT16 HuffACNum, UINT16 QuantNum, INT16 *dcCoef)
   {
   INT16 array2[8][8];
   INT32 d; UINT16 XPos; UINT16 YPos;
//   UINT32 Sum;
   UINT16 bits; UINT16 zeros; INT32 bitVal; UINT16 ACCount;
   UINT16 x; UINT16 y;
//   UINT16 v; UINT16 u;
//   INT32 temp;
   INT16 temp0;
//   UINT16 Add1 = 0;
   UINT16 ZigIndex;

   EOI = 0;

   for (x=0; x<8; x++)
      for (y=0; y<8; y++)
         array2[x][y] = 0;

   if (HuffDCNum)
      temp0 = JPGDecode(HuffmanDC1);   // Get the DC coefficient
   else
      temp0 = JPGDecode(HuffmanDC0);   // Get the DC coefficient
//   if (EOI) d = 0;
   *dcCoef = (INT16) (*dcCoef + JPGReceiveBits(temp0));
   array2[0][0] = *dcCoef;//* Quant[QuantNum][0][0];

   XPos = 0; YPos = 0;
   ZigIndex = 1;
   ACCount = 1;
   do
      {
      if (HuffACNum)
         d = JPGDecode(HuffmanAC1);
      else
         d = JPGDecode(HuffmanAC0);
//      if (EOI) d = 0;

      zeros = (UINT16) (d >> 4);
      bits = (UINT16) (d & 15);
      bitVal = JPGReceiveBits(bits);

      if (bits)
         {
         ZigIndex = (UINT16) (ZigIndex+zeros);
         ACCount = (UINT16) (ACCount+zeros);
         if (ACCount >= 64) break;

         XPos = JPGZig1[ZigIndex];
         YPos = JPGZig2[ZigIndex];
         ZigIndex++;

         //Read(XPos, YPos);
         array2[XPos][YPos] = (INT16) bitVal; // * Quant[QuantNum][XPos][YPos];
         ACCount++;
         }
      else
         {
         if (zeros != 15) break;
         ZigIndex += 15;
         ACCount += 16;
         }

      }
   while (ACCount < 64);

//   if (HuffDCNum == Image.HuffDCTableY) Add1 = 128;

   jpeg_idct_ifast (array2, vector, QuantNum);

   }

UINT16 JPGGetHuffTables (void)
   {
   UINT16 HuffAmount[17]; //1-16
   UINT32 l0;
   UINT16 c0;
   UINT16 temp0;
   INT16 temp1;
   UINT16 total;
   UINT16 i;
   UINT16 t0;
   INT32 CurNum;
   UINT16 CurIndex;
   UINT16 j;

   l0 = JPGGetWord();
   c0 = 2;
   do
      {
      temp0 = JPGGetByte();
      c0++;
      t0 = (UINT16) ((temp0 & 16) >> 4);
      temp0 &= 15;
      switch (t0)
         {
         case 0:        // DC Table
            total = 0;
            for (i=1; i<16+1; i++)
               {
               temp1 = JPGGetByte();
               c0++;
               total = (INT16) (total+temp1);
               HuffAmount[i] = temp1;
               }
            for (i=0; i<total; i++)
               {
               if (temp0)
                  HuffmanDC1[i].Code = JPGGetByte();
               else
                  HuffmanDC0[i].Code = JPGGetByte();
               c0++;
               }
            CurNum = 0;
            CurIndex = (UINT16) -1;
            for (i=1; i<16+1; i++)
               {
               for (j=1; j<HuffAmount[i]+1; j++)
                  {
                  CurIndex++;
                  if (temp0)
                     {
                     HuffmanDC1[CurIndex].Index = (UINT16) CurNum;
                     HuffmanDC1[CurIndex].Length = (UINT16) i;
                     }
                  else
                     {
                     HuffmanDC0[CurIndex].Index = (UINT16) CurNum;
                     HuffmanDC0[CurIndex].Length = (UINT16) i;
                     }
                  CurNum++;
                  }
               CurNum *= 2;
               }
            DCTables++;
            break;
         case 1:
            total = 0;
            for (i=1; i<16+1; i++)
               {
               temp1 = JPGGetByte();
               c0++;
               total = (UINT16) (total+temp1);
               HuffAmount[i] = temp1;
               }
            for (i=0; i<total; i++)
               {
               if (temp0)
                  HuffmanAC1[i].Code = JPGGetByte();
               else
                  HuffmanAC0[i].Code = JPGGetByte();
               c0++;
               }

            CurNum = 0;
            CurIndex = (UINT16) -1;
            for (i=1; i<16+1; i++)
               {
               for (j=1; j<HuffAmount[i]+1; j++)
                  {
                  CurIndex++;
                  if (temp0)
                     {
                     HuffmanAC1[CurIndex].Index = (UINT16) CurNum;
                     HuffmanAC1[CurIndex].Length = (UINT16) i;
                     }
                  else
                     {
                     HuffmanAC0[CurIndex].Index = (UINT16) CurNum;
                     HuffmanAC0[CurIndex].Length = (UINT16) i;
                     }
                  CurNum++;
                  }
               CurNum *= 2;
               }
            ACTables++;
            break;
         }
      }
   while (c0 < l0);

   return(1);
   }

UINT16 JPGGetImageAttr (void)
   {
   UINT32 temp4;
   UINT16 temp0;
   UINT16 temp1;
   UINT16 i;
   UINT16 id;

   temp4 = JPGGetWord();          //Length of segment
   temp0 = JPGGetByte();          // Data precision
   if (temp0 != 8)
      return(0);                  // we do not support 12 or 16-bit samples
   
   Image.Rows = JPGGetWord();
   Image.Cols = JPGGetWord();

   temp0 = JPGGetByte();          // Number of components
   for (i=1; i<temp0+1; i++)
      {
      id = JPGGetByte();
      switch (id)
         {
         case 1:
            temp1 = JPGGetByte();
            Image.SamplesY = (UINT16) ((temp1 & 15) * (temp1 >> 4));
            Image.QuantTableY = JPGGetByte();
            break;
         case 2:
         case 3:
            temp1 = JPGGetByte();
            Image.SamplesCbCr = (UINT16) ((temp1 & 15) * (temp1 >> 4));
            Image.QuantTableCbCr = JPGGetByte();
            break;
         }
      }

   return(1);
   }

UINT8 JPGGetQuantTables(void)
   {
   UINT32 l0 = JPGGetWord();
   UINT16 c0 = 2;
   UINT16 temp0;
   UINT16 xp;
   UINT16 yp;
   UINT16 i;
   UINT16 ZigIndex;
   
   do
      {
      temp0 = JPGGetByte();
      c0++;
      if (temp0 & 0xf0)
         return(0);        //we don't support 16-bit tables

      temp0 &= 15;
      ZigIndex = 0;
      xp = 0;
      yp = 0;
      for (i=0; i<64; i++)
         {
         xp = JPGZig1[ZigIndex];
         yp = JPGZig2[ZigIndex];
         ZigIndex++;
         /* For AA&N IDCT method, multipliers are equal to quantization
          * coefficients scaled by scalefactor[row]*scalefactor[col], where
          *   scalefactor[0] = 1
          *   scalefactor[k] = cos(k*PI/16) * sqrt(2)    for k=1..7
          */
         QuantTable[temp0][xp][yp] = (UINT16) ((JPGGetByte() * aanscales[(xp<<3) + yp]) >> 12);
         c0++;
         }
      QTables++;
      }
   while (c0 < l0);

   return(1);
   }

UINT16 JPGGetSOS (void)
   {
   UINT32 temp4;
   UINT16 temp0;
   UINT16 temp1;
   UINT16 temp2;
   UINT16 i;

   temp4 = JPGGetWord();
   temp0 = JPGGetByte();

   if ((temp0 != 1) && (temp0 != 3))
      return(0);
   Image.NumComp = temp0;
   for (i=1; i<temp0+1; i++)
      {
      temp1 = JPGGetByte();
      switch (temp1)
         {
         case 1:
            temp2 = JPGGetByte();
            Image.HuffACTableY = (UINT16) (temp2 & 15);
            Image.HuffDCTableY = (UINT16) (temp2 >> 4);
            break;
         case 2:
            temp2 = JPGGetByte();
            Image.HuffACTableCbCr = (UINT16) (temp2 & 15);
            Image.HuffDCTableCbCr = (UINT16) (temp2 >> 4);
            break;
         case 3:
            temp2 = JPGGetByte();
            Image.HuffACTableCbCr = (UINT16) (temp2 & 15);
            Image.HuffDCTableCbCr = (UINT16) (temp2 >> 4);
            break;
         default:
            return(0);
         }
      }
   findex += 3;
   return(1);
   }

//UINT32 gfxJpeg4 (UINT8 x0, UINT8 y0, UINT8 *vram)
UINT32 
JpgTinyRead(UINT8 *src, UINT8 **dst, INT32 *width, INT32 *height, INT32 *depth)
   {
	UINT8 x0 = 0;
	UINT8 y0 = 0;
   UINT8 exit = 1;
//   UINT16 u,v;
//   INT16 x;
   INT16 y;
   UINT32 size = 0;
//   float t;
   UINT16 Restart = 0;
   UINT16 XPos; UINT16 YPos;
   INT16 dcY; INT16 dcCb; INT16 dcCr;
   UINT16 xindex; UINT16 yindex;
   UINT16 mcu;
   INT16 YVector1[8][8];              // 4 vectors for Y attribute
   INT16 YVector2[8][8];              // (not all may be needed)
   INT16 YVector3[8][8];
   INT16 YVector4[8][8];
   INT16 CbVector[8][8];              // 1 vector for Cb attribute
   INT16 CrVector[8][8];              // 1 vector for Cr attribute

   UINT16 i,j;
   UINT16 i2; UINT16 j2;
   INT16 cb; INT16 cr;
   UINT16 xj;
   UINT16 yi;
   INT16 r; INT16 g; INT16 b;

   jpgdata = src;

   QTables = 0;     // Initialize some checkpoint variables
   ACTables = 0;
   DCTables = 0;

   findex = 0;

   if (JPGGetByte() == 0xff)
      {
      if (JPGGetByte() == 0xd8)
         exit = 0;
      }

   // Exit if not a JPEG file
   if (exit)
      return(0);

   while (!exit)
      {
      if (JPGGetByte() == 0xff)
         {
         switch (JPGGetByte())
            {
            case 0x00: //not important
               break;
            case 0xc0: //SOF0
               JPGGetImageAttr();
               break;
            case 0xc1: //SOF1
               JPGGetImageAttr();
               break;
            case 0xc4: //DHT
               if ((ACTables < 2) || (DCTables < 2))
                  JPGGetHuffTables();
               break;
            case 0xc9: //SOF9
               break;
            case 0xd9: //EOI
               exit = 1;
               break;
            case 0xda: //SOS
               JPGGetSOS();
               if ( ((DCTables == 2) &&
                     (ACTables == 2) &&
                     (QTables == 2)) ||
                     (Image.NumComp == 1) )
                  {
                  EOI = 0;
                  exit = 1;        // Go on to secondary control loop
                  }
               break;
            case 0xdb: //DQT
               if (QTables < 2)
                  JPGGetQuantTables();
               break;
            case 0xdd: //DRI
               Restart = JPGGetWord();
               break;
            case 0xe0: //APP0
               (void) JPGGetWord();        // Length of segment
               findex += 5;
               (void) JPGGetByte();        // Major revision
               (void) JPGGetByte();        // Minor revision
               (void) JPGGetByte();        // Density definition
               (void) JPGGetByte();        // X density
               (void) JPGGetByte();        // Y density
               (void) JPGGetByte();        // Thumbnail width
               (void) JPGGetByte();        // Thumbnail height
               break;
            case 0xfe: //COM
               break;
            }
         }
      }

   XPos = 0;
   YPos = 0;                            // Initialize active variables
   dcY = 0; dcCb = 0; dcCr = 0;
   xindex = 0; yindex = 0; mcu = 0;
   r = 0; g = 0; b = 0;

   

	 *width = Image.Cols;
	 *height = Image.Rows;
	 *depth = 32;

	 size = Image.Cols*Image.Rows*sizeof(UINT32);	 
	 *dst = (UINT8 *) malloc(size); /* here you should link to your memory manager */

	 vram = *dst;

   curBits = 128;                // Start with the seventh bit
   curByte = JPGGetByte();       // Of the first byte

   switch (Image.NumComp)        // How many components does the image have?
     {
     case 3:                     // 3 components (Y-Cb-Cr)
       {
       switch (Image.SamplesY)   // What's the sampling ratio of Y to CbCr?
         {
         case 4:                 // 4 pixels to 1

           do                    // Process 16x16 blocks of pixels
             {
             JPGGetBlock (YVector1, Image.HuffDCTableY,    Image.HuffACTableY,    Image.QuantTableY,    &dcY);
             JPGGetBlock (YVector2, Image.HuffDCTableY,    Image.HuffACTableY,    Image.QuantTableY,    &dcY);
             JPGGetBlock (YVector3, Image.HuffDCTableY,    Image.HuffACTableY,    Image.QuantTableY,    &dcY);
             JPGGetBlock (YVector4, Image.HuffDCTableY,    Image.HuffACTableY,    Image.QuantTableY,    &dcY);
             JPGGetBlock (CbVector, Image.HuffDCTableCbCr, Image.HuffACTableCbCr, Image.QuantTableCbCr, &dcCb);
             JPGGetBlock (CrVector, Image.HuffDCTableCbCr, Image.HuffACTableCbCr, Image.QuantTableCbCr, &dcCr);
             // YCbCr vectors have been obtained

             for (i=0; i<8; i++)  // Draw top left 8x8 pixels
               for (j=0; j<8; j++)
                 {
                 y = YVector1[i][j];
                 i2 = (UINT16) (i >> 1);
                 j2 = (UINT16) (j >> 1);
                 cb = CbVector[i2][j2];
                 cr = CrVector[i2][j2];
                 JPGToRGB (y, cb, cr, &r, &g, &b);
                 xj = (UINT16) (xindex + j);
                 yi = (UINT16) (yindex + i);
                 if ( (xj < Image.Cols) && (yi < Image.Rows) )
                   gfxPixel ((INT16) (xj + x0), (INT16) (yi + y0), (UINT32) RGB32(r, g, b), (UINT32)vram);
                 }
             for (i=0; i<8; i++)  // Draw top right 8x8 pixels
               for (j=8; j<16; j++)
                 {
                 y = YVector2[i][j - 8];
                 i2 = (UINT16) (i >> 1);
                 j2 = (UINT16) (j >> 1);
                 cb = CbVector[i2][j2];
                 cr = CrVector[i2][j2];
                 JPGToRGB (y, cb, cr, &r, &g, &b);
                 xj = (UINT16) (xindex + j);
                 yi = (UINT16) (yindex + i);
                 if ((xj < Image.Cols) && (yi < Image.Rows))
                   gfxPixel ((INT16) (xj + x0), (INT16) (yi + y0), (UINT32) RGB32(r, g, b), (UINT32)vram);
                 }
             for (i=8; i<16; i++)  // Draw bottom left 8x8 pixels
               for (j=0; j<8; j++)
                 {
                 y = YVector3[i - 8][j];
                 i2 = (UINT16) (i >> 1);
                 j2 = (UINT16) (j >> 1);
                 cb = CbVector[i2][j2];
                 cr = CrVector[i2][j2];
                 JPGToRGB (y, cb, cr, &r, &g, &b);
                 xj = (UINT16) (xindex + j);
                 yi = (UINT16) (yindex + i);
                 if ((xj < Image.Cols) && (yi < Image.Rows))
                   gfxPixel ((INT16) (xj + x0), (INT16) (yi + y0), (UINT32) RGB32(r, g, b), (UINT32)vram);
                 }
             for (i=8; i<16; i++)          // Draw bottom right 8x8 pixels
               for (j=8; j<16; j++)
                 {
                 y = YVector4[i - 8][j - 8];
                 i2 = (UINT16) (i >> 1);
                 j2 = (UINT16) (j >> 1);
                 cb = CbVector[i2][j2];
                 cr = CrVector[i2][j2];
                 JPGToRGB (y, cb, cr, &r, &g, &b);
                 xj = (UINT16) (xindex + j);
                 yi = (UINT16) (yindex + i);
                 if ((xj < Image.Cols) && (yi < Image.Rows))
                   gfxPixel ((INT16) (xj + x0), (INT16) (yi + y0), (UINT32) RGB32(r, g, b), (UINT32)vram);
                 }
             xindex += 16;
             if (xindex >= Image.Cols)
                {
                xindex = 0; yindex += 16; mcu = 1;
                }
             if ((mcu == 1) && (Restart != 0))    //Execute the restart interval
                {
                curByte = JPGGetByte(); curByte = JPGGetByte(); curByte = JPGGetByte();
                curBits = 128;
                dcY = 0; dcCb = 0; dcCr = 0; mcu = 0;  //Reset the DC value
                }
             }
//           while ((findex < flen) && (yindex < Image.Rows));
           while (yindex < Image.Rows);
           break;
         case 2:           // 2 pixels to 1
           do
             {
             JPGGetBlock (YVector1, Image.HuffDCTableY,    Image.HuffACTableY,    Image.QuantTableY,    &dcY);
             JPGGetBlock (YVector2, Image.HuffDCTableY,    Image.HuffACTableY,    Image.QuantTableY,    &dcY);
             JPGGetBlock (CbVector, Image.HuffDCTableCbCr, Image.HuffACTableCbCr, Image.QuantTableCbCr, &dcCb);
             JPGGetBlock (CrVector, Image.HuffDCTableCbCr, Image.HuffACTableCbCr, Image.QuantTableCbCr, &dcCr);
             // YCbCr vectors have been obtained

             for (i=0; i<8; i++)       // Draw left 8x8 pixels
               for (j=0; j<8; j++)
                 {
                 y = YVector1[i][j];
                 i2 = (UINT16) (i >> 1);
                 j2 = (UINT16) (j >> 1);
                 cb = CbVector[i2][j2];
                 cr = CrVector[i2][j2];
                 JPGToRGB (y, cb, cr, &r, &g, &b);
                 xj = (UINT16) (xindex + j);
                 yi = (UINT16) (yindex + i);
                 if ((xj < Image.Cols) && (yi < Image.Rows))
                   gfxPixel ((INT16) (xj + x0), (INT16) (yi + y0), (UINT32) RGB32(r, g, b), (UINT32)vram);
                 }

             for (i=0; i<8; i++)       // Draw right 8x8 pixels
               for (j=8; j<16; j++)
                 {
                 y = YVector2[i][j - 8];
                 i2 = (UINT16) (i >> 1);
                 j2 = (UINT16) (j >> 1);
                 cb = CbVector[i2][j2];
                 cr = CrVector[i2][j2];
                 JPGToRGB (y, cb, cr, &r, &g, &b);
                 xj = (UINT16) (xindex + j);
                 yi = (UINT16) (yindex + i);
                 if ((xj < Image.Cols) && (yi < Image.Rows))
                   gfxPixel ((INT16) (xj + x0), (INT16)(yi + y0), (UINT32) RGB32(r, g, b), (UINT32)vram);
                 }
             xindex += 16;
             if (xindex >= Image.Cols)
               {
               xindex = 0; yindex += 8; mcu = 1;
               }
             if ((mcu == 1) && (Restart != 0))  // execute the restart interval
               {
               curByte = JPGGetByte(); curByte = JPGGetByte(); curByte = JPGGetByte();
               curBits = 128;
               dcY = 0; dcCb = 0; dcCr = 0; mcu = 0;
               }
             }
//           while ((findex < flen) && (yindex < Image.Rows));
           while (yindex < Image.Rows);
           break;
         case 1:        // 1 pixel to 1
           do
             {
             JPGGetBlock (YVector1, Image.HuffDCTableY,    Image.HuffACTableY,    Image.QuantTableY,    &dcY);
             JPGGetBlock (CbVector, Image.HuffDCTableCbCr, Image.HuffACTableCbCr, Image.QuantTableCbCr, &dcCb);
             JPGGetBlock (CrVector, Image.HuffDCTableCbCr, Image.HuffACTableCbCr, Image.QuantTableCbCr, &dcCr);
             // YCbCr vectors have been obtained

             for (i=0; i<8; i++)            // Draw 8x8 pixels
               for (j=0; j<8; j++)
                 {
                 y = YVector1[i][j];
                 i2 = (UINT16) (i >> 1);
                 j2 = (UINT16) (j >> 1);
                 cb = CbVector[i2][j2];
                 cr = CrVector[i2][j2];
                 JPGToRGB (y, cb, cr, &r, &g, &b);
                 xj = (UINT16) (xindex + j);
                 yi = (UINT16) (yindex + i);
                 if ((xj < Image.Cols) && (yi < Image.Rows))
                   gfxPixel ((INT16) (xj + x0), (INT16) (yi + y0), (UINT32) RGB32(r, g, b), (UINT32)vram);
                 }

             xindex += 8;
             if (xindex >= Image.Cols)
               {
               xindex = 0; yindex += 8; mcu = 1;
               }
             if ((mcu == 1) && (Restart != 0))  // execute the restart interval
               {
               curByte = JPGGetByte(); curByte = JPGGetByte(); curByte = JPGGetByte();
               curBits = 128;
               dcY = 0; dcCb = 0; dcCr = 0; mcu = 0;
               }
             }
//           while ((findex < flen) && (yindex < Image.Rows));
           while (yindex < Image.Rows);
           break;
         }  // Ratio
       }
     case 1:
       do
         {
         JPGGetBlock (YVector1, Image.HuffDCTableY, Image.HuffACTableY, Image.QuantTableY, &dcY);
         // Y vector has been obtained

         for (i=0; i<8; i++)           // Draw 8x8 pixels
           for (j=0; j<8; j++)
             {
             y = YVector1[i][j];
             if (y < 0) y = 0;
             if (y > 255) y = 255;
             xj = (UINT16) (xindex + j);
             yi = (UINT16) (yindex + i);
             if ((xj < Image.Cols) && (yi < Image.Rows))
               gfxPixel ((INT16) (xj + x0), (INT16) (yi + y0), (UINT32) RGB32(y, y, y), (UINT32)vram);
             }

// Get setup to draw next block
         xindex += 8;
         if (xindex >= Image.Cols)
           {
           xindex = 0; yindex += 8; mcu = 1;
           }

         if ((mcu == 1) && (Restart != 0))   // execute the restart interval
           {
           curByte = JPGGetByte(); curByte = JPGGetByte(); curByte = JPGGetByte();
           curBits = 128;
           dcY = 0; mcu = 0;
           }
         }
//       while ((findex <flen) && (yindex < Image.Rows));
       while (yindex < Image.Rows);
       break;
     }

   return(1);
   }

INT16 JPGReceiveBits (UINT16 cat)
   {
   UINT32 temp0 = 0;
   UINT16 i;
   INT32 ReceiveBits;

   for (i=0; i<cat; i++)
      temp0 = temp0 * 2 + JPGNextBit();
   if ((temp0*2) >= (JPGpower2(cat)) )
      ReceiveBits = temp0;
   else
      ReceiveBits = (INT32) (-(INT32) (JPGpower2(cat) - 1) + temp0);
   return (INT16) ReceiveBits;
   }

void JPGToRGB (INT16 y0, INT16 cb0, INT16 cr0, INT16 *r0, INT16 *g0, INT16 *b0)
   {
   // Do color space conversion from YCbCr to RGB
   *r0 = (INT16) ((y0) + (((cr0-128) * 45) >> 5));
   *g0 = (INT16) ((y0) - (((cb0-128) * 11) >> 5) - (((cr0-128) * 23) >> 5));
   *b0 = (INT16) ((y0) + (((cb0-128) * 57) >> 5));
   if (*r0 > 255) *r0 = 255;
   if (*r0 < 0) *r0 = 0;
   if (*g0 > 255) *g0 = 255;
   if (*g0 < 0) *g0 = 0;
   if (*b0 > 255) *b0 = 255;
   if (*b0 < 0) *b0 = 0;
   }
