#include "fusa_framebuffer.h"
#include "../graphicIncludes.h"
#include "fusa_texture.h"
#include <stack>
using namespace fusa;


std::stack<cFrameBuffer*> cFrameBuffer::fbStack;
cFrameBuffer::cFrameBuffer(int sizeX,int sizeY,int bufferTypes)
{
  m_size.x = sizeX;
  m_size.y = sizeY;
  m_buffTypes = bufferTypes;
  m_autoClear = true;
  m_clearColor.set(1.0,1.0,1.0,0.0);
}


void cFrameBuffer::setClearColor(const cVec4f &clearCol)
{
  m_clearColor = clearCol;
}

const cVec4f& cFrameBuffer::getClearColor()const
{
  return m_clearColor;
}

const cVec2f& cFrameBuffer::getSize()const
{
  return m_size;
}

void cFrameBuffer::setAutoClear(bool trueFalse)
{
  m_autoClear = trueFalse;
}

bool cFrameBuffer::getAutoClear()const
{
  return m_autoClear;
}

GLint cFrameBuffer::getGLBufferTypes()
{
  return m_buffTypes;
}

void cFrameBuffer::resume()
{
  glViewport(0,0,getSize().x,getSize().y);
}

void cFrameBuffer::stackResume()
{
 fbStack.pop();
  if(fbStack.size()!=0)
    fbStack.top()->resume();
}

//DIRECT FRAMEBUFFER

cDirectFrameBuffer::cDirectFrameBuffer(int sizeX,int sizeY,GLint bufferTypes):cFrameBuffer(sizeX,sizeY,bufferTypes)
{
}

void cDirectFrameBuffer::start()
{
  glViewport(0,0,getSize().x,getSize().y);
  glClearColor(getClearColor().x,getClearColor().y,
		 getClearColor().z,getClearColor().w);
  if(getAutoClear())
    {
      clear();
    }
  fbStack.push(this);
}

void cDirectFrameBuffer::finish()
{
  stackResume();
}

void cDirectFrameBuffer::clear()
{
  glClear(getGLBufferTypes());
}


//RENDERBUFFER

cRenderBuffer::cRenderBuffer(int width,int height,GLint storage)
{
  glGenRenderbuffersEXT(1,&m_renderBuffer);
  glBindRenderbufferEXT(GL_RENDERBUFFER_EXT,m_renderBuffer);
  glRenderbufferStorageEXT(GL_RENDERBUFFER_EXT,
			storage,width,height);
}


//OFFSCREEN FRAMEBUFFER

cOffscreenFrameBuffer::cOffscreenFrameBuffer(int sizeX,
					     int sizeY,GLint bufferTypes):cFrameBuffer(sizeX,sizeY,bufferTypes)
{
  glGenFramebuffersEXT(1,&m_framebuffer);
}

void cOffscreenFrameBuffer::start()
{
  glBindFramebufferEXT(GL_FRAMEBUFFER_EXT,m_framebuffer);
  glViewport(0,0,getSize().x,getSize().y);
  glClearColor(getClearColor().x,getClearColor().y,
		 getClearColor().z,getClearColor().w);
  if(getAutoClear())
    {
      clear();
    }
  fbStack.push(this);
}

void cOffscreenFrameBuffer::finish()
{
   glBindFramebufferEXT(GL_FRAMEBUFFER_EXT,0);
   for(unsigned int i=0; i < attachedTextures.size(); i++)
     {
      if(attachedTextures[i]->isMipmapped())
	{
	  attachedTextures[i]->bindTexture();	  
	  glGenerateMipmapEXT(GL_TEXTURE_2D);
	}

     }

   stackResume();
}

void cOffscreenFrameBuffer::clear()
{
  glClear(getGLBufferTypes());
}

void cOffscreenFrameBuffer::attachRenderBuffer(cRenderBuffer* buff,
					       GLint attachmentPoint)
{
  glBindFramebufferEXT(GL_FRAMEBUFFER_EXT,m_framebuffer);
  glFramebufferRenderbufferEXT(GL_FRAMEBUFFER_EXT,
			    attachmentPoint,GL_RENDERBUFFER_EXT,
			    buff->m_renderBuffer);
  
  glBindFramebufferEXT(GL_FRAMEBUFFER_EXT,0);
}

void cOffscreenFrameBuffer::attachTexture(cRenderableTexture* texture,
					  GLint attachmentPoint,
					  int miplevel)
{
  glBindFramebufferEXT(GL_FRAMEBUFFER_EXT,m_framebuffer);
  if(texture->m_target==GL_TEXTURE_2D)
    {
      if(texture->isMipmapped())
	{
	  texture->bindTexture();
	  glGenerateMipmapEXT(GL_TEXTURE_2D);
	}
      glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT,attachmentPoint,
				texture->m_target,texture->m_id,miplevel);
      attachedTextures.push_back(texture);
    }
  
  glBindFramebufferEXT(GL_FRAMEBUFFER_EXT,0);
		   
}

bool cOffscreenFrameBuffer::checkCompleteness()
{
  GLenum status;
  glBindFramebufferEXT(GL_FRAMEBUFFER_EXT,m_framebuffer);
  status = glCheckFramebufferStatusEXT(GL_FRAMEBUFFER_EXT);
  glBindFramebufferEXT(GL_FRAMEBUFFER_EXT,0);
  if(status==GL_FRAMEBUFFER_COMPLETE_EXT)
    return true;
  if(status==GL_FRAMEBUFFER_UNSUPPORTED_EXT)
    {
      std::cout<<"framebuffer unsupported!"<<std::endl;
      return false;
    }

  if(status== GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT_EXT)
    {
      std::cout<<"GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT_EXT\n";
      return false;
     
    }
  if(status== GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT_EXT)
    {
      std::cout<<"GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT_EXT\n";
      return false;
     
    }
  if(status== GL_FRAMEBUFFER_INCOMPLETE_DIMENSIONS_EXT)
    {
      std::cout<<"GL_FRAMEBUFFER_INCOMPLETE_DIMENSIONS_EXT\n";
      return false;
    }
  if(status== GL_FRAMEBUFFER_INCOMPLETE_FORMATS_EXT)
    {
      std::cout<<"GL_FRAMEBUFFER_INCOMPLETE_FORMATS_EXT\n";
      return false;
    }
  if(status== GL_FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER_EXT)
    {
      std::cout<<"GL_FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER_EXT\n";
      return false;
     
    }
  if(status== GL_FRAMEBUFFER_INCOMPLETE_READ_BUFFER_EXT)
    {
      std::cout<<"incomplete read buffer oops"<<std::endl;
      return false;
    }
  
  std::cout<<"framebuffer error, some bad stuff happen"<<std::endl;
  return false;

}

void cOffscreenFrameBuffer::setDrawAndReadBuffer(GLint drawBuffer,
						 GLint readBuffer)
{
   glBindFramebufferEXT(GL_FRAMEBUFFER_EXT,m_framebuffer);
   glDrawBuffer(drawBuffer);
   glReadBuffer(readBuffer);
   glBindFramebufferEXT(GL_FRAMEBUFFER_EXT,0);
}
void cOffscreenFrameBuffer::setDrawBuffers(const GLenum *drawBuffers,int numBuffs)
{
	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT,m_framebuffer);
   glDrawBuffers(numBuffs,drawBuffers);
   glBindFramebufferEXT(GL_FRAMEBUFFER_EXT,0);
}