;  _________              .__.__                                     __          
; /   _____/ ____   _____ |__|  | _____    ____   ____  ____ _____ _/  |______   
; \_____  \_/ __ \ /     \|  |  | \__  \  /    \_/ ___\/ __ \\__  \\   __\__  \  
; /        \  ___/|  Y Y  \  |  |__/ __ \|   |  \  \__\  ___/ / __ \|  |  / __ \_
;/_______  /\___  >__|_|  /__|____(____  /___|  /\___  >___  >____  /__| (____  /
;        \/     \/      \/             \/     \/     \/    \/     \/          \/ 
; DMA Soundchip Player (beta)
; Grim/Arkos^Semilanceata

; This lib give the abilitie to play music (at any frequency) with a single DMA
; channel and one call from time to time (to fill the audio-buffer).
;
; An audio-buffer (practically, a DMA-List) is used to periodically send data
; to the AY3. This audio-buffer is filled with the AY values calculated by any
; soundchip player (which must be modified to fill this buffer instead of OUTing
; these values directly to the AY).
;
; The depth of the audio-buffer define how long the DMA can keep playing the
; music without any call to the fill routine (e.g. while loading a disk-sector,
; executing a CPU intensive routine, etc).
;
; You can fill the audio-buffer with a CALL to the dma_fill() routine in your
; program. The more often you call it, the less data will have to be calculated
; by the music-player. Beware of under-run (when buffer is empty), it will
; produce glitch in the audio.
;
; The trick which make it all possible is to use the AY3 register 15 to store
; the last audio-frame marker played by the DMA, because the ASIC does not
; provide any mean to get the current position of it's DMA-pointer.
; AY3 register 15 is a 8bit I/O data register (juste like register 14) but is
; not wired on the AY3-8912 used in the CPC/Plus.
; In order to use this trick, we must take care that bit7 in the mixer (AY3 Reg7),
; must always be set (1).
;
; Requirements
; ------------
; The audio-buffer MUST be in the base 64kb RAM since the ASIC can't access
; the extra-RAM.
;
; Restrictions
; ------------
; - the lib will map the ASIC I/O in memory [#4000,#7FFF], so it's obvious
;   it can not be compiled within this address range.
;
; - Since there's no DMA interrupts involved, DMA0 is used so that the fully
;   working DMA1 and DMA2 remain free to use (there's an ASIC bug on DMA0-Int
;   which is sometime confused with the raster-interrupt).
;
; Credits
; -------
; You are free to use and modify this lib for whatever you want, but if you do
; so, I will be glad to be credited for my work and noticed about what you did
; with it (or what you've modified). grim(at)cpcscene.com

org &100
jp dma_init
jp dma_start
jp dma_fill

							; *********************
							; *** CONFIGURATION ***
							; *********************

; if your program run with all interruptions disabled, you can comment the
; line below to remove all di/ei from the lib & players.
dma_cnf_mutitasking_friendly		equ 1


							; ********************
							; *** REQUIREMENTS ***
							; ********************

							if $ and &C000 - &4000
							else
								print "*** DMA SOUNDCHIP FATAL ERROR ***"
								print "This library can not be compiled within #4000/#7FFF"
							endif

							ifdef IOASIC_SAR0
							else
								read "asic.equ.asm"
							endif
							
							; ******************************
							; *** INTERNAL CONFIGURATION ***
							; ******************************
							
							; Buffer structure indexes
dma_idx_R0					equ 0
dma_idx_R1					equ 2
dma_idx_R2					equ 4
dma_idx_R3					equ 6
dma_idx_R4					equ 8
dma_idx_R5					equ 10
dma_idx_R6					equ 12
dma_idx_R7					equ 14
dma_idx_R8					equ 16
dma_idx_R9					equ 18
dma_idx_RA					equ 20
dma_idx_RB					equ 22
dma_idx_RC					equ 24
dma_idx_RDv					equ 26
dma_idx_RDi					equ 27
dma_idx_RF					equ 28

_dma_cnf_buffer_framesize		equ _dma_data_buffer_frame_end - _dma_data_buffer_frame_start


							; *********************************
							; *** DMA Initialize DMA player ***
							; *********************************
							;
							; Input
							;  HL=Replay period-15	[0,4096]
							;  BC=Player address	[#0000,#FFFF]
							;  DE=Buffer address	[#0000,#FFFF]
							;   A=Buffer depth		[0,255]
							
							; Output
							;  - Consider HL,DE,BC,AF,IX,IY corrupts,
							;    but it also depend on the music player.
							;  - Buffer is prefilled and DMA ready to be
							;    started.
							;
							; notes
							; With a PAL CRTC configuration (64us scanlines),
							; to replay a music at 300Hz, HL=52-15
							; to replay a music at 150Hz, HL=104-15
							; to replay a music at 100Hz, HL=156-15
							; to replay a music at 50Hz,  HL=312-15
							; to replay a music at 25Hz,  HL=624-15
							;
							; max replay freq is 976Hz (HL=0)
							; min replay freq is 3.8Hz (HL=4095)
							
dma_init:
							push af
							
							; setup the player address
							ld (_dma_var_music_player),bc
							
							; setup the SAR pointer to re/start the DMA 
							ld (_dma_var_sar_start),de
							
							; Build the DMA-PAUSE instruction
							ld a,h
							and %00001111
							or  %00010000
							ld h,a
							; Update DMA-List chunk
							ld (_dma_data_buffer_pause),hl
							
							; Generate the audio-buffer/DMA list
							ld hl,_dma_data_buffer
							ldi
							ldi
							
							; setup the audio-buffer loopback address
							ld (_dma_fill_var_from),de
							ld (_dma_fill_var_loopback),de
							
							; prepare buffer's audio-frames
							ld ix,_dma_data_buffer_frame_start
							pop af
_dma_init_loop
							; update chunk
							; disable Env. Shape update (AY3 Reg 13)
							ld (ix+dma_idx_RDi),&0F
							ld (ix+dma_idx_RDv),a
							; set frame marker
							ld (ix+dma_idx_RF),a
							; copy chunk
							ld hl,_dma_data_buffer_frame_start
							ld bc,_dma_cnf_buffer_framesize
							ldir
							; loop buffer-depth
							dec a
							jp p,_dma_init_loop
							; copy the DMA-LOOP opcode
							ldi
							ldi
							; update the dma_stop()
							ld (_dma_var_sar_stop),hl
							; copy the DMA-STOP sequence
							ld bc,_dma_data_buffer_end - _dma_data_buffer_frame_stop
							ldir
							
							; Initialize AY3 with a dummy frame-marker
							ld de,&0700+%10111111
							call _dma_ay3_write
							ld de,&0F01
							call _dma_ay3_write
							
							; *****************************
							; *** Fill the audio-buffer ***
							; *****************************
							
dma_fill:						; execute the DMA fill
							call _dma_fill
							; save the audio-buffer pointer
							ld (_dma_fill_var_from),iy
							ret
_dma_fill
							; audio-buffer pointer from which the fill
							; must begin.
							ld iy,0
_dma_fill_var_from				equ $-2

							; Execute the fill loop until the fill pointer
							; in the audio-buffer reach the last audio-frame
							; played by the DMA.
_dma_fill_loop					
							; *** Read AY3 register 15 to get the last frame-marker ***
							ld a,15
							
							ifdef dma_cnf_mutitasking_friendly
								di
							endif
							
							ifdef ay3_read
								; use AY3.lib if it is found
								call ay3_read
							else
								; custom AY3 read register
								di
								ld b,&f4
								out (c),a
								ld bc,&f6c0
								out (c),c
								dw &71ED
								ld bc,&f792
								out (c),c
								ld bc,&F640
								out (c),c
								ld a,&f4
								in a,(&FF)
								dw &71ED
								ld bc,&F782
								out (c),c
							endif
							
							ifdef dma_cnf_mutitasking_friendly
								ei
							endif
							
							; detect DMA-List restart
							or a
							call z,dma_start
							
							; compare DMA marker with the last frame marker filled
							cp &FF
_dma_fill_var_lastmarker			equ $-1
							ret z
							
							; Replay music
							; IY hold the audio-frame pointer to be filled
							call 0
_dma_var_music_player			equ $-2
							
							; detect the last audio-frame of the buffer
							xor a
							or (iy+dma_idx_RF)
							ld (_dma_fill_var_lastmarker),a
							jr z,_dma_fill_loopback

							; move forward in the audio-buffer
							ld bc,_dma_cnf_buffer_framesize
							add iy,bc
							jr _dma_fill_loop
							
							; we've reached the end of the audio-buffer
							; loop back at the begining
_dma_fill_loopback				ld iy,0
_dma_fill_var_loopback			equ $-2
							jr _dma_fill_loop
							
							
							
							
							
							; *****************************
							; *** Start the DMA channel ***
							; *****************************
dma_start:
							ld hl,0
_dma_var_sar_start				equ $-2
							jr _dma_ctrl

							; ****************************
							; *** Stop the DMA channel ***
							; ****************************
dma_stop:
							ld hl,0
_dma_var_sar_stop				equ $-2

_dma_ctrl
							push af
							push hl
							; enable ASIC I/O mapping
							ifdef kl_asic_enable
								; use kernel memory manager
								call kl_asic_enable
							else
								; use direct memory mapping
								ld bc,&7FB8
								out (c),c
							endif
							
							; reset PPR=0
							xor a
							ld (IOASIC_PPR0),a
							
							; reset SAR pointer
							pop hl
							ld (IOASIC_SAR0),hl
							
							; enable DMA channel
							ld hl,IOASIC_DCSR
							set 0,(hl)
							
							; disable ASIC I/O mapping
							ifdef kl_asic_disable
								; use kernel memory manager
								call kl_asic_disable
							else
								; use direct memory mapping
								ld bc,&7FA0
								out (c),c
							endif
							
							pop af
							ret
							
							
							ifdef ay3_write
								; use AY3.lib if it is found
_dma_ay3_write:					equ ay3_write
							else
_dma_ay3_write:
								ld b,&F4
								ld a,&E6
								out (&FF),a	; select register mode
								out (c),d		; put ay register index
								ld a,&A6
								out (&FF),a	; write register mode
								out (c),e 	; put ay register value
								ld b,a
								dw &71ED		; standby mode
								ret
							endif
							
							; ************
							; *** DATA ***
							; ************
							
_dma_data_buffer
							dw &2FFF				; Repeat 4095x
_dma_data_buffer_frame_start
							dw &0000,&0100			; Tone period A
							dw &0200,&0300			; Tone period B
							dw &0400,&0500			; Tone period C
							dw &0600				; Noise period
							dw &0700+%10111111		; Mixer
							dw &0800,&0900,&0A00	; Volume A,B,C
							dw &0B00,&0C00			; Env. period
							dw &0F00				; Env. shape
							dw &0F00				; Frame-marker
_dma_data_buffer_pause			dw &1000				; Pause
_dma_data_buffer_frame_end
							dw &4001				; loop-repeat
_dma_data_buffer_frame_stop
							; Silence before the DMA stop
							dw &0700+%10111111		; Mixer no tone, no noise
							dw &0800,&0900,&0A00	; Volume A,B,C to 0
							dw &4020				; STOP DMA
_dma_data_buffer_end
