/*
   Reverie
   amigasphere.cpp
   Copyright (C)2003 Dan Potter
*/

#include "global.h"
#include "amigasphere.h"
#include "delaycube.h"

/*

This is an Amiga "boing" like object for Reverie, based on the "bubbles"
sample in KOS.

*/

AmigaSphere::AmigaSphere(DelayCube * st, bool startBounce) {
	m_dc = st;
	if (m_dc)
		m_thetaset = false;
	else
		m_thetaset = true;
	m_theta = 0;
	m_dt = 0;
	m_yaw = 0;
	m_bouncing = startBounce;
	m_size = 6.0f;

	if (m_bouncing) {
		m_dt = 0.0f;
		m_size = 3.0f;
		m_lastvel = m_velocity = Vector(0,2.0f,0);
	}
}

AmigaSphere::~AmigaSphere() {
}

void AmigaSphere::setBouncing() {
	m_bouncing = true;
	m_dt = 1.0f;
}

/*
   Sphere routine from Parallax bubbles example
*/
void AmigaSphere::drawSphere(Color c) {
	int	i, j, foomajor, foominor;
	float	pitch, pitch2;
	float	x, y, z, r, g, b;
	float	yaw, yaw2;
	pvr_dr_state_t	dr_state;

	// If we haven't yet, lock onto the theta of the old sphere.
	if (!m_thetaset) {
		m_theta = m_dc->m_theta;
		m_thetaset = true;
	}

	const float radius = m_size;
	const int slices = 11;
	const int stacks = 20;

	/* Setup our Direct Render state: pick a store queue and setup QACR0/1 */
	pvr_dr_init(dr_state);

	/* Initialize xmtrx with the values from KGL */
	plx_mat_identity();
	plx_mat3d_apply(PLX_MAT_SCREENVIEW);
	plx_mat3d_apply(PLX_MAT_PROJECTION);
	plx_mat3d_apply(PLX_MAT_MODELVIEW);

	/* Iterate over stacks */
	foomajor = 0;
	for ( i=0; i<stacks; i++ ) {
		pitch = 2*M_PI * ( (float)i/(float)stacks );
		pitch2 = 2*M_PI * ( (float)(i+1)/(float)stacks );

		/* Iterate over slices: each entire stack will be one
		   long triangle strip. */
		foominor = 0;
		for ( j=0; j<=slices; j++ ) {
			yaw = 2*M_PI * ( (float)j/(float)(slices*2) );
			yaw2 = 2*M_PI * ( (float)(j+1)/(float)(slices*2) );

			/* x, y+1 */
			x = radius * fcos( yaw ) * fcos( pitch2 );
			y = radius * fsin( pitch2 );
			z = radius * fsin( yaw ) * fcos( pitch2 );
			mat_trans_single(x, y, z);			/* Use ftrv to transform */
			if ((foomajor ^ foominor) & 1) {
				r = 1.0f; g = b = 0.0f;
			} else {
				r = b = 1.0f; g = 1.0f;
			}
			z += 0.001f;	// Put us in front of the delaysphere
			plx_vert_fnd(&dr_state, PVR_CMD_VERTEX,
				x, y, z,
				c.a, c.r*r, c.g*g, c.b*b);
                
			/* x, y */
			x = radius * fcos( yaw ) * fcos( pitch );
			y = radius * fsin( pitch );
			z = radius * fsin( yaw ) * fcos( pitch );
			mat_trans_single(x, y, z);
			if ((foomajor ^ foominor) & 1) {
				r = 1.0f; g = b = 0.0f;
			} else {
				r = b = 1.0f; g = 1.0f;
			}

			z += 0.001f;
			if (j == slices) {
				plx_vert_fnd(&dr_state, PVR_CMD_VERTEX_EOL,
					x, y, z,
					c.a, c.r*r, c.g*g, c.b*b);
			} else {
				plx_vert_fnd(&dr_state, PVR_CMD_VERTEX,
					x, y, z,
					c.a, c.r*r, c.g*g, c.b*b);
			}
			foominor++;
		}
		foomajor++;
	}
}

void AmigaSphere::draw(int list) {
	Color col = getColor();
	if (col.a >= 1.0f) {
		if (list != PLX_LIST_OP_POLY)
			return;
	} else {
		if (list != PLX_LIST_TR_POLY)
			return;
	}

	// Make and submit the context. We want flat shading
	// here to get the checkerboard effect.
	pvr_poly_cxt_t cxt;
	pvr_poly_hdr_t hdr;

	pvr_poly_cxt_col(&cxt, list);
	cxt.gen.culling = PVR_CULLING_NONE;
	cxt.gen.shading = PVR_SHADE_FLAT;
	pvr_poly_compile(&hdr, &cxt);
	pvr_prim(&hdr, sizeof(hdr));

	Vector pos = getTranslate();
	plx_mat3d_identity();
	if (m_bouncing) {
		plx_mat3d_translate(0.0f, 0.0f, -30.0f + -30.0f*(1.0f - m_dt));
		plx_mat3d_rotate(m_theta*2.0f*(1.0f-m_dt), 0.0f, 1.0f, 0.0f);
		plx_mat3d_translate(pos.x, pos.y, pos.z);
	} else {
		plx_mat3d_translate(0.0f, 0.0f, -30.0f);
	}
	plx_mat3d_rotate(m_theta*0.3f, 1.0f, 0.0f, 0.0f);
	plx_mat3d_rotate(m_theta*1.5f, 0.0f, 1.0f, 0.0f);
	if (!m_bouncing || m_dt > 0.0f) {
		if (m_bouncing) {
			plx_mat3d_translate(10.0f * m_dt, 0.0f, 0.0f);
		} else {
			plx_mat3d_translate(10.0f, 0.0f, 0.0f);
		}
		plx_mat3d_rotate(m_theta, 1.0f, 0.0f, 0.0f);
		plx_mat3d_rotate(m_theta*0.5f, 0.0f, 0.1f, 0.0f);
		plx_mat3d_rotate(m_theta*0.3f, 0.0f, 0.0f, 0.1f);
	} else {
		plx_mat3d_rotate(m_theta, 1.0f, 0.0f, 0.0f);
		plx_mat3d_rotate(m_theta*0.5f, 0.0f, 0.1f, 0.0f);
		plx_mat3d_rotate(m_theta*0.3f, 0.0f, 0.0f, 0.1f);
	}

	// Figure out how far the sphere is from the camera so we can shade it.
	// This is kinda cheating since we don't share every poly but who
	// can tell the difference when the camera is in the same place as the light. ;)
	Vector pnt = Vector(0,0,0,1);
	plx_mat_identity();
	plx_mat3d_apply(PLX_MAT_MODELVIEW);
	plx_mat_tfip_3dw(pnt.x, pnt.y, pnt.z, pnt.w);

	// -40 is full bright
	// -100 is full dark
	float cf = 1.0f - ((-pnt.z - 40) / (100 - 40.0));
	if (cf < 0.2f) cf = 0.2f;
	if (cf > 1.0f) cf = 1.0f;
	col.r *= cf;
	col.g *= cf;
	col.b *= cf;

	// Draw a cube at our current position/rotation. We make a snapshot
	// of the full translation matrix after projection so that we can
	// just apply new modelview matrices for the delay cubes.
	drawSphere(col);
}

void AmigaSphere::nextFrame() {
	if (m_dt > 0.0f) {
		m_dt -= 1.0f / 120.0f;
		if (m_dt <= 0.001f) {
			m_dt = 0.0f;
			m_velocity = Vector(0,2.0f,0);
		}

		//Vector p = getTranslate();
		//setTranslate(Vector(p.x, p.y, -40.0f * (1.0f - m_dt)));
		m_size = 6.0f - 3.0f*(1.0f - m_dt);
	}

	m_theta++;

	if (m_bouncing && m_dt <= 0.0f) {
		Vector p = getTranslate();
		p += m_velocity;

		if (p.y < -20.0f) {
			p.y = -20.0f;
			m_velocity.y = -(m_velocity.y/1.1f);
			//printf("bounce %.2f %.2f\n", (double)p.y, (double)m_velocity.y);

			float d = m_lastvel.y - m_velocity.y;
			d = d < 0 ? -d : d;
			if (d < 0.01f)
				m_velocity.y = 3.0f;
		}

		m_lastvel = m_velocity;
		m_velocity.y -= 0.125f;

		setTranslate(p);
	}

	Drawable::nextFrame();
}
