/*
  Flectrum 1.0
  2016/08/15
  Christian Corti
  Neoart Costa Rica
*/

const BackgroundMode = Object.freeze(Object.create(null, {
  "0": { value:"clone"    }, clone   : { value:0, enumerable:true },
  "1": { value:"gradient" }, gradient: { value:1, enumerable:true },
  "2": { value:"color"    }, color   : { value:2, enumerable:true },
  "3": { value:"image"    }, image   : { value:3, enumerable:true }
}));

const Channels = Object.freeze(Object.create(null, {
  "0": { value:"both"   }, both  : { value:0, enumerable:true },
  "1": { value:"left"   }, left  : { value:1, enumerable:true },
  "2": { value:"right"  }, right : { value:2, enumerable:true },
  "3": { value:"stereo" }, stereo: { value:3, enumerable:true }
}));

const Domain = Object.freeze(Object.create(null, {
  "0": { value:"frequency" }, frequency: { value:0, enumerable:true },
  "1": { value:"time"      }, time     : { value:1, enumerable:true }
}));

const MeterMode = Object.freeze(Object.create(null, {
  "0": { value:"gradient"   }, gradient  : { value:0, enumerable:true },
  "1": { value:"image"      }, image     : { value:1, enumerable:true },
  "2": { value:"color"      }, color     : { value:2, enumerable:true },
  "3": { value:"foreground" }, foreground: { value:3, enumerable:true }
}));

const Visual = Object.freeze(Object.create(null, {
  "0": { value:"basic"   }, basic  : { value:0, enumerable:true },
  "1": { value:"split"   }, split  : { value:1, enumerable:true },
  "2": { value:"stripe"  }, stripe : { value:2, enumerable:true },
  "3": { value:"inward"  }, inward : { value:3, enumerable:true },
  "4": { value:"outward" }, outward: { value:4, enumerable:true }
}));

(function() {
"use strict";

  if (!window.neoart) {
    window.neoart = Object.create(null);
  }

  if (!window.neoart.audioContext) {
    window.neoart.audioContext = new AudioContext();
  }

  window.neoart.Flectrum = function(node, cols_no = 32, rows_no = 64) {
    if (!(node instanceof HTMLElement)) { return false; }

    class Flectrum {
      get background() { return background; };
      set background(value) {
        if (value != background) {
          if (value) {
            container.insertBefore(ground.canvas, screen.canvas);
            background = true;
          } else {
            container.removeChild(ground.canvas);
            background = false;
          }
        }
      };

      get backgroundAbove() { return ground.canvas.style.zIndex != ""; };
      set backgroundAbove(value) {
        if (value) {
          ground.canvas.style.zIndex = 1;
        } else {
          ground.canvas.style.removeProperty("z-index");
        }
      };

      get backgroundBeat() { return backBeat; };
      set backgroundBeat(value) {
        if (value) {
          ground.canvas.style.willChange = "opacity";
          backBeat = true;
        } else {
          ground.canvas.style.removeProperty("will-change");
          backBeat = false;
        }
      };

      get backgroundColor() { return backColor; };
      set backgroundColor(value) {
        if (isColor(value)) {
          backColor = value;

          if (backMode == BackgroundMode.color) {
            invalidate(BitGround);
          }
        }
      };

      get backgroundGradient() { return backGradient; };
      set backgroundGradient(value) {
        if (Array.isArray(value)) {
          backGradient = value;

          if (backMode == BackgroundMode.gradient) {
            invalidate(BitGround);
          }
        }
      };

      get backgroundImage() { return backImage; };
      set backgroundImage(value) {
        if (isImage(value)) {
          backImage = value;

          if (backMode == BackgroundMode.image) {
            invalidate(BitGround);
          }
        }
      };

      get backgroundMode() { return backMode; };
      set backgroundMode(value) {
        if (value in BackgroundMode) {
          value = parseInt(value);

          if (value != backMode) {
            backMode = value;
            invalidate(BitGround);
          }
        }
      };

      get backgroundOpacity() { return backOpacity; };
      set backgroundOpacity(value) {
        if (isNumeric(value)) {
          backOpacity = range(value);
          ground.canvas.style.opacity = backOpacity;
        }
      };

      get beatIntensity() { return beatIntensity; };
      set beatIntensity(value) {
        if (isNumeric(value)) {
          beatIntensity = range(value);
        }
      };

      get channels() { return channels; };
      set channels(value) {
        if (value in Channels) {
          channels = parseInt(value);
          invalidate(BitChannel);
        }
      };

      get columns() { return cols; };
      set columns(value) {
        value = parseInt(value);

        if (Number.isInteger(value)) {
          value = range(value, 2, 256);

          if (value != cols) {
            cols = value;
            invalidate(BitCombo1);
          }
        }
      };

      get columnWidth() { return colWidth; };
      set columnWidth(value) {
        value = parseInt(value);

        if (Number.isInteger(value)) {
          value = range(value, 1, 50);

          if (value != colWidth) {
            colWidth = value;
            invalidate(BitCombo1);
          }
        }
      };

      get columnSpacing() { return colSpacing; };
      set columnSpacing(value) {
        value = parseInt(value);

        if (Number.isInteger(value)) {
          value = range(value, 0, 20);

          if (value != colSpacing) {
            colSpacing = value;
            invalidate(BitCombo1);
          }
        }
      };

      get dataDomain() { return dataDomain; };
      set dataDomain(value) {
        if (value in Domain) {
          dataDomain = parseInt(value);
        }
      };

      get decay() { return decay; };
      set decay(value) {
        decay = (value) ? true : false;
      };

      get fades() { return fades; };
      set fades(value) {
        fades = (value) ? true : false;
      };

      get fadeGradient() { return fadeGradient; };
      set fadeGradient(value) {
        if (Array.isArray(value)) {
          fadeGradient = value;
          invalidate(BitFader);
        }
      };

      get foregroundImage() { return foreImage; };
      set foregroundImage(value) {
        if (isImage(value)) {
          foreImage = value;

          if (meterMode == MeterMode.foreground) {
            invalidate(BitMeter);
          }
        }
      };

      get frameRate() { return frameRate; };
      set frameRate(value) {
        value = parseInt(value);

        if (Number.isInteger(value)) {
          frameRate = range(value, 1, 60);
          interval = 1000 / frameRate;
        }
      };

      get height() { return height; };

      get leftMeterColor() { return meterColor[0]; };
      set leftMeterColor(value) {
        if (isColor(value)) {
          meterColor[0] = value;

          if (meterMode == MeterMode.color) {
            invalidate(BitMeter);
          }
        }
      };

      get rightMeterColor() { return meterColor[1]; };
      set rightMeterColor(value) {
        if (isColor(value)) {
          meterColor[1] = value;

          if (meterMode == MeterMode.color) {
            invalidate(BitMeter);
          }
        }
      };

      get meterDecay() { return meterDecay; };
      set meterDecay(value) {
        if (isNumeric(value)) {
          meterDecay = range(value, 0.02, 0.20);
        }
      };

      get leftMeterGradient() { return meterGradient[0]; };
      set leftMeterGradient(value) {
        if (Array.isArray(value)) {
          meterGradient[0] = value;

          if (meterMode == MeterMode.gradient) {
            invalidate(BitMeter);
          }
        }
      };

      get rightMeterGradient() { return meterGradient[1]; };
      set rightMeterGradient(value) {
        if (Array.isArray(value)) {
          meterGradient[1] = value;

          if (meterMode == MeterMode.gradient) {
            invalidate(BitMeter);
          }
        }
      };

      get leftMeterImage() { return meterImage[0]; };
      set leftMeterImage(value) {
        if (isImage(value)) {
          meterImage[0] = value;

          if (meterMode == MeterMode.image) {
            invalidate(BitMeter);
          }
        }
      };

      get rightMeterImage() { return meterImage[1]; };
      set rightMeterImage(value) {
        if (isImage(value)) {
          meterImage[1] = value;

          if (meterMode == MeterMode.image) {
            invalidate(BitMeter);
          }
        }
      };

      get meterMode() { return meterMode; };
      set meterMode(value) {
        if (value in MeterMode) {
          value = parseInt(value);

          if (value != meterMode) {
            meterMode = value;
            invalidate(BitMeter);
          }
        }
      };

      get meterOpacity() { return meterOpacity; };
      set meterOpacity(value) {
        if (isNumeric(value)) {
          value = range(value);
          meterOpacity = value;
          alphas[0] = "rgba(0,0,0,"+ value +")";
        }
      };

      get peaks() { return peaks; };
      set peaks(value) {
        peaks = (value) ? true : false;
      };

      get peakDecay() { return peakDecay; };
      set peakDecay(value) {
        if (isNumeric(value)) {
          peakDecay = range(value, 0.01, 0.10);
        }
      };

      get peakOpacity() { return peakOpacity; };
      set peakOpacity(value) {
        if (isNumeric(value)) {
          peakOpacity = range(value);
          alphas[1] = "rgba(0,0,0,"+ peakOpacity +")";
        }
      };

      get rows() { return rows; };
      set rows(value) {
        value = parseInt(value);

        if (Number.isInteger(value)) {
          value = range(value, 5, 255);

          if (value != rows) {
            rows = value;
            invalidate(BitCombo2);
          }
        }
      };

      get rowHeight() { return rowHeight; };
      set rowHeight(value) {
        value = parseInt(value);

        if (Number.isInteger(value)) {
          value = range(value, 1, 25);

          if (value != rowHeight) {
            rowHeight = value;
            invalidate(BitCombo2);
          }
        }
      };

      get rowSpacing() { return rowSpacing; };
      set rowSpacing(value) {
        value = parseInt(value);

        if (Number.isInteger(value)) {
          value = range(value, 0, 20);

          if (value != rowSpacing) {
            rowSpacing = value;
            invalidate(BitCombo2);
          }
        }
      };

      get smoothingTime() { return smoothTime; };
      set smoothingTime(value) {
        if (isNumeric(value)) {
          smoothTime = range(value);

          if (analyser1) { analyser1.smoothingTimeConstant = smoothTime; }
          if (analyser2) { analyser2.smoothingTimeConstant = smoothTime; }
        }
      };

      get stopSpeed() { return stopSpeed; };
      set stopSpeed(value) {
        value = parseInt(value);

        if (Number.isInteger(value)) {
          stopSpeed = value;
        }
      };

      get trails() { return trails; };
      set trails(value) {
        trails = (value) ? true : false;

        if (!trails) {
          screen.globalCompositeOperation = "copy";
        }
      };

      get trailOpacity() { return trailOpacity; };
      set trailOpacity(value) {
        if (isNumeric(value)) {
          trailOpacity = range(value, 0.1, 0.9);
        }
      };

      get visual() { return visual; };
      set visual(value) {
        if (value in Visual) {
          visual = parseInt(value);
          composer.visual = new Classes[value]();
          invalidate(BitMeter);
        }
      };

      get width() { return width; };

      about() {
        console.info("Flectrum 1.0\n2016/08/15\nChristian Corti\nNeoart Costa Rica");
      };

      append() {
        parent.appendChild(container);

        if (!instances.has(composer)) {
          instances.set(composer, true);
        }
      };

      connect(node) {
        if (node instanceof AudioNode) {
          destination = node;
          composer.createChannels();

          if (!instances.has(composer)) {
            instances.set(composer, true);
          }

          composer.processMode();

          if (!step_id) {
            step_id = window.requestAnimationFrame(step);
          }
        }

        return node;
      };

      disconnect() {
        composer.fadeMode();
        destination = null;
      };

      remove() {
        parent.removeChild(container);
        instances.delete(composer);
      };
    }

    class Composer {
      constructor() {
        this.visual = new Basic();

        faders.canvas.width = 2;

        ground.canvas.style.position =
        screen.canvas.style.position = "absolute";

        ground.canvas.style.opacity = backOpacity;

        container.style.position  = "relative";
        container.style.textAlign = "left";

        container.appendChild(ground.canvas);
        container.appendChild(screen.canvas);
      };

      fade() {
        if (remains--) {
          if (!invalid) {
            this.visual.update();

            if (restore) {
              ground.canvas.style.opacity -= restore;
            }
          }
        } else {
          instances.delete(this);
          screen.clearRect(0,0,width,height);
          this.processMode();

          peakDecay /= stopSpeed;
          meterDecay /= stopSpeed;

          if (!instances.size) {
            window.cancelAnimationFrame(step_id);
            step_id = 0;
          }
        }
      };

      fadeMode() {
        if (this.update == this.process) {
          peakDecay *= stopSpeed;
          meterDecay *= stopSpeed;

          if (peaks) {
            remains = Math.ceil(1.0 / peakDecay);
          } else {
            remains = Math.ceil(1.0 / meterDecay);
          }

          restore = (ground.canvas.style.opacity - backOpacity) / remains;

          reset();
          this.update = this.fade;
        }
      };

      process() {
        if (!invalid) {
          if (dataDomain == Domain.time) {
            this.sampleTime();
          } else {
            this.sampleFrequency();
          }

          this.visual.update();

          if (backBeat) {
            ground.canvas.style.opacity = level * beatIntensity;
          }
        }
      };

      processMode() {
        this.update = this.process;
      };

      createBackground() {
        var i, l;

        switch (backMode) {
          case BackgroundMode.gradient:
            let data = backGradient;
            let grad = ground.createLinearGradient(0,0,0,height);

            for (i = 0, l = data.length; i < l; i++) {
              grad.addColorStop(data[i], data[++i]);
            }

            ground.fillStyle = grad;
            ground.fillRect(0,0,width,height);
            break;
          case BackgroundMode.color:
            ground.fillStyle = backColor;
            ground.fillRect(0,0,width,height);
            break;
          case BackgroundMode.image:
            if (backImage) {
              ground.drawImage(backImage, 0,0,backImage.width,backImage.height, 0,0,width,height);
            }
            break;
          default:
            ground.drawImage(bitmap.canvas,0,0);
            break;
        }
      };

      createChannels() {
        if (destination) {
          reset();

          analyser1 = audio.createAnalyser();
          analyser1.fftSize = freqBin;
          analyser1.smoothingTimeConstant = smoothTime;

          if (channels == Channels.stereo) {
            destination.connect(analyser1);
          } else {
            splitter = audio.createChannelSplitter(2);

            switch (channels) {
              case Channels.left:
                splitter.connect(analyser1, 0, 0);
                break;
              case Channels.right:
                splitter.connect(analyser1, 1, 0);
                break;
              default:
                analyser1.fftsize = freqBin >> 1;

                analyser2 = audio.createAnalyser();
                analyser2.fftsize = freqBin >> 1;
                analyser2.smoothingTimeConstant = smoothTime;

                splitter.connect(analyser1, 0, 0);
                splitter.connect(analyser2, 1, 0);
                break;
            }

            destination.connect(splitter);
          }
        }
      };

      createData() {
        var cols = this.visual.cols;

        freqBin = Math.pow(2, Math.round(Math.log(cols) / Math.log(2))) * 16;

        if (analyser2) {
          analyser1.fftSize = freqBin >> 1;
          analyser2.fftSize = freqBin >> 1;
        } else if (analyser1) {
          analyser1.fftSize = freqBin;
        }

        aldata = new Uint8Array(freqBin);
        ardata = new Uint8Array(freqBin);

        fldata = new Float32Array(cols);
        frdata = new Float32Array(cols);
        cldata = new Float32Array(cols);
        crdata = new Float32Array(cols);
        pldata = new Float32Array(cols);
        prdata = new Float32Array(cols);
      };

      createMask() {
        var tb = faders.createLinearGradient(0,0,0,height);
        var bt = faders.createLinearGradient(0,height,0,0);
        var color, i, l, stop;

        for (i = 0, l = fadeGradient.length; i < l; i++) {
          stop = fadeGradient[i];
          color = fadeGradient[++i];

          tb.addColorStop(stop, color);
          bt.addColorStop(stop, color);
        }

        faders.clearRect(0,0,width,height);
        faders.fillStyle = tb;
        faders.fillRect(0,0,1,height);
        faders.fillStyle = bt;
        faders.fillRect(1,0,1,height);
      };

      resize() {
        colSize = colWidth + colSpacing;
        rowSize = rowHeight + rowSpacing;

        width = (colSize * cols) - colSpacing;
        height = (rowSize * rows) - rowSpacing;

        ground.canvas.width =
        bitmap.canvas.width =
        buffer.canvas.width =
        masker.canvas.width =
        screen.canvas.width = width;

        ground.canvas.height =
        bitmap.canvas.height =
        buffer.canvas.height =
        masker.canvas.height =
        faders.canvas.height =
        screen.canvas.height = height;

        ground.imageSmoothingEnabled =
        bitmap.imageSmoothingEnabled =
        buffer.imageSmoothingEnabled =
        masker.imageSmoothingEnabled =
        faders.imageSmoothingEnabled =
        screen.imageSmoothingEnabled = false;

        ground.globalCompositeOperation =
        screen.globalCompositeOperation = "copy";

        container.style.width = width +"px";
        container.style.height = height +"px";
      };

      sampleFrequency() {
        var spread = (freqBin / (this.visual.cols * 2)) >> 0;
        var c = freqBin >> 2;
        var x = 0;
        var i, val;

        level = 0;
        analyser1.getByteFrequencyData(aldata);

        if (analyser2) {
          analyser2.getByteFrequencyData(ardata);

          for (i = 0; i < c; i += spread) {
            val = aldata[i] / 255;
            level += val;
            fldata[x] = val;

            val = ardata[i] / 255;
            level += val;
            frdata[x] = val;

            x++;
          }
        } else {
          for (i = 0; i < c; i += spread) {
            val = aldata[i] / 255;
            level += val;
            fldata[x] = val;

            val = aldata[i + c] / 255;
            level += val;
            frdata[x] = val;

            x++;
          }
        }

        level /= cols;
      };

      sampleTime() {
        var spread = (freqBin / this.visual.cols) >> 0;
        var abs = Math.abs;
        var c = freqBin >> 1;
        var x = 0;
        var i, val;

        level = 0;
        analyser1.getByteTimeDomainData(aldata);

        if (analyser2) {
          analyser2.getByteTimeDomainData(ardata);

          for (i = 0; i < c; i += spread) {
            val = abs(aldata[i] - 128) / 128;
            level += val;
            fldata[x] = val * (2.0 - val);

            val = abs(ardata[i] - 128) / 128;
            level += val;
            frdata[x] = val * (2.0 - val);

            x++;
          }
        } else {
          for (i = 0; i < c; i += spread) {
            val = abs(aldata[i] - 128) / 128;
            level += val;
            fldata[x] = val * (2.0 - val);

            val = abs(aldata[i + c] - 128) / 128;
            level += val;
            frdata[x] = val * (2.0 - val);

            x++;
          }
        }

        level /= cols;
      };
    }

    class Basic {
      setup() {
        var i, l;

        this.cols = cols;
        this.half = Math.ceil(cols / 2);
        this.midx = this.half * colSize;

        switch (meterMode) {
          case MeterMode.image:
            if (meterImage[0]) {
              let meter = meterImage[0];
              let x = 0;

              for (i = 0, l = cols; i < l; i++) {
                if (i == this.half) {
                  meter = meterImage[1];
                }

                bitmap.drawImage(meter, 0,0,meter.width,meter.height, x,0,colWidth,height);
                x += colSize;
              }
            }
            break;
          case MeterMode.color:
            bitmap.fillStyle = meterColor[0];
            bitmap.fillRect(0,0,this.midx,height);

            bitmap.fillStyle = meterColor[1];
            bitmap.fillRect(this.midx,0,width,height);
            break;
          case MeterMode.foreground:
            if (foreImage) {
              bitmap.drawImage(foreImage, 0,0,foreImage.width,foreImage.height, 0,0,width,height);
            }
            break;
          default:
            let data = meterGradient[0];
            let grad = bitmap.createLinearGradient(0,0,0,height);

            for (i = 0, l = data.length; i < l; i++) {
              grad.addColorStop(data[i], data[++i]);
            }

            bitmap.fillStyle = grad;
            bitmap.fillRect(0,0,this.midx,height);

            data = meterGradient[1];
            grad = bitmap.createLinearGradient(0,0,0,height);

            for (i = 0, l = data.length; i < l; i++) {
              grad.addColorStop(data[i], data[++i]);
            }

            bitmap.fillStyle = grad;
            bitmap.fillRect(this.midx,0,width,height);
            break;
        }

        this.grid();
        composer.createData();
      };

      grid() {
        var i, l, p;

        if (rowSpacing) {
          p = rowHeight;

          for (i = 0, l = rows - 1; i < l; i++) {
            bitmap.clearRect(0,p,width,rowSpacing);
            p += rowSize;
          }
        }

        if (meterMode == MeterMode.image) { return; }

        if (colSpacing) {
          p = colWidth;

          for (i = 0, l = cols - 1; i < l; i++) {
            bitmap.clearRect(p,0,colSpacing,height);
            p += colSize;
          }
        }
      };

      before() {
        if (trails) {
          screen.globalCompositeOperation = "copy";
          screen.globalAlpha = trailOpacity;
          screen.drawImage(screen.canvas,0,0);
          screen.globalAlpha = 1.0;
          screen.globalCompositeOperation = "source-over";
        }

        buffer.drawImage(bitmap.canvas,0,0);
        masker.clearRect(0,0,width,height);
      };

      after() {
        buffer.globalCompositeOperation = "destination-in";
        buffer.drawImage(masker.canvas,0,0);
        buffer.globalCompositeOperation = "copy";

        screen.drawImage(buffer.canvas,0,0);
      };

      update() {
        var am = alphas[0];
        var ap = alphas[1];
        var xl = 0;
        var xr = this.midx;
        var i = 0;
        var l = this.half;
        var lh, lv, rh, rv;

        this.before();

        for (; i < l; i++) {
          lv = fldata[i];
          rv = frdata[i];

          if (decay) {
            if (lv < cldata[i]) {
              lv = (cldata[i] -= meterDecay);
              if (lv < 0.0) { lv = 0.0; }
            } else {
              cldata[i] = lv;
            }

            if (rv < crdata[i]) {
              rv = (crdata[i] -= meterDecay);
              if (rv < 0.0) { rv = 0.0; }
            } else {
              crdata[i] = rv;
            }
          }

          lh = ((lv * rows) << 0) * rowSize - rowSpacing;
          rh = ((rv * rows) << 0) * rowSize - rowSpacing;

          if (fades) {
            masker.globalAlpha = meterOpacity;

            if (lh) {
              masker.drawImage(faders.canvas, 0,0,1,lh, xl,height-lh,colWidth,lh);
            }

            if (rh) {
              masker.drawImage(faders.canvas, 0,0,1,rh, xr,height-rh,colWidth,rh);
            }

            masker.globalAlpha = 1.0;
          } else {
            masker.fillStyle = am;
            masker.fillRect(xl,height-lh,colWidth,lh);
            masker.fillRect(xr,height-rh,colWidth,rh);
          }

          if (peaks) {
            masker.fillStyle = ap;

            if (lv > pldata[i]) { pldata[i] = lv; }

            lh = height - ((pldata[i] * rows << 0) * rowSize - rowSpacing);
            masker.fillRect(xl,lh,colWidth,rowHeight);
            pldata[i] -= peakDecay;

            if (rv > prdata[i]) { prdata[i] = rv; }

            rh = height - ((prdata[i] * rows << 0) * rowSize - rowSpacing);
            masker.fillRect(xr,rh,colWidth,rowHeight);
            prdata[i] -= peakDecay;
          }

          xl += colSize;
          xr += colSize;
        }

        this.after();
      };
    }

    class Split extends Basic {
      setup() {
        var i, l;

        this.cols = cols;
        this.half = Math.ceil(cols / 2);
        this.midx = this.half * colSize;

        switch (meterMode) {
          case MeterMode.image:
            if (meterImage[0]) {
              let meter = meterImage[0];
              let x = 0;

              for (i = 0, l = cols; i < l; i++) {
                if (i == this.half) {
                  meter = meterImage[1];

                  bitmap.save();
                  bitmap.translate(0,height);
                  bitmap.scale(1,-1);
                }

                bitmap.drawImage(meter, 0,0,meter.width,meter.height, x,0,colWidth,height);
                x += colSize;
              }

              bitmap.restore();
            }
            break;
          case MeterMode.color:
            bitmap.fillStyle = meterColor[0];
            bitmap.fillRect(0,0,this.midx,height);

            bitmap.fillStyle = meterColor[1];
            bitmap.fillRect(this.midx,0,width,height);
            break;
          case MeterMode.foreground:
            if (foreImage) {
              bitmap.drawImage(foreImage, 0,0,foreImage.width,foreImage.height, 0,0,width,height);
            }
            break;
          default:
            let data = meterGradient[0];
            let grad = bitmap.createLinearGradient(0,0,0,height);

            for (i = 0, l = data.length; i < l; i++) {
              grad.addColorStop(data[i], data[++i]);
            }

            bitmap.fillStyle = grad;
            bitmap.fillRect(0,0,this.midx,height);

            data = meterGradient[1];
            grad = bitmap.createLinearGradient(0,height,0,0);

            for (i = 0, l = data.length; i < l; i++) {
              grad.addColorStop(data[i], data[++i]);
            }

            bitmap.fillStyle = grad;
            bitmap.fillRect(this.midx,0,width,height);
            break;
        }

        this.grid();
        composer.createData();
      };

      update() {
        var am = alphas[0];
        var ap = alphas[1];
        var xl = 0;
        var xr = this.midx;
        var i = 0;
        var l = this.half;
        var lh, lv, rh, rv;

        this.before();

        for (; i < l; i++) {
          lv = fldata[i];
          rv = frdata[i];

          if (decay) {
            if (lv < cldata[i]) {
              lv = (cldata[i] -= meterDecay);
              if (lv < 0.0) { lv = 0.0; }
            } else {
              cldata[i] = lv;
            }

            if (rv < crdata[i]) {
              rv = (crdata[i] -= meterDecay);
              if (rv < 0.0) { rv = 0.0; }
            } else {
              crdata[i] = rv;
            }
          }

          lh = ((lv * rows) << 0) * rowSize - rowSpacing;
          rh = ((rv * rows) << 0) * rowSize - rowSpacing;

          if (fades) {
            masker.globalAlpha = meterOpacity;

            if (lh) {
              masker.drawImage(faders.canvas, 0,0,1,lh, xl,height-lh,colWidth,lh);
            }

            if (rh) {
              masker.drawImage(faders.canvas, 1,height-rh,1,rh, xr,0,colWidth,rh);
            }

            masker.globalAlpha = 1.0;
          } else {
            masker.fillStyle = am;
            masker.fillRect(xl,height-lh,colWidth,lh);
            masker.fillRect(xr,0,colWidth,rh);
          }

          if (peaks) {
            masker.fillStyle = ap;

            if (lv > pldata[i]) { pldata[i] = lv; }

            lh = height - ((pldata[i] * rows << 0) * rowSize - rowSpacing);
            masker.fillRect(xl,lh,colWidth,rowHeight);
            pldata[i] -= peakDecay;

            if (rv > prdata[i]) { prdata[i] = rv; }

            rh = ((prdata[i] * rows << 0) * rowSize - rowSpacing) - rowHeight;
            masker.fillRect(xr,rh,colWidth,rowHeight);
            prdata[i] -= peakDecay;
          }

          xl += colSize;
          xr += colSize;
        }

        this.after();
      };
    }

    class Stripe extends Basic {
      setup() {
        var x = 0;
        var i, l;

        this.cols = cols;
        this.half = Math.ceil(cols / 2);
        this.step = colSize * 2;

        switch (meterMode) {
          case MeterMode.image:
            if (meterImage[0]) {
              let meter = meterImage[0];

              for (i = 0, l = this.half; i < l; i++) {
                bitmap.drawImage(meter, 0,0,meter.width,meter.height, x,0,colWidth,height);
                x += this.step;
              }

              meter = meterImage[1];

              bitmap.save();
              bitmap.translate(0,height);
              bitmap.scale(1,-1);
              x = colSize;

              for (i = 0, l = this.half; i < l; i++) {
                bitmap.drawImage(meter, 0,0,meter.width,meter.height, x,0,colWidth,height);
                x += this.step;
              }

              bitmap.restore();
            }
            break;
          case MeterMode.color:
            for (i = 0, l = this.half; i < l; i++) {
              bitmap.fillStyle = meterColor[0];
              bitmap.fillRect(x,0,colWidth,height);
              x += colSize;

              bitmap.fillStyle = meterColor[1];
              bitmap.fillRect(x,0,colWidth,height);
              x += colSize;
            }
            break;
          case MeterMode.foreground:
            if (foreImage) {
              bitmap.drawImage(foreImage, 0,0,foreImage.width,foreImage.height, 0,0,width,height);
            }
            break;
          default:
            let data1 = meterGradient[0];
            let data2 = meterGradient[1];
            let grad1 = bitmap.createLinearGradient(0,0,0,height);
            let grad2 = bitmap.createLinearGradient(0,height,0,0);

            for (i = 0, l = data1.length; i < l; i++) {
              grad1.addColorStop(data1[i], data1[++i]);
            }

            for (i = 0, l = data2.length; i < l; i++) {
              grad2.addColorStop(data2[i], data2[++i]);
            }

            for (i = 0, l = this.half; i < l; i++) {
              bitmap.fillStyle = grad1;
              bitmap.fillRect(x,0,colWidth,height);
              x += colSize;

              bitmap.fillStyle = grad2;
              bitmap.fillRect(x,0,colWidth,height);
              x += colSize;
            }
            break;
        }

        this.grid();
        composer.createData();
      };

      update() {
        var am = alphas[0];
        var ap = alphas[1];
        var xl = 0;
        var xr = colSize;
        var i = 0;
        var l = this.half;
        var lh, lv, rh, rv;

        this.before();

        for (; i < l; i++) {
          lv = fldata[i];
          rv = frdata[i];

          if (decay) {
            if (lv < cldata[i]) {
              lv = (cldata[i] -= meterDecay);
              if (lv < 0.0) { lv = 0.0; }
            } else {
              cldata[i] = lv;
            }

            if (rv < crdata[i]) {
              rv = (crdata[i] -= meterDecay);
              if (rv < 0.0) { rv = 0.0; }
            } else {
              crdata[i] = rv;
            }
          }

          lh = ((lv * rows) << 0) * rowSize - rowSpacing;
          rh = ((rv * rows) << 0) * rowSize - rowSpacing;

          if (fades) {
            masker.globalAlpha = meterOpacity;

            if (lh) {
              masker.drawImage(faders.canvas, 0,0,1,lh, xl,height-lh,colWidth,lh);
            }

            if (rh) {
              masker.drawImage(faders.canvas, 1,height-rh,1,rh, xr,0,colWidth,rh);
            }

            masker.globalAlpha = 1.0;
          } else {
            masker.fillStyle = am;
            masker.fillRect(xl,height-lh,colWidth,lh);
            masker.fillRect(xr,0,colWidth,rh);
          }

          if (peaks) {
            masker.fillStyle = ap;

            if (lv > pldata[i]) { pldata[i] = lv; }

            lh = height - ((pldata[i] * rows << 0) * rowSize - rowSpacing);
            masker.fillRect(xl,lh,colWidth,rowHeight);
            pldata[i] -= peakDecay;

            if (rv > prdata[i]) { prdata[i] = rv; }

            rh = ((prdata[i] * rows << 0) * rowSize - rowSpacing) - rowHeight;
            masker.fillRect(xr,rh,colWidth,rowHeight);
            prdata[i] -= peakDecay;
          }

          xl += this.step;
          xr += this.step;
        }

        this.after();
      };
    }

    class Inward extends Basic {
      setup() {
        var i, l;

        this.cols = cols * 2;
        this.half = Math.ceil(rows / 2);
        this.midy = this.half * rowSize;

        switch (meterMode) {
          case MeterMode.image:
            if (meterImage[0]) {
              let meter = meterImage[0];
              let x = 0;

              for (i = 0, l = cols; i < l; i++) {
                bitmap.drawImage(meter, 0,0,meter.width,meter.height, x,this.midy,colWidth,this.midy);
                x += colSize;
              }

              meter = meterImage[1];

              bitmap.save();
              bitmap.translate(0,height);
              bitmap.scale(1,-1);
              x = 0;

              for (i = 0, l = cols; i < l; i++) {
                bitmap.drawImage(meter, 0,0,meter.width,meter.height, x,this.midy,colWidth,this.midy);
                x += colSize;
              }

              bitmap.restore();
            }
            break;
          case MeterMode.color:
            bitmap.fillStyle = meterColor[0];
            bitmap.fillRect(0,0,width,this.midy);

            bitmap.fillStyle = meterColor[1];
            bitmap.fillRect(0,this.midy,width,height);
            break;
          case MeterMode.foreground:
            if (foreImage) {
              bitmap.drawImage(foreImage, 0,0,foreImage.width,foreImage.height, 0,0,width,height);
            }
            break;
          default:
            let data = meterGradient[0];
            let grad = bitmap.createLinearGradient(0,this.midy,0,0);

            for (i = 0, l = data.length; i < l; i++) {
              grad.addColorStop(data[i], data[++i]);
            }

            bitmap.fillStyle = grad;
            bitmap.fillRect(0,0,width,this.midy);

            data = meterGradient[1];
            grad = bitmap.createLinearGradient(0,this.midy,0,height);

            for (i = 0, l = data.length; i < l; i++) {
              grad.addColorStop(data[i], data[++i]);
            }

            bitmap.fillStyle = grad;
            bitmap.fillRect(0,this.midy,width,height);
            break;
        }

        this.grid();
        composer.createData();
      };

      update() {
        var am = alphas[0];
        var ap = alphas[1];
        var i = 0;
        var l = cols;
        var x = 0;
        var lh, lv, rh, rv;

        this.before();

        for (; i < l; i++) {
          lv = fldata[i];
          rv = frdata[i];

          if (decay) {
            if (lv < cldata[i]) {
              lv = (cldata[i] -= meterDecay);
              if (lv < 0.0) { lv = 0.0; }
            } else {
              cldata[i] = lv;
            }

            if (rv < crdata[i]) {
              rv = (crdata[i] -= meterDecay);
              if (rv < 0.0) { rv = 0.0; }
            } else {
              crdata[i] = rv;
            }
          }

          lh = ((lv * this.half) << 0) * rowSize - rowSpacing;
          rh = ((rv * this.half) << 0) * rowSize - rowSpacing;

          if (fades) {
            masker.globalAlpha = meterOpacity;

            if (lh) {
              masker.drawImage(faders.canvas, 1,height-lh,1,lh, x,0,colWidth,lh);
            }

            if (rh) {
              masker.drawImage(faders.canvas, 0,0,1,rh, x,height-rh,colWidth,rh);
            }

            masker.globalAlpha = 1.0;
          } else {
            masker.fillStyle = am;
            masker.fillRect(x,0,colWidth,lh);
            masker.fillRect(x,height-rh,colWidth,rh);
          }

          if (peaks) {
            masker.fillStyle = ap;

            if (lv > pldata[i]) { pldata[i] = lv; }

            lh = ((pldata[i] * this.half << 0) * rowSize - rowSpacing) - rowHeight;
            masker.fillRect(x,lh,colWidth,rowHeight);
            pldata[i] -= peakDecay;

            if (rv > prdata[i]) { prdata[i] = rv; }

            rh = height - ((prdata[i] * this.half << 0) * rowSize - rowSpacing);
            masker.fillRect(x,rh,colWidth,rowHeight);
            prdata[i] -= peakDecay;
          }

          x += colSize;
        }

        this.after();
      };
    }

    class Outward extends Basic {
      setup() {
        var i, l;

        this.cols = cols * 2;
        this.half = Math.ceil(rows / 2);
        this.midy = this.half * rowSize;

        switch (meterMode) {
          case MeterMode.image:
            if (meterImage[0]) {
              let meter = meterImage[0];
              let x = 0;

              for (i = 0, l = cols; i < l; i++) {
                bitmap.drawImage(meter, 0,0,meter.width,meter.height, x,0,colWidth,this.midy);
                x += colSize;
              }

              meter = meterImage[1];

              bitmap.save();
              bitmap.translate(0,height);
              bitmap.scale(1,-1);
              x = 0;

              for (i = 0, l = cols; i < l; i++) {
                bitmap.drawImage(meter, 0,0,meter.width,meter.height, x,0,colWidth,this.midy);
                x += colSize;
              }

              bitmap.restore();
            }
            break;
          case MeterMode.color:
            bitmap.fillStyle = meterColor[0];
            bitmap.fillRect(0,0,width,this.midy);

            bitmap.fillStyle = meterColor[1];
            bitmap.fillRect(0,this.midy,width,height);
            break;
          case MeterMode.foreground:
            if (foreImage) {
              bitmap.drawImage(foreImage, 0,0,foreImage.width,foreImage.height, 0,0,width,height);
            }
            break;
          default:
            let data = meterGradient[0];
            let grad = bitmap.createLinearGradient(0,0,0,this.midy);

            for (i = 0, l = data.length; i < l; i++) {
              grad.addColorStop(data[i], data[++i]);
            }

            bitmap.fillStyle = grad;
            bitmap.fillRect(0,0,width,this.midy);

            data = meterGradient[1];
            grad = bitmap.createLinearGradient(0,height,0,this.midy);

            for (i = 0, l = data.length; i < l; i++) {
              grad.addColorStop(data[i], data[++i]);
            }

            bitmap.fillStyle = grad;
            bitmap.fillRect(0,this.midy,width,height);
            break;
        }

        this.grid();
        composer.createData();
      };

      update() {
        var am = alphas[0];
        var ap = alphas[1];
        var i = 0;
        var l = cols;
        var x = 0;
        var lh, lv, rh, rv;

        this.before();

        for (; i < l; i++) {
          lv = fldata[i];
          rv = frdata[i];

          if (decay) {
            if (lv < cldata[i]) {
              lv = (cldata[i] -= meterDecay);
              if (lv < 0.0) { lv = 0.0; }
            } else {
              cldata[i] = lv;
            }

            if (rv < crdata[i]) {
              rv = (crdata[i] -= meterDecay);
              if (rv < 0.0) { rv = 0.0; }
            } else {
              crdata[i] = rv;
            }
          }

          lh = ((lv * this.half) << 0) * rowSize;
          rh = ((rv * this.half) << 0) * rowSize - rowSpacing;

          if (fades) {
            masker.globalAlpha = meterOpacity;

            if (lh) {
              masker.drawImage(faders.canvas, 0,0,1,lh, x,this.midy-lh,colWidth,lh);
            }

            if (rh) {
              masker.drawImage(faders.canvas, 1,height-rh,1,rh, x,this.midy,colWidth,rh);
            }

            masker.globalAlpha = 1.0;
          } else {
            masker.fillStyle = am;
            masker.fillRect(x,this.midy-lh,colWidth,lh);
            masker.fillRect(x,this.midy,colWidth,rh);
          }

          if (peaks) {
            masker.fillStyle = ap;

            if (lv > pldata[i]) { pldata[i] = lv; }
            lh = this.midy - ((pldata[i] * this.half << 0) * rowSize);

            if (lh < this.midy) {
              masker.fillRect(x,lh,colWidth,rowHeight);
              pldata[i] -= peakDecay;
            }

            if (rv > prdata[i]) { prdata[i] = rv; }
            rh = this.midy + ((prdata[i] * this.half << 0) * rowSize - rowSpacing) - rowHeight;

            if (rh > this.midy) {
              masker.fillRect(x,rh,colWidth,rowHeight);
              prdata[i] -= peakDecay;
            }
          }

          x += colSize;
        }

        this.after();
      };
    }

    function reset() {
      if (destination) {
        if (splitter) {
          destination.disconnect(splitter);
        } else if (analyser1) {
          destination.disconnect(analyser1);
        }
      } else if (instances.has(composer)) {
        instances.delete(composer);
      }

      fldata.fill(0);
      frdata.fill(0);

      analyser1 = null;
      analyser2 = null;
      splitter  = null;
    }

    function invalidate(bits) {
      state |= bits;

      if (!invalid) {
        invalid = 1;
        window.requestAnimationFrame(validate);
      }
    }

    function validate(e) {
      if (state & BitSize) { composer.resize(); }

      if (state & BitMeter) {
        composer.visual.setup();

        if (backMode == BackgroundMode.clone) {
          composer.createBackground();
          state ^= BitGround;
        }
      }

      if (state & BitGround) { composer.createBackground(); }

      if (state & BitFader) { composer.createMask(); }

      if (state & BitChannel) { composer.createChannels(); }

      invalid = state = 0;
    }

    const Classes = {
      "0": Basic,
      "1": Split,
      "2": Stripe,
      "3": Inward,
      "4": Outward
    };

    const BitSize    = 1;
    const BitMeter   = 2;
    const BitGround  = 4;
    const BitFader   = 8;
    const BitChannel = 16;
    const BitCombo1  = 7;
    const BitCombo2  = 15;

    var background    = true;
    var backBeat      = false;
    var backColor     = "#000000";
    var backGradient  = [0.0, "#fceabb", 0.5, "#fccd4d", 0.5, "#f8b500", 1.0, "#fbdf93"];
    var backImage     = null;
    var backMode      = BackgroundMode.clone;
    var backOpacity   = 0.2;
    var beatIntensity = 0.7;
    var channels      = Channels.both;
    var cols          = cols_no;
    var colWidth      = 10;
    var colSpacing    = 1;
    var dataDomain    = Domain.frequency;
    var decay         = true;
    var fades         = false;
    var fadeGradient  = [0.0, "rgba(0,0,0,0)", 0.065, "rgba(0,0,0,0.5)", 0.13, "rgba(0,0,0,1.0)"];
    var foreImage     = null;
    var meterColor    = ["#50d020", "#50d020"];
    var meterDecay    = 0.08;
    var meterGradient = [[0.039, "#ff3939", 0.235, "#ffb320", 0.431, "#fff820", 0.941, "#50d020"], [0.039, "#ff3939", 0.235, "#ffb320", 0.431, "#fff820", 0.941, "#50d020"]];
    var meterImage    = [null, null];
    var meterMode     = MeterMode.gradient;
    var meterOpacity  = 1.0;
    var peaks         = true;
    var peakDecay     = 0.02;
    var peakOpacity   = 1.0;
    var rows          = rows_no;
    var rowHeight     = 3;
    var rowSpacing    = 1;
    var smoothTime    = 0.0;
    var stopSpeed     = 1;
    var trails        = false;
    var trailOpacity  = 0.7;
    var visual        = Visual.basic;

    var audio = window.neoart.audioContext;
    var analyser1;
    var analyser2;
    var destination;
    var splitter;

    var aldata, ardata;
    var fldata, frdata;
    var cldata, crdata;
    var pldata, prdata;

    var alphas  = ["rgba(0,0,0,1.0)", "rgba(0,0,0,1.0)"];
    var colSize = 0;
    var freqBin = 0;
    var height  = 0;
    var invalid = 0;
    var level   = 0;
    var parent  = node;
    var remains = 0;
    var restore = 0.0;
    var rowSize = 0;
    var state   = 0;
    var width   = 0;

    var ground = document.createElement("canvas").getContext("2d", false);
    var bitmap = document.createElement("canvas").getContext("2d", false);
    var buffer = document.createElement("canvas").getContext("2d", false);
    var masker = document.createElement("canvas").getContext("2d", false);
    var faders = document.createElement("canvas").getContext("2d", false);
    var screen = document.createElement("canvas").getContext("2d", false);

    var container = document.createElement("div");

    var composer = new Composer();
    invalidate(BitCombo2);

    node.appendChild(container);
    return Object.seal(new Flectrum());
  }

  function isColor(value) {
    var span = document.createElement("span");
    span.style.color = value;
    return span.style.color.trim() !== "";
  }

  function isImage(value) {
    return (value instanceof HTMLImageElement || value instanceof HTMLCanvasElement);
  }

  function isNumeric(value) {
    return !Number.isNaN(value) && isFinite(value);
  }

  function range(value, min = 0.0, max = 1.0) {
    if (value < min) {
      value = min;
    } else if (value > max) {
      value = max;
    }

    return value;
  }

  var instances = new Map();
  var before    = window.performance.now();
  var frameRate = 15;
  var interval  = 1000 / frameRate;
  var step_id   = 0;

  function step(now) {
    var delta = now - before;
    var obj;

    step_id = window.requestAnimationFrame(step);

    if (delta >= interval) {
      for (obj of instances.keys()) { obj.update(); }
      before = now - (delta % interval);
    }
  }
})();