#include "module_manager.h"
#include "file_system/file_system.h"

PSPOS_SingletonCpp( CModuleManager );

CModuleManager::CModuleManager( void )
{
    PSPOS_CreateSingleton( CModuleManager );

    Initialize();
}

CModuleManager::~CModuleManager( void )
{
    PSPOS_DestroySingleton( CModuleManager );

    Finalize();
}

bool CModuleManager::FindModuleByName( CModule ** module, ulong * index, CString module_name ) const
{
    ulong module_index;

    for( module_index = 0 ; module_index < ModuleArray.GetSize() ; module_index ++ )
    {
        if( (*ModuleArray[ module_index ])->GetName() == module_name )
        {
            *module = (*ModuleArray[ module_index ]);
            *index = module_index;

            return true;
        }
    }

    return false;
}

bool CModuleManager::FindModuleByType( CModule ** module, ulong * index, const MODULE_TYPE & module_type ) const
{
    ulong module_index;

    for( module_index = 0 ; module_index < ModuleArray.GetSize() ; module_index ++ )
    {
        if( (*ModuleArray[ module_index ])->GetType() == module_type )
        {
            *module = (*ModuleArray[ module_index ]);
            *index = module_index;

            return true;
        }
    }

    return false;
}

bool CModuleManager::FindModuleNameByType( CString & module_name, const MODULE_TYPE & module_type ) const
{
    CModule * module;
    ulong index;

    if( !FindModuleByType( &module, &index, module_type ) )
        return false;

    module_name = module->GetName();

    return true;
}

bool CModuleManager::FindModuleIndex( ulong * index, CModule & module ) const
{
    ulong module_index;

    for( module_index = 0 ; module_index < ModuleArray.GetSize() ; module_index ++ )
    {
        if( (*ModuleArray[ module_index ]) == &module )
        {
            *index = module_index;

            return true;
        }
    }

    return false;
}

void CModuleManager::Initialize( void )
{
}

void CModuleManager::Finalize( void )
{
    while( ModuleArray.GetSize() > 0 )
        StopModule( *(*ModuleArray[ 0 ]) );

    ModuleArray.Erase();
}

void CModuleManager::Update( void )
{    
    ulong module_index;

    for( module_index = 0 ; module_index < ModuleArray.GetSize() ; module_index ++ )
        (*ModuleArray[ module_index ])->Update();
}

void CModuleManager::StartModules( void )
{
    ulong module_index;

    for( module_index = 0 ; module_index < ModuleArray.GetSize() ; module_index ++ )
        StartModule( *(*ModuleArray[ module_index ]), false );
}

void CModuleManager::StopModules( void )
{
    ulong module_index;

    for( module_index = 0 ; module_index < ModuleArray.GetSize() ; module_index ++ )
        StopModule( *(*ModuleArray[ module_index ]), false );
}

bool CModuleManager::StartModule( CModule & module, bool it_has_to_manage_array )
{
    if( module.Start() )
    {
        if( module.GetType() < MODULE_TYPE_Kernel )
            AttachKernelModule( module );

        if( it_has_to_manage_array )
        {
            ModuleArray.Add( &module );
            UpdateLibraries();
        }

        return true;
    }

    return false;
}

bool CModuleManager::StopModule( CModule & module, bool it_has_to_manage_array )
{
    if( module.Stop() )
    {
        if( module.GetType() < MODULE_TYPE_Kernel )
            DetachKernelModule( module );

        if( it_has_to_manage_array )
        {
            ulong module_index;

            if( !FindModuleIndex( &module_index, module ) )
                return false;

            delete (*ModuleArray[ module_index ]);
            ModuleArray.Remove( module_index );
            UpdateLibraries();
        }

        return true;
    }

    return false;
}

bool CModuleManager::StartModule( CString module_name, CString module_path )
{
    CModule * module;

    module = new CModule( module_name, module_path );

    return StartModule( *module );
}

bool CModuleManager::StopModule( CString module_name )
{
    CModule * module;
    ulong module_index;

    if( !FindModuleByName( &module, &module_index, module_name ) )
        return false;

    return StopModule( *module );
}

void CModuleManager::AttachKernelModule( CModule & module )
{
    CreateProcFile( module );

    switch( module.GetType() )
    {
        case MODULE_TYPE_KernelDisplay:
            break;

        case MODULE_TYPE_KernelKeyboard:
            break;

        case MODULE_TYPE_KernelTerminal:
            break;
    }
}

void CModuleManager::DetachKernelModule( CModule & module )
{
    switch( module.GetType() )
    {
        case MODULE_TYPE_KernelDisplay:
            break;

        case MODULE_TYPE_KernelKeyboard:
            break;

        case MODULE_TYPE_KernelTerminal:
            break;
    }

    DestroyProcFile( module );
}

void CModuleManager::CreateProcFile( CModule & module )
{
    int proc_file;
    CString proc_file_path;

    if( KERNEL_FileSystem().FindFullPath( proc_file_path, "proc/" ) )
    {
        proc_file_path += module.GetName();
        proc_file = sceIoOpen( proc_file_path.GetBuffer(), PSP_O_WRONLY | PSP_O_CREAT | PSP_O_TRUNC, 0777 );
        sceIoClose( proc_file );
    }
}

void CModuleManager::DestroyProcFile( CModule & module )
{
    CString proc_file_path;
    CString full_path;

    proc_file_path = "proc/";
    proc_file_path += module.GetName();

    if( KERNEL_FileSystem().FindFullPath( full_path, proc_file_path ) )
    {
        sceIoRemove( full_path.GetBuffer() );
    }
}

void CModuleManager::UpdateLibraries( void )
{
    ulong module_index;

    for( module_index = 0 ; module_index < ModuleArray.GetSize() ; module_index ++ )
        (*ModuleArray[ module_index ])->UpdateLibrary();

    LIBS_Update();
}
