#include "keyboard.h"

extern "C"
{
    #include <pikeylib.h>
    #include "pikeytypes.h"
}

CKeyboard::CKeyboard( void ) :
    CurrentKey( NONE )
{
    CreateButtons();
}

CKeyboard::~CKeyboard( void )
{
    DestroyButtons();
}

bool CKeyboard::IsKeyPressed( uint32 key_code ) const
{
    return ( *ButtonArray[ key_code ] )->IsPressed();
}

bool CKeyboard::IsKeyJustPressed( uint32 key_code ) const
{
    return ( *ButtonArray[ key_code ] )->IsJustPressed();
}

uint32 CKeyboard::GetKeyJustPressed( void ) const
{
    return CurrentKey;
}

CString CKeyboard::GetStringFromKey( uint32 key_code ) const
{
    int32 alt_key, ctrl_key, shift_key;

    GetMetaKeys( &alt_key, &ctrl_key, &shift_key );

    if( alt_key != 0 )
        return GetStringFromKeyAlt( key_code );

    if( shift_key != 0 )
        return GetStringFromKeyShift( key_code );

    return GetStringFromKeyDefault( key_code );
}

void CKeyboard::GetMetaKeys( int32 * alt_key, int32 * ctrl_key, int32 * shift_key ) const
{
    *alt_key = MetaKeys[ 0 ];
    *ctrl_key = MetaKeys[ 1 ];
    *shift_key = MetaKeys[ 2 ];
}

bool CKeyboard::Initialize( void )
{
    if( !initPikey() )
        return false;

    sceKernelDelayThread( 1000000 );
    
    if( !requestExclusive() )
        return false;

    if( setMode( PIKEY_KEYMODE_KEYPRESS ) != PIKEY_SUCCESS )
        return false;

    for( uint32 i=0 ; i<10 ; i++ )
        sceKernelDelayThread( 1000000 );

    return isPikeyLoaded();
}

void CKeyboard::Finalize( void )
{
    releaseExclusive();
}

void CKeyboard::Update( void )
{
    uint32 button_index;

    getMetaKeys( &MetaKeys[ 0 ], &MetaKeys[ 1 ], &MetaKeys[ 2 ] );

    CurrentKey = NONE;

    for( button_index = 0 ; button_index < KEY_MAX ; button_index ++ )
    {
        if( ( *ButtonArray[ button_index ] )->Update() )
            CurrentKey = button_index;
    }
}

void CKeyboard::CreateButtons( void )
{
    uint32 button_index;

    for( button_index = 0 ; button_index < KEY_MAX ; button_index ++ )
        ButtonArray.Add( new CKeyboardButton( button_index ) );
}

void CKeyboard::DestroyButtons( void )
{
    uint32 button_index;

    for( button_index = 0 ; button_index < ButtonArray.GetSize() ; button_index ++ )
    {
        delete *ButtonArray[ button_index ];
        *ButtonArray[ button_index ] = NULL;
    }

    ButtonArray.Erase();
}

CString CKeyboard::GetStringFromKeyDefault( uint32 key_code ) const
{
    switch( key_code )
    {
        case KEY_1:             return "1";
        case KEY_2:             return "2";
        case KEY_3:             return "3";
        case KEY_4:             return "4";
        case KEY_5:             return "5";
        case KEY_6:             return "6";
        case KEY_7:             return "7";
        case KEY_8:             return "8";
        case KEY_9:             return "9";
        case KEY_0:             return "0";
        case KEY_MINUS:         return "-";
        case KEY_EQUAL:         return "=";
        case KEY_TAB:           return "    ";
        case KEY_Q:             return "q";
        case KEY_W:             return "w";
        case KEY_E:             return "e";
        case KEY_R:             return "r";
        case KEY_T:             return "t";
        case KEY_Y:             return "y";
        case KEY_U:             return "u";
        case KEY_I:             return "i";
        case KEY_O:             return "o";
        case KEY_P:             return "p";
        case KEY_LEFTBRACE:     return "(";
        case KEY_RIGHTBRACE:    return ")";
        case KEY_A:             return "a";
        case KEY_S:             return "s";
        case KEY_D:             return "d";
        case KEY_F:             return "f";
        case KEY_G:             return "g";
        case KEY_H:             return "h";
        case KEY_J:             return "j";
        case KEY_K:             return "k";
        case KEY_L:             return "l";
        case KEY_SEMICOLON:     return ";";
        case KEY_APOSTROPHE:    return "'";
        case KEY_GRAVE:         return "^";
        case KEY_BACKSLASH:     return "\\";
        case KEY_Z:             return "z";
        case KEY_X:             return "x";
        case KEY_C:             return "c";
        case KEY_V:             return "v";
        case KEY_B:             return "b";
        case KEY_N:             return "n";
        case KEY_M:             return "m";
        case KEY_COMMA:         return ",";
        case KEY_DOT:           return ".";
        case KEY_SLASH:         return "/";
        case KEY_SPACE:         return " ";
    }

    return "";
}

CString CKeyboard::GetStringFromKeyAlt( uint32 key_code ) const
{
    switch( key_code )
    {
        case KEY_1:             return "1";
        case KEY_2:             return "2";
        case KEY_3:             return "3";
        case KEY_4:             return "4";
        case KEY_5:             return "5";
        case KEY_6:             return "6";
        case KEY_7:             return "7";
        case KEY_8:             return "8";
        case KEY_9:             return "9";
        case KEY_0:             return "0";
        case KEY_MINUS:         return "-";
        case KEY_EQUAL:         return "=";
        case KEY_TAB:           return "    ";
        case KEY_Q:             return "Q";
        case KEY_W:             return "W";
        case KEY_E:             return "E";
        case KEY_R:             return "R";
        case KEY_T:             return "T";
        case KEY_Y:             return "Y";
        case KEY_U:             return "U";
        case KEY_I:             return "I";
        case KEY_O:             return "O";
        case KEY_P:             return "P";
        case KEY_LEFTBRACE:     return "[";
        case KEY_RIGHTBRACE:    return "]";
        case KEY_A:             return "A";
        case KEY_S:             return "S";
        case KEY_D:             return "D";
        case KEY_F:             return "F";
        case KEY_G:             return "G";
        case KEY_H:             return "H";
        case KEY_J:             return "J";
        case KEY_K:             return "K";
        case KEY_L:             return "L";
        case KEY_SEMICOLON:     return ":";
        case KEY_APOSTROPHE:    return "\"";
        case KEY_GRAVE:         return "";
        case KEY_BACKSLASH:     return "\\";
        case KEY_Z:             return "Z";
        case KEY_X:             return "X";
        case KEY_C:             return "C";
        case KEY_V:             return "V";
        case KEY_B:             return "B";
        case KEY_N:             return "N";
        case KEY_M:             return "M";
        case KEY_COMMA:         return "<";
        case KEY_DOT:           return ">";
        case KEY_SLASH:         return "/";
        case KEY_SPACE:         return " ";
    }

    return "";
}

CString CKeyboard::GetStringFromKeyShift( uint32 key_code ) const
{
    switch( key_code )
    {
        case KEY_1:             return "!";
        case KEY_2:             return "@";
        case KEY_3:             return "#";
        case KEY_4:             return "$";
        case KEY_5:             return "|";
        case KEY_6:             return "^";
        case KEY_7:             return "&";
        case KEY_8:             return "*";
        case KEY_9:             return "(";
        case KEY_0:             return ")";
        case KEY_MINUS:         return "_";
        case KEY_EQUAL:         return "+";
        case KEY_TAB:           return "    ";
        case KEY_Q:             return "Q";
        case KEY_W:             return "W";
        case KEY_E:             return "E";
        case KEY_R:             return "R";
        case KEY_T:             return "T";
        case KEY_Y:             return "Y";
        case KEY_U:             return "U";
        case KEY_I:             return "I";
        case KEY_O:             return "O";
        case KEY_P:             return "P";
        case KEY_LEFTBRACE:     return "[";
        case KEY_RIGHTBRACE:    return "]";
        case KEY_A:             return "A";
        case KEY_S:             return "S";
        case KEY_D:             return "D";
        case KEY_F:             return "F";
        case KEY_G:             return "G";
        case KEY_H:             return "H";
        case KEY_J:             return "J";
        case KEY_K:             return "K";
        case KEY_L:             return "L";
        case KEY_SEMICOLON:     return ":";
        case KEY_APOSTROPHE:    return "\"";
        case KEY_GRAVE:         return "";
        case KEY_BACKSLASH:     return "\\";
        case KEY_Z:             return "Z";
        case KEY_X:             return "X";
        case KEY_C:             return "C";
        case KEY_V:             return "V";
        case KEY_B:             return "B";
        case KEY_N:             return "N";
        case KEY_M:             return "M";
        case KEY_COMMA:         return "<";
        case KEY_DOT:           return ">";
        case KEY_SLASH:         return "/";
        case KEY_SPACE:         return " ";
    }

    return "";
}

