//-------------------------------------------------------------------------
//
// File:		GraphicsDeviceI.h
// Desc:		Graphics device interface.
// Author:		memon <memon@inside.org>
//
//-------------------------------------------------------------------------
//	Copyright (c) 2000 Moppi Productions. All Rights Reserved.
//  This file is part of Moppi Demopaja SDK. For conditions of 
//  distribution and use, see the accompanying license.txt file.
//  http://moppi.inside.org/demopaja/
//-------------------------------------------------------------------------

#ifndef __DEMOPAJA_GRAPHICSDEVICEI_H__
#define __DEMOPAJA_GRAPHICSDEVICEI_H__

#define WIN32_LEAN_AND_MEAN
#include <windows.h>

#include "PajaTypes.h"
#include "BBox2C.h"
#include "ColorC.h"
#include "Vector2C.h"
#include "DeviceInterfaceI.h"
#include "DataBlockI.h"
#include "ClassIdC.h"
#include "DeviceFeedbackC.h"

namespace PajaSystem {

	#define DP_END_PREVIEW		(WM_APP + 110)

	//! Graphics device super class ID.
	const PluginClass::SuperClassIdC		SUPERCLASS_GRAPHICSDEVICE = PluginClass::SuperClassIdC( 0x3000001 );


	//! Graphics device creation flags.
	enum CreateFlagsE {
		GRAPHICSDEVICE_CREATE_EDITOR_CHILD = 1,	//!< Create a child window, to be used inside the editor.
		GRAPHICSDEVICE_CREATE_WINDOWED,			//!< Create a window, to be used in windowed preview.
		GRAPHICSDEVICE_CREATE_FULLSCREEN,		//!< Create a full screen window for preview.
		GRAPHICSDEVICE_CREATE_CHILD,			//!< Creates a child window which behaves like windowed mode.
	};

	//! Graphics device clear flags.
	enum ClearFlagsE {
		GRAPHICSDEVICE_COLORBUFFER =	0x01,	//!< Clear color buffer.
		GRAPHICSDEVICE_DEPTHBUFFER =	0x02,	//!< Clear depth buffer.
		GRAPHICSDEVICE_STENCILBUFFER =	0x04,	//!< Clear stencil buffer.
		GRAPHICSDEVICE_ACCUMBUFFER =	0x08,	//!< Clear accum buffer.
		GRAPHICSDEVICE_ALLBUFFERS =		0x0f,	//!< Clear all buffers.
	};

	//! Graphics device class.
	/*!
	*/
	class GraphicsDeviceI : public DeviceInterfaceI
	{
	public:
		//! Create new graphics device.
		virtual Edit::DataBlockI*				create() = 0;

		//! Returns super class ID.
		virtual PluginClass::SuperClassIdC		get_super_class_id() const;
		//! Returns true, if the device is excluseive.
		/*! If this method returns true, there can be only one instance of this device
			in the device context.
		*/
		virtual bool							get_exclusive() const;

		//! Intialises a created graphics device.
		virtual bool							init( HINSTANCE hInstance, HWND hParent,
													PajaTypes::int32 i32ID, PajaTypes::uint32 ui32Flags,
													PajaTypes::uint32 ui32Width = 0, PajaTypes::uint32 ui32Height = 0,
													PajaTypes::uint32 ui32BPP = 0,
													PajaSystem::DeviceFeedbackC* pFeedback = 0 ) = 0;
		//! Destroys a graphics device.
		virtual void							destroy() = 0;
		//! Flushes the current rendering buffer to screen.
		virtual void							flush() = 0;
		//! Activates this device.
		virtual void							activate() = 0;
		//! Returns the wondow handle of this device.
		virtual HWND							get_hwnd() = 0;
		//! Launches the configuration dialog.
		virtual bool							configure() = 0;

		//! Sets windowed or child device to full screen.
		virtual bool							set_fullscreen( PajaTypes::uint32 ui32Width, PajaTypes::uint32 ui32Height,
													PajaTypes::uint32 ui32BPP = 0 ) = 0;
		//! Sets windowed or child device to back to normal state.
		virtual bool							set_windowed() = 0;

		//! Clears the device.
		virtual void							clear_device( PajaTypes::uint32 ui32Flags,
													const PajaTypes::ColorC& rColor = PajaTypes::ColorC(),
													PajaTypes::float32 f32Depth = 1.0f, PajaTypes::int32 i32Stencil = 0 ) = 0;

		//! Starts drawing block.
		virtual void							begin_draw() = 0;
		//! Ends drawing block.
		virtual void							end_draw() = 0;

		//! Pushes current state and prepares effect drawing block.
		virtual void							begin_effects() = 0;
		//! Ends effect drawing block.
		virtual void							end_effects() = 0;

		//! Resizes the device viewport.
		virtual void							set_size( PajaTypes::int32 int32X, PajaTypes::int32 int32Y,
													PajaTypes::int32 i32Width, PajaTypes::int32 i32Height ) = 0;

	protected:
		GraphicsDeviceI();
		virtual ~GraphicsDeviceI();

	};

};

#endif // __DEMOPAJA_GRAPHICSDEVICEI_H__
