#pragma once

#include <xmmintrin.h>


class vec3
{
public:
	union
	{
		struct
		{
			float x;
			float y;
			float z;
		};
		struct
		{
			float v[3];
		};
	};

	vec3(float x, float y, float z)
	{
		v[0] = x;
		v[1] = y;
		v[2] = z;
	}

	vec3(const vec3& other) { v[0] = other.x; v[1] = other.y; v[2] = other.z; }

	explicit vec3(float x)
	{
		v[0] = v[1] = v[2] = x;
	}

	vec3() {}

	vec3 xyz() const { return vec3(x, y, z); }
	vec3 zyx() const { return vec3(z, y, x); }
	vec3 zxy() const { return vec3(z, x, y); }

#define xyz xyz()
#define zyx zyx()
#define zxy zxy()

};


inline float min(float a, float b) { return a < b ? a : b; }
inline float max(float a, float b) { return a > b ? a : b; }
inline float sign(float a) { return a >= 0 ? 1.0f : -1.0f; }
inline float myabs(float a) { return a >= 0 ? a : -a; }

inline vec3 operator+ (vec3 a, vec3 b) { a.x += b.x; a.y += b.y; a.z += b.z; return a; }
inline vec3 operator- (vec3 a, vec3 b) { a.x -= b.x; a.y -= b.y; a.z -= b.z; return a; }
inline vec3 operator* (vec3 a, vec3 b) { a.x *= b.x; a.y *= b.y; a.z *= b.z; return a; }
inline vec3 operator* (vec3 a, float b) { a.x *= b; a.y *= b; a.z *= b; return a; }
inline vec3 operator* (float a, vec3 b) { b.x *= a; b.y *= a; b.z *= a; return b; }
inline vec3& operator+= (vec3& a, vec3 b) { a = a + b; return a; }
inline vec3& operator-= (vec3& a, vec3 b) { a = a - b; return a; }
inline vec3& operator*= (vec3& a, vec3 b) { a = a * b; return a; }
inline vec3& operator*= (vec3& a, float b) { a = a * b; return a; }
inline vec3 min(vec3 a, vec3 b) { a.x = min(a.x, b.x); a.y = min(a.y, b.y); a.z = min(a.z, b.z); return a; }
inline vec3 max(vec3 a, vec3 b) { a.x = max(a.x, b.x); a.y = max(a.y, b.y); a.z = max(a.z, b.z); return a; }
inline vec3 operator-(vec3 a) { return vec3(-a.x, -a.y, -a.z); }

inline vec3 cross(const vec3& a, const vec3& b)
{
	// x  <-  a.y*b.z - a.z*b.y
	// y  <-  a.z*b.x - a.x*b.z
	// z  <-  a.x*b.y - a.y*b.x
	// We can save a shuffle by grouping it in this wacky order:
	return (a.zxy * b - a * b.zxy).zxy;
}

inline vec3 clamp(vec3 t, vec3 a, vec3 b) { return min(max(t, a), b); }
inline float sum3(vec3 v) { return v.x + v.y + v.z; }
inline float dot3(vec3 a, vec3 b) { return sum3(a * b); }

inline float mysqrt(float v)
{
	__m128 vv = _mm_set1_ps(v);
	vv = _mm_sqrt_ps(vv);
	return _mm_cvtss_f32(vv);
}

inline vec3 normalize(const vec3& v)
{
	float d2 = dot3(v, v);
	// alternatively, will save ~12 bytes:
	//__m128 d2sse = _mm_set1_ps(d2);
	//__m128 rsqsse = _mm_rsqrt_ss(d2sse);
	//return v * _mm_cvtss_f32(rsqsse);
	float m = 1.0f / mysqrt(d2);
	return v * m;
}



// not actually used anymore
//void CalcProjectionMatrix(float fovHalfYRadians, float aspect, float nearPlane, float farPlane, float matrix[16]);
//void CalcViewMatrix(vec3 from, float matrix[16]); // always looks at 0,0,0
//void CalcOrthoMatrix(float width, float height, float znear, float zfar, float matrix[16]);


struct Plane
{
	vec3 normal;
	float distance;

	float PointDistance(vec3 p) const
	{
		return dot3(p, normal) - distance;
	}
};
