#include "planet.h"
#include "../resources/resource.h"

GLuint mode = GL_OBJECT_LINEAR;
GLuint plane = GL_OBJECT_PLANE;

unsigned int glsl_loc_pass1_sunspotTexture;

unsigned int glsl_loc_pass2_sunspotTexture;
unsigned int glsl_loc_pass2_1stPassTexture;
unsigned int glsl_loc_pass2_surfaceWidth;
unsigned int glsl_loc_pass2_surfaceHeight;

unsigned int glsl_loc_pass3_2ndPassTexture;
unsigned int glsl_loc_pass3_fireGradientTexture;
unsigned int glsl_loc_pass3_surfaceWidth;
unsigned int glsl_loc_pass3_surfaceHeight;

unsigned int glsl_loc_pass4_textureToBlur;
unsigned int glsl_loc_pass4_surfaceWidth;
unsigned int glsl_loc_pass4_surfaceHeight;

unsigned int glsl_loc_pass5_baseTexture;
unsigned int glsl_loc_pass5_blendTexture;

unsigned int glsl_loc_bloom_baseTexture;

planet::planet(GLfloat radius, GLint slices, GLint stacks, bool isSun, float distanceFromParent, int numSats, int textureResourceId, float rotationSpeed)
{
	distanceModifier = 1;
	enabledSunspots = false;
	sunspotsFadingUp = false;

	_rotationSpeed = rotationSpeed;
	_textureResourceId = textureResourceId;
	_numSats = numSats;
	this->satellites = new planet*[numSats];
	_distanceFromParent = distanceFromParent;
	_isSun = isSun;
	_radius = radius;
	_slices = slices;
	_stacks = stacks;
	_tsphere = new tesSphere(radius, 3);
	HandleTexture();
}

void planet::EnableEffect(float time)
{
	enabledEffect = true;
	effectFadingUp = true;
	startTimeEffect = time;
}

void planet::HandleTexture()
{
	msys_debugPrintf("PLANET: Handle Textures\n");
	GLint width, height, icomps;
	GLenum eFormat;
	GLbyte *pixels = gltLoadJPG(_textureResourceId, &width, &height, &icomps, &eFormat );
	glGenTextures(1, &_textureMap);
	glBindTexture(GL_TEXTURE_2D, _textureMap);
	glTexImage2D(GL_TEXTURE_2D, 0, icomps, width, height, 0, eFormat, GL_UNSIGNED_BYTE, pixels );
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR); // Linear Filtering
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR); // Linear Filtering

	msys_mallocFree(pixels);
}

void planet::EnableSunSpots(float time)
{
	if (!enabledSunspots)
	{
		sunSpotsAlpha = 0;
		enabledSunspots = true;
		startTimeSunspots = time;
	}
}

void planet::initFboSun()
{
	msys_debugPrintf("PLANET: Init Sun Fbo\n");
	// sun spots fbo 1
	// generate namespace for the frame buffer, colorbuffer and depthbuffer
	oglGenFramebuffersEXT(1, &fboSunspotsPass1);
	glGenTextures(1, &fboTextureSunspotsPass1);
	oglGenRenderbuffersEXT(1, &depthTextureSunspotsPass1);

	//switch to our fbo so we can bind stuff to it
	oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fboSunspotsPass1);

	//create the colorbuffer texture and attach it to the frame buffer
	glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass1);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB16F_ARB, globalScreenInfo.fboWidth, globalScreenInfo.fboHeight, 0, GL_BGRA, GL_FLOAT, NULL);

	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR); // Linear Filtering
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR); // Linear Filtering
	// OpenGL will then generate all the required mipmap data for you so that your texture is ready to be used.
	oglGenerateMipmapEXT(GL_TEXTURE_2D); 

	oglFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT,	GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, fboTextureSunspotsPass1, 0);

	// create a render buffer as our depth buffer and attach it

	oglBindRenderbufferEXT(GL_RENDERBUFFER_EXT, depthTextureSunspotsPass1);
	oglRenderbufferStorageEXT(GL_RENDERBUFFER_EXT,
		GL_DEPTH_COMPONENT16,globalScreenInfo.fboWidth, globalScreenInfo.fboHeight);

	oglFramebufferRenderbufferEXT(GL_FRAMEBUFFER_EXT,
		GL_DEPTH_ATTACHMENT_EXT,
		GL_RENDERBUFFER_EXT, depthTextureSunspotsPass1);

	// Go back to regular frame buffer rendering
	oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

	GLenum status = oglCheckFramebufferStatusEXT(GL_FRAMEBUFFER_EXT);
	if(status != GL_FRAMEBUFFER_COMPLETE_EXT)
	{
		//DebugBreak();
	}

	// sun spots fbo 2
	// generate namespace for the frame buffer, colorbuffer and depthbuffer
	oglGenFramebuffersEXT(1, &fboSunspotsPass2);
	glGenTextures(1, &fboTextureSunspotsPass2);
	oglGenRenderbuffersEXT(1, &depthTextureSunspotsPass2);

	//switch to our fbo so we can bind stuff to it
	oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fboSunspotsPass2);

	//create the colorbuffer texture and attach it to the frame buffer
	glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass2);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB16F_ARB, globalScreenInfo.fboHeight, globalScreenInfo.fboHeight, 0, GL_BGRA, GL_FLOAT, NULL);

	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR); // Linear Filtering
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR); // Linear Filtering
	// OpenGL will then generate all the required mipmap data for you so that your texture is ready to be used.
	oglGenerateMipmapEXT(GL_TEXTURE_2D); 

	oglFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT,	GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, fboTextureSunspotsPass2, 0);

	// create a render buffer as our depth buffer and attach it

	oglBindRenderbufferEXT(GL_RENDERBUFFER_EXT, depthTextureSunspotsPass2);
	oglRenderbufferStorageEXT(GL_RENDERBUFFER_EXT,
		GL_DEPTH_COMPONENT16,globalScreenInfo.fboWidth, globalScreenInfo.fboHeight);

	oglFramebufferRenderbufferEXT(GL_FRAMEBUFFER_EXT,
		GL_DEPTH_ATTACHMENT_EXT,
		GL_RENDERBUFFER_EXT, depthTextureSunspotsPass2);

	// Go back to regular frame buffer rendering
	oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

	status = oglCheckFramebufferStatusEXT(GL_FRAMEBUFFER_EXT);
	if(status != GL_FRAMEBUFFER_COMPLETE_EXT)
	{
		//DebugBreak();
	}

	// sun spots fbo 3
	// generate namespace for the frame buffer, colorbuffer and depthbuffer
	oglGenFramebuffersEXT(1, &fboSunspotsPass3);
	glGenTextures(1, &fboTextureSunspotsPass3);
	oglGenRenderbuffersEXT(1, &depthTextureSunspotsPass3);

	//switch to our fbo so we can bind stuff to it
	oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fboSunspotsPass3);

	//create the colorbuffer texture and attach it to the frame buffer
	glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass3);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB16F_ARB, globalScreenInfo.fboWidth, globalScreenInfo.fboHeight, 0, GL_BGRA, GL_FLOAT, NULL);

	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR); // Linear Filtering
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR); // Linear Filtering
	// OpenGL will then generate all the required mipmap data for you so that your texture is ready to be used.
	oglGenerateMipmapEXT(GL_TEXTURE_2D); 

	oglFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT,	GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, fboTextureSunspotsPass3, 0);

	// create a render buffer as our depth buffer and attach it

	oglBindRenderbufferEXT(GL_RENDERBUFFER_EXT, depthTextureSunspotsPass3);
	oglRenderbufferStorageEXT(GL_RENDERBUFFER_EXT,
		GL_DEPTH_COMPONENT16,globalScreenInfo.fboWidth, globalScreenInfo.fboHeight);

	oglFramebufferRenderbufferEXT(GL_FRAMEBUFFER_EXT,
		GL_DEPTH_ATTACHMENT_EXT,
		GL_RENDERBUFFER_EXT, depthTextureSunspotsPass3);

	// Go back to regular frame buffer rendering
	oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

	status = oglCheckFramebufferStatusEXT(GL_FRAMEBUFFER_EXT);
	if(status != GL_FRAMEBUFFER_COMPLETE_EXT)
	{
		msys_debugPrintf("PLANET: INIT FBO SUN FAILED...\n");
	}
}

void planet::initSunspotTextures()
{
	msys_debugPrintf("PLANET: initSunspotTextures\n");
	GLint width, height, icomps;
	GLenum eFormat;
	GLbyte *pixels = gltLoadPNG(IDR_PNG_SUNSPOTS1_TEXTURE, &width, &height, &icomps, &eFormat );
	glGenTextures(1, &textureSunspot1);
	glBindTexture(GL_TEXTURE_2D, textureSunspot1);
	glTexImage2D(GL_TEXTURE_2D, 0, icomps, width, height, 0, eFormat, GL_UNSIGNED_BYTE, pixels );
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR); // Linear Filtering
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR); // Linear Filtering

	msys_mallocFree(pixels);

	pixels = gltLoadPNG(IDR_PNG_SUNSPOTS2_TEXTURE, &width, &height, &icomps, &eFormat );
	glGenTextures(1, &textureSunspot2);
	glBindTexture(GL_TEXTURE_2D, textureSunspot2);
	glTexImage2D(GL_TEXTURE_2D, 0, icomps, width, height, 0, eFormat, GL_UNSIGNED_BYTE, pixels );
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR); // Linear Filtering
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR); // Linear Filtering

	msys_mallocFree(pixels);

	pixels = gltLoadPNG(IDR_PNG_FIREGRAD_TEXTURE, &width, &height, &icomps, &eFormat );
	glGenTextures(1, &textureFireGradient);
	glBindTexture(GL_TEXTURE_2D, textureFireGradient);
	glTexImage2D(GL_TEXTURE_2D, 0, icomps, width, height, 0, eFormat, GL_UNSIGNED_BYTE, pixels );
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR); // Linear Filtering
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR); // Linear Filtering

	msys_mallocFree(pixels);
}


void planet::InitSun()
{
	msys_debugPrintf("PLANET: InitSun\n");
#ifdef DEBUG
	
	initShader(&shaderSunspotsPass1, NULL, LoadShaderText("shaders\\sunspotsPass1.fs"), "Sunspots 1");
	initShader(&shaderSunspotsPass2, LoadShaderText("shaders\\sunspotsPass2.vs"), LoadShaderText("shaders\\sunspotsPass2.fs"), "Sunspots 2");
	initShader(&shaderSunspotsPass3, LoadShaderText("shaders\\sunspotsPass3.vs"), LoadShaderText("shaders\\sunspotsPass3.fs"), "Sunspots 3");
	initShader(&shaderSunspotsPass4, LoadShaderText("shaders\\sunspotsPass4.vs"), LoadShaderText("shaders\\sunspotsPass4.fs"), "Sunspots 4");
	initShader(&shaderSunspotsPass5, LoadShaderText("shaders\\sunspotsPass5.vs"), LoadShaderText("shaders\\sunspotsPass5.fs"), "Sunspots 5");
	initShader(&shaderBloom, NULL, LoadShaderText("shaders\\bloom.fs"), "Bloom");
	
	/*
	initShader(&shaderSunspotsPass1, NULL, shaderFragFromResource(IDS_SHADER_SUNSPOTS1_FRAG), "Sunspots Pass1");
	initShader(&shaderSunspotsPass2, shaderFragFromResource(IDS_SHADER_SUNSPOTS2_VERT), shaderFragFromResource(IDS_SHADER_SUNSPOTS2_FRAG), "Sunspots Pass2");
	initShader(&shaderSunspotsPass3, shaderFragFromResource(IDS_SHADER_SUNSPOTS3_VERT), shaderFragFromResource(IDS_SHADER_SUNSPOTS3_FRAG), "Sunspots Pass3");
	initShader(&shaderSunspotsPass4, shaderFragFromResource(IDS_SHADER_SUNSPOTS4_VERT), shaderFragFromResource(IDS_SHADER_SUNSPOTS4_FRAG), "Sunspots Pass4");
	initShader(&shaderSunspotsPass5, shaderFragFromResource(IDS_SHADER_SUNSPOTS5_VERT), shaderFragFromResource(IDS_SHADER_SUNSPOTS5_FRAG), "Sunspots Pass5");
	initShader(&shaderBloom, NULL, shaderFragFromResource(IDS_SHADER_BLOOM_FRAG), "BLOOM");*/
#else
	initShader(&shaderSunspotsPass1, NULL, shaderFragFromResource(IDS_SHADER_SUNSPOTS1_FRAG));
	initShader(&shaderSunspotsPass2, shaderFragFromResource(IDS_SHADER_SUNSPOTS2_VERT), shaderFragFromResource(IDS_SHADER_SUNSPOTS2_FRAG));
	initShader(&shaderSunspotsPass3, shaderFragFromResource(IDS_SHADER_SUNSPOTS3_VERT), shaderFragFromResource(IDS_SHADER_SUNSPOTS3_FRAG));
	initShader(&shaderSunspotsPass4, shaderFragFromResource(IDS_SHADER_SUNSPOTS4_VERT), shaderFragFromResource(IDS_SHADER_SUNSPOTS4_FRAG));
	initShader(&shaderSunspotsPass5, shaderFragFromResource(IDS_SHADER_SUNSPOTS5_VERT), shaderFragFromResource(IDS_SHADER_SUNSPOTS5_FRAG));
	initShader(&shaderBloom, NULL, shaderFragFromResource(IDS_SHADER_BLOOM_FRAG));
#endif


	initFboSun();
	initSunspotTextures();

	msys_debugPrintf("PLANET: Get Uniforms\n");

	// pass 1
	glsl_loc_pass1_sunspotTexture = oglGetUniformLocation(shaderSunspotsPass1, "sunspotTexture");
	oglUniform1i(glsl_loc_pass1_sunspotTexture,0);

	// pass 2
	glsl_loc_pass2_sunspotTexture = oglGetUniformLocation(shaderSunspotsPass2, "sunspotTexture");
	oglUniform1i(glsl_loc_pass2_sunspotTexture,0);

	glsl_loc_pass2_1stPassTexture = oglGetUniformLocation(shaderSunspotsPass2, "firstPassTexture");
	oglUniform1i(glsl_loc_pass2_1stPassTexture,1);

	glsl_loc_pass2_surfaceWidth = oglGetUniformLocation(shaderSunspotsPass2, "surfaceWidth");
	oglUniform1f(glsl_loc_pass2_surfaceWidth, (float)globalScreenInfo.fboWidth);

	glsl_loc_pass2_surfaceHeight = oglGetUniformLocation(shaderSunspotsPass2, "surfaceHeight");
	oglUniform1f(glsl_loc_pass2_surfaceHeight, (float)globalScreenInfo.fboHeight);

	// pass 3
	glsl_loc_pass3_fireGradientTexture = oglGetUniformLocation(shaderSunspotsPass3, "fireTexture");
	oglUniform1i(glsl_loc_pass2_1stPassTexture,1);

	glsl_loc_pass3_2ndPassTexture = oglGetUniformLocation(shaderSunspotsPass3, "secondPassFboTexture");
	oglUniform1i(glsl_loc_pass2_1stPassTexture,1);

	glsl_loc_pass3_surfaceWidth = oglGetUniformLocation(shaderSunspotsPass3, "surfaceWidth");
	oglUniform1f(glsl_loc_pass3_surfaceWidth, (float)globalScreenInfo.fboWidth);

	glsl_loc_pass3_surfaceHeight = oglGetUniformLocation(shaderSunspotsPass3, "surfaceHeight");
	oglUniform1f(glsl_loc_pass3_surfaceHeight, (float)globalScreenInfo.fboHeight);

	// bloom
	glsl_loc_bloom_baseTexture = oglGetUniformLocation(shaderBloom, "bgl_RenderedTexture");
	oglUniform1i(glsl_loc_bloom_baseTexture,0);

	// pass 4
	glsl_loc_pass4_textureToBlur = oglGetUniformLocation(shaderSunspotsPass4, "textureToBlur");
	oglUniform1i(glsl_loc_pass4_textureToBlur,0);

	glsl_loc_pass4_surfaceWidth = oglGetUniformLocation(shaderSunspotsPass4, "surfaceWidth");
	oglUniform1f(glsl_loc_pass4_surfaceWidth, (float)globalScreenInfo.fboWidth);

	glsl_loc_pass4_surfaceHeight = oglGetUniformLocation(shaderSunspotsPass4, "surfaceHeight");
	oglUniform1f(glsl_loc_pass4_surfaceHeight,(float)globalScreenInfo.fboHeight);

	// pass 5
	glsl_loc_pass5_baseTexture = oglGetUniformLocation(shaderSunspotsPass5, "baseTexture");
	oglUniform1i(glsl_loc_pass5_baseTexture,0);

	glsl_loc_pass5_blendTexture= oglGetUniformLocation(shaderSunspotsPass5, "blendTexture");
	oglUniform1i(glsl_loc_pass5_blendTexture,0);
}

void planet::DrawRenderedSunspots()
{
	// gl one sunspots blur
	glEnable(GL_BLEND);
	glBlendFunc(GL_ONE, GL_ONE);
	glDisable(GL_LIGHTING);
	glDepthMask(FALSE); // disable depth buffer writing or planets won't be visible
	glPushMatrix();
	{		
		//glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		glLoadIdentity();	
		ViewOrtho(globalScreenInfo.width, globalScreenInfo.height);
		glColor4f(sunSpotsAlpha,sunSpotsAlpha,sunSpotsAlpha,sunSpotsAlpha);
		
		glEnable(GL_TEXTURE_2D);
			
		if (globalScreenInfo.postFx)
		{
			glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass1);
		}else{
			glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass3);
		}
		

		float border = 10;

		float w = (float)globalScreenInfo.width;
		float h = (float)globalScreenInfo.height;
		
		glBegin(GL_QUADS);
		{
			glTexCoord2f(0,1);
			glVertex2f(0-border,0-border);

			glTexCoord2f(0,0);
			glVertex2f(0-border, h+border);

			glTexCoord2f(1,0);
			glVertex2f(w+border, h+border);

			glTexCoord2f(1,1);
			glVertex2f(w+border,0-border);
		}		
		glEnd();

		ViewPerspective();
	}
	glPopMatrix();
	glDepthMask(TRUE);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE);
	glEnable(GL_LIGHTING);
	glDisable(GL_BLEND); // disable blending so planets down appear below sunspots
}

vertex planet::ScreenSpacePosition()
{
	vertex v;

	GLdouble pos3D_x, pos3D_y, pos3D_z;
	// arrays to hold matrix information
	GLdouble model_view[16];
	glGetDoublev(GL_MODELVIEW_MATRIX, model_view);
	GLdouble projection[16];
	glGetDoublev(GL_PROJECTION_MATRIX, projection);
	GLint viewport[4];
	glGetIntegerv(GL_VIEWPORT, viewport);	

	/*
	viewport[2] = _screenInfo->fboWidth;
	viewport[3] = _screenInfo->fboHeight;
	*/

	// get 3D coordinates based on window coordinates
	gluProject(0, 0, 0, model_view, projection, viewport, &pos3D_x, &pos3D_y, &pos3D_z);
	//gluUnProject(_screenInfo->fboWidth/2, _screenInfo->fboHeight/2, 0.01,	model_view, projection, viewport,	&pos3D_x, &pos3D_y, &pos3D_z); 
	//gluProject(0, 0, 0, )
	v.x = (float)pos3D_x / (float)viewport[2];
	v.y = (float)pos3D_y/ (float)viewport[3];
	v.z = (float)pos3D_z;
	return v;
}

void planet::BlendAndBlur()
{
	if (globalScreenInfo.shaderLevel != ShaderlLevel_All)
	{
		// no shaders or no post fx
		return;
	}
	glPushMatrix();
	{
		// MIP MAPS to 4th fbo
		//oglUniform1i(glsl_loc_pass4_3rdPassTexture,0);
		float sWidth = (float)globalScreenInfo.width/2;
		float sHeight = (float)globalScreenInfo.height/2;
		float yoffset = 0;
		float xoffset = 0;
		glColor4f(1,1,1,1);
		glEnable(GL_TEXTURE_2D);

		glLoadIdentity();	
		ViewOrtho(globalScreenInfo.fboWidth, globalScreenInfo.fboHeight);

		// blur (fbo 3 into 1)
		{
			oglUseProgram(shaderSunspotsPass4);
			oglUniform1i(glsl_loc_pass4_textureToBlur,0);
			oglUniform1f(glsl_loc_pass4_surfaceWidth, (float)globalScreenInfo.fboWidth);
			oglUniform1f(glsl_loc_pass4_surfaceHeight, (float)globalScreenInfo.fboHeight);
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fboSunspotsPass1);
			glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
			oglActiveTextureARB(GL_TEXTURE0);
			glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass3);

			glDrawBuffer(GL_COLOR_ATTACHMENT0_EXT);

			glBegin(GL_QUADS);
			{
				glTexCoord2f(0, 1);
				glVertex2f(0,0);

				glTexCoord2f(0, 0);
				glVertex2i(0,globalScreenInfo.fboHeight);

				glTexCoord2f(1, 0);
				glVertex2i(globalScreenInfo.fboWidth, globalScreenInfo.fboHeight);

				glTexCoord2f(1, 1);
				glVertex2i(globalScreenInfo.fboWidth,0);
			}	
			glEnd();	

			glBindTexture(GL_TEXTURE_2D, 0);
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
			oglUseProgram(0);
		}

		// bloom (fbo 1 into 3)
		{
			oglUseProgram(shaderBloom);
			oglUniform1i(glsl_loc_pass4_textureToBlur,0);
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fboSunspotsPass3);
			glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
			oglActiveTextureARB(GL_TEXTURE0);
			glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass1);

			glDrawBuffer(GL_COLOR_ATTACHMENT0_EXT);

			glBegin(GL_QUADS);
			{
				glTexCoord2f(0, 1);
				glVertex2f(0,0);

				glTexCoord2f(0, 0);
				glVertex2i(0,globalScreenInfo.fboHeight);

				glTexCoord2f(1, 0);
				glVertex2i(globalScreenInfo.fboWidth, globalScreenInfo.fboHeight);

				glTexCoord2f(1, 1);
				glVertex2i(globalScreenInfo.fboWidth,0);
			}	
			glEnd();	

			glBindTexture(GL_TEXTURE_2D, 0);
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
			oglUseProgram(0);
		}

		// blur (fbo 3 into 1)
		{
			oglUseProgram(shaderSunspotsPass4);
			oglUniform1f(glsl_loc_pass4_surfaceWidth, (float)globalScreenInfo.fboWidth);
			oglUniform1f(glsl_loc_pass4_surfaceHeight, (float)globalScreenInfo.fboHeight);
			oglUniform1i(glsl_loc_pass4_textureToBlur,0);
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fboSunspotsPass1);
			glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
			oglActiveTextureARB(GL_TEXTURE0);
			glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass3);

			glDrawBuffer(GL_COLOR_ATTACHMENT0_EXT);

			glBegin(GL_QUADS);
			{
				glTexCoord2f(0, 1);
				glVertex2f(0,0);

				glTexCoord2f(0, 0);
				glVertex2i(0,globalScreenInfo.fboHeight);

				glTexCoord2f(1, 0);
				glVertex2i(globalScreenInfo.fboWidth, globalScreenInfo.fboHeight);

				glTexCoord2f(1, 1);
				glVertex2i(globalScreenInfo.fboWidth,0);
			}	
			glEnd();	

			glBindTexture(GL_TEXTURE_2D, 0);
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
			oglUseProgram(0);
		}

		
		/*
		// create mips maps in single fbo texture on fbo 1
		{
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fboSunspotsPass1);
			glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
			oglActiveTextureARB(GL_TEXTURE0);
			glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass3);

			glDrawBuffer(GL_COLOR_ATTACHMENT0_EXT);

			// create manual mipmaps - largest first
			for(int i=0; i<5; i++)
			{
				glBegin(GL_QUADS);
				{
					glTexCoord2f(0, 0);
					glVertex2f(0,0+yoffset);

					glTexCoord2f(0, 1);
					glVertex2f(0,sHeight+yoffset);

					glTexCoord2f(1, 1);
					glVertex2f(sWidth, sHeight+yoffset);

					glTexCoord2f(1, 0);
					glVertex2f(sWidth,0+yoffset);
				}	
				yoffset+=sHeight;
				glEnd();
				sWidth /=2;
				sHeight /=2;			
			}
			glBindTexture(GL_TEXTURE_2D, 0);
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
		}

		// blend mipmaps into fbo 4
		{
			oglUseProgram(shaderSunspotsPass4);
			oglUniform1i(glsl_loc_pass4_textureToBlur,0);
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fboSunspotsPass4);
			glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
			oglActiveTextureARB(GL_TEXTURE0);
			glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass1);

			glDrawBuffer(GL_COLOR_ATTACHMENT0_EXT);

			glBegin(GL_QUADS);
			{
				glTexCoord2f(0, 1);
				glVertex2f(0,0);

				glTexCoord2f(0, 0);
				glVertex2i(0,_screenInfo->height);

				glTexCoord2f(1, 0);
				glVertex2i(_screenInfo->width, _screenInfo->height);

				glTexCoord2f(1, 1);
				glVertex2i(_screenInfo->height,0);
			}	
			glEnd();	

			glBindTexture(GL_TEXTURE_2D, 0);
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
			oglUseProgram(0);
		}


		/*
		// draw 16 (scaled x2) from FBO 4 to FBO 1
		{
			float vsizex = (float)_screenInfo->width/2/2/2/2/2;
			float vsizey = (float)_screenInfo->height/2/2/2/2/2;
			float tsize = 1.0f/2/2/2/2/2;
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fboSunspotsPass1);
			glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
			oglActiveTextureARB(GL_TEXTURE0);
			glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass4);

			glDrawBuffer(GL_COLOR_ATTACHMENT0_EXT);

			glBegin(GL_QUADS);
			{	
				glTexCoord2f(0, 1);
				glVertex2f(0,0);

				glTexCoord2f(0, 0);
				glVertex2i(0,_screenInfo->height);

				glTexCoord2f(1, 0);
				glVertex2i(_screenInfo->width, _screenInfo->height);

				glTexCoord2f(1, 1);
				glVertex2i(_screenInfo->width,0);
			}
			glEnd();

			glBindTexture(GL_TEXTURE_2D, 0);
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
		}*/

		/*
		
		// test

		float tsize= 0.03125f;
		float vsizex = 25*2;
		float vsizey = 18.75*2;

	
		// draw 16 (scaled x2) from FBO 4 to FBO 1
		{
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fboSunspotsPass1);
			glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
			oglActiveTextureARB(GL_TEXTURE0);
			glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass4);

			glDrawBuffer(GL_COLOR_ATTACHMENT0_EXT);

			glBegin(GL_QUADS);
			{	
				glTexCoord2f(0, tsize);
				glVertex2f(0,0);

				glTexCoord2f(0, tsize*2);
				glVertex2f(0, vsizey*2);

				glTexCoord2f(tsize, tsize*2);
				glVertex2f(vsizex*2, vsizey*2);

				glTexCoord2f(tsize, tsize);
				glVertex2f(vsizex*2,0);
			}
			glEnd();

			glBindTexture(GL_TEXTURE_2D, 0);
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
		}
		
		tsize*=2;
		vsizex*=2;
		vsizey*=2;

		// draw 32 from FBO 4 mixed with scaled 16 from FBO 1 into FBO 2
		{
			oglUseProgram(shaderSunspotsPass4);
			
			oglUniform1i(glsl_loc_pass5_baseTexture,0);
			oglUniform1i(glsl_loc_pass5_blendTexture,1);
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fboSunspotsPass2);
			glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

			oglActiveTextureARB(GL_TEXTURE0);
			glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass1);

			oglActiveTextureARB(GL_TEXTURE1);
			glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass4);

			glDrawBuffer(GL_COLOR_ATTACHMENT0_EXT);

			glBegin(GL_QUADS);
			{	
				glTexCoord2f(0, tsize);
				glVertex2f(0,0);

				glTexCoord2f(0, tsize*2);
				glVertex2f(0, vsizey);

				glTexCoord2f(tsize, tsize*2);
				glVertex2f(vsizex, vsizey);

				glTexCoord2f(tsize, tsize);
				glVertex2f(vsizex,0);
			}
			glEnd();

			oglActiveTextureARB(GL_TEXTURE1);
			glBindTexture(GL_TEXTURE_2D, 0);
			oglActiveTextureARB(GL_TEXTURE0);
			glBindTexture(GL_TEXTURE_2D, 0);

			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
			oglUseProgram(0);
		}	

		// draw 32 (scaled x2) from FBO 2 to FBO 1
		tsize*=2;
		{
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fboSunspotsPass1);
			glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
			oglActiveTextureARB(GL_TEXTURE0);
			glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass2);

			glDrawBuffer(GL_COLOR_ATTACHMENT0_EXT);

			glBegin(GL_QUADS);
			{								
				glTexCoord2f(0, 1);
				glVertex2f(0,0);

				glTexCoord2f(0,  1-tsize);
				glVertex2f(0, vsizey*2);

				glTexCoord2f(tsize,  1-tsize);
				glVertex2f(vsizex*2, vsizey*2);

				glTexCoord2f(tsize,  1);
				glVertex2f(vsizex*2,0);
			}
			glEnd();

			glBindTexture(GL_TEXTURE_2D, 0);
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
		}

		// draw 64 from FBO 4 mixed with scaled 16 from FBO 1 into FBO 2
		vsizex*=2;
		vsizey*=2;
		{
			oglUseProgram(shaderSunspotsPass5);

			oglUniform1i(glsl_loc_pass5_baseTexture,0);
			oglUniform1i(glsl_loc_pass5_blendTexture,1);
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fboSunspotsPass2);
			glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

			oglActiveTextureARB(GL_TEXTURE0);
			glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass1);

			oglActiveTextureARB(GL_TEXTURE1);
			glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass4);

			glDrawBuffer(GL_COLOR_ATTACHMENT0_EXT);

			glBegin(GL_QUADS);
			{	
				glTexCoord2f(0, tsize);
				glVertex2f(0,0);

				glTexCoord2f(0, tsize*2);
				glVertex2f(0, vsizey);

				glTexCoord2f(tsize, tsize*2);
				glVertex2f(vsizex, vsizey);

				glTexCoord2f(tsize, tsize);
				glVertex2f(vsizex,0);
			}
			glEnd();

			oglActiveTextureARB(GL_TEXTURE1);
			glBindTexture(GL_TEXTURE_2D, 0);
			oglActiveTextureARB(GL_TEXTURE0);
			glBindTexture(GL_TEXTURE_2D, 0);

			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
			oglUseProgram(0);
		}

		// draw 64 (scaled x2) from FBO 2 to FBO 1
		tsize*=2;
		{
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fboSunspotsPass1);
			glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
			oglActiveTextureARB(GL_TEXTURE0);
			glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass2);

			glDrawBuffer(GL_COLOR_ATTACHMENT0_EXT);

			glBegin(GL_QUADS);
			{								
				glTexCoord2f(0, 1);
				glVertex2f(0,0);

				glTexCoord2f(0,  1-tsize);
				glVertex2f(0, vsizey*2);

				glTexCoord2f(tsize,  1-tsize);
				glVertex2f(vsizex*2, vsizey*2);

				glTexCoord2f(tsize,  1);
				glVertex2f(vsizex*2,0);
			}
			glEnd();

			glBindTexture(GL_TEXTURE_2D, 0);
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
		}
	
		// draw 128 (scaled x2) from FBO 2 to FBO 1
		tsize*=2;
		{
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fboSunspotsPass1);
			glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
			oglActiveTextureARB(GL_TEXTURE0);
			glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass2);

			glDrawBuffer(GL_COLOR_ATTACHMENT0_EXT);

			glBegin(GL_QUADS);
			{								
				glTexCoord2f(0, 1);
				glVertex2f(0,0);

				glTexCoord2f(0,  1-tsize);
				glVertex2f(0, vsizey*2);

				glTexCoord2f(tsize,  1-tsize);
				glVertex2f(vsizex*2, vsizey*2);

				glTexCoord2f(tsize,  1);
				glVertex2f(vsizex*2,0);
			}
			glEnd();

			glBindTexture(GL_TEXTURE_2D, 0);
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
		}

		// draw 128 from FBO 4 mixed with scaled 16 from FBO 1 into FBO 2
		vsizex*=2;
		vsizey*=2;
		{
			oglUseProgram(shaderSunspotsPass5);

			oglUniform1i(glsl_loc_pass5_baseTexture,0);
			oglUniform1i(glsl_loc_pass5_blendTexture,1);
			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fboSunspotsPass2);
			glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

			oglActiveTextureARB(GL_TEXTURE0);
			glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass1);

			oglActiveTextureARB(GL_TEXTURE1);
			glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass4);

			glDrawBuffer(GL_COLOR_ATTACHMENT0_EXT);

			glBegin(GL_QUADS);
			{	
				glTexCoord2f(0, tsize);
				glVertex2f(0,0);

				glTexCoord2f(0, tsize*2);
				glVertex2f(0, vsizey);

				glTexCoord2f(tsize, tsize*2);
				glVertex2f(vsizex, vsizey);

				glTexCoord2f(tsize, tsize);
				glVertex2f(vsizex,0);
			}
			glEnd();

			oglActiveTextureARB(GL_TEXTURE1);
			glBindTexture(GL_TEXTURE_2D, 0);
			oglActiveTextureARB(GL_TEXTURE0);
			glBindTexture(GL_TEXTURE_2D, 0);

			oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
			oglUseProgram(0);
		}*/

		ViewPerspective();
	}
	glPopMatrix();

	//oglUseProgram(0);

}

void planet::DrawSunSpots(float time)
{
	glViewport(0, 0, globalScreenInfo.fboWidth, globalScreenInfo.fboHeight); // change viewport to match fbo size

	glDisable(GL_LIGHTING);
	glMatrixMode(GL_MODELVIEW);

	/*
	GLenum buffers[] = { GL_COLOR_ATTACHMENT0_EXT };
	oglDrawBuffers(1, buffers);
*/
	float rotAmount = (time+100)*45;

	oglUseProgram(shaderSunspotsPass1);

	oglUniform1i(glsl_loc_pass1_sunspotTexture,0);

	
	glDisable(GL_BLEND);
	glBlendFunc(GL_ONE, GL_ONE);

	// 1st pass

	oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fboSunspotsPass1);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	oglActiveTextureARB(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, textureSunspot1);

	glPushMatrix();
	{	
		glRotatef((time+100)*45*_rotationSpeed, 0, 1.5f, 0);	

		glTranslatef(_distanceFromParent, 0, 0);

		// flip 90
		glRotatef(rotAmount, 0.1f, 0.2f, 0.3f);

		glColor4f(1,1,1,1);

		_tsphere->Draw();

		// flip -90
		glRotatef(90, -1, 0, 0);
	}
	glPopMatrix();

	oglUseProgram(0);
	
	oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
	
	// *** 2nd pass

	oglUseProgram(shaderSunspotsPass2);

	oglUniform1i(glsl_loc_pass2_sunspotTexture,0);
	oglUniform1i(glsl_loc_pass2_1stPassTexture,1);
	oglUniform1f(glsl_loc_pass2_surfaceWidth, (float)globalScreenInfo.fboWidth);
	oglUniform1f(glsl_loc_pass2_surfaceHeight, (float)globalScreenInfo.fboHeight);

	oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fboSunspotsPass2);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	
	// bind texture unit 0 to sunspot texture
	oglActiveTextureARB(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, textureSunspot2);

	//  bind texture unit 0 to 1st pass fbo texture
	oglActiveTextureARB(GL_TEXTURE1);
	glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass1);
	
	glPushMatrix();
	{	
		glRotatef((time+100)*45*_rotationSpeed, 0, 1.5f, 0);	

		glTranslatef(_distanceFromParent, 0, 0);

		// flip 90
		glRotatef(-rotAmount, 0.3f, 0.1f, 0.2f);

		glColor4f(1,1,1,1);

		_tsphere->Draw();

		// flip -90
		glRotatef(90, -1, 0, 0);
	}
	glPopMatrix();

	oglUseProgram(0);

	oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

	glBlendFunc(GL_SRC_ALPHA, GL_ONE);
	glBindTexture(GL_TEXTURE_2D, 0);
	oglActiveTextureARB(GL_TEXTURE0);

	// *** 3rd  pass

	oglUseProgram(shaderSunspotsPass3);
	oglUniform1f(glsl_loc_pass3_surfaceWidth, (float)globalScreenInfo.fboWidth);
	oglUniform1f(glsl_loc_pass3_surfaceHeight, (float)globalScreenInfo.fboHeight);
	oglUniform1i(glsl_loc_pass3_fireGradientTexture,0);
	oglUniform1i(glsl_loc_pass3_2ndPassTexture,1);

	oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fboSunspotsPass3);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);


	// bind texture unit 0 to sunspot texture
	oglActiveTextureARB(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, textureFireGradient);

	//  bind texture unit 0 to 1st pass fbo texture
	oglActiveTextureARB(GL_TEXTURE1);
	glBindTexture(GL_TEXTURE_2D, fboTextureSunspotsPass2);

	glPushMatrix();
	{	
		glRotatef((time+100)*45*_rotationSpeed, 0, 1.5f, 0);	

		glTranslatef(_distanceFromParent, 0, 0);

		// flip 90
		glRotatef(rotAmount, 0.3f, 0.1f, 0.2f);

		glColor4f(1,1,1,1);

		_tsphere->Draw();

		// flip -90
		glRotatef(90, -1, 0, 0);
	}
	glPopMatrix();

	oglUseProgram(0);

	oglBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

	glBlendFunc(GL_SRC_ALPHA, GL_ONE);
	glBindTexture(GL_TEXTURE_2D, 0);
	oglActiveTextureARB(GL_TEXTURE0);

	if (globalScreenInfo.postFx)
	{
		BlendAndBlur();
	}
	

	glViewport(0, 0, globalScreenInfo.width, globalScreenInfo.height); // change viewport to match fbo size
}

void planet::HandleSunRotation(float time)
{
	//glRotatef(-time, 0, 1, 0);
}

void planet::Draw(float time, bool onlyShadows)
{
	glMatrixMode(GL_MODELVIEW);

	glPushMatrix();
	{	
		if (_isSun)
		{
			HandleSunRotation(time);
		}

		glRotatef((time+100)*45*_rotationSpeed, 0, 1.5f, 0);	

		glTranslatef(_distanceFromParent*distanceModifier, 0, 0);

		// flip 90
		glRotatef(90, 1, 0, 0);

		if (onlyShadows && !_isSun)
		{
			glColor4f(0,0,0,1);	
		}else{

			glColor4f(1,1,1,1);

			glEnable( GL_TEXTURE_2D );

			glBindTexture(GL_TEXTURE_2D, _textureMap);
		}

		_tsphere->Draw();


		/* Sync planets with chime sound, looked crap so not used
		if (!onlyShadows && enabledEffect)
		{
			float effectLength = 2;
			float effectTime = (time - startTimeEffect);
			if (effectTime < effectLength)
			{				
				float amount = effectLength - (time - startTimeEffect);
				glEnable(GL_LIGHTING);
				//glBlendFunc(GL_ONE, GL_ONE);	
				glEnable( GL_COLOR_MATERIAL );
				glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
				glPushMatrix();
				{
					glScalef(1.0f+amount, 1.0f+amount, 1.0f+amount);
					glColor4f(1.0f, 1.0f, 1.0f, 0.1f);
					_tsphere->Draw();
				}
				glPopMatrix();
				glColor4f(1,1,1,1);
				glEnable(GL_LIGHTING);				
				glDisable( GL_COLOR_MATERIAL );
				//glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
			}
		}
		*/

		if (enabledSunspots)
		{
			float dif = (time - startTimeSunspots) * 0.001f;
			if (dif < 1.0f)
			{
				sunSpotsAlpha += dif;
			}else
			{
				sunSpotsAlpha = 1.0f;
			}
			DrawRenderedSunspots();
		}

		// flip -90
		glRotatef(90, -1, 0, 0);

		for(int i=0; i<_numSats; i++)
		{
			satellites[i]->distanceModifier = distanceModifier;
			satellites[i]->Draw(time, onlyShadows);
		}
	}
	glPopMatrix();
}

planet::~planet(void)
{
}
