unit GUS;

interface

(* PUBLIC *)

uses
	Dos;

const
	(* BALANCE *)
	Left    = 0;
	Middle  = 8;
	Right   = 15;

	(* MODE *)
	Bit8    = 0;
	Bit16   = 4;
	LoopOff = 0;
	LoopOn  = 8;
	UniDir  = 0;
	BiDir   = 16;
	Forw    = 0;
	Backw   = 64;
	Up      = 0;
	Down    = 64;

	(* OUTPUT *)
	LineIn    = -1;
	LineOut   = -2;
	MicIn     = 4;

var
	GusBase        : Word;         (* GUS I/O BASE 2x0 *)
	GusMemory      : Word;         (* GUS MEMORY SIZE 256K, 512K, 768K, 1024K *)
	GusEnvironment : string[20];   (* GUS "ULTRASND" ENVIRONMENT STRING *)

	GusVoices      : Byte;         (* NUMBER OF ACTIVE VOICES *)
	GusDataConvert : Boolean;      (* CONVERT DATA FOR GUSWRITE AND GUSREAD *)
	GusData16Bits  : Boolean;      (* 16BITS DATA FOR GUSWRITE AND GUSREAD *)

function  GusFind : Boolean;
function  MegaEm : Boolean;
function init(MaxVoices:Byte):boolean;
procedure GusMixer (Output : ShortInt);

procedure GusPoke (Address : LongInt ; Value : Byte);
function  GusPeek (Address : LongInt) : Byte;
procedure GusWrite (Address : LongInt; var Buffer; Count : Word);
procedure GusRead (Address : LongInt; var Buffer; Count : Word);

procedure VoiceInit (Voice : Byte);

procedure VoiceBalance (Voice : Byte; Balance : Byte);
procedure VoiceVolume (Voice : Byte; Volume : Word);
procedure VoiceFreq (Voice : Byte; Freq : Word);
procedure VoiceStart (Voice : Byte);
procedure VoiceStop (Voice : Byte);
procedure VoiceMode (Voice : Byte; Mode : Byte);
procedure PlaySmp (Voice : Byte; Start, LoopStart, LoopEnd : LongInt);

procedure RampStart (Voice : Byte);
procedure RampStop (Voice : Byte);
procedure RampMode (Voice : Byte; Mode : Byte);
procedure RampRate (Voice : Byte; Scale, Rate : Byte);
procedure RampRange (Voice : Byte; Lower, Upper : Word);

function GetVoiceLoc (Voice : Byte; Location : Byte) : LongInt;
function VoiceActive (Voice : Byte) : Boolean;

(* PRIVATE *)

const
	(* OUTPUT *)
	Disabled  = 8 + 2 + 1;

	(* FREQUENCY *)
	FREQ_DIV_TABLE : array [0..31] of Word =
									 ($250, $128, $0C5, $094, $076, $063, $055, $04A,
										$042, $03C, $036, $032, $02E, $02B, $028, $025,
										$023, $021, $01F, $01E, $01C, $01B, $01A, $019,
										$018, $017, $016, $015, $014, $014, $013, $012);

var
	(* GUS DATA *)
	GusDataHigh         : Byte;      (* HIGH PART OF THE 16BITS GUS DATA *)

	(* VARIABLES *)
  GF1_ACTIVE_VOICE    : Byte;
  GF1_FREQ_DIV        : Word;

  (* I/O MAP *)
  MIX_CONTROL_REG     : Word;
  IRQ_STATUS_REG      : Word;
  TIMER_CONTROL_REG   : Word;
  TIMER_DATA          : Word;
  IRQ_DMA_CONTROL_REG : Word;
  GF1_PAGE_REG        : Word;
  GF1_REG_SELECT      : Word;
  DRAM                : Word;

procedure GF1_DELAY;

procedure GF1_OUT_BYTE;
procedure GF1_OUT_WORD;
procedure GF1_OUT_ADDR;

function GF1_IN_BYTE : Byte;
function GF1_IN_WORD : Word;
function GF1_IN_ADDR : LongInt;

procedure GF1_VOICE;
procedure GF1_VOICE_CONTROL;
procedure GF1_RAMP_CONTROL;

implementation

procedure GF1_DELAY; assembler;
asm
  push	ax
  push  dx
  mov   ah, 80

  @lab1:
  mov	dx, DRAM
  in    al, dx
  dec   ah
  jnz   @lab1

  pop   dx
  pop	ax
end;

procedure GF1_OUT_BYTE; assembler;
(*
   INPUT: AL -> GF1_REGISTER_SELECT
          BL -> GF1_DATA
*)
asm
  mov   dx, GF1_REG_SELECT
  out   dx, al
  inc   dx
  inc   dx
  mov   al, bl
  out   dx, al
end;

procedure GF1_OUT_WORD; assembler;
(*
   INPUT  : AL -> GF1_REGISTER_SELECT
            BX -> GF1_DATA
*)
asm
  mov   dx, GF1_REG_SELECT
  out   dx, al
  inc   dx
  mov   ax, bx
  out   dx, ax
end;

procedure GF1_OUT_ADDR; assembler;
(*
   INPUT  : AL(+1) -> GF1_REGISTER_SELECT
            BX:SI  -> GF1_DATA
*)
asm
  mov   dx, GF1_REG_SELECT
  out   dx, al
  mov   cl, al
  inc   dx
  mov   ax, si
  shr   ax, 7
  ror   bx, 7
  or    ax, bx
  out   dx, ax
  dec   dx
  mov   al, cl
  inc   al
  out   dx, al
  inc   dx
  mov   ax, si
  shl   ax, 9
  out   dx, ax
end;

function GF1_IN_BYTE : Byte; assembler;
(*
   INPUT  : AL -> GF1_REGISTER_SELECT
   OUTPUT : AL <- GF1_DATA
*)
asm
  mov   dx, GF1_REG_SELECT
  out   dx, al
  inc   dx
  inc   dx
  in    al, dx
end;

function GF1_IN_WORD : Word; assembler;
(*
   INPUT  : AL -> GF1_REGISTER_SELECT
   OUTPUT : AX <- GF1_DATA
*)
asm
  mov   dx, GF1_REG_SELECT
  out   dx, al
  inc   dx
  in    ax, dx
end;

function GF1_IN_ADDR : LongInt; assembler;
(*
   INPUT  : AL(+1) -> GF1_REGISTER_SELECT
   OUTPUT : AX:DX  <- GF1_DATA
*)
asm
  mov   dx, GF1_REG_SELECT
  out   dx, al
  mov   cl, al
  inc   dx
  in    ax, dx
  mov   di, ax
  shr   di, 9
  and   di, 15
  mov   si, ax
  shl   si, 7
  dec   dx
  mov   al, cl
  inc   al
  out   dx, al
  inc   dx
  in    ax, dx
  shr   ax, 9
  or    ax, si
  mov   dx, di
end;

procedure GF1_VOICE; assembler;
(*
   INPUT: AL -> GF1_PAGE_REG
*)
asm
  cmp   GF1_ACTIVE_VOICE, al
  je    @lab1
  mov   GF1_ACTIVE_VOICE, al

  mov   dx, GF1_PAGE_REG
  out   dx, al
  @lab1:
end;

procedure GF1_VOICE_CONTROL; assembler;
(*
   INPUT: BL -> VOICE CONTROL REGISTER (AND)
          BH -> VOICE CONTROL REGISTER (OR)
*)
asm
  mov   al, 80h
  call  GF1_IN_BYTE

  and   bl, al
  or    bl, bh
  xor   al, al
  call  GF1_OUT_BYTE
end;

procedure GF1_RAMP_CONTROL; assembler;
(*
   INPUT: BL -> VOICE CONTROL REGISTER (AND)
          BH -> VOICE CONTROL REGISTER (OR)
*)
asm
  mov   al, 8Dh
  call  GF1_IN_BYTE

  and   bl, al
  or    bl, bh
  mov   al, 0Dh
  call  GF1_OUT_BYTE
end;

function GusMemSize : Word; assembler;
asm
  (* SIZE := 1024 *)
  mov   cx, 1024

  @lab1:
  (* ADDRESS LOW  := FFFFh *)
  mov   al, 43h
  mov   bx, 0FFFFh
  call  GF1_OUT_WORD

  (* ADDRESS HIGH := SIZE SHR 6 - 1 *)
  mov   al, 44h
  mov   bx, cx
  shr   bx, 6
  dec   bx
  call  GF1_OUT_WORD

  (* TEST MEMORY *)
  mov   dx, DRAM
  in    al, dx
  mov   bl, al
  xor   al, 0FFh
  out   dx, al
  call  GF1_DELAY
  in    al, dx
  xor   al, 0FFh
  out   dx, al
  cmp   bl, al
  je    @lab2
  (* SIZE := SIZE - 256 *)
  sub   cx, 256
  jnz   @lab1
  jmp   @lab3

  @lab2:
  (* TEST GUS *)
  mov   dx, GF1_REG_SELECT
  inc   dx
  in    al, dx
  cmp   al, 0FFh
  je    @lab3
  inc   dx
  in    al, dx
  cmp   al, 0FFh
  jne   @lab4

  @lab3:
  (* MEM OR GUS NOT O.K. -> SIZE := 0 *)
  xor   cx, cx

  @lab4:
  (* GUSMEMSIZE := SIZE *)
  mov   ax, cx
end;

function GusNrOfVoices : Byte; assembler;
asm
  (* NR OF VOICES *)
  mov   al, 8Eh
  call  GF1_IN_BYTE
  and   al, 31
end;

(* PUBLIC *)

function GusFind : Boolean;
var
	Code : Integer;
begin
	GusEnvironment := GetEnv ('ULTRASND');

	if (GusEnvironment <> '') then
	begin
		Val ('$' + Copy (GusEnvironment, 1, 3), GusBase, Code);
		if (Code <> 0) or (GusBase < $210) or
			 (GusBase > $2A0) or (GusBase and $F <> 0) then GusBase := $200
	end
		else GusBase := $200;

	Dec (GusBase, $10);

	repeat
		(* GUSBASE *)
		Inc (GusBase, $10);

		(* INTERNAL *)
		asm
			push   ax
			(* 2x? *)
			mov    ax, GusBase
			mov    MIX_CONTROL_REG, ax
			add    ax, 6h
			mov    IRQ_STATUS_REG, ax
			inc    ax
			inc    ax
			mov    TIMER_CONTROL_REG, ax
			inc    ax
			mov    TIMER_DATA, ax
			inc    ax
			inc    ax
			mov    IRQ_DMA_CONTROL_REG, ax
			(* 3x? *)
			mov    ax, GusBase
			add    ax, 102h
			mov    GF1_PAGE_REG, ax
			inc    ax
			mov    GF1_REG_SELECT, ax
			add    ax, 4h
			mov    DRAM, ax
			pop    ax
		end;

		(* MEMORY *)
		GusMemory := GusMemSize;
	until (GusMemory <> 0) or (GusBase = $2A0);

	if (GusMemory = 0) then
	begin
		GusBase := 0;
		GusFind := False;
	end
		else GusFind := True;
end;

function MegaEm : Boolean; assembler;
asm
	(* CHECK INT 21H FOR MEGA-EM *)
	mov   ax, 0FD12h
	mov   bx, 3457h
  int   21h
  cmp   ax, 5678h
  jne   @NoMegaEm
  cmp   bx, 1235h
  jne   @NoMegaEm
	mov   ax, True
	jmp   @lab1

	@NoMegaEm:
	xor   ax, ax
	@lab1:
end;

function init(MaxVoices:byte):boolean;
procedure GUSReset;
	Procedure GUSDelay; Assembler;

	{ Pause for approx. 7 cycles. }

	ASM
		mov   dx, 0300h
		in    al, dx
		in    al, dx
		in    al, dx
		in    al, dx
		in    al, dx
		in    al, dx
		in    al, dx
	End;

	{ An incomplete routine to initialise the GUS for output. }

begin
	port[GUsBase+$103]:=$4C;
	port[GUsBase+$105]:=1;
	GUSDelay;
	port[GUsBase+$103]:=$4C;
	port[GUsBase+$105]:=7;
	port[GUsBase+$103]:=$0E;
	port[GUsBase+$105]:=(14 OR $0C0);
end;

begin
	GusFind;
	GusVoices      := GusNrOfVoices + 1;
	GusDataConvert := False;
	GusData16Bits  := False;

	(* PRIVATE *)
	GF1_FREQ_DIV     := FREQ_DIV_TABLE [GusVoices - 1];
	GF1_ACTIVE_VOICE := $FF;

	GusReset;
	GusMixer(LineOut);

	asm
	(* RESET REG *)
	mov	al, 4Ch
	xor	bl, bl
	call  GF1_OUT_BYTE
	call	GF1_DELAY

	(* RESET REG *)
	mov	al, 4Ch
	mov	bl, 1h
	call  GF1_OUT_BYTE
	call	GF1_DELAY

	(* DRAM DMA CONTROL REG *)
	mov	al, 41h
	xor	bl, bl
	call  GF1_OUT_BYTE
	call	GF1_DELAY

	(* TIMER CONTROL *)
	mov	al, 45h
	xor	bl, bl
	call  GF1_OUT_BYTE
	call	GF1_DELAY

	(* SAMPLING CONTROL REG *)
	mov	al, 49h
	xor	bl, bl
	call  GF1_OUT_BYTE
	call	GF1_DELAY

	(* ACTIVE VOICES *)
	mov	bl, MaxVoices
	mov   GusVoices, bl
	dec   bl
	mov	si, bx
	and	si, 0FFh
	shl	si, 1
	mov	ax, [offset FREQ_DIV_TABLE + si]
	mov	GF1_FREQ_DIV, ax
	or	bl, 0C0h
	mov   al, 0Eh
	call  GF1_OUT_BYTE
	call	GF1_DELAY

	xor   cx, cx
	@lab1:
	(* VOICE *)
	mov   al, cl
	call  GF1_VOICE
	call	GF1_DELAY

	(* VOICE CONTROL *)
	xor   al, al
	mov   bl, 3h
	call  GF1_OUT_BYTE

	(* VOLUME CONTROL *)
	mov   al, 0Dh
	mov   bl, 3h
	call  GF1_OUT_BYTE

	(* FREQUENCY = 0  (BX) *)
	xor   bx, bx
	mov   al, 1h
	call  GF1_OUT_WORD

	(* STARTING ADDRESS = 0 (BX) *)
	mov   al, 2h
	call  GF1_OUT_WORD
	mov   al, 3h
	call  GF1_OUT_WORD

	(* END ADDRESS = 0 (BX) *)
	mov   al, 4h
	call  GF1_OUT_WORD
	mov   al, 5h
	call  GF1_OUT_WORD

	(* VOLUME RAMP START = 0 (BL) *)
	mov   al, 7h
	call  GF1_OUT_BYTE

	(* VOLUME RAMP END = 0 (BL) *)
	mov   al, 8h
	call  GF1_OUT_BYTE

	(* CURRENT VOLUME = 0 (BX) *)
	mov   al, 9h
	call  GF1_OUT_WORD

	(* CURRENT ADDRESS = 0 (BX) *)
	mov   al, 0Ah
	call  GF1_OUT_WORD
	mov   al, 0Bh
	call  GF1_OUT_WORD

	(* VOLUME RAMP RATE = 0 (BL) *)
	mov   al, 6h
	call  GF1_OUT_BYTE

	(* PAN POSITION *)
	mov   al, 0Ch
	mov   bl, Middle
	call  GF1_OUT_BYTE

	inc   cx
	cmp   cx, 20h
	jne   @lab1

	(* RESET REG *)
	mov	al, 4Ch
	mov	bl, 7h
	call  GF1_OUT_BYTE
	call	GF1_DELAY
	end;

	init:=GusBase>0;
end;

procedure GusMixer (Output : ShortInt); assembler;
asm
	mov   dx, MIX_CONTROL_REG
	mov   al, Disabled
	xor   al, 80
	add   al, Output
	xor   al, 80
	out   dx, al
end;

procedure GusPoke (Address : LongInt; Value : Byte); assembler;
asm
  (* ADDRESS LOW *)
  mov   al, 43h
  mov   bx, [bp + offset Address + 0]
  call  GF1_OUT_WORD

  (* ADDRESS HIGH *)
  mov   al, 44h
  mov   bl, [bp + offset Address + 2]
  call  GF1_OUT_BYTE

  (* WRITE DRAM *)
  mov   dx, DRAM
  mov   al, Value
  out   dx, al
end;

function GusPeek (Address : LongInt) : Byte; assembler;
asm
  (* ADDRESS LOW *)
  mov   al, 43h
  mov   bx, [bp + offset Address + 0]
  call  GF1_OUT_WORD

  (* ADDRESS HIGH *)
  mov   al, 44h
  mov   bl, [bp + offset Address + 2]
  call  GF1_OUT_BYTE

  (* READ DRAM *)
  mov   dx, DRAM
  in    al, dx
end;

procedure GusWrite (Address : LongInt; var Buffer; Count : Word); assembler;
asm
	(* PUSH ES *)
	push  es

	(* TEST COUNT *)
	cmp   Count, 0
	je    @stop

	(* SET REGISTERS  CX:SI=ADDRESS  ES:DI=BUFFER *)
	mov   si, [bp + offset Address + 0]
	mov   cx, [bp + offset Address + 2]

	mov   di, [bp + offset Buffer + 0]
	mov   es, [bp + offset Buffer + 2]

	(* GUSDATAHIGH := FALSE *)
	mov   GusDataHigh, 1

	(* ADDRESS HIGH *)
	mov   al, 44h
	mov   bl, cl
	call  GF1_OUT_BYTE
	(* SELECT ADDRESS LOW *)
	mov   dx, GF1_REG_SELECT
	mov   al, 43h
	out   dx, al

	@lab1:
	(* ADDRESS LOW *)
	mov   dx, GF1_REG_SELECT
	inc   dx
	mov   ax, si
	out   dx, ax

	(* READ BUFFER *)
	mov   al, es:[di]
	inc   di

	(* 2CMP / 16BITS *)
	cmp   GusDataConvert, 0
	je    @signed

	cmp   GusData16Bits, 0
	je    @unsigned

	dec   GusDataHigh
	jnz   @unsigned
	mov   GusDataHigh, 2

	jmp   @signed

	@unsigned:
	xor   al, 80h

	@signed:
	(* WRITE DRAM *)
	mov   dx, DRAM
	out   dx, al

	(* INC ADDRESS *)
	inc   si
  jnz   @lab2
  inc   cx

  (* ADDRESS HIGH *)
  mov   al, 44h
  mov   bl, cl
  call  GF1_OUT_BYTE
  (* SELECT ADDRESS LOW *)
  mov   dx, GF1_REG_SELECT
  mov   al, 43h
  out   dx, al

  @lab2:
  (* LOOP UNTIL COUNT = 0 *)
  dec   Count
  jnz   @lab1

  @stop:
  (* POP ES *)
  pop   es
end;

procedure GusRead (Address : LongInt; var Buffer; Count : Word); assembler;
asm
  (* PUSH ES *)
  push  es

  (* TEST COUNT *)
  cmp   Count, 0
  je    @stop

  (* SET REGISTERS  CX:SI=ADDRESS  ES:DI=BUFFER *)
  mov   si, [bp + offset Address + 0]
  mov   cx, [bp + offset Address + 2]

	mov   di, [bp + offset Buffer + 0]
  mov   es, [bp + offset Buffer + 2]

  (* CLD *)
  cld

  (* GUSDATAHIGH := FALSE *)
  mov   GusDataHigh, 1

  (* ADDRESS HIGH *)
  mov   al, 44h
  mov   bl, cl
  call  GF1_OUT_BYTE
  (* SELECT ADDRESS LOW *)
  mov   dx, GF1_REG_SELECT
  mov   al, 43h
  out   dx, al

  @lab1:
  (* ADDRESS LOW *)
  mov   dx, GF1_REG_SELECT
  inc   dx
  mov   ax, si
  out   dx, ax

  (* READ DRAM *)
  mov   dx, DRAM
  in    al, dx

  (* 2CMP / 16BITS *)
  cmp   GusDataConvert, 0
  je    @signed
	cmp   GusData16Bits, 0
  je    @unsigned
  dec   GusDataHigh
  jnz   @unsigned
  mov   GusDataHigh, 2
  jmp   @signed
  @unsigned:
  xor   al, 80h

  @signed:
  (* WRITE BUFFER *)
  stosb

	(* INC ADDRESS *)
  inc   si
  jnz   @lab2
  inc   cx

  (* ADDRESS HIGH *)
  mov   al, 44h
  mov   bl, cl
  call  GF1_OUT_BYTE
  (* SELECT ADDRESS LOW *)
  mov   dx, GF1_REG_SELECT
  mov   al, 43h
  out   dx, al

  @lab2:
  (* LOOP UNTIL COUNT = 0 *)
  dec   Count
  jnz   @lab1

  @stop:
  (* POP ES *)
  pop   es
end;

procedure VoiceInit(Voice : Byte);
begin
	VoiceStop (Voice);
	RampStop (Voice);

	RampMode (Voice, LoopOff+UniDir+Up);
	RampRange (Voice, $000, $000);
	RampRate (Voice, 0, 0);

	VoiceMode (Voice, Bit8 + LoopOff + UniDir + Forw);
	VoiceFreq (Voice, 0);
	VoiceBalance (Voice, Middle);
	VoiceVolume (Voice, $000);
	PlaySmp (Voice, 0, 0, 0);
end;

procedure VoiceBalance (Voice : Byte; Balance : Byte); assembler;
asm
	(* VOICE *)
	mov   al, Voice
	call  GF1_VOICE

	(* PANNING *)
	mov   al, 0Ch
	mov   bl, Balance
	call  GF1_OUT_BYTE
end;

procedure VoiceVolume (Voice : Byte; Volume : Word); assembler;
asm
	(* VOICE *)
	mov   al, Voice
	call  GF1_VOICE

	(* VOLUME *)
	mov   al, 9h
	mov   bx, Volume
	shl   bx, 4
	call  GF1_OUT_WORD
end;

procedure VoiceFreq (Voice : Byte; Freq : Word); assembler;
asm
	(* VOICE *)
	mov   al, Voice
	call  GF1_VOICE

	(* FREQUENCY *)
	xor   dx, dx
	mov   ax, Freq
	div   GF1_FREQ_DIV
	mov   bx, ax
	mov   al, 1h
	call  GF1_OUT_WORD
end;

procedure VoiceStart (Voice : Byte); assembler;
asm
	(* VOICE *)
	mov   al, Voice
	call  GF1_VOICE

	(* VOICE CONTROL: START *)
	mov   bl, 0FCh
	xor   bh, bh
	call  GF1_VOICE_CONTROL;
end;

procedure VoiceStop (Voice : Byte); assembler;
asm
	(* VOICE *)
	mov   al, Voice
	call  GF1_VOICE

	(* VOICE CONTROL: STOP *)
	mov   bl, 0FCh
	mov   bh, 3h
	call  GF1_VOICE_CONTROL;
end;

procedure VoiceMode (Voice : Byte; Mode : Byte); assembler;
asm
  (* VOICE *)
  mov   al, Voice
  call  GF1_VOICE

  (* VOICE CONTROL: START *)
  mov   bl, 83h
  mov   bh, Mode
  call  GF1_VOICE_CONTROL;
end;

procedure PlaySmp (Voice : Byte; Start, LoopStart, LoopEnd : LongInt); assembler;
asm
  (* VOICE *)
  mov   al, Voice
  call  GF1_VOICE

  (* CURRENT LOCATION *)
  mov   al, 0Ah
	mov   si, [bp + offset Start + 0]
  mov   bx, [bp + offset Start + 2]
  call  GF1_OUT_ADDR

  (* LOOP START *)
  mov   al, 2h
  mov   si, [bp + offset LoopStart + 0]
  mov   bx, [bp + offset LoopStart + 2]
  call  GF1_OUT_ADDR

  (* LOOP END *)
	mov   al, 4h
  mov   si, [bp + offset LoopEnd + 0]
  mov   bx, [bp + offset LoopEnd + 2]
  call  GF1_OUT_ADDR
end;

function GetVoiceLoc (Voice : Byte; Location : Byte) : LongInt; assembler;
asm
	(* VOICE *)
  mov   al, Voice
  call  GF1_VOICE

  (* ADDRESS LOCATION: 8Ah | 82h | 84h *)
  mov   al, Location
  call  GF1_IN_ADDR
end;

function VoiceActive (Voice : Byte) : Boolean; assembler;
asm
  (* VOICE *)
  mov   al, Voice
  call  GF1_VOICE

  (* VOICE ACTIVE *)
  mov   al, 80h
  call  GF1_IN_BYTE
  and   al, 3h  (* ?? bit 0 not allways set ?? *)
	jz    @lab1
  mov   al, 1h
  @lab1:
  xor   al, 1h
end;

procedure  RampStart (Voice : Byte); assembler;
asm
  (* VOICE *)
  mov   al, Voice
  call  GF1_VOICE

  (* RAMP CONTROL: START *)
  mov   bl, 0FCh
  xor   bh, bh
  call  GF1_RAMP_CONTROL;
end;

procedure  RampStop (Voice : Byte); assembler;
asm
  (* VOICE *)
  mov   al, Voice
	call  GF1_VOICE

  (* RAMP CONTROL: STOP *)
  mov   bl, 0FCh
  mov   bh, 3
  call  GF1_RAMP_CONTROL
end;

procedure  RampMode (Voice : Byte; Mode : Byte); assembler;
asm
  (* VOICE *)
  mov   al, Voice
  call  GF1_VOICE

  (* RAMP CONTROL: MODE *)
  mov   bl, 0A7h
	mov   bh, Mode
  call  GF1_RAMP_CONTROL
end;

procedure  RampRate (Voice : Byte; Scale, Rate : Byte); assembler;
asm
	(* VOICE *)
  mov   al, Voice
  call  GF1_VOICE

  (* RAMP RATE *)
  mov   al, 6h
  mov   bl, Scale
  shl   bl, 6
  or    bl, Rate
  call  GF1_OUT_BYTE
end;

procedure  RampRange (Voice : Byte; Lower, Upper : Word); assembler;
asm
  (* VOICE *)
  mov   al, Voice
	call  GF1_VOICE

  (* RAMP RANGE *)
	mov   al, 7h
  mov   bx, Lower
  shr   bx, 4
	call  GF1_OUT_BYTE

	mov   al, 8h
	mov   bx, Upper
	shr   bx, 4
	call  GF1_OUT_BYTE
end;

end.
