#include "mathematics.hpp"

// Mersenne twister defines
#define M 397
#define MATRIX_A 0x9908b0dfUL   /* constant vector a */
#define UMASK 0x80000000UL /* most significant w-r bits */
#define LMASK 0x7fffffffUL /* least significant r bits */
#define MIXBITS(u,v) ( ((u) & UMASK) | ((v) & LMASK) )
#define TWIST(u,v) ((MIXBITS(u,v) >> 1) ^ ((v)&1UL ? MATRIX_A : 0UL))

Vector Math::sphereToCartesian(float radius, float phi, float theta)
{
	float sp = (float)sin(phi);
	float cp = (float)cos(phi);
	float st = (float)sin(theta);
	float ct = (float)cos(theta);
	Vector value = Vector(radius*sp*ct, radius*sp*st, radius*cp);
	return value;
}
Vector Math::cartesianToSphere(float x, float y, float z)
{
	float rho = (float)sqrt(x*x+y*y+z*z);
	if (rho == 0.0f) 
		return Vector(0,0,0);
/*
    float phi = (float)acos(z/rho);
	float theta = (float)acos(x/(rho*sin(phi)));
*/

    float phi = atan2f(y, x);
    float theta = acosf(z / rho);
	Vector value = Vector(rho, theta, phi);
	return value;
}
float Math::integrateSimpson(float startpoint, float endpoint, int steps, float(*function)(float))
{
	float val = 0.0f;
	float h = (endpoint-startpoint)/steps;
	val += function(startpoint);
	val += function(endpoint);

	for (int i=0;i<steps-1;i++)
	{
		if (!(i&1)) //parillinen
			val += 4*function(h+i*h);
		else
			val += 2*function(h+i*h);
	}
	val *= h/3.0f;
	return val;
}
void Math::antiRotate(Vector3 *x, Vector3 *y, Vector3 *z)
{
	float matrix[16];
	glGetFloatv(GL_MODELVIEW_MATRIX, matrix);	
	
	*x = (Vector(matrix[0], matrix[4], matrix[8]));
	*y = (Vector(matrix[1], matrix[5], matrix[9]));
	*z = Vector(matrix[2], matrix[6], matrix[10]);
}

float Math::calcSaturate(float value, float limit1, float limit2, float multiply)
{
	float pos, arvo;
	if (value <= limit1)
		return 0;
	if (value >= limit2)
		return 0;
	else
		pos = (float)(value-limit1)/(limit2-limit1);

	arvo = (float)sin(pos*3.141592f)*multiply;
	if (arvo < 0) 
		arvo = 0;
	if (arvo > 1)
		arvo = 1;

	return arvo;
}

float Math::calcPosFloat(float value, float limit1, float limit2)
{
	if (value <= limit1)
		return 0;
	if (value >= limit2)
		return 1;
	else
		return (float)(value-limit1)/(limit2-limit1);
}

float Math::calcSmoothStep(float value, float limit1, float limit2)
{ 
  if (value < limit1) return 0;
  if (value >= limit2) return 1; 
  float x = clampVal((value - limit1)/(limit2 - limit1),0,1); 
  return (x*x * (3 - 2*x)); 
}
             
float Math::calcPosCos(float value, float limit1, float limit2)
{
	if (value <= limit1)
		return 0;
	if (value >= limit2)
		return 1;
	
	const float val = 1.0f-cosf(1.5707963f*Math::calcPosFloat(value, limit1, limit2));
	
	const float accuracy_factor = 0.001f;

	// palautetaan 0 jos ollaan kipupisteen alapuolella 
	//(cosf ei anna nollaa, koska PII/2 ei ole jaollinen)
	if(val < accuracy_factor) return 0.0f;
	else return val;
}


float Math::randFloat()
{
	return (float)((rand()%10000)/10000.0f);
}

float Math::randBetween(float min, float max)
{
    return min + randFloat() * (max - min);
}

float Math::randFloat(float i) {
	if(i<=0) return 0.0f;
	return randFloat()*i;
}

float Math::minimum(float val, float mini)
{
	if(val > mini) return mini;
	return val;
}

float Math::maximum(float val, float maxi)
{
	if(val < maxi) return maxi;
	return val;
}

float Math::clampVal(float val, float mini, float maxi)
{
	if(val < mini) return mini;
	if(val > maxi) return maxi;
	return val;
}

int Math::randInt(int i) {
	if(i<=0) return 0;
	return rand()%i;
}

Vector Math::randVector()
{
	return Vector(randFloat()-0.5f, randFloat()-0.5f, randFloat()-0.5f);
}

Vector Math::randVector(float dx, float dy, float dz)
{
	return Vector(dx*(randFloat()-0.5f), dy*(randFloat()-0.5f), dz*(randFloat()-0.5f));

}

Vector Math::randVector2()
{
	return Vector(randFloat(), randFloat(), randFloat());

}
Vector Math::randVectSphere()
{
	float rho = randFloat()*3.141592f;
	float theta = randFloat()*3.141592f*2;

	return sphereToCartesian(1, theta, rho);
}
Vector Math::randVectPlane()
{
    float a = randFloat()*2*3.141592f;
    return Vector(cosf(a), 0, sinf(a));
}

float Math::getClosest(float value, float first, float second) 
{
	  if(fabsf(value - first) < fabsf(value - second))
		return first;
	  return second;
}

// Mersenne twister
void Math::init_genrand(unsigned long s)
{
    int j;
    state[0]= s & 0xffffffffUL;
    for (j=1; j<N; j++) {
        state[j] = (1812433253UL * (state[j-1] ^ (state[j-1] >> 30)) + j); 
        /* See Knuth TAOCP Vol2. 3rd Ed. P.106 for multiplier. */
        /* In the previous versions, MSBs of the seed affect   */
        /* only MSBs of the array state[].                        */
        /* 2002/01/09 modified by Makoto Matsumoto             */
        state[j] &= 0xffffffffUL;  /* for >32 bit machines */
    }
    left = 1; //initf = 1;
}

void Math::next_state()
{
    unsigned long *p=state;
    int j;

    /* if init_genrand() has not been called, */
    /* a default initial seed is used         */
    /* initf is disabled, the class explicitly needs to be initialized
     * via a constructor which leaves the object to a known state.
     * moved the default seed to the header. -- Jetro Lauha 2004-10-07
     */
    //if (initf==0) init_genrand(5489UL);

    left = N;
    next = state;
    
    for (j=N-M+1; --j; p++) 
        *p = p[M] ^ TWIST(p[0], p[1]);

    for (j=M; --j; p++) 
        *p = p[M-N] ^ TWIST(p[0], p[1]);

    *p = p[M-N] ^ TWIST(p[0], state[0]);
}

/* generates a random number on [0,0xffffffff]-interval */
unsigned long Math::genrand_int32(void)
{
    unsigned long y;

    if (--left == 0) next_state();
    y = *next++;

    /* Tempering */
    y ^= (y >> 11);
    y ^= (y << 7) & 0x9d2c5680UL;
    y ^= (y << 15) & 0xefc60000UL;
    y ^= (y >> 18);

    return y;
}

/* generates a random number on [0,0x7fffffff]-interval */
long Math::genrand_int31(void)
{
    unsigned long y;

    if (--left == 0) next_state();
    y = *next++;

    /* Tempering */
    y ^= (y >> 11);
    y ^= (y << 7) & 0x9d2c5680UL;
    y ^= (y << 15) & 0xefc60000UL;
    y ^= (y >> 18);

    return (long)(y>>1);
}

/* generates a random number on [0,1]-real-interval */
float Math::genrand_real1(void)
{
    unsigned long y;

    if (--left == 0) next_state();
    y = *next++;

    /* Tempering */
    y ^= (y >> 11);
    y ^= (y << 7) & 0x9d2c5680UL;
    y ^= (y << 15) & 0xefc60000UL;
    y ^= (y >> 18);

    return (float)y * (1.0f/4294967295.0f); 
    /* divided by 2^32-1 */ 
}

/* generates a random number on [0,1)-real-interval */
float Math::genrand_real2(void)
{
    unsigned long y;

    if (--left == 0) next_state();
    y = *next++;

    /* Tempering */
    y ^= (y >> 11);
    y ^= (y << 7) & 0x9d2c5680UL;
    y ^= (y << 15) & 0xefc60000UL;
    y ^= (y >> 18);

    return (float)y * (1.0f/4294967296.0f);
    /* divided by 2^32 */
}

/* generates a random number on (0,1)-real-interval */
float Math::genrand_real3(void)
{
    unsigned long y;

    if (--left == 0) next_state();
    y = *next++;

    /* Tempering */
    y ^= (y >> 11);
    y ^= (y << 7) & 0x9d2c5680UL;
    y ^= (y << 15) & 0xefc60000UL;
    y ^= (y >> 18);

    return ((float)y + 0.5f) * (1.0f/4294967296.0f); 
    /* divided by 2^32 */
}

/* generates a random number on [0,1) with 53-bit resolution*/
float Math::genrand_res53(void) 
{ 
    unsigned long a=genrand_int32()>>5, b=genrand_int32()>>6; 
    return(a*67108864.0f+b)*(1.0f/9007199254740992.f); 
} 

void Math::getQuadraticBezierPoint(Vector3 p1, Vector3 p2, Vector3 p3, float t, Vector3 *point)
{
	*point =  p1 * ((1 - t) * (1 - t)) + p2 * (2 * t * (1 - t)) + p3 * (t * t);
}