require 'rexml/parsers/baseparser'
require 'rexml/document'
require 'socket'
require 'timeout'

module Machines
  def command name, args = {}
    decl = REXML::XMLDecl.new
    decl.encoding = "UTF-8"
    doc = REXML::Document.new
    doc << decl
    doc.add_element name, Hash[*args.map { |name, value| [name, value.to_s] }.flatten]
    doc.to_s
  end

  class Client
    include Machines
  
    attr_reader :socket, :name, :id
    attr_accessor :game, :skin, :score, :slot
    
    def initialize server, socket, id
      @server = server
      @socket = socket
      @id = id
      name, args = receive
      raise "Invalid command: #{name}" unless name == "login"
      @name = args["name"]
      @server.log "#@name connected"
    end
    
    def receive
      raise Errno::EINVAL if @socket.eof?
      data = @socket.gets("\0")
      raise Errno::EINVAL unless data
      @server.log "#{@name}: #{data[0..-2]}"
      parser = REXML::Parsers::BaseParser.new data
      type, name, args = parser.pull
      if type == :xmldecl
        type, name, args = parser.pull
      end
      [name, args]
    end

    def send data
      return if disconnected?
      begin
        @socket.write data + "\0"
      rescue Errno::EINVAL, Errno::ECONNABORTED
        @socket = nil
      end
    end
    
    def disconnect
      @socket.close rescue Exception
      @socket = nil
    end

    def disconnected?
      @socket.nil?
    end

    def error message
      send command("error", "message"=>message)
    end

    def message message
      send command("message", "text"=>message)
    end

    def inspect
      result = "<Client #{@name.inspect}"
      result << " on #{@game.name.inspect}" if @game
      result << " disconnected" if disconnected?
      result << ">"
      result
    end
  end

  class Wall
    attr_accessor :x, :y, :z
  
    def initialize values
      @x, @y, @z = @values = values.split(",").map { |x| x.to_i }
    end
    
    def to_s
      @values.join(",")
    end
    
    def cells
      case @z
      when 0
        [
          [@x, @y - 1],
          [@x, @y],
        ]
      when 1
        [
          [@x, @y],
          [@x - 1, @y],
        ]
      else
        raise "Invalid Z: #{@values[2]}"
      end
    end
  end

  class Game
    include Machines

    attr_reader :name, :rows, :cols, :players, :host, :walls
    def initialize server, host, name, rows, cols, round_time
      @server = server
      @host = host
      @name = name
      @rows = rows
      @cols = cols
      @players = Array.new(4)
      @started = false
      @player = 0
      @done = false
      @cells = Array.new(@cols).map { Array.new(@rows, 0) }
      @rows.times { |row|
        @cells[0][row] += 1
        @cells[-1][row] += 1
      }
      @cols.times { |col|
        @cells[col][0] += 1
        @cells[col][-1] += 1
      }
      @switched = 0
      @round_time = round_time
      @walls = (0..(@rows+1)).map {
        (0..(@cols+1)).map {
          Array.new(2, false)
        }
      }
      @cols.times { |x|
        @walls[x][0][0] = true
        @walls[x][@rows][0] = true
      }
      @rows.times { |y|
        @walls[0][y][1] = true
        @walls[@cols][y][1] = true
      }
    end
    
    def done?
      @done
    end
    
    def time
      if @round_start
        stop = @round_start + @round_time
        remaining = stop - Time.now
        if remaining < 0
          0
        else
          remaining
        end
      else
        @round_time
      end
    end
    
    def free_walls
      walls = []
      (@rows).times { |y|
        (@cols).times { |x|
          2.times { |z|
            if !@walls[x][y][z]
              walls << Wall.new("#{x},#{y},#{z}")
            end
          }
        }
      }
      walls
    end
    
    def check_time
      if time == 0
        walls = free_walls
        wall = walls[rand(walls.size)]
        play current_player, wall
      end
    end
    
    def start
      @started = true
      connected_players.each_with_index { |player, i| player.skin = i }
    end
    
    def started?
      @started
    end
    
    def current_player
      @players[@player]
    end

    def next_player
      @player = (@player + 1) % @players.size
    end
    
    def new_turn
      return if connected_players.all? { |player| player.game != self }
      while current_player.nil? or current_player.disconnected?
        next_player
      end
      connected_players.each { |player|
        player.send command("new_turn", "player"=>current_player.id, "slot"=>current_player.slot, "time"=>@round_time)
      }
      if current_player.disconnected?
        connected_players.each { |player|
          player.send command("not_played", "player"=>current_player.id, "slot"=>current_player.slot)
        }
        new_turn
      else
        current_player.send command("play")
      end
      @round_start = Time.now
    end
    
    def play client, wall
      begin
        raise "You can't play now" unless client == current_player
        raise "Wall already used: #{wall}" if @walls[wall.x][wall.y][wall.z]
        commands = []
        commands << command("played", "player"=>client.id, "skin"=>client.skin, "position"=>wall, "slot"=>client.slot)
        @walls[wall.x][wall.y][wall.z] = true
        switched_any = false
        wall.cells.each { |x, y|
          next if x < 0 or x >= @cols or y < 0 or y >= @rows
          @cells[x][y] += 1
          if @cells[x][y] == 4
            client.score += 1
            commands << command("switch", "player"=>client.id, "slot"=>client.slot, "skin"=>client.skin, "player"=>client.id, "skin"=>client.skin, "position"=>[x, y].join(","), "score"=>client.score)
            @switched += 1
            switched_any = true
          end
        }
        if @switched == @rows * @cols
          commands << command("end")
          @done = true
        end
        connected_players.each { |player|
          commands.each { |command|
            player.send command
          }
        }
        next_player unless switched_any
        new_turn
      rescue RuntimeError => e
        client.error e.message
      end
    end
    
    def map
      text = ""
      line = "-" * @cols + "\n"
      text << line
      @rows.times { |row|
        @cols.times { |col|
          text << @cells[col][row].to_s
        }
        text << "\n"
      }
      text << line
      text
    end
    
    def connected_players
      @players.dup.compact
    end

    def add_player client
      slot = @players.index nil
      raise "Game is full" unless slot
      client.game = self
      client.slot = slot
      client.score = 0
      client.send command("joined",  "name"=>@name, "rows"=>@rows, "cols"=>@cols, "slot"=>slot)
      @players[slot] = client
      connected_players.each { |player|
        next unless player
        client.send command("player",  "id"=>player.id, "name"=>player.name, "slot"=>player.slot)
      }
      connected_players.each { |player|
        next unless player and player != client
        player.send command("player",  "id"=>client.id, "name"=>client.name, "slot"=>client.slot)
      }
    end

    def kick player
      player.send command("left")
      player.game = nil
      connected_players.each { |other_player|
        next if other_player == player
        other_player.message "#{player.name} left the game"
        other_player.send command("player_left", "player"=>player.id, "slot"=>player.slot)
      }
      @players[player.slot] = nil
      if started? and @player == player.slot
        connected_players.each { |other_player|
          other_player.send command("not_played", "player"=>player.id, "slot"=>player.slot)
        }
        new_turn
      end
      @done = true if connected_players.all? { |player| player.game != self }
    end
  end

  class Server
    include Machines

    attr_reader :clients, :games
    attr_accessor :log_streams, :now, :round_time
    
    def initialize
      @clients = []
      @games = []
      @next_id = 0
      @log_streams = []
      @now = nil
      @round_time = 20
    end
    
    def now
      @now || Time.now
    end

    def log text
      @log_streams.each { |stream| stream.puts text }
    end
    
    def send_games player
      @games.each { |game|
        next if game.started?
        player.send command("new_game", "name" => game.name)
      }
    end
    
    def receive_data client
      name, args = client.receive
      case name
      when "create"
        raise "Already in game" if client.game
        name = args["name"]
        game = @games.find { |game| game.name == name }
        if game
          client.error "Game #{name.inspect} already exists"
        else
          game = Game.new(self, client, name, args['rows'].to_i, args['cols'].to_i, @round_time)
          @clients.each { |other_client|
            next if other_client == client or other_client.game
            other_client.send command("new_game", "name" => game.name)
          }
          @games << game
          game.add_player client
        end
      when "cancel"
        raise "You aren't the host" unless client == client.game.host
        game = client.game
        @games.delete game
        @clients.each { |other_client|
          next if other_client.game
          other_client.send command("del_game", "name" => game.name)
        }
        game.connected_players.each { |player|
          player.send command("left")
          player.game = nil
          send_games player
        }
      when "leave"
        game = client.game
        game.kick client
        send_games client
      when "join"
        raise "Already in game" if client.game
        name = args["name"]
        game = @games.find { |game| game.name == name }
        if game
          if game.started?
            client.error "Game already started"
          else
            game.add_player client
          end
        else
          client.error "Invalid game: #{name.inspect}"
        end
      when "start"
        game = client.game
        game.start
        @clients.each { |other_client|
          other_client.send command("del_game", "name"=>game.name) unless other_client.game
        }
        game.connected_players.each { |other_client|
          other_client.send command("started")
        }
        game.new_turn
      when "play"
        game = client.game
        game.play client, Wall.new(args["position"])
      when "chat"
        message = args["message"]
        client.game.connected_players.each { |player|
          player.send command("chat", "message"=>message, "player"=>client.id)
        }
      else
        raise "Invalid command: #{name}"
      end
    end
  
    def clean_games
      @games.delete_if { |game|
        if game.done?
          @clients.each { |other_client|
            next if other_client.game
            other_client.send command("del_game", "name" => game.name)
          }
          true
        else
          false
        end
      }
    end
  
    def start
      @server = TCPServer.new 8856
      begin
        loop do
          sockets = [@server] + @clients.map { |client| client.socket }.compact
          
          timeout = if @games.empty?
            nil
          else
            min_time = @round_time
            @games.each { |game|
              min_time = game.time if game.time < min_time
            }
            min_time
          end

          clean_games
          readable, = select sockets, nil, nil, timeout
          @games.each { |game| game.check_time }

          readable ||= []
          readable.each { |socket|
            client = @clients.find { |client| client.socket == socket }
            if client
              begin
                receive_data client
              rescue Errno::EINVAL
                log "#{client.name} disconnected"
                client.disconnect
              rescue Exception => e
                client.send command("error", "message"=>e.message)
                log "#{e.message} (#{e.class})"
                e.backtrace.each { |line| log line } unless e.class == RuntimeError
              end
              
              if client.disconnected?
                if client.game
                  client.game.kick client
                end
                @clients.delete client
              end
            else
              begin
                socket = @server.accept
                client = Client.new(self, socket, @next_id)
                @clients << client
                send_games client
                @next_id += 1
              rescue Exception => e
                log "#{e.message} (#{e.class})"
                e.backtrace.each { |line| log line }
              end
            end
          }
        end
      rescue Interrupt
      ensure
        @server.close
      end
    end
  end
end

if __FILE__ == $0
  server = Machines::Server.new
  File.open 'machines_server.log', 'a' do |f|
    server.log_streams = [STDOUT, f]
    server.start
  end
end

