require 'test/unit'
require 'socket'
require 'timeout'
require "machines_server"

class TC_MachinesServer < Test::Unit::TestCase
  include Machines

  class Client
    include Machines
    
    attr_reader :name, :socket, :id
    attr_accessor :skin, :score, :slot
    
    def initialize name, id
      @name = name
      @id = id
      @score = 0
      @socket = TCPSocket.new 'localhost', 8856
      #send command("login", "name"=>name)
      send "<login name=#{name.inspect}/>"
    end
    
    def send data
      @socket.write data + "\0"
    end
    
    def create name, rows, cols
      send command('create', 'name'=>name, "rows"=>rows, "cols"=>cols)
    end

    def cancel
      send command('cancel')
    end
    
    def join name
      send command("join", "name"=>name)
    end
    
    def play position
      send command("play", "position"=>position)
    end
    
    def start
      send command("start")
    end
    
    def chat message
      send command("chat", "message"=>message)
    end
    
    def leave
      send command("leave")
    end
    
    def disconnect
      @socket.close
    end
  end
  
  def setup
    @next_id = 0
    @server = Machines::Server.new
    @server.log_streams = [File.open("test.log", "w")]
    @thread = Thread.new do
      Thread.current.abort_on_exception = true
      @server.start
    end
    Thread.pass
  end
  
  def teardown
    @thread.kill
    @thread.value
  end
  
  def assert_receive data, players, msg = nil
    players = [players] unless players.is_a? Array
    players.each { |player|
      begin
        timeout(1) {
          msg ||= player.name
          received = player.socket.gets("\0")
          #commands = received.scan(/<\?xml version='1.0' encoding='UTF-8'\?>(.*)\0/).first
          #assert commands, "invalid reply: #{received.inspect}"
          assert_equal data, received[0..-2], msg
        }
      rescue TimeoutError
        raise "#{data.inspect} not received by #{player.name}"
      end
    }
  end
  
  def assert_new_game name, players
    assert_receive command("new_game", "name"=>name), players
  end

  def assert_joined name, rows, cols, players, client
    client.slot = players.index client
    assert_receive command("joined", "name"=>name, "rows"=>rows, "cols"=>cols, "slot"=>client.slot), client
    players.each { |player|
      assert_player player, client
    }
  end

  def assert_left client
    assert_receive command("left"), client
  end
  
  def assert_del_game name, players
    assert_receive command("del_game", "name"=>name), players
  end

  def assert_error message, players
    assert_receive command("error", "message"=>message), players
  end

  def assert_player player, client
    assert_receive command("player", "name"=>player.name, "id"=>player.id, "slot"=>player.slot), client
  end

  def assert_new_turn player, players
    assert_receive command("new_turn", "player"=>player.id, "slot"=>player.slot, "time"=>@server.round_time), players
  end
  
  def assert_played player, position, players
    assert_receive command("played", "player"=>player.id, "slot"=>player.slot, "position"=>position, "skin"=>player.skin), players
  end

  def assert_not_played player, players
    assert_receive command("not_played", "player"=>player.id, "slot"=>player.slot), players
  end

  def assert_play player
    assert_receive command("play"), player
  end
  
  def assert_started players
    assert_receive command("started"), players
  end

  def assert_message msg, players
    assert_receive command("message", "text"=>msg), players
  end

  def new_client name
    client = Client.new name, @next_id
    @next_id += 1
    client
  end
  
  def assert_switch player, switch, players, msg = nil
    player.score += 1
    assert_receive command("switch", "player"=>player.id, "slot"=>player.slot, "skin"=>player.skin, "position"=>switch, "score"=>player.score), players, msg
  end
  
  def assert_player_left player, players
    assert_receive command("player_left", "player"=>player.id, "slot"=>player.slot), players
  end


  def play players, steps
    current = 0
    steps.each_with_index { |values, i|
      position, *switches = values
      player = players[current]
      assert_new_turn player, players
      assert_play player
      player.play position
      assert_played player, position, players
      switches.each { |switch|
        assert_switch player, switch, players, "turn #{i}"
      }
      current = (current + 1) % players.size if switches.empty?
    }
  end

  def end_game players
    assert_receive command("end"), players
  end

  def create_game rows, cols, *names
    host, client, = names.map { |name| new_client name }
    players = [host, client]
    host.create "a game", rows, cols
    assert_receive command("new_game", "name"=>'a game'), client
    assert_joined 'a game', rows, cols, [host], host
    client.join "a game"
    assert_joined 'a game', rows, cols, [host, client], client
    assert_player client, host
    players
  end
  
  def start_game host, players
    host.start
    assert_started players
    players.each_with_index { |player, i| player.skin = i }
  end

  def test_game
    players = create_game 2, 3, "bob", "mock"
    start_game players.first, players

    play players, [
      ["0,1,0"],
      ["1,0,1", "0,0"],
      ["1,1,0"],
      ["2,1,1"],
      ["1,1,1", "1,1", "0,1"],
      ["2,0,1", "1,0"],
      ["2,1,0", "2,0", "2,1"],
    ]
    end_game players
  end

  def test_create_two_games
    bob = new_client 'bob'
    mock = new_client 'mock'
    foo = new_client 'foo'
    players = [bob, mock, foo]
    bob.create "bob's game", 3, 3
    assert_joined "bob's game", 3, 3, [bob], bob
    assert_new_game "bob's game", [mock, foo]
    foo.create "foo's game", 3, 3
    assert_joined "foo's game", 3, 3, [foo], foo
    assert_new_game "foo's game", [mock]
    foo.start
    assert_started foo
    assert_del_game "foo's game", mock
  end
  
  def test_cancel_game
    bob = new_client 'bob'
    bob.create "my game", 2, 2
    assert_joined "my game", 2, 2, [bob], bob
    
    mock = new_client 'mock'
    assert_new_game "my game", mock
    mock.create "other game", 2, 2
    assert_joined "other game", 2, 2, [mock], mock
    
    foo = new_client 'foo'
    assert_new_game "my game", foo
    assert_new_game "other game", foo
    foo.join "my game"
    assert_joined "my game", 2, 2, [bob, foo], foo
    assert_player foo, bob
    foo.cancel
    assert_error "You aren't the host", foo
    
    bob.cancel
    [bob, foo].each { |client|
      assert_left client
      assert_new_game "other game", client
    }
  end
  
  def test_canceled_game_deleted
    bob = new_client 'bob'
    bob.create "my game", 2, 2
    assert_joined "my game", 2, 2, [bob], bob
    
    mock = new_client 'mock'
    assert_new_game "my game", mock
    
    bob.cancel
    assert_left bob
    assert_del_game "my game", mock
    
    bob.create "another game", 3, 3
    assert_joined "another game", 3, 3, [bob], bob
    assert_new_game "another game", mock

    bob.disconnect
    assert_del_game "another game", mock
  end
  
  def test_game_list_at_connect
    bob = new_client 'bob'
    bob.create "my game", 2, 2
    assert_joined "my game", 2, 2, [bob], bob

    mock = new_client 'mock'
    assert_new_game "my game", mock
  end
  
  def test_join_invalid_game
    bob = new_client 'bob'
    bob.join "a game"
    assert_error "Invalid game: \"a game\"", bob
  end
  
  def test_disconnect
    bob = new_client 'bob'
    mock = new_client 'mock'
    bob.disconnect
    mock.join 'foo'
    assert_error "Invalid game: \"foo\"", mock
  end
  
  def test_create_with_existant_name
    bob = new_client 'bob'
    mock = new_client 'mock'
    mock.create 'foo', 2, 2
    assert_joined 'foo', 2, 2, [mock], mock
    assert_new_game 'foo', bob
    bob.create 'foo', 2, 2
    assert_error "Game \"foo\" already exists", bob
  end

  def test_disconnect_in_game
    bob, mock = create_game 3, 2, "bob", "mock"
    start_game bob, [bob, mock]
    assert_new_turn bob, [bob, mock]
    assert_play bob
    bob.disconnect
    assert_message "bob left the game", mock
    assert_player_left bob, mock
    assert_not_played bob, mock
    assert_new_turn mock, mock
    assert_play mock
    mock.play "0,1,0"
    assert_played mock, "0,1,0", mock
    assert_new_turn mock, mock
    mock.disconnect

    foo = new_client "foo"
    assert_del_game "a game", foo
    foo.join "foo"
    assert_error "Invalid game: \"foo\"", foo
  end
  
  def test_invalid_play
    bob, mock = players = create_game(3, 2, "bob", "mock")
    start_game bob, players
    play players, [["0,1,0"]]
    assert_new_turn mock, players
    assert_play mock
    mock.play "0,1,0"
    assert_error "Wall already used: 0,1,0", mock
  end
  
  def test_not_your_turn
    bob, mock = players = create_game(3, 2, "bob", "mock")
    start_game bob, players
    assert_new_turn bob, players
    assert_play bob
    mock.play "0,1,0"
    assert_error "You can't play now", mock
  end
  
  def test_timeout
    srand 123456
    @server.round_time = 0.2
    bob, mock = players = create_game(2, 2, "bob", "mock")
    start_game bob, players
    play players, [["0,1,0"], ["1,1,1", "0,1"]]
    
    assert_new_turn mock, players
    assert_play mock
    sleep 0.3
    assert_played mock, "1,0,1", players
    assert_switch mock, "0,0", players
    assert_new_turn mock, players
    assert_play mock
    srand
  end
  
  def test_initial_walls
    bob, mock = players = create_game(2, 3, "bob", "mock")
    game = @server.games.find { |game| game.name == "a game" }
    [
      [0,0,0, true],
      [1,0,0, true],
      [2,0,0, true],
      [0,0,1, true],
      [0,1,1, true],
      [0,2,0, true],
      [1,2,0, true],
      [2,2,0, true],
      [3,0,1, true],
      [3,1,1, true],
    ].each { |x, y, z, value|
      assert_equal value, game.walls[x][y][z], "#{x},#{y},#{z}"
    }
  end
  
  def test_join_started_game
    bob, mock = players = create_game(3, 2, "bob", "mock")
    start_game bob, players
    foo = new_client "foo"
    foo.join "a game"
    assert_error "Game already started", foo
  end
  
  def test_join_two_games
    bob, mock = players = create_game(3, 2, "bob", "mock")
    bob.create "foo", 3, 3
    assert_error "Already in game", bob
    
    bob.join "foo"
    assert_error "Already in game", bob
  end
  
  def test_leave_game
    bob, mock = players = create_game(3, 2, "bob", "mock")
    mock.leave
    assert_left mock
    assert_message "mock left the game", bob
    assert_player_left mock, bob
  end

  def test_leave_started_game
    bob, mock = players = create_game(3, 2, "bob", "mock")
    start_game bob, players
    assert_new_turn bob, players
    assert_play bob

    foo = new_client "foo"
    foo.create "foo's", 3, 3
    assert_joined "foo's", 3, 3, [foo], foo
    
    mock.leave
    assert_left mock
    assert_message "mock left the game", bob
    assert_player_left mock, bob
    
    assert_new_game "foo's", mock
  end

  def assert_chat player, message, players
    assert_receive command("chat", "player"=>player.id, "message"=>message), players
  end

  def test_chat
    bob, mock = players = create_game(3, 2, "bob", "mock")
    bob.chat "hello"
    assert_chat bob, "hello", players
  end
  
  def test_chat_not_in_game
  end
  
  def test_noone_left_in_game
    bob, mock = players = create_game(3, 2, "bob", "mock")
    bob.leave
    assert_left bob
    mock.leave
    assert_message "bob left the game", mock
    assert_player_left bob, mock
    assert_left mock
    sleep 1
    assert_equal [], @server.games
  end
  
  def test_player_left_game
    bob, mock = players = create_game(3, 2, "bob", "mock")
    bob.disconnect
    foo = new_client 'foo'
    assert_new_game "a game", foo
    foo.join "a game"
    assert_joined "a game", 3, 2, [foo, mock], foo
  end
  
  def ___disabled___test_send_invalid_command
    bob = new_client "bob"
    mock = new_client "mock"
    bob.socket.write "<foo>"
    mock.join "game"
    assert_error "Invalid game: \"game\"", mock
    assert_error "Invalid command: foo", bob
  end
end
