; Hugi compo #14's birth was quite problematic... the first proposed
; task was an interpreter for a LOGO-like language, called HOGO. TAD
; conceived it before lightning struck his computer, I only added full
; 360-degrees capabilities. You will find the rules at the end of this file.
;
; Here is my version.  It's probably possible to make it much smaller,
; maybe below the 256-byte magic mark. Feel free to lift the sine/cosine
; code, which I am really proud of... :-)
;
;--------------------------------------------------------------------------
;
; HOGO interpreter
; coded by Bonz
;
; 435 TAD's example
; 508 extension of TAD's code to handle 360-degrees
;
; 298 I'm satisfied of this attempt
;     basic changes: removed gratuitous subs and error checking,
;     rewritten ScanNumber, store constants as self-modifying code,
;     table-driven dispatcher, and of course some tactical stuff
;
; 296 used zero as the end-of-command-table guard
; 294 moved subret inside the command table
;
; Assemble with:
; nasm -fbin -ohogo.com hogo.asm
;

; Compute parity of a byte and check if two bytes have the same parity

%define parity4(xx)	    (( ((xx)/8) ^ ((xx)/4) ^ ((xx)/2) ^ ((xx)/1) ) & 1)
%define parity(xx)	    (parity4((xx)/16) ^ parity4(xx))
%define same_parity(xx, yy) (1 ^ parity((xx)) ^ parity(yy))

	org 256
;	mov	si, 100h
;	mov	bp, 900h

start:
	mov	bl,[si-80h]		; BX = length of command line
	mov	[bx+si-7Fh],al		; put a zero after the file name
					; these two lines by ElpDragon

	mov	ah, 3Dh
	mov	dx, 82h			; DX = 82h
	mov	ch, dl			; prepare for length when reading
	int	21h
	jc	quit
	xchg	bx, ax			; handle in BX, zero in AX

	mov	al, 13h
	int	10h			; 320x200x256c

	mov	dx, bp
	mov	ah, 3Fh			; read the data
	int	21h

	mov	si, bp
	call	execute
	int	16h			; wait/read key (AH = 0)
	mov	ax, 3
	int	10h			; 80x25 text mode
quit:
	ret

;----------------------------------------------------------------------------

xy:	mov	bl, ch
	add	word [bx], byte 32	; add 0.5
	sar	word [bx], 6		; round
	call	get
	shl	word [bx], 6		; back to fixed point
	ret

;----------------------------------------------------------------------------

subr:					; on entry CH = 0ffh
	dec	si			; undo lodsb or cmpsb
	mov	al, '['			; look for a label
	repne	scasb
	cmpsb				; then check if it is the right one
	jnz	subr
	pop	ax			; pop our return address
	push	si			; push current IP
	mov	si, di			; and set new IP
	call	execute			; and execute the sub
	pop	si			; when done, get back old IP

execute:
	lodsb
decode:
	mov	di, table
again:
	mov	cx, [di]		; CH = data, CL = address
	jcxz	execute
	scasw				; skip address & data byte
	scasb
	jnz	again

	mov	bl, cl			; BL = address
	mov	bh, 1			; invariant across the program
	mov	di, bp			; DI = pointer to program
	cbw				; AH = 0
	lodsb				; load a byte
	call	bx
	jmp	short decode

;----------------------------------------------------------------------------

draw:
	call	get
	pusha

	mov	ax, 0			; Initial angle
ang	equ	$-2

	mov	si, ax			; Save the angle
	call	sine			; Compute sin[angle]
	xchg	si, ax			; Store it in SI, angle in AX
	call	cosine			; Compute the cosine
	xchg	dx, ax			; And store it in DX

; The parities of Xpos and Ypos must differ; if not, by inserting 
; a nop either before or after the move, we satisfy this constraint
; (we gain four bytes if we do this, so it is always convenient).

nop_after	 equ same_parity($-$$+1, $-$$+4) & (1^same_parity($-$$+1, $-$$+5))
nop_before equ same_parity($-$$+1, $-$$+4) & (1^nop_after)

	times	nop_before nop
	mov	di, 0			; Initial X
Xpos	equ	$-2

	times	nop_after nop
	mov	bp, 0			; Initial Y
Ypos	equ	$-2
	mov	ax, 7			; Initial color
color	equ	$-2
	mov	cx, 0			; Initial distance
dist	equ	$-2
	jcxz	no_draw

	push	word 0a000h		; ES --> video ram
	pop	es

draw_loop:
	pusha
	add	di, byte 32		; round
	add	bp, byte 32
	sar	di, 6			; divide by scale factor
	sar	bp, 6

	mov	si, 320
	cmp	di, si
	jae	short pp_clip		; Xpos < 0 || Xpos > 319 ?
	cmp	bp, 199
	ja	short pp_clip		; Ypos < 0 || Ypos > 199 ?
	imul	bp, si
	mov	[es:bp+di], al		; video-mem[Xpos+Ypos*320] = color
pp_clip:
	popa

	add	di, si			; Xpos += sin[angle]
	sub	bp, dx			; Ypos -= cos[angle]
	loop	draw_loop

no_draw:
	push	ds
	pop	es
	mov	[Xpos], di
	mov	[Ypos], bp
	popa
	ret

;----------------------------------------------------------------------------
angle:
	mov	bl, Xpos - start
round:
	add	word [bx], byte 32		; add 0.5
	and	[bx], byte -64			; and round

	xor	bl, (Ypos - start) ^ (Xpos - start)
%if parity(Xpos - start)
	jpe	round
%else
	jpo	round
%endif

;----------------------------------------------------------------------------

get:
	cwd				; initialize accumulator to 0
	mov	bl, opcode - start	; where to store opcode
	mov	cl, 89h			; default opcode is MOV
	cmp	al, '-'			; if -, opcode must be SUB (29h)
	je	subtract
	cmp	al, '+'			; if -, opcode must be SUB (29h)
	jne	first
	mov	cl, 1			; this is an ADD
	db	0a9h			; test ax, ... to skip next instr
subtract:
	mov	cl, 29h
	mov	al, 0

decimal:
	imul	dx, byte 10
	add	dx, ax			; add last digit
	lodsb				; and read next one
first:
	mov	[bx], cl		; here to avoid prefetching bugs
	sub	al, '0'
	cmp	al, 9			; If not a digit, we're done
	jbe	decimal			; else loop back

done:
	add	al, '0'			; compensate for previous SUB

	mov	bl, ch			; set offset where we do the operation
opcode:
	mov	[bx], dx		; do the operation (self-modifying)
	ret

;----------------------------------------------------------------------------

; subret-start executes a POP BX        HLT          RET
;                           '['    F4h=subret-start  0c3h
;

; Command table in format <lsb of displacement> <value in CH> <char>
table:	db	draw - start, dist - start, 'D'
	db	angle - start, ang - start, 'A'
	db	subret - start, 0c3h, '['
subret	equ	$ - 1
	db	subret - start, 0c3h, ']'
	db	subr - start, 0ffh, '#'
	db	xy - start, Xpos - start, 'X'
	db	xy - start, Ypos - start, 'Y'
	db	get - start, color - start, 'C'
	dw	0




;----------------------------------------------------------------------------
; Here is the transcript of a Mathematica session which gives a pretty good
; approximation to the sine function.
; 
; Here I load a package to do approximations.  I approximate Sin[x]/x
; because MinimaxApproximation does not like having zeros inside the
; approximation range.  That function gives extremely good approximations
; (7 exacts digits) so we have some liberty to twiddle the coefficients
; to suit our purposes.
; 
;    <<NumericalMath`Approximations`
;    Off[MiniMaxApproximation::extalt];
;    mm = MiniMaxApproximation[Sin[x]/x, {x, {-3.1, 3.1}, 2, 2}];
;    f = Chop[x mm[[2]] ]
; 
; 
;             x (1 - 0.1028614926641981 x^2)
;    Out[1] = ------------------------------
;                  1 + 0.0719131 x^2
; 
; 
; Here we substitute our range for the domain and codomain.  Mathematica
; has the /. operator which means substitute.
; 
;    g = 64 f /. {x -> N[Pi] x / 128}
; 
; 
;             1.5708 x (1 - 0.000061963 x^2)
;    Out[2] = ------------------------------
;                  1 + 0.0000433199 x^2
; 
; 
; Now we scale coefficients.  We will make 1.5708 become 32768.  Here,
; the second statement avoids that Mathematica simplifies everything.
; 
;    factor = 65536 / Pi;
;    Expand[factor Numerator[g]] / Expand[factor Denominator[g]]
; 
; 
;             32768 x - 2.0403 x^3
;    Out[3] = --------------------
;              20860 + 0.9037 x^2
; 
; However this is not good enough for fixed-point math... it's better
; to have something like x (a - 2x^2) / (b+x^2) so that we can do fewer
; multiplies and divides.  But if we change the coefficients for x^2 and
; x^3, we also have to adjust the other two coefficients.  With a little
; trial and error we get...
;
;               x (32682 - 2 x^2)
;    Out[197] = -----------------
;                 (20582 + x^2)
;
; ... er ... yes, Out[197] ... it was actually a lot of trial and error.
;
; To use this formula, *always* truncate the result except when |x|=64.
; In fact, when |x|=64 you can consider sin x = x.
;----------------------------------------------------------------------------

cosine:
	add	al, 40h			; Delta for cosine

sine:	cbw			; Sign extend
	cmp	al, 64		; special case these two
	je	sine_end	; (they give 63 instead of 64)
	cmp	al, 192
	je	sine_end

	mov	cx, ax		; CX = x^2
	imul	cx, cx
	mov	bx, 16341	; BX = 16341 - x^2
	sub	bx, cx
	shr	cx, 1
	add	cx, 10291	; CX = 10291 + x^2/2
	imul	bx		; DX:AX = numerator
	idiv	cx		; divide -- sine in AX
sine_end:
	ret

;----------------------------------------------------------------------------


;--------.-----.------------.-----..---------.
; offsets|     | offsets    |     || offsets |
; without|     | with a NOP |	    || that are|
; NOPs   |fine?| after Xpos |fine?|| picked  |
;--------|-----|------------|-----||---------|  This table explains the
; 00 03  |  NO | 00 04      | yes || 00 04   |  `NOP' trick used to
; 01 04  |  NO | 01 05      | yes || 01 05   |  give different parity
; 02 05  | yes | 02 06      | yes || 02 05   |  values to Xpos and Ypos
; 03 06  |  NO | 03 07      | yes || 03 07   |
; 04 07  |  NO | 04 08      |  NO || 05 08   |
; 05 08  | yes | 05 09      |  NO || 05 08   |
; 06 09  |  NO | 06 0a      |  NO || 07 0a   |
; 07 0a  | yes | 07 0b      |  NO || 07 0a   |
; 08 0b  |  NO | 08 0c      | yes || 08 0c   |
; 09 0c  |  NO | 09 0d      | yes || 09 0d   |
; 0a 0d  | yes | 0a 0e      | yes || 0a 0d   |
; 0b 0e  |  NO | 0b 0f      | yes || 0b 0f   |
; 0c 0f  |  NO | 0c 10      | yes || 0c 10   |
; 0d 10  |  NO | 0d 11      | yes || 0d 11   |
; 0e 11  | yes | 0e 12      | yes || 0e 11   |
; 0f 12  |  NO | 0f 13      | yes || 0f 13   |
;--------|-----|------------|-----||---------|
; 1c 1f  |  NO | 1c 20      |  NO || 1d 20   |
; 1d 20  | yes | 1d 21      |  NO || 1d 20   |
; 1e 21  |  NO | 1e 22      |  NO || 1f 22   |
; 1f 22  | yes | 1f 23      |  NO || 1f 22   |
;--------'-----'------------'-----''---------'


;               Simple Size-Coding Compo Idea by TAD & Bonz
;               ===========================================
; 
;                 ***************************************
;                 ****** NOTE: rough example draft ******
;                 ***************************************
; 
; _____________________________________________________________________________
; 
; HOGO:
;         "HOGO" is basically a very simple LOGO-like language with
;         just a few  commands in the  form of single-letter  ASCII
;         characters followed by decimal numbers.
; 
;         The commands are used to control an invisible HOGO screen
;         cursor (xpos,ypos) and  to draw a  number of pixels  of a
;         certain color, at a certain angle.
; 
; _____________________________________________________________________________
; Angles:
;         Angles are represented  with 256  `degrees' in a  circle. 
;         Here are the (dX,dY) increments for the HOGO  turtle  for
;         the main 8 directions.
; 
;                 Angle   dX      dY      direction
;                 -----   --      --      ---------
;                  0       0      -1      up
;                 32      +1      -1      up, right
;                 64      +1       0      right
;                 96      +1      +1      down, right
;                128       0      +1      down
;                160      -1      +1      down, left
;                192      -1       0      left
;                224      -1      -1      up, left
; 
;         After  each  pixel is  drawn  using the  'D'  command the
;         (xpos,ypos)  co-ordinates are  updated using  one of  the
;         angle increments above.
; 
; _____________________________________________________________________________
; Meaning:
;         +               - relative number prefix (an increment)
;         -               - relative number prefix (a decrement)
;         <num>           - decimal number '0' to '32767' or ''
;                           '' means the same as '0' = 0
;         <c>             - single character sub-program label
; 
; Commands:
;         D<num>          - draw pixels           (dist = <n>)
;         D+<num>         - draw <n> more pixels  (dist = dist + <n>)
;         D-<num>         - draw <n> less pixels  (dist = dist - <n>)
; 
;         C<num>          - change color          (color = <n>)
;         C+<num>         - increment color       (color = color + <n>)
;         C-<num>         - decrement color       (color = color - <n>)
; 
;         A<num>          - set angle             (angle = <n>)
;         A+<num>         - increment angle       (angle = angle + <n>)
;         A-<num>         - decrement angle       (angle = angle - <n>)
; 
;         X<num>          - set xpos              (xpos = <n>)
;         X+<num>         - increment xpos        (xpos = xpos + <n>)
;         X-<num>         - decrement xpos        (xpos = xpos - <n>)
; 
;         Y<num>          - set ypos              (ypos = <n>)
;         Y+<num>         - increment ypos        (ypos = ypos + <n> )
;         Y-<num>         - decrement ypos        (ypos = ypos - <n> )
; 
;         #<c>            - execute a sub-program <c>
;         [               - return() (exit from program/sub-program)
;         ]               - return() (exit from program/sub-program)
; 
;         ??              - all other characters are ignored
;                           (including NUL, EOF, TAB, LF, CR etc..)
; 
; Sub-programs:
;         [<c>            - define a sub-program label <c>
;         ]               - end of a sub-program, return()
; 
;         A sub-program is a HOGO sub-routine which can be called
;         using the '#<c>' command. The <c> is a single-character
;         label such as 'f' or '~' or '&' and is used to identify
;         each sub-program.
; 
;         When a '#' is found by the interpreter, it scans the entire
;         HOGO program looking for the matching <c> label after a '['
;         character. This allows both forward-referencing and
;         backward-referencing of labels.
; 
;         The end of a sub-program is identified by the ']' character.
;         This is treated like a 'return()' or 'RET' and so will
;         return back to the calling program/sub-program.
; 
; _____________________________________________________________________________
; HOGO programs:
; --------------
;         A  HOGO program  is a  file called  '<filename>.HOG' and
;         contains all the commands needed to draw a HOGO image  on
;         the screen.
; 
; "EG1.HOG"
;         1>       X160Y100C4A0D10[
; 
;         The  above  .HOG  program  will  move  to  (160,100)  set
;         color(4) angle(0) and draw(10) pixels.
; 
; 
; "EG2.HOG"
;         1>       X160 Y100 C4 A0 D10 [
; 
;         The above .HOG program will do exactly the same thing  as
;         "EG1.HOG", but this time extra space characters have been
;         used to make it more readable.
; 
; "SUB1.HOG"
;         1>       X160 Y100 #b #i #b #i #b
;         2>       [b A64 D10 A128 D10 A192 D10 A0 D10]
;         3>       [i X+20]
; 
;         The  above  .HOG program  will  draw 3  boxes  using some
;         simple  [...]  sub-program  calls.  The  1st  sub-program
;         called 'b' will draw a 10x10 pixel unfilled box. The  2nd
;         sub-program called 'i' will  increase xpos by 20,  moving
;         the cursor right by 20 pixels.
; 
; NOTE:   The '[' on the 2nd line will cause the HOGO interpreter
;         to return from the main program, and therefore, quit.
; 
; "null.HOG"
;         1>       [
; 
; "null2.HOG"
;         1>       ]
; 
;         The '[' and ']' characters  are used to denote the  start
;         and end  of a  HOGO sub-program  (sub-routine). They also
;         provide a  common way  to return/quit  from a program/sub
;         -program.
; 
;         ALL unused ASCII characters should be ignored as they are
;         not valid HOGO commands.  These characters (including the
;         lower-case 'a' 'd' 'c' 'm') can safely be used to  insert
;         comments and  line-breaks in  a program  to make  it more
;         readable.
; 
; _____________________________________________________________________________
; HOGO interpreter:
; -----------------
;         The interpretation  of the  HOGO language  is very simple
;         and has lots  of repeated code  fragments which means  it
;         should be easy to optimize #;o)
; 
;         Any  pixels  whose co-ordinates  are  outside the  screen
;         boundaries (x = 0...319 and  y = 0...199) are NOT  drawn,
;         but the (xpos,  ypos) co-ordinates  are still  updated by
;         the  appropriate   xstep,    ystep   increment    values.
;         (see PlotPixel() for an example)
; 
; -----------------------------------------------------------------------------
; Pseudo-code:
;         <set mode 13 hex>
;         <load HOGO program>
; 
;         xpos = 0
;         ypos = 0
;         color = 255
;         angle = 0
;         dist = 0
;         pc = 0
;         Execute()
; 
;         <wait for a key using AH=0, INT 16h>
; 
;         <set mode 03 hex>
;         <quit cleanly to DOS>
; 
; -----------------------------------------------------------------------------
; xstep[8] =  0, +1, +1, +1,  0, -1, -1, -1
; ystep[8] = -1, -1,  0, +1, +1, +1,  0, -1
; 
; -----------------------------------------------------------------------------
; Execute:
;         b = program[pc++]
; 
;         if b == 'D'
;                 v = dist
;                 dist = ScanNumber()
;                 count = dist
;                 while count>0
;                         PlotPixel()
;                         xpos += sin[angle * pi / 128]
;                         ypos -= cos[angle * pi / 128]
;                         count--
;                 goto Execute
; 
;         if b == 'C'
;                 v = color
;                 color = ScanNumber()
;                 goto Execute
; 
;         if b == 'A'
;                 v = angle
;                 angle = ScanNumber()
;                 goto Execute
; 
;         if b == 'X'
;                 v = xpos
;                 xpos = ScanNumber()
;                 goto Execute
; 
;         if b == 'Y'
;                 v = ypos
;                 ypos = ScanNumber()
;                 goto Execute
; 
; 
;         if b == '#'
;                 c = program[pc++]
;                 push pc
;                 pc = 0
;                 while program[pc] <> '[' && program[pc+1] <> c
;                         pc++
;                 Execute()
;                 pop pc
;                 goto Execute
; 
;         if b == '['
;                 return()
; 
;         if b == ']'
;                 return()
; 
;         goto Execute
; 
; -----------------------------------------------------------------------------
; PlotPixel:
;         if xpos < 0 or xpos > 319 then return()
;         if ypos < 0 or ypos > 199 then return()
; 
;         videomem[xpos+(ypos*320)] = color
;         return()
; 
; -----------------------------------------------------------------------------
; ScanNumber:
;         if program[pc] == '-'
;                         pc++
;                         n = ScanDecimal()
;                         v = v - n
;                         return(n)
; 
;         if program[pc] == '+'
;                         pc++
;                         n = ScanDecimal()
;                         v = v + n
;                         return(v)
; 
;         n = ScanDecimal()
;         v = n
;         return (v)
; 
; -----------------------------------------------------------------------------
; ScanDecimal:
;         n = 0
;         c = 0
; Digit:
;         n = n*10 + c
;         c = program[pc++] - '0'
;         if c <= 9
;                 goto Digit
;         pc--
;         return(n)
; -----------------------------------------------------------------------------
;  
; 
;         Well, thats enough for a quick rough draft + example.
;  
;  
;  regards
;         TAD & Bonz
;  
;        
