program test_tictactoe;
{$X+,R-,S-,G+}
{$M 4096,0,0}

uses crt, dos;

const
  { Table to check for black blank characters.  The mask represents
    the bits that must be 0 in the attribute to obtain a non-blinking
    black space: $70 (background) for 0/32/255, $8F (blinking+foreground)
    for 219, $7F (background+foreground) for everything else. }

  blank_tab: array [#0..#255] of byte = (
    $70,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F, {   0 <}
    $7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F, {  16  }
    $70,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F, {  32 <}
    $7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F, {  48  }
    $7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F, {  64  }
    $7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F, {  80  }
    $7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F, {  96  }
    $7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F, { 112  }
    $7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F, { 128  }
    $7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F, { 144  }
    $7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F, { 160  }
    $7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F, { 176  }
    $7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F, { 192  }
    $7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$8F,$7F,$7F,$7F,$7F, { 208 <}
    $7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F, { 224  }
    $7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$7F,$70  { 240 <}
  );

type
  cellstate = (bad, dont_care, blank, naught, cross, free,
               hline, vline, hvline);

  video = record ch: char; attr: byte end;

  gamestate = (Naughts, Crosses, None, Draw);

var
  screen: array[0..40*25-1] of cellstate;
  vram: array[0..40*25-1] of video absolute $B800:0;
  save_mode: byte;

  OldInt20, OldInt16: pointer;
  key: word;

  games_draw, games_tested, games_played: word;
  b_naughts, b_crosses: word;
  stack: array[0..10] of word;
  played: array[0..10] of word;
  sp: integer;
  current: integer;
  errors: set of (mode_1, mode_3, esc_pressed, screen_mismatch, dos_error,
                  match_lost, bad_move);

procedure set_screen_cell(x,y: integer; state: cellstate);
begin
  screen[y*40+x] := state;
end;

procedure reset_screen;
var
  x, y: word;
begin
  b_naughts := 0;
  b_crosses := 0;

  for x := 0 to 40*25-1 do
    screen[x] := blank;

  for x := 0 to 2 do
    for y := 0 to 2 do
      set_screen_cell(16+x*4, 10+y*2, free);

  for x := 0 to 1 do begin
    set_screen_cell(18+4*x, 10, vline);
    set_screen_cell(18+4*x, 11, hvline);
    set_screen_cell(18+4*x, 12, vline);
    set_screen_cell(18+4*x, 13, hvline);
    set_screen_cell(18+4*x, 14, vline);
    set_screen_cell(15, 11+2*x, hline);
    set_screen_cell(16, 11+2*x, hline);
    set_screen_cell(17, 11+2*x, hline);
    set_screen_cell(19, 11+2*x, hline);
    set_screen_cell(20, 11+2*x, hline);
    set_screen_cell(21, 11+2*x, hline);
    set_screen_cell(23, 11+2*x, hline);
    set_screen_cell(24, 11+2*x, hline);
    set_screen_cell(25, 11+2*x, hline);
  end;
end;

function position(i: word): word;
begin
  case i of
    40*10+16: position := $100;
    40*10+20: position := $80;
    40*10+24: position := $40;
    40*12+16: position := $20;
    40*12+20: position := $10;
    40*12+24: position := $8;
    40*14+16: position := $4;
    40*14+20: position := $2;
    40*14+24: position := $1;
  end;
end;

function is_mode(expected: byte): boolean; assembler;
asm
  PUSH WORD PTR 0
  POP ES
  MOV AL, ES:[449h]
  MOV BL, AL

  XOR AL, [expected]
  CMP AL, 2
  MOV AL, 1             { Set result }
  JB  @ok

  DEC AX                { Fix result if wrong }
  MOV [save_mode], BL   { And save the wrong mode }
@ok:
end;

function won: gamestate;
const
  wins: array[1..8] of word = ($1C0, $38, $7, $124, $92, $49, $111, $54);
var
  i: word;
begin
  if (b_naughts or b_crosses) = $1ff then
    won := Draw
  else
    won := None;

  for i := 1 to 8 do
    if (b_naughts and wins[i]) = wins[i] then
      won := Naughts
    else if (b_crosses and wins[i]) = wins[i] then
      won := Crosses;

end;

{ Leave blanks as they are, but check that the characters are right
  and turn them into dont_care or bad values (resp. if correct or
  incorrect). }
procedure check_msg (s: string);
var
  x: word;
begin
  inc (games_tested);
  x := word(vram[40*20+17].ch);
  for x := 17 to 23 do
    if (s[x-16] <> ' ') then
      if (vram[40*20+x].ch = s[x-16]) then
        screen[40*20+x] := dont_care
      else
        screen[40*20+x] := bad;
end;

function check_screen(must_fill_one: boolean): boolean;
var
  found, i: word;
  ok: boolean;
begin
  check_screen := false;
  found := 0;

  for i := 0 to 40*25-1 do
    with vram[i] do begin
      case screen[i] of
        bad: ok := false;
        dont_care: ok := attr = 7;
        naught: ok := (ch = 'O') and (attr = 7);
        cross: ok := (ch = 'X') and (attr = 7);
        hline: ok := (ch = '') and (attr = 7);
        vline: ok := (ch = '') and (attr = 7);
        hvline: ok := (ch = '') and (attr = 7);
        hvline: ok := (ch = '') and (attr = 7);
        blank: ok := (attr and blank_tab[ch]) = 0;
        free:
          if (attr and blank_tab[ch]) > 0 then begin
            ok := (ch = 'O') and (attr = 7);
            screen[i] := naught;
            b_naughts := b_naughts or position(i);
            played[current*2-1] := position(i);
            inc (found);
            case won of
              Draw: begin
                inc (games_draw);
                check_msg ('A draw!');
                key := $11B;
              end;
              Naughts: begin
                check_msg ('O wins!');
                key := $11B;
              end
            end
          end else
            ok := true;
      end;
      if not ok then begin
        errors := errors + [ screen_mismatch ];
        exit;
      end;
    end;

  if (found > 1) or (must_fill_one <> (found = 1)) then
    errors := errors + [bad_move]
  else
    check_screen := true;

end;

procedure find_next_move;
var
  filled, chosen: word;
begin
  if (key = $11B) then begin
    errors := errors + [ esc_pressed ];
    exit;
  end;

  if not is_mode (1) then begin
    errors := errors + [ mode_1 ];
    key := $11B;
    exit;
  end;

  case won of
    Draw: begin
      inc (games_draw);
      check_msg ('A draw!');
      check_screen (false);
      key := $11B;
      exit;
    end;
    Crosses: begin
      errors := errors + [ match_lost ];
      key := $11B;
      exit;
    end;
  end;

  key := 0;
  if not check_screen (current > 0) then
    key := $11B;

  if key > 0 then exit;

  chosen := stack[current];
  filled := b_naughts or b_crosses;
  while ((filled and chosen) > 0) and (sp >= 0) do begin
    chosen := chosen shr 1;

    if chosen = 0 then begin
      { Backtrack }
      key := $11B;
      exit;
    end;
  end;

  { Prepare the next cell in the stack }
  if current = sp then begin
    stack[current] := chosen;
    stack[sp+1] := $100;
    inc(sp);
  end;

  played[current*2] := chosen;
  inc (current);
  case chosen of
    $200:
      key := $4030;
    $100: begin
      key := $4737;
      set_screen_cell(16,10, cross);
    end;
    $80: begin
      key := $4838;
      set_screen_cell(20,10, cross);
    end;
    $40: begin
      key := $4939;
      set_screen_cell(24,10, cross);
    end;
    $20: begin
      key := $4434;
      set_screen_cell(16,12, cross);
    end;
    $10: begin
      key := $4535;
      set_screen_cell(20,12, cross);
    end;
    $8: begin
      key := $4636;
      set_screen_cell(24,12, cross);
    end;
    $4: begin
      key := $4131;
      set_screen_cell(16,14, cross);
    end;
    $2: begin
      key := $4232;
      set_screen_cell(20,14, cross);
    end;
    $1: begin
      key := $4333;
      set_screen_cell(24,14, cross);
    end;
  end;

  b_crosses := b_crosses or (chosen and $1FF);
end;

procedure NewInt20; assembler;
asm
  POP AX                        { Pop IP }
  POP AX                        { Pop CS }
  POPF                          { Pop the flags }
  MOV AX, 4C00H                 { Exit through a more polite function! }
  INT 21H
end;

procedure NewInt16; assembler;
asm
  STI
  PUSH DS
  PUSH ES

  PUSH SEG @Data
  POP  DS
  PUSH 0
  POP  ES

  CMP  [key], 0                 { Do move if there is a key to stuff }
  JNE  @key_stuffed

  PUSH WORD PTR DS:[OldInt16]   { Save state }
  POP  WORD PTR ES:[16h*4]
  PUSH WORD PTR DS:[OldInt16+2]
  POP  WORD PTR ES:[16h*4+2]
  PUSHA
  CALL find_next_move           { Run hook }
  POPA

  PUSH 0
  POP  ES
  PUSH OFFSET NewInt16          { Restore state }
  POP  WORD PTR ES:[16h*4]
  PUSH CS
  POP  WORD PTR ES:[16h*4+2]

@key_stuffed:
  CMP  [key], 0                 { Pass all functions if no key to stuff }
  JE   @old

  CMP  AH, 0                    { Pass all functions but 0/1 to old handler }
  JNE  @not_zero
  MOV  AX, [key]                { Return key of choice }
  MOV  [key], 0                 { Clear pseudo-keyboard buffer }
@done:
  POP  ES
  POP  DS                       { Return }
  IRET

@not_zero:
  CMP  AH, 1                    { Pass all functions but 0/1 to old handler }
  JNE  @old
  OR   AX, AX                   { Reset ZF }
  POP  ES
  POP  DS
  RETF 2

@old:
  POP ES
  POP DS
  JMP FAR PTR [OldInt16]
end;

procedure show_error;
var
  i: integer;
  s: string[9];
  peg: char;
begin
  if current = 0 then
    exit;

  writeln ('Error in move sequence: ');
  s := '---------';
  for i := 0 to current*2-1 do begin
    if odd(i) then peg := 'O' else peg := 'X';
    case played[i] of
      $100: s[1] := peg;
      $80: s[2] := peg;
      $40: s[3] := peg;
      $20: s[4] := peg;
      $10: s[5] := peg;
      $8: s[6] := peg;
      $4: s[7] := peg;
      $2: s[8] := peg;
      $1: s[9] := peg;
    end;
    writeln (s);
  end;
end;

var
  prog: string;
begin
  if ParamCount < 1 then begin
    writeln ('Usage: ttt_test PATH_TO_ENTRY.COM');
    halt (1);
  end;
  prog := FExpand(ParamStr(1));

  errors := [];
  stack[0] := $200;
  sp := 0;
  games_played := 0;
  games_tested := 0;
  games_draw := 0;
  while sp >= 0 do begin
    current := 0;
    key := 0;
    save_mode := 0;
    GetIntVec ($16, OldInt16);
    SetIntVec ($16, @NewInt16);
    GetIntVec ($20, OldInt20);
    SetIntVec ($20, @NewInt20);
    reset_screen;
    SwapVectors;
    Exec(Prog, '');
    SwapVectors;
    if dosError > 0 then
      errors := errors + [ dos_error ];
    SetIntVec ($16, OldInt16);
    SetIntVec ($20, OldInt20);

    inc (games_played);

    { Backtrack the losing moves }
    sp := current;
    repeat
      dec (sp);
      if (sp >= 0) then stack[sp] := stack[sp] shr 1;
    until (sp < 0) or (stack[sp] > 0);

    if not is_mode(3) then begin
      errors := errors + [ mode_3 ];
      asm MOV AX,3; INT 10H; end;
    end;

    if errors <> [] then begin
      writeln ('Failed. (', games_played, ' executions, ',
                games_tested, ' games played)');
      if dos_error in errors then writeln ('DOS error ', doserror, '.');
      if mode_1 in errors then writeln ('Mode ', save_mode, ' set, 0/1 expected.');
      if screen_mismatch in errors then writeln ('Screen mismatch.');
      if bad_move in errors then writeln ('Computer made an incorrect move.');
      if match_lost in errors then writeln ('Computer lost a match.');
      if esc_pressed in errors then writeln ('Did not exit upon ESC.');
      if mode_3 in errors then writeln ('Mode ', save_mode, ' set, 2/3 expected.');
      show_error;
      halt (1);
    end;
  end;

  clrscr;
  writeln ('Passed! (', games_played, ' executions, ',
           games_tested, ' games played, ',
           games_draw, ' draws)');

end.
