#include<stdio.h>
#include<malloc.h>

/* Constants to tune the compression. */
#define MAXREPEAT               63
#define MAXLOOKBACK		1024

/* Useful macro */
#define	min(a, b)		((a<b) ? (a) : (b))

/* The number of zeroes to place before the file.  Saves a few bytes by not initializing DI. */
#define         OFFSET          2


/*
 * Saves the buffer to a file.
 */
void SaveFile(char *FileName, unsigned char *Buffer, int Size)
{
	FILE		*fp;

	fp = fopen(FileName, "wb");
	if(fp == NULL) return;
	fwrite(Buffer, 1, Size, fp);
	fclose(fp);
}


/*
 * Loads a file.
 */
int LoadFile(char *FileName, unsigned char *Buffer, int Size)
{
	FILE		*fp;

	fp = fopen(FileName, "rb");
	if(fp == NULL) return 0;
	memset(Buffer, 0, Size);
	Size = fread(&Buffer[OFFSET], 1, Size - OFFSET, fp);
	fclose(fp);
	return Size;
}


/*
 * Search for repeats.
 */
int DictSearch(unsigned char *Buffer, int Start, int Length, int *LookBack, int MaxLookBack)
{
	int		MaxMatch = 1;
	int		i, j;

	if(Buffer[Start] == 0) {
		for(j=0; j<Length; j++) {
			if(Buffer[j + Start] != 0) {
				break;
			}
		}
		if(j > MaxMatch) {
			MaxMatch = j;
			if(LookBack != NULL) {
				*LookBack = 0;
			}
		}
	}

	for(i=1; i<=min(Start, MaxLookBack); i++) {
		for(j=0; j<Length; j++) {
			if(Buffer[j+Start] != Buffer[j+Start-i]) {
				break;
			}
		}
		if(j > MaxMatch) {
			MaxMatch = j;
			if(LookBack != NULL) {
				*LookBack = i;
			}
		}
	}

	return MaxMatch;
}


/*
 * Compress the buffer.
 */
int CompressBuffer(unsigned char *InBuffer, unsigned char *OutBuffer, int Size)
{
	int		i, j;
	int		OutputLength;
	int		ControlOffset;
	int		n;
	int		Len;
	int		NewLen;
	int		LookBack;

	i = j = 0;
	OutputLength = 1;
	ControlOffset = 0;
	OutBuffer[0] = 0;
	while(i < Size) {
		n = min(Size - i, MAXREPEAT);

		Len = DictSearch(InBuffer, i, n, &LookBack, MAXLOOKBACK);
		if(Len >= 2) {
			n = min(Size - i - 1, MAXREPEAT);
			NewLen = DictSearch(InBuffer, i+1, n, NULL, MAXLOOKBACK);
			if(NewLen > Len) {
				Len = 1;
			}
		}

		if(Len == 1) {
			OutBuffer[OutputLength] = InBuffer[i];
			OutputLength++;
			OutBuffer[ControlOffset] |= 1<<j;
		} else {
			OutBuffer[OutputLength+1] = LookBack >> 2;
			OutBuffer[OutputLength] = (LookBack << 6) + Len;
			OutputLength += 2;
		}

		j++;
		if(j==8) {
			ControlOffset = OutputLength;
			OutBuffer[OutputLength] = 0;
			OutputLength++;
			j = 0;
		}

		i += Len;
	}

	OutBuffer[OutputLength++] = 0;
	OutBuffer[OutputLength++] = 0;

	return OutputLength;
}


/*
 * Decompress the buffer to verify the compression.
 */
int DecompressBuffer(unsigned char *InBuffer, unsigned char *OutBuffer)
{
	unsigned char	Code;
	int				InOffset = 0;
	int				OutOffset = 0;
	int				i, j;
	int				Length = 1;
	int				Dict = 1;

	while(Dict && Length && OutOffset < 0xdc0) {
		Code = InBuffer[InOffset];
		InOffset++;
		for(i=0; i<8; i++) {
			if(Code & 1) {
				OutBuffer[OutOffset] = InBuffer[InOffset];
				OutOffset++;
				InOffset++;
			} else {
				Length = InBuffer[InOffset] & 0x3f;
				Dict = InBuffer[InOffset+1];
				Dict = (Dict << 2) | (InBuffer[InOffset] >> 6);
				if(Length == 0) break;
				for(j=0; j<Length; j++) {
					OutBuffer[OutOffset + j] = OutBuffer[OutOffset + j - Dict];
				}
				InOffset += 2;
				OutOffset += Length;
			}
			Code >>= 1;
		}
	}

	return OutOffset;
}


// Best: 797
int main(int argc, char *argv[])
{
        unsigned char   *InBuffer;
        unsigned char   *OutBuffer;
        unsigned char   *CompBuffer;
	int				InSize;
	int				DecompSize;
	int				CompSize;
	int				i;

        /* Allocate buffers. */
        InBuffer = malloc(5000);
        OutBuffer = malloc(5000);
        CompBuffer = malloc(5000);
	
	/* Load the file. */
        InSize = LoadFile("IMAGE.BIN", InBuffer, 5000);

	/* Change some bytes for better compression. */
	for(i=OFFSET; i<4000+OFFSET; i+=2) {
		if(InBuffer[i] == 0x20) {
			if(InBuffer[i+1] > 0x0f) {
				InBuffer[i+1] >>= 4;
				InBuffer[i] = 0xdb;
			} else {
				InBuffer[i] = InBuffer[i+1] = 0x00;
			}
		}
	}

	/* Compress the file. */
	CompSize = CompressBuffer(InBuffer, CompBuffer, 0xdc0+OFFSET);
	printf("Compressed Size: %d\n", CompSize);

	/* Decompress the compressed file. */
        memset(OutBuffer, 0x00, 5000);
	DecompSize = DecompressBuffer(CompBuffer, OutBuffer);

	/* Check the compression. */
	if(memcmp(InBuffer, OutBuffer, DecompSize) != 0) {
		for(i=0; i<DecompSize; i++)
			if(InBuffer[i] != OutBuffer[i]) break;
		printf("Decompression Error at offset %d!\n", i);
	}

	/* Save the compressed file. */
        SaveFile("IMAGE.CMP", CompBuffer, CompSize);

        /* Free buffers */
        free(InBuffer);
        free(OutBuffer);
        free(CompBuffer);

	return 0;
}
