#include <cstdio>
#include <cstdarg>
#include <boost/shared_ptr.hpp>
#include <vector>
#include <iterator>
#include <algorithm>
#include <sstream>


unsigned log2(unsigned n)
{
    if (n <= 1)
        return 1;

    return 1+log2(n/2);    
}

class code_writer
{
public:

    virtual void comment(char const* s, ...) const = 0;
    virtual void constant(char const* name, char const* suffix, unsigned value) const = 0;
    virtual void begin_byte_array(char const* name, char const* suffix) const = 0;
    virtual void end_byte_array() const = 0;
    virtual void array_element(unsigned char value) const = 0;

    virtual void begin_word_array(char const* name, char const* suffix) const = 0;
    virtual void end_word_array() const = 0;
    virtual void array_element(unsigned short value) const = 0;
};

class asm_writer : public code_writer
{
    boost::shared_ptr<FILE> fout;
    mutable unsigned array_len;
public:
    asm_writer(boost::shared_ptr<FILE> f) 
        : fout(f)
        , array_len(0)
    { }

protected:

    void comment(char const* s, ...) const
    {
        va_list args;
        va_start(args, s);

        std::fprintf(&*fout, "; ");		
        std::vfprintf(&*fout, s, args);
        std::fprintf(&*fout, "\n");
    }

    void constant(char const* name, char const* suffix, unsigned value) const
    {
        std::fprintf(&*fout, "%s%s equ %u\n", name, suffix, value);
    }

    void begin_byte_array(char const* name, char const* suffix) const
    {
        std::fprintf(&*fout, "%s%s:\n", name, suffix);
        array_len = 0;
    }

    void begin_word_array(char const* name, char const* suffix) const
    {
        return begin_byte_array(name, suffix);
    }

    void end_byte_array() const
    {
        std::fprintf(&*fout, "\n\n");
    }

    void end_word_array() const
    {
        return end_byte_array();
    }

    void array_element(unsigned char value) const
    {
        if (array_len == 0) std::fprintf(&*fout, "db\t");
        if (array_len > 0) std::fprintf(&*fout, ",");
        std::fprintf(&*fout, "0x%02x", value);
        if (++array_len % 10 == 0) std::fprintf(&*fout, " \\\n\t");
    }

    void array_element(unsigned short value) const
    {
        if (array_len == 0) std::fprintf(&*fout, "dw\t");
        if (array_len > 0) std::fprintf(&*fout, ",");
        std::fprintf(&*fout, "0x%04x", value);
        if (++array_len % 8 == 0) std::fprintf(&*fout, " \\\n\t");
    }
};

class cxx_writer : public code_writer
{
    boost::shared_ptr<FILE> fout;
    mutable unsigned array_len;
public:
    cxx_writer(boost::shared_ptr<FILE> f) 
        : fout(f)
        , array_len(0)
    { }

protected:

    void comment(const char* s, ...) const
    {
        va_list args;
        va_start(args, s);

        std::fprintf(&*fout, "// ");
        std::vfprintf(&*fout, s, args);
        std::fprintf(&*fout, "\n");
    }

    void constant(char const* name, char const* suffix, unsigned value) const
    {
        std::fprintf(&*fout, "unsigned const %s%s = %u;\n", name, suffix, value);
    }

    void begin_byte_array(char const* name, char const* suffix) const
    {
        std::fprintf(&*fout, "unsigned char const %s%s[] = {\n\t", name, suffix);
        array_len = 0;
    }

    void begin_word_array(char const* name, char const* suffix) const
    {
        std::fprintf(&*fout, "unsigned short const %s%s[] = {\n\t", name, suffix);
        array_len = 0;
    }

    void end_byte_array() const
    {
        std::fprintf(&*fout, "};\n\n");
    }

    void end_word_array() const
    {
        std::fprintf(&*fout, "};\n\n");
    }

    void array_element(unsigned char value) const
    {
        if (array_len > 0) std::fprintf(&*fout, ",");
        std::fprintf(&*fout, "0x%02x", value);
        if (++array_len % 10 == 0) std::fprintf(&*fout, "\n\t");
    }

    void array_element(unsigned short value) const
    {
        if (array_len > 0) std::fprintf(&*fout, ",");
        std::fprintf(&*fout, "0x%04x", value);
        if (++array_len % 8 == 0) std::fprintf(&*fout, "\n\t");
    }
};

void write_bits(std::vector<bool>& out, unsigned bits, unsigned value)
{
    for (unsigned i = 0; i < bits; ++i)
    {
        out.push_back(value & 1);
        value >>= 1;
    }
}

void write_rle(std::vector<bool>& out, unsigned len, unsigned len_bits, unsigned char byte, unsigned byte_bits)
{
    for (unsigned j = 0; j < (len >> len_bits); ++j)
    {
        write_bits(out, byte_bits, byte);
        write_bits(out, len_bits, ~0);
    }

    if ((len & ((1 << len_bits)-1)) != 0)
    {
        write_bits(out, byte_bits, byte);
        write_bits(out, len_bits, len - 1);
    }
}

static void _nop(FILE*) { }

int main(int argc, char** argv)
{
    if (argc != 4)
    {
        std::fprintf(stderr, "Usage:\n\n\trle2 label_prefix len_bits infile > outfile\n\n");
        return 1;
    }

	unsigned const len_bits = atoi(argv[2]);

    // TODO: file names from cmdline arguments.
    boost::shared_ptr<FILE> fin;
    boost::shared_ptr<FILE> fout(stdout, _nop);

    fin.reset(fopen(argv[3], "rb"), fclose);
    if (fin == NULL)
    {
        std::fprintf(stderr, "Failed to open file '%s'.\n", argv[3]);
        return 1;
    }

    boost::shared_ptr<code_writer> code;
    if (strstr(argv[0], "rle_cxx") != 0)
        code.reset(new cxx_writer(fout));

    else
        code.reset(new asm_writer(fout));

    unsigned char buf[2000];
    size_t const bytes = fread(buf, 1, 2000, fin.get());

    unsigned char counter[256] = {0};

    for (unsigned i = 0; i < bytes; ++i)
    {
        counter[buf[i]] = 1;
    }

    unsigned char lookup[256] = {0};

    unsigned count_of_dif_chrs = 0;
    for (unsigned i = 0; i < 256; ++i)
    {
        if (counter[i] != 0)
        {
            lookup[i] = count_of_dif_chrs;
            ++count_of_dif_chrs;
        }
    }

    unsigned const min_char_len = log2(count_of_dif_chrs);
    code->comment("Number of different characters: %u", count_of_dif_chrs);
    code->constant(argv[1], "_chars", count_of_dif_chrs);

    code->comment("Minimum length of charater: %u bits", min_char_len);

    // perform the substitution

    code->comment("Substitution table:");
    unsigned idx = 0;
    for (unsigned i = 0; i < 256; ++i)
    {
        if (counter[i] != 0)
        {
            code->comment("0x%02x : 0x%02x", idx, i);
            ++idx;
        }
    }

    idx = 0;
    code->begin_byte_array(argv[1], "_subst");
    for (unsigned i = 0; i < 256; ++i)
    {
        if (counter[i] != 0)
        {
            code->array_element((unsigned char)i);
            lookup[i] = idx++;
        }
    }
    code->end_byte_array();

    code->comment("end of substitution table\n");

    code->constant(argv[1], "_blk_bits", len_bits+4);

    // perform the compression
    code->comment("");
    code->comment("Compressed data:");
    code->begin_word_array(argv[1], "_data");

    unsigned char ch = buf[0];
    unsigned len = 1;
    unsigned data_len = 0;
    //unsigned const max_count = 1 << (8-min_char_len);
    std::vector<bool> out;
    for (unsigned i = 1; i < bytes; ++i)
    {
        if (buf[i] == ch)
        {
            ++len;
        }
        else
        {
            write_rle(out, len, len_bits, lookup[ch], 4);
            code->comment("ch=0x%02x, lookup[ch]=0x%02x, len=%u", ch, lookup[ch], len);
            ch = buf[i];
            len = 1;
        }
    }

    write_rle(out, len, len_bits, lookup[ch], 4);

    {
        std::ostringstream oss;
        std::copy(out.begin(), out.end(), std::ostream_iterator<unsigned>(oss));
        code->comment(oss.str().c_str());
    }

    unsigned short b = 0;
    unsigned r = 16 - out.size() % 16;
    unsigned const _len = out.size();
    if (r == 16) r = 0;
    for (; r != 0; --r)
    {
        out.push_back(false);
    }

    #if defined(REVERSE)
        std::reverse(out.begin(), out.end());
    #endif

    for (unsigned i = 0; i < out.size()+r; ++i)
    {
    	b >>= 1;
        if (i < out.size()) b |= out[i] ? (1<<15) : 0;

        if (i % 16 == 15)
        {
            code->array_element(b);
            b = 0;
        }
    }

    code->end_word_array();
    code->comment("Number of bits:");
    code->constant(argv[1], "_len", _len);
    code->comment("End of data");

    return 0;
}
