{
  Hugi Size Coding Competition 27 - ANSI Compression

  This is the compressor program.

  ImageData contains the data from image.asm, manually converted to
  Turbo Pascal syntax.

  Data format: this is a really stupid implementation of LZ77, with
  hand-tuned sort-of-huffman encoding. LZ77 with its backreferences
  seems very well suited for lossy compression of this type, because
  it allows "fuzzy" matches (i.e. a Full Block matches a Space if we
  can invert its color).

  A compressed data stream consists of a list of elements of the
  following form (bitstream):

    0.0.nnn              Frequent literal (table lookup)
    0.1.nnnnnnnn         Infrequent literal
    1.aaa.AAAA.bbb.BBBB  Backreference, AAAA is distance, BBBB is length.
                         Here, aaa is the number of "mantissa" bits in
                         AAAA after an implied initial '1'. For example,
                         17 (=10001b) is encoded as 100.0001, 1 (most
                         frequent case) is encoded as single 0 bit.

  We compress the character stream independantly from the color stream.
  This allows better compression and also allows easier feedback from
  lossy character compression.

  By Stefan <Streu@gmx.de>

  04/Oct/2008  First version: lossless compression
  05/Oct/2008  Lossy compression: swap block characters and colors if useful
               Bit-wise storage
               Separate format for frequent/infrequent literals
  25/Oct/2008  Make use of the minimum length of a backreference
               Optimized frequent literal tables
}

{$F+  We'll be using function pointers}

{
  $DB  219  Full Block
  $DC  220  Lower Half
  $DD  221  Left Half
  $DE  222  Right Half
  $DF  223  Upper Half
}
CONST ImageData : ARRAY[1..4000] OF BYTE = (
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $B1,$06,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$DC,$0E,$DC,$0E,$DB,$0E,$DB,$0E,$DB,$0E,$DB,$0E,$DB,$0E,
    $DB,$0E,$DB,$0E,$DB,$0E,$DB,$0E,$DB,$0E,$DC,$0E,$DC,$0E,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$B1,$06,
    $B0,$06,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $DC,$0E,$DB,$0E,$DB,$0E,$DB,$0E,$DB,$0E,$DB,$0E,$DB,$0E,$DB,$0E,
    $DB,$0E,$DB,$0E,$DB,$0E,$DB,$0E,$DB,$0E,$DB,$0E,$DB,$0E,$DC,$0E,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$B1,$06,
    $B0,$06,$B2,$06,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $B2,$06,$B2,$06,$B0,$06,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $DB,$0E,$DB,$0E,$DB,$0E,$DB,$0E,$DB,$0E,$DB,$0E,$DB,$0E,$DB,$0E,
    $DB,$0E,$DB,$0E,$DF,$0E,$DF,$0E,$DB,$0E,$DB,$0E,$DB,$0E,$DB,$0E,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $DC,$07,$DC,$07,$DC,$07,$DC,$08,$DC,$08,$DC,$08,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$B1,$06,$B1,$06,
    $20,$06,$B2,$06,$20,$06,$20,$06,$20,$06,$B1,$06,$B1,$06,$B2,$06,
    $20,$06,$20,$06,$20,$06,$B1,$06,$B0,$06,$20,$06,$20,$06,$20,$06,
    $20,$06,$DB,$0E,$DB,$0E,$DB,$0E,$DF,$0E,$DC,$0E,$DC,$0E,$DF,$0E,
    $20,$0E,$DC,$0E,$DB,$0E,$DB,$0E,$DC,$0E,$DB,$0E,$DB,$0E,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$DC,$07,$DC,$07,$20,$78,$20,$78,$B0,$78,
    $B1,$78,$B1,$78,$DB,$08,$DC,$08,$DC,$08,$DC,$08,$DF,$08,$DF,$08,
    $DB,$08,$DC,$08,$DC,$08,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$B2,$06,$B1,$06,
    $B0,$06,$B1,$06,$B1,$06,$20,$06,$B0,$06,$B1,$06,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$B1,$06,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$DF,$0E,$DF,$0E,$DB,$0E,$DB,$0E,$DB,$0E,$DB,$0E,
    $DB,$0E,$DB,$0E,$DB,$0E,$DB,$0E,$DF,$0E,$DF,$0E,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $DC,$07,$20,$78,$B0,$78,$B1,$78,$B1,$78,$B1,$78,$B2,$78,$B2,$78,
    $DB,$08,$B2,$78,$DB,$78,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,
    $DC,$08,$DC,$08,$DF,$08,$DF,$08,$DB,$08,$DC,$08,$20,$07,$20,$07,
    $B0,$06,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$B1,$06,$B2,$06,
    $B1,$06,$B1,$06,$B1,$06,$B2,$06,$B1,$06,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$B0,$06,$B0,$06,$B1,$06,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$DC,$0F,$DC,$0F,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$78,
    $DB,$07,$B0,$78,$B1,$78,$B2,$78,$DB,$08,$B1,$78,$DB,$08,$B2,$78,
    $DB,$08,$DB,$78,$DB,$78,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,
    $DB,$08,$DB,$08,$DB,$08,$DB,$08,$DC,$08,$DF,$08,$DB,$08,$20,$07,
    $B0,$06,$B1,$06,$20,$06,$20,$06,$B1,$06,$B0,$06,$B1,$06,$B2,$06,
    $B2,$06,$B0,$06,$20,$06,$B1,$06,$B1,$06,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $B0,$06,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$DC,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,
    $DB,$0F,$DC,$0F,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$78,
    $B0,$78,$B1,$78,$B2,$78,$DB,$08,$DB,$08,$B2,$78,$DB,$78,$DB,$08,
    $DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,
    $DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DD,$08,$DB,$08,$20,$07,
    $20,$08,$20,$08,$B0,$06,$B0,$06,$B1,$06,$20,$06,$B0,$06,$B1,$06,
    $DB,$06,$B0,$06,$B0,$06,$B1,$06,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$DE,$0F,$DB,$0F,$DF,$0F,$DF,$0F,$DB,$0F,$DB,$0F,$DF,$0F,
    $DF,$0F,$DB,$0F,$DB,$0F,$20,$0F,$20,$0F,$20,$0F,$20,$0F,$DE,$78,
    $B0,$78,$B1,$78,$DB,$08,$DB,$08,$DB,$78,$DB,$08,$DB,$08,$DB,$08,
    $DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,
    $DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DD,$08,$DB,$08,$20,$07,
    $20,$08,$B0,$06,$B1,$06,$20,$06,$20,$06,$20,$06,$B0,$06,$B1,$06,
    $DB,$06,$B0,$06,$B0,$06,$B1,$06,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $DF,$0F,$DC,$0F,$DC,$0F,$DB,$0F,$DC,$0F,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$DB,$0F,$DC,$0F,$DC,$0F,$DB,$0F,$DB,$0F,$DC,$0F,$DC,$0F,
    $DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$20,$0F,$20,$0F,$20,$0F,$DE,$78,
    $DB,$08,$B1,$78,$DB,$78,$FE,$70,$52,$70,$65,$70,$73,$70,$74,$70,
    $20,$70,$69,$70,$6E,$70,$20,$70,$20,$70,$50,$70,$65,$70,$61,$70,
    $63,$70,$65,$70,$FE,$70,$DB,$08,$DB,$08,$DD,$08,$DB,$08,$20,$07,
    $B0,$06,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$B2,$06,$B1,$06,
    $B2,$06,$B1,$06,$B0,$06,$B1,$06,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $DF,$0F,$DC,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DC,$0F,
    $DC,$0F,$DC,$0F,$20,$0F,$20,$0F,$20,$0F,$20,$0F,$DC,$0F,$DC,$0F,
    $DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DC,$0F,$20,$0F,$20,$0F,$DF,$0F,
    $DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$20,$0F,$20,$0F,$DB,$08,
    $DB,$08,$B2,$78,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,
    $DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,
    $DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DD,$08,$DB,$08,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$B2,$06,$20,$06,
    $B2,$06,$DB,$06,$B1,$06,$B1,$06,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $DF,$0F,$20,$0F,$20,$0F,$20,$0F,$DF,$0F,$DB,$0F,$DB,$0F,$DB,$0F,
    $DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,
    $DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,
    $DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DC,$0F,$DC,$0F,$20,$0F,
    $DF,$08,$DF,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,
    $DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,
    $DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DD,$08,$DB,$08,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$B1,$06,$20,$06,
    $B2,$06,$DB,$06,$B2,$06,$B1,$06,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$DC,$08,$DC,$08,
    $DC,$08,$DC,$08,$DC,$08,$DC,$08,$20,$08,$20,$08,$20,$08,$DF,$0F,
    $DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,
    $DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,
    $DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$B2,$7F,$B1,$7F,
    $DB,$07,$DC,$07,$DC,$07,$DC,$07,$DF,$08,$DB,$08,$DF,$08,$DC,$07,
    $DF,$08,$DF,$08,$DC,$07,$20,$07,$DF,$08,$DB,$08,$DB,$08,$DB,$08,
    $DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DD,$08,$DB,$08,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$B1,$06,$B2,$06,
    $DB,$06,$20,$06,$B2,$06,$B1,$06,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$DC,$08,$DB,$08,$DB,$08,$48,$0E,$61,$0E,
    $70,$0E,$70,$0E,$79,$0E,$DB,$08,$DB,$08,$DB,$08,$DC,$08,$20,$08,
    $20,$08,$20,$08,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,
    $DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,
    $DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$B2,$7F,$B1,$7F,$B0,$7F,
    $DB,$07,$DB,$07,$20,$78,$B0,$78,$B1,$78,$DC,$07,$DC,$07,$DC,$07,
    $B0,$78,$B0,$78,$B1,$78,$DC,$07,$DF,$07,$DC,$08,$DB,$08,$DB,$08,
    $DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DD,$08,$DB,$08,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$DB,$06,$B2,$06,
    $DB,$06,$B0,$06,$B2,$06,$B2,$06,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$DB,$08,$DB,$08,$48,$0E,$61,$0E,$6C,$0E,$6C,$0E,
    $6F,$0E,$77,$0E,$65,$0E,$65,$0E,$6E,$0E,$21,$0E,$DB,$08,$DB,$08,
    $20,$08,$20,$08,$20,$08,$DF,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,
    $DB,$0F,$DB,$7F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,
    $DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$B2,$7F,$B1,$7F,
    $B0,$7F,$DB,$07,$B0,$78,$DB,$07,$B0,$78,$B0,$78,$B0,$78,$B1,$78,
    $B0,$78,$B0,$78,$B1,$78,$DF,$07,$DF,$07,$DC,$08,$DB,$08,$DB,$08,
    $DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DD,$08,$DB,$08,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$B2,$06,$B2,$06,$20,$06,
    $B0,$06,$B1,$06,$DB,$06,$B2,$06,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$DB,$08,$DB,$08,$B1,$08,$DB,$08,$DB,$08,$B2,$08,
    $B2,$08,$DB,$08,$B2,$08,$DB,$08,$DB,$08,$B2,$08,$DB,$08,$B2,$08,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$DF,$0F,$DF,$0F,$B1,$7F,
    $B1,$7F,$DB,$7F,$B2,$7F,$DB,$7F,$DB,$7F,$DB,$0F,$DB,$0F,$DB,$0F,
    $DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$B2,$7F,$B1,$7F,
    $B0,$7F,$DB,$07,$DB,$07,$B0,$78,$B0,$78,$B0,$78,$DF,$07,$DF,$07,
    $20,$07,$DC,$08,$DC,$08,$DC,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,
    $DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DD,$08,$DB,$08,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$DB,$06,$B1,$06,$B0,$06,
    $20,$06,$B1,$06,$DB,$06,$B2,$06,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$DB,$08,$B2,$08,$B0,$08,$B2,$08,$B2,$08,$B1,$08,
    $B2,$08,$DB,$08,$B1,$08,$B2,$08,$DB,$08,$B2,$08,$DB,$08,$B2,$08,
    $20,$08,$20,$08,$20,$08,$20,$08,$DC,$08,$DB,$08,$DB,$08,$20,$08,
    $DB,$07,$DB,$07,$DB,$07,$DF,$7F,$B1,$7F,$DB,$7F,$DB,$0F,$DB,$0F,
    $DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$B2,$7F,
    $B1,$7F,$B0,$7F,$DB,$07,$DB,$07,$DF,$07,$20,$07,$DB,$08,$DB,$08,
    $DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,
    $DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DD,$08,$DB,$08,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$B2,$06,$20,$06,$DB,$06,
    $B1,$06,$B1,$06,$DB,$06,$B1,$06,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$B2,$08,$B1,$08,$B0,$08,$B1,$08,$B2,$08,$B1,$08,
    $B1,$08,$B2,$08,$B0,$08,$B2,$08,$B2,$08,$B1,$08,$DB,$08,$B1,$08,
    $20,$08,$20,$08,$20,$08,$20,$08,$B2,$08,$DB,$08,$B2,$08,$B1,$08,
    $DC,$08,$DF,$07,$DB,$07,$DB,$07,$B1,$7F,$DB,$7F,$DB,$7F,$DB,$0F,
    $DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$B2,$7F,$B1,$7F,
    $DB,$07,$DF,$07,$DC,$08,$DC,$08,$DC,$08,$DB,$08,$DB,$08,$DB,$08,
    $DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,
    $DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DD,$08,$DB,$08,$20,$07,
    $20,$08,$20,$08,$20,$08,$20,$08,$B2,$06,$DB,$06,$20,$06,$B1,$06,
    $20,$06,$B0,$06,$B2,$06,$B1,$06,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$B2,$08,$B0,$08,$B0,$08,$B0,$08,$B1,$08,$B0,$08,
    $B1,$08,$B1,$08,$B0,$08,$B1,$08,$B1,$08,$B1,$08,$B2,$08,$B1,$08,
    $20,$08,$20,$08,$20,$08,$20,$08,$B1,$08,$B2,$08,$B1,$08,$B0,$08,
    $B2,$08,$B2,$08,$DC,$08,$DF,$07,$B1,$7F,$DB,$0F,$DB,$0F,$DB,$0F,
    $DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$7F,$B2,$7F,
    $B1,$7F,$B0,$7F,$DC,$07,$DC,$07,$DC,$07,$20,$07,$DF,$08,$DF,$08,
    $DF,$08,$DF,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,
    $DB,$08,$DB,$08,$DB,$08,$DB,$08,$DB,$08,$DD,$08,$DB,$08,$20,$07,
    $20,$08,$20,$08,$20,$08,$B2,$06,$DB,$06,$20,$06,$B2,$06,$B0,$06,
    $B0,$06,$B0,$06,$B2,$06,$B0,$06,$B0,$06,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$B1,$08,$B0,$08,$20,$08,$B0,$08,$B0,$08,$B0,$08,
    $B0,$08,$B0,$08,$20,$08,$B0,$08,$B1,$08,$B0,$08,$B1,$08,$B0,$08,
    $20,$08,$20,$08,$20,$08,$20,$08,$B0,$08,$B1,$08,$B0,$08,$20,$08,
    $B1,$08,$B1,$08,$B1,$08,$B1,$08,$20,$08,$DF,$0F,$DF,$0F,$DB,$0F,
    $DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$DB,$0F,$B1,$7F,
    $B0,$7F,$20,$7F,$DB,$07,$DB,$07,$20,$78,$20,$78,$B0,$78,$B0,$78,
    $B0,$78,$B1,$78,$DC,$07,$20,$07,$DF,$08,$DF,$08,$DF,$08,$DF,$08,
    $DF,$08,$DF,$08,$DB,$08,$DB,$08,$DB,$08,$DD,$08,$DB,$08,$20,$07,
    $20,$08,$20,$08,$B2,$06,$B2,$06,$20,$06,$20,$06,$B1,$06,$B0,$06,
    $B0,$06,$20,$06,$B2,$06,$B1,$06,$B0,$06,$B0,$06,$20,$06,$20,$06,
    $20,$06,$20,$06,$B0,$08,$20,$08,$20,$08,$B0,$08,$B0,$08,$B0,$08,
    $20,$08,$B0,$08,$20,$08,$B0,$08,$B0,$08,$20,$08,$B0,$08,$B0,$08,
    $20,$08,$20,$08,$20,$08,$20,$08,$B0,$08,$B0,$08,$20,$08,$20,$08,
    $B1,$08,$B0,$08,$B0,$08,$B0,$08,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$DF,$0F,$DF,$0F,$DF,$0F,$DF,$0F,$DF,$0F,$DF,$07,
    $DB,$07,$DB,$07,$DB,$07,$DB,$07,$B0,$78,$B0,$78,$B0,$78,$B0,$78,
    $B0,$78,$DB,$07,$B0,$78,$B0,$78,$B1,$78,$B1,$78,$DB,$07,$DB,$07,
    $DB,$07,$DC,$07,$DC,$07,$DC,$07,$20,$07,$20,$07,$DF,$08,$20,$07,
    $DC,$06,$B0,$06,$B1,$06,$73,$60,$63,$60,$B0,$06,$B1,$06,$B2,$06,
    $DC,$06,$DC,$06,$DC,$06,$DC,$06,$B0,$06,$B0,$06,$B0,$06,$DC,$06,
    $DC,$06,$DC,$06,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $DC,$06,$DC,$06,$DC,$06,$DC,$06,$DC,$06,$DC,$06,$DC,$06,$DC,$06,
    $DC,$06,$DC,$06,$DC,$06,$DC,$06,$DC,$06,$DC,$06,$DC,$06,$DC,$06,
    $DC,$06,$DC,$06,$DC,$06,$DC,$06,$DC,$06,$DC,$06,$DC,$06,$DF,$68,
    $DF,$68,$DC,$06,$DC,$06,$DC,$06,$DC,$06,$DC,$06,$DC,$06,$DC,$76,
    $DC,$76,$DC,$76,$DC,$76,$DC,$76,$DC,$76,$DC,$76,$DC,$76,$DC,$76,
    $DC,$76,$DC,$76,$DC,$76,$DC,$76,$DC,$76,$DC,$76,$DC,$76,$DF,$6F,
    $DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,
    $DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,
    $DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,
    $DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,
    $DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,
    $DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,
    $DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,
    $DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,
    $DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,
    $DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,$DF,$06,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,
    $20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07,$20,$07
);

CONST
  NUM_CHARACTERS  = 2000;
  MAX_COMPRESSION = 5000;

  MIN_LENGTH      = 3;

TYPE
  TCompressibleStuff  = ARRAY[1..NUM_CHARACTERS] OF BYTE;
  TCompressionResult  = RECORD
                          length : INTEGER;
                          data   : ARRAY[1..MAX_COMPRESSION] OF RECORD
                                     count : INTEGER; { 0 means literal }
                                     dist  : INTEGER; { look-back distance, or literal }
                                   END;
                        END;
  TComparisonFunction = FUNCTION (orig, comp : INTEGER):BOOLEAN;

VAR
  CharacterData    : TCompressibleStuff;
  ColorData        : TCompressibleStuff;
  Compressed       : TCompressionResult;
  CompressedColors : TCompressionResult;
  i, j, k          : INTEGER;
  EffNumCharacters : INTEGER;

FUNCTION IsBlank(ch:BYTE):BOOLEAN;
BEGIN
  IsBlank := (ch=32) OR (ch=0) OR (ch=255);
END;

FUNCTION IsFull(ch:BYTE):BOOLEAN;
BEGIN
  IsFull := (ch=219);
END;

FUNCTION IsHalfV(ch:BYTE):BOOLEAN;
BEGIN
  IsHalfV := (ch=220) OR (ch=223);
END;

FUNCTION IsHalfH(ch:BYTE):BOOLEAN;
BEGIN
  IsHalfH := (ch=221) OR (ch=222);
END;

FUNCTION IsInvertibleColor(c:BYTE):BOOLEAN;
BEGIN
  IsInvertibleColor := (c AND 8)=0;
END;

FUNCTION InvertColor(c:BYTE):BYTE;
BEGIN
  InvertColor := (c SHR 4) OR ((c SHL 4) AND $F0);
END;

FUNCTION AdjustColor(orig, repl, color:BYTE):INTEGER;
BEGIN
  IF orig = repl THEN BEGIN
    AdjustColor := color;
  END ELSE BEGIN
    CASE orig OF
      { We can replace a space by any character by adjusting the color. Not using
        this rule saves 5 bytes. }
      {.$20: AdjustColor := 17*(color SHR 4); }
      {.$00: AdjustColor := 17*(color SHR 4); }
      { We can replace a box by any character by adjusting the color if the color is not bright }
      {.$DB: IF IsInvertibleColor(color) THEN AdjustColor := 17*(color AND 15) ELSE AdjustColor := -1;}
      { We can replace a half by other half by inverting }
      $DC: IF IsInvertibleColor(color) AND (repl=$DF) THEN AdjustColor := InvertColor(color) ELSE AdjustColor := -1;
      $DD: IF IsInvertibleColor(color) AND (repl=$DE) THEN AdjustColor := InvertColor(color) ELSE AdjustColor := -1;
      $DE: IF IsInvertibleColor(color) AND (repl=$DD) THEN AdjustColor := InvertColor(color) ELSE AdjustColor := -1;
      $DF: IF IsInvertibleColor(color) AND (repl=$DC) THEN AdjustColor := InvertColor(color) ELSE AdjustColor := -1;
    ELSE AdjustColor := -1;
    END;
  END;
END;

FUNCTION CompareCharacters(orig, comp:INTEGER):BOOLEAN;
BEGIN
  CompareCharacters := AdjustColor(CharacterData[orig], CharacterData[comp], ColorData[orig]) >= 0;
END;

FUNCTION CompareColors(orig, comp:INTEGER):BOOLEAN;
BEGIN
  CompareColors := (ColorData[orig] = ColorData[comp])
                OR ((CharacterData[orig] = $20) AND ((ColorData[orig] AND $F0) = (ColorData[comp] AND $F0)))
                OR ((CharacterData[orig] = $00) AND ((ColorData[orig] AND $F0) = (ColorData[comp] AND $F0)))
                OR ((CharacterData[orig] = $DB) AND ((ColorData[orig] AND $0F) = (ColorData[comp] AND $0F)));
END;

PROCEDURE Compress(VAR data : TCompressibleStuff;
                   VAR out : TCompressionResult;
                   compare : TComparisonFunction);
VAR
  index                  : INTEGER;
  offset, length         : INTEGER;
  bestoffset, bestlength : INTEGER;
  i                      : INTEGER;
  tmp                    : TCompressibleStuff;
BEGIN
  out.length := 0;
  index := 1;
  WHILE index <= EffNumCharacters DO BEGIN
    { Try to find maximum-length run starting at 'index' }
    bestoffset := 0;
    bestlength := 0;
    offset := 1;
    WHILE (offset <= index-1) AND (offset < 256) DO BEGIN
      length := 0;
      tmp := data;
      WHILE (length <= EffNumCharacters - index) AND (length < 255 { no '+MIN_LENGTH' here, so length fits in 8 bit })
       AND compare(index + length, index - offset + length)
        DO BEGIN
          data[index + length] := data[index - offset + length];
          Inc(length);
        END;
      data := tmp;
      IF length > bestlength THEN BEGIN
        bestlength := length;
        bestoffset := offset;
      END;
      Inc(offset);
    END;
    { What did we find? }
    Inc(out.length);
    IF bestlength > MIN_LENGTH THEN BEGIN
      { Update data }
      FOR i:=0 TO bestlength-1 DO
        data[index + i] := data[index - bestoffset + i];
      { Backreference }
      out.data[out.length].count := bestlength;
      out.data[out.length].dist  := bestoffset;
      Inc(index, bestlength);
    END ELSE BEGIN
      { Literal }
      out.data[out.length].count := 0;
      out.data[out.length].dist  := data[index];
      Inc(index);
    END;
  END;
END;

PROCEDURE Decompress(CONST data : TCompressionResult; VAR out_);
VAR
  out   : ARRAY[1..10000] OF BYTE ABSOLUTE out_;
  index : INTEGER;
  i, j  : INTEGER;
BEGIN
  index := 1;
  FOR i:=1 TO data.length DO BEGIN
    IF data.data[i].count = 0 THEN BEGIN
      out[index] := data.data[i].dist;
      Inc(index, 2);
    END ELSE BEGIN
      FOR j:=1 TO data.data[i].count DO BEGIN
        out[index] := out[index - 2*data.data[i].dist];
        Inc(index, 2);
      END;
    END;
  END;
END;

VAR bitValue, bitIndex, byteCount : INTEGER;

PROCEDURE BitFlush;
BEGIN
  IF bitIndex<>8 THEN Writeln('; ', bitIndex, ' bits used in following byte:');
  Writeln(#9'db ',bitValue);
  bitValue := 0;
  bitIndex := 0;
  Inc(byteCount);
END;

PROCEDURE Bit(value:BYTE);
BEGIN
  IF bitIndex >= 8 THEN BitFlush;
  bitValue := bitValue OR (value SHL bitIndex);
  Inc(bitIndex);
END;

PROCEDURE Bits(value:INTEGER; count:INTEGER);
VAR i:INTEGER;
BEGIN
  FOR i:=1 TO count DO Bit((value SHR (count-i)) AND 1);
END;

PROCEDURE VarBits(value:INTEGER; CONST what:STRING);
VAR count:INTEGER;
BEGIN
  IF value<=0 THEN BEGIN
    Writeln('** Error: value=', value, ' not allowed');
    Halt(1);
  END;
  count := 0;
  WHILE (value SHR count) > 1 DO Inc(count);
  IF count > 7 THEN BEGIN
    Writeln('** Error: count=', count, ' not allowed');
    Halt(1);
  END;
  Writeln(#9'; encoding ', what, ' ', value, ' with ', count, ' value bits');
  Bits(count, 3);
  Bits(value, count);
END;

PROCEDURE ShowCompression(VAR c:TCompressionResult; colors:BOOLEAN);
VAR
  i, j : INTEGER;
  ctl  : INTEGER;
  bytes : INTEGER;

CONST FREQUENT_LITERALS = #32#101#176#177#178#219#220#223;
CONST FREQUENT_COLORS =  #6#7#8#112#120#127#96;

BEGIN
 {FOR i:=1 TO c.length DO BEGIN
    IF c.data[i].count=0 THEN Writeln('  Literal ', c.data[i].dist)
                         ELSE Writeln('  Backref ', c.data[i].count, ' chars at ', c.data[i].dist);
  END;}
 {i := 1;
  bytes := 0;
  WHILE i <= c.length DO BEGIN
    ctl := 0;
    FOR j:=0 TO 7 DO IF (i+j <= c.length) AND (c.data[i+j].count=0) THEN ctl := ctl OR (1 SHL j);
    Writeln(#9'db ', ctl, #9'; control word');
    Inc(bytes);
    FOR j:=0 TO 7 DO IF (i <= c.length) THEN BEGIN
      IF c.data[i].count=0 THEN BEGIN
        Writeln(#9'db ', c.data[i].dist, #9'; literal');
        Inc(bytes);
      END ELSE BEGIN
        Writeln(#9'db ', c.data[i].dist, ',', c.data[i].count, #9'; backref');
        Inc(bytes, 2);
      END;
      Inc(i);
    END;
  END;
  Writeln('; ', bytes, ' bytes');}
  FOR i:=1 TO c.length DO BEGIN
    IF c.data[i].count=0 THEN BEGIN
      IF colors THEN j := Pos(Chr(c.data[i].dist), FREQUENT_COLORS)
                ELSE j := Pos(Chr(c.data[i].dist), FREQUENT_LITERALS);
      Bit(0);
      IF j>0 THEN BEGIN
        Writeln('; frequent literal ', c.data[i].dist);
        Bit(1);
        Bits(j-1, 3);
      END ELSE BEGIN
        Writeln('; literal ', c.data[i].dist);
        Bit(0);
        Bits(c.data[i].dist, 8);
      END;
    END ELSE BEGIN
      Writeln('; backref ', c.data[i].count, ' bytes ', c.data[i].dist, ' away');
      Bit(1);
      VarBits(c.data[i].dist, 'distance');
      VarBits(c.data[i].count-MIN_LENGTH, 'count');
    END;
  END;
  Writeln('; ', byteCount, ' bytes so far');
END;

PROCEDURE DoLiteralStatistic;
TYPE TLiteralStatistic   = ARRAY[BYTE] OF INTEGER;

  PROCEDURE ShowStatistic(CONST s:TLiteralStatistic);
  VAR i, j:INTEGER;
  BEGIN
    j:=0;
    FOR i:=0 TO 255 DO BEGIN
      IF s[i]>1 THEN Writeln(';     ', i, ' appears ', s[i], ' times');
      Inc(j, s[i]);
    END;
    Writeln(';     => total: ', j, ' literals');
  END;

VAR
  a, b, c : TLiteralStatistic;
  i       : INTEGER;
BEGIN
  FOR i:=0 TO 255 DO BEGIN
    a[i] := 0;
    b[i] := 0;
    c[i] := 0;
  END;
  FOR i:=1 TO Compressed.length DO
    IF Compressed.data[i].count=0 THEN BEGIN
      Inc(a[Compressed.data[i].dist]);
      Inc(c[Compressed.data[i].dist]);
    END;
  FOR i:=1 TO CompressedColors.length DO
    IF CompressedColors.data[i].count=0 THEN BEGIN
      Inc(b[CompressedColors.data[i].dist]);
      Inc(c[CompressedColors.data[i].dist]);
    END;
  Writeln('; Frequent text literals:');
  ShowStatistic(a);
  Writeln('; Frequent color literals:');
  ShowStatistic(b);
  Writeln('; Frequent overall literals:');
  ShowStatistic(c);
END;

BEGIN
  { Optimize input data a little }
  j := NUM_CHARACTERS;
  WHILE (j > 0) AND (ImageData[2*j-1]=32) AND (ImageData[2*j]<$10) DO Dec(j);
  EffNumCharacters := j;
  j:=1;
  WHILE (j <= NUM_CHARACTERS) AND (ImageData[2*j-1]=32) AND (ImageData[2*j]<$10) DO Inc(j);
  IF (j <= NUM_CHARACTERS) AND (ImageData[2*j]<$10) THEN BEGIN
    { The image starts with a row of black spaces followed by a brown-on-black character.
      This makes all the spaces brown-on-black, and thus saves two entries in compression. }
    FOR i:=1 TO j-1 DO ImageData[2*i] := ImageData[2*j];
    { Alternative: This makes all the spaces character-on-black and saves the entries in
      the image, not the color array. Needs 7 bits more. }
    {FOR i:=1 TO j-1 DO BEGIN
      ImageData[2*i-1] := ImageData[2*j-1];
      ImageData[2*i] := 0;
    END;}
  END;
  Writeln('NUM_CHARACTERS = ', EffNumCharacters);

  { Copy character data }
  FOR i:=1 TO NUM_CHARACTERS DO BEGIN
    CharacterData[i] := ImageData[2*i-1];
    ColorData[i] := ImageData[2*i];
  END;
  Compress(CharacterData, Compressed, CompareCharacters);
  Writeln('; Characters compressed to ', Compressed.length, ' elements');
  Writeln('CharacterData:');
  ShowCompression(Compressed, FALSE);

  { Update color data }
  FOR i:=1 TO NUM_CHARACTERS DO BEGIN
    j := AdjustColor(ImageData[2*i-1], CharacterData[i], ImageData[2*i]);
    IF j<0 THEN BEGIN
      Writeln('*** Error');
    END ELSE BEGIN
      ColorData[i] := j;
    END;
  END;
  Compress(ColorData, CompressedColors, CompareColors);
  Writeln('; Colors compressed to ', CompressedColors.length, ' elements');
  Writeln('ColorData:');
  ShowCompression(CompressedColors, TRUE);
  BitFlush;
  DoLiteralStatistic;

  { Test: display memory }
  IF NOT TRUE THEN BEGIN
    FOR i:=1 TO NUM_CHARACTERS DO BEGIN
      Mem[SegB800 : 2*i-2] := CharacterData[i];
      Mem[SegB800 : 2*i-1] := ColorData[i];
    END;
  END;

  { Test: display it }
  IF NOT TRUE THEN BEGIN
    {ASM mov ax, 3; int 10h; END;}
    Decompress(Compressed, Ptr(SegB800, 0)^);
    Decompress(CompressedColors, Ptr(SegB800, 1)^);
  END;
END.
