// =========================================================================
//  raw2pre v1.15 -- Hugi Compo #7 entry by Jibz '99
// =========================================================================
//
// Takes a raw image file and removes all zero entries (except the first of
// cause). Then it applies a simple RLE preprocessing to the raw image data
// and saves the optimized palette as ENTRY.PAL, and the preprocessed image
// data as ENTRY.PRE
//

#include <string.h>
#include <stdio.h>
#include <io.h>
#include <dos.h>
#include <conio.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <errno.h>

// Possible errors
enum { IN_ERR = 1, SIZE_ERR, ABORT_ERR, OUT_ERR };

// Make sure we have more than enough mem to work with ;)
#define BUFFERSIZE (65536)

// Program name and version
const char *versionstr = "raw2pre v1.15";

unsigned char inbuffer[BUFFERSIZE];
unsigned char outbuffer[BUFFERSIZE];
unsigned char palbuffer[2048];
unsigned char tagbuffer[BUFFERSIZE];

// Global variables used
int infile = -1, outfile = -1;

unsigned int inpos, outpos, palpos;
unsigned int tagpos, bitcount;

// =========================================================================
//  FUNCTIONS
// =========================================================================

inline int log2x(int n)
{
   if (n < 2) return (100);
   if (n < 4) return (2);
   if (n < 8) return (4);
   if (n < 16) return (6);
   if (n < 32) return (8);
   if (n < 64) return (10);
   if (n < 128) return (12);
   if (n < 256) return (14);
   if (n < 512) return (16);
   if (n < 1024) return (18);
   if (n < 2048) return (20);
   if (n < 4096) return (22);
   if (n < 8192) return (24);
   if (n < 16384) return (26);
   if (n < 32768) return (28);
   if (n < 65536) return (30);
   return (32);
}

static void advancetagbyte(int bit)
{
   // Check tagpos and then decrement
   if (!bitcount--)
   {
      bitcount = 7;
      tagpos++;
   }

   // Shift in bit
   if (bit)
   {
      tagbuffer[tagpos] = (tagbuffer[tagpos] >> 1) + 0x80;
   } else {
      tagbuffer[tagpos] = tagbuffer[tagpos] >> 1;
   }
}

// Output Gamma2-code for val in range [2..?] ...
static void outputGAMMA(unsigned int val)
{
   int invertlen = 0;
   unsigned int invert;

   do {
      invert = (invert << 1) + (val & 0x0001);
      invertlen++;
   } while ((val >>= 1) > 1);

   while (--invertlen)
   {
      advancetagbyte(invert & 0x0001);
      advancetagbyte(1);
      invert >>= 1;
   }
   advancetagbyte(invert & 0x0001);
   advancetagbyte(0);
}

unsigned int getPALDISTlength(unsigned int oldval, unsigned int newval)
{
   unsigned int dist = abs(oldval - newval);

   switch (dist)
   {
      case 0  : return (2); break;
      case 1  :
      case 2  : return (3); break;
      default : return (3 + log2x(dist - 1)); break;
   }
}

void swap_colors(unsigned int c1, unsigned int c2)
{
   unsigned char tmp;

   c1 *= 3; c2 *= 3;

   tmp = palbuffer[c1];
   palbuffer[c1] = palbuffer[c2];
   palbuffer[c2] = tmp;

   tmp = palbuffer[c1 + 1];
   palbuffer[c1 + 1] = palbuffer[c2 + 1];
   palbuffer[c2 + 1] = tmp;

   tmp = palbuffer[c1 + 2];
   palbuffer[c1 + 2] = palbuffer[c2 + 2];
   palbuffer[c2 + 2] = tmp;
}

// =========================================================================
//  MISC. FUNCTIONS
// =========================================================================

void cursoff();
#pragma aux cursoff = \
        "mov   ah, 3"    \
        "xor   ebx, ebx" \
        "int   10h"      \
        "or    ch, 20h"  \
        "mov   ah, 1"    \
        "int   10h"      \
        modify exact [eax ebx ecx edx];

void curson();
#pragma aux curson = \
        "mov   ah, 3"    \
        "xor   ebx, ebx" \
        "int   10h"      \
        "and   ch, 1fh"  \
        "mov   ah, 1"    \
        "int   10h"      \
        modify exact [eax ebx ecx edx];

// Some I/O error handler :)
void myerror(int n)
{
   // Print out error information
   printf("\n\n\007ERR: ");
   switch (n)
   {
      case IN_ERR    : printf("Unable to open input-file!\n"); break;
      case SIZE_ERR  : printf("File has wrong size (not 64768 bytes)!\n"); break;
      case ABORT_ERR : printf("Aborted by user!\n"); break;
      case OUT_ERR   : printf("Unable to open output-file!\n"); break;
      default        : printf("An unknown error occured!\n");
   }

   // Free memory and close files
   if (infile != -1) close(infile);
   if (outfile != -1) close(outfile);

   // Turn cursor on
   curson();

   // Exit with errorlevel = error no.
   exit(n);
}

// =========================================================================
//  MAIN
// =========================================================================

int main(int argc, char *argv[])
{
   unsigned char ch;

   unsigned char *rotator = "-/|\\";
   int rotatorpos = 0;

   unsigned int size, i, j, temp;
   unsigned char lastbyte;

   // Write name and copyright notice
   printf("---------------------------------------\\/--------------------------------------\n");
   printf("%-20s                            Hugi Compo #7 entry by Jibz '99\n", versionstr);
   printf("---------------------------------------/\\--------------------------------------\n\n");

   // Write syntax if not correct number of parameters
   if (argc != 2)
   {
      printf("Syntax:   RAW2PRE <input file>\n\n");
      exit(1);
   }

   // Turn cursor off
   cursoff();

   // Open input file
   printf("- Opening file\n");
   if ((infile = open(argv[1], O_RDONLY | O_BINARY)) == -1) myerror(IN_ERR);
   lseek(infile, 0, SEEK_SET);

   // Get file size and check it
   if (filelength(infile) != 64768) myerror(SIZE_ERR);

   // Read palette and image data
   read(infile, palbuffer, 768);
   read(infile, inbuffer, 64000);

   // This table is used for each round of swapping
   unsigned int table[256];

   // This table stores where each color goes after all the work
   unsigned int orgtable[256];
   for (i = 0; i < 256; i++) orgtable[i] = i;

   // Remove zero palette entries
   printf("- Optimizing palette ............ ");
   for (i = 0; i < 256; i++) table[i] = 0;

   j = 1;
   for (i = 1; i < 256; i++)
   {
      if ((palbuffer[i*3] != 0) || (palbuffer[i*3+1] != 0) || (palbuffer[i*3+2] != 0))
      {
         palbuffer[j*3  ] = palbuffer[i*3  ];
         palbuffer[j*3+1] = palbuffer[i*3+1];
         palbuffer[j*3+2] = palbuffer[i*3+2];
         table[i] = j;
         j++;
      } else {
         table[i] = 0;
      }
   }
   palpos = j;
   while (j < 256)
   {
      palbuffer[j*3  ] = 0;
      palbuffer[j*3+1] = 0;
      palbuffer[j*3+2] = 0;
      j++;
   }

   // Update orgtable accordingly
   for (i = 0; i < 256; i++) orgtable[i] = table[orgtable[i]];

   printf("%d entries removed\n", 256 - palpos);

   // Sort palette so linear color distance is minimized
   printf("- Sorting palette\n");
   for (i = 0; i < 256; i++) table[i] = i;

   for (i = 0; i < (palpos - 1); i++)
   {
      unsigned int min_distance = 4800;
      unsigned int min_distance_pos = i + 1;

      for (j = i + 1; j < palpos; j++)
      {
         unsigned int cur_distance = abs(palbuffer[j*3] - palbuffer[i*3]) +
                                     abs(palbuffer[j*3+1] - palbuffer[i*3+1]) +
                                     abs(palbuffer[j*3+2] - palbuffer[i*3+2]);

         if (cur_distance < min_distance)
         {
            min_distance = cur_distance;
            min_distance_pos = j;
         }
      }

      if (min_distance_pos != i + 1)
      {
         swap_colors(i + 1, min_distance_pos);

         temp = table[i + 1];
         table[i + 1] = table[min_distance_pos];
         table[min_distance_pos] = temp;

         for (j = 0; j < 256; j++) orgtable[j] = table[orgtable[j]];
         for (j = 0; j < 256; j++) table[j] = j;
      }
   }

   // Try to optimize sorting
   printf("- Optimizing sorted palette");
   for (i = 0; i < 256; i++) table[i] = i;

   int changes;
   int round = 0;

   do { // while (changes)

      changes = 0;
      round++;

      // Variables
      int rgb_bitlength = 0;
      int best_rgb_bitlength = 0;
      int rval = 0;
      int gval = 0;
      int bval = 0;

      // Get current bitlength
      for (i = 0; i < palpos; i++)
      {
         rgb_bitlength += getPALDISTlength(palbuffer[i*3], rval);
         rval = palbuffer[i*3];
         rgb_bitlength += getPALDISTlength(palbuffer[i*3+1], gval);
         gval = palbuffer[i*3+1];
         rgb_bitlength += getPALDISTlength(palbuffer[i*3+2], bval);
         bval = palbuffer[i*3+2];
      }
      best_rgb_bitlength = rgb_bitlength;

      // Try swapping all pairs of colors, and apply if better
      for (i = 0; i < (palpos - 1); i++)
      {
         // Show progress and check for ESC every now and then..
         if ((i & 0x0003) == 0)
         {
            printf("\r%c Optimizing sorted palette, round %d, %u%% done          ", rotator[rotatorpos], round, (i*100)/(palpos - 1));
            rotatorpos = (rotatorpos + 1) & 0x0003;

            // Check for ESC-hit
            if (kbhit())
            {
               ch = getch();
               if (ch == 0) ch = getch();
               if (ch == 27) myerror(ABORT_ERR);
            }
         }

         for (j = i + 1; j < palpos; j++)
         {
            swap_colors(i, j);

            rgb_bitlength = 0;
            rval = gval = bval = 0;

            unsigned int x;

            for (x = 0; x < palpos; x++)
            {
               rgb_bitlength += getPALDISTlength(palbuffer[x*3], rval);
               rval = palbuffer[x*3];
               rgb_bitlength += getPALDISTlength(palbuffer[x*3+1], gval);
               gval = palbuffer[x*3+1];
               rgb_bitlength += getPALDISTlength(palbuffer[x*3+2], bval);
               bval = palbuffer[x*3+2];
            }

            if (rgb_bitlength >= best_rgb_bitlength)
            {
               swap_colors(i, j);
            } else {
               changes++;

               best_rgb_bitlength = rgb_bitlength;

               temp = table[i];
               table[i] = table[j];
               table[j] = temp;

               for (x = 0; x < 256; x++) orgtable[x] = table[orgtable[x]];
               for (x = 0; x < 256; x++) table[x] = x;
            }
         } // for (j = i + 1; j < palpos; j++)
      } // for (i = 0; i < (palpos - 1); i++)
   } while (changes);
   printf("\r- Optimizing sorted palette, round %d, 100%% done          \n", round);

   // Update image data accordingly
   for (i = 0; i < 64000; i++) inbuffer[i] = orgtable[inbuffer[i]];

   // Preprocess image data 1
   printf("- Preprocessing image data 1 .... ");
   tagpos = 0;
   tagbuffer[tagpos++] = 1;
   bitcount = 8;

   outpos = 0;

   lastbyte = inbuffer[0] + 128;

   for (inpos = 0; inpos < 64000; inpos++)
   {
      if (inbuffer[inpos] != lastbyte)
      {
         advancetagbyte(1);
         lastbyte = inbuffer[inpos];
         outbuffer[outpos++] = lastbyte;
      } else {
         advancetagbyte(0);
      }
   }

   // Shift last tagbits into position
   tagbuffer[tagpos] = tagbuffer[tagpos] >> bitcount;
   tagpos++;

   printf("Done (64000 bytes -> %d bytes)\n", outpos + tagpos);

   // Write palette to ENTRY.PAL
   if ((outfile = open("ENTRY.PAL", O_WRONLY | O_CREAT | O_BINARY | O_TRUNC, S_IREAD | S_IWRITE)) == -1) myerror(OUT_ERR);
   lseek(outfile, 0, SEEK_SET);

   printf("- Writing ENTRY.PAL ............. %d bytes\n", palpos*3);
   write(outfile, palbuffer, palpos*3);

   // Write image data to ENTRY.PRE
   if (outfile != -1) close(outfile);
   if ((outfile = open("ENTRY.PRE", O_WRONLY | O_CREAT | O_BINARY | O_TRUNC, S_IREAD | S_IWRITE)) == -1) myerror(OUT_ERR);
   lseek(outfile, 0, SEEK_SET);

   printf("- Writing ENTRY.PRE ............. %d bytes\n", outpos + tagpos);
   write(outfile, outbuffer, outpos);
   write(outfile, tagbuffer, tagpos);

   // Close in and out files
   printf("- Closing files\n");
   if (infile != -1) close(infile);
   if (outfile != -1) close(outfile);

   // Turn cursor back on
   curson();

   // We're happy :)
   return (0);
}
// =========================================================================
//  END
// =========================================================================
