// TinyPTC future concept (floating point)
// Glenn "Gaffer" Fiedler, 30/5/2003
// http://www.gaffer.org/tinyptc

// THE IDEA: 
//
// Lets exploit floating point texture formats on new DirectX 9.0 compatible cards
// to provide a true floating point framebuffer with linear [0,1] intensity for all pixel
// color components (r,g,b,a), and automatic clamping of color values outside this range.
//
// On graphics cards that do not support such coolness, lets convert from the floating
// point values down to 32bit, 24bit, 16bit pixel formats ourselves in software, just
// for compatibility - this is not expected to be fast, its just a fallback!
//
// Why floating point you ask? Because we can, because its got really high precision 
// and you can do some really cool shit knowing that its automatically clamped.
// With a true linear colorspace anti-aliasing is perfect (no stair stepping), and
// honestly it would make a quite cool display output for a raytracer, or for a program
// exploring high end software rendering with *high dynamic range textures* in software,
// lots of uses, many of them probably wont quite be realtime yet, but the future
// beckons... one day, all computers and display hardware will be fast enough that
// we'll work in linear floating point color space all the time!
// 
// ps. I hope you have a pentium4 and like your SSE2... >B)
//
// Feedback appreciated - send mail to ptc@gaffer.org

namespace ptc
{
    // lets start off with a cool event listener interface so your application knows about whats going on...

    class Listener
    {
    public:
        virtual void onKeyUp(int code) = 0;
        virtual void onKeyDown(int code) = 0;
        virtual void onMouseButtonDown(int button, int x, int y) = 0;
        virtual void onMouseMove(int button, int x, int y) = 0;
        virtual void onActivate(bool active) = 0;                   // user activated the window (alt-tab away and back, also called when app starts...)
        virtual void onClose() = 0;                                 // user closed the window
    };

    // and a convenient pixel class that maps 1:1 to each pixel in the frame buffer

    class Pixel
    {
    public:
        float r,g,b,a;
    };

    // the new display class provides all of the functionality of the old tinyptc functions: ptc_open, ptc_update, ptc_close, and adds user input callback support via listener

    class Display
    {
    public:
        Display();
        Display(const char title[], int width, int height, bool windowed = true);
        ~Display();

        void open(const char title[], int width, int height, bool windowed = true);
        void close();

        void update(Pixel pixels[]);            // floating point framebuffer: each pixel is float r,g,b in range [0,1].

        void listener(Listener *listener);      // pass a valid listener interface to get notification, null pointer disables listener
    };

    // we may as well add a nice timer class too...

    class Timer
    {
    public:
        double time();
        double delta();
        double resolution();
    };

    // you should now have pretty much everything you need to make simple framebuffer application,
    // have i missed anything? is there anything else that would be useful?
}

// Here is what a basic program using this library would look like:

#include <vector>
#include "tinyptc.hpp"

using namespace tinyptc;


class Application : public Listener
{
public:

    void run()
    {
        // open the display and create our pixel array

        const int width = 640;
        const int height = 480;
        const int size = width * height;

        Display display("Example", width, height, true);

        pixels.resize(size);

        while (true)
        {
            // draw a gradient across the screen

            int i = 0;

            for (int y=0; y<height; y++)
            {
                for (int x=0; x<width; x++)
                {
                    pixels[i].r = x*0.01f + y*0.02f;
                    pixels[i].g = x*0.03f + y*0.01f;
                    pixels[i].b = x*0.02f + y*0.03f;
                    pixels[i].a = 0.0f;
                    i++;
                }
            }

            // send the updated pixels to the display

            display.update(&pixels[0]);
        }
    }

private:

    // event handling code, anybody who has used java should find this rather familiar...

    virtual void onKeyDown(int code)
    {
        // close the application if the user presses any key
        // note: there will be key codes defined like this, Key::ESCAPE...

        display.close();
        exit(0);
    }

    virtual void onKeyUp(int code)
    {
    }

    virtual void onMouseButtonDown(int button, int x, int y)
    {
    }

    virtual void onMouseButtonUp(int button, int x, int y)
    {
    }

    virtual void onMouseMove(int button, int x, int y)
    {
    }

    virtual void onActivate(bool active)
    {
    }

    virtual void onClose()
    {
        // close the application if the user closes the window, eg: ALT-F4...
        
        display.close();
        exit(0);
    }

    Display display;
    std::vector<Pixel> pixels
};


void main()
{
    Application application;
    application.run();
}
