/**
	lightmodel: (http://en.wikipedia.org/wiki/Phong_shading)

	ia = ambient light
	ka = ambient reflection constant

	kd = diffuse reflection constant (intensity of light)
	Lm = direction vector from light to current point
	N = normal of surface
	id = diffuse light

	Ip = ka * ia + sum(kd * (Lm dot N) * id);
**/

#include "lightmodel.h"
#include "light.h"


/** 
 *	Initialize data-members
 */
LightModel::LightModel(float const *lightInfo, size_t nLights) 
:
	d_lights(0),
	d_nLights(nLights)
{
	d_ambient.rgba = 0;

	d_lights = new Light*[nLights];
	for (size_t idx = 0; idx < nLights; ++idx) {
		d_lights[idx] = new Light((float *) lightInfo);
		lightInfo += 7;
	}
	
}

/**
 *	This method determines the color at a certain position in the world
 */
union color LightModel::colorAt(Gfx &gfx, vector3 const &position, vector3 const &normal) {
	
	const float 
		infAmbient = 0.1f,
		infDiffuse = 0.9f;
	
	float r = 0, g = 0, b = 0;
	
	// ambient term	
	r = infAmbient * d_ambient.r;
	g = infAmbient * d_ambient.g;
	b = infAmbient * d_ambient.b;

	vector3 dVector;
	
	// calculcate diffuse term
	for (size_t idx = 0; idx < d_nLights; ++idx) {
		Light *l = d_lights[idx];

		union color diffuseColor = l->rgb();

		// get ia
		float intensity = l->intensityAt(position);
		
		if (intensity > 0) {
			// get Lm term
			vec_direction_vector(dVector, position, l->position());
		
			// (Lm dot N)
			float dotProduct = vec_dotproduct(dVector, normal);

			// kd * (Lm dot N) * id		
			r += infDiffuse * dotProduct * (intensity * diffuseColor.r);
			g += infDiffuse * dotProduct * (intensity * diffuseColor.g);
			b += infDiffuse * dotProduct * (intensity * diffuseColor.b);
		}
		
	}
	
	// cut off at white.
	if (r > 255) r = 255;
	if (g > 255) g = 255;
	if (b > 255) b = 255;
	
	// return color.
	return gfx.color((Uint8) r, (Uint8) g, (Uint8) b);
}

