//
// Tiny Console class for OpenPTC 1.0 Java Implementation
// Copyright (c) 1999 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
// See http://www.gnu.org/copyleft/lgpl.html for details
//

// package name
package tinyptc;

// import classes
import java.awt.Image;
import java.awt.Toolkit;
import java.awt.Graphics;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.image.ImageProducer;
import java.awt.image.ImageConsumer;
import java.awt.image.ColorModel;
import java.awt.image.IndexColorModel;
import java.awt.image.DirectColorModel;



public class Console implements ImageProducer
{
    public void open(Component component,Format format)
    {
        // get component size
        Dimension size = component.size();

        // setup console data
        _width = size.width;
        _height = size.height;
        _format = format;
        _component = component;

        // setup color model
        _model = model(format,new int[256]);

        // create image using default toolkit
        _image = Toolkit.getDefaultToolkit().createImage(this);
    }

    public void update()
    {
        // paint
        paint();
    }

    public synchronized void load(Object pixels,int width,int height,int pitch,Format format,int palette[])
    {
        // check consumer reference
        if (_consumer==null) return;

        // check bits
        switch (format.bits())
        {
            case 32:
            {
                // copy integer pixel data to image consumer
                _consumer.setPixels(0,0,width,height,model(format,palette),(int[])pixels,0,pitch/4);
            }
            break;

            case 8:
            {
                // copy byte pixel data to image consumer
                _consumer.setPixels(0,0,width,height,_model,(byte[])pixels,0,pitch);
            }
            break;
        }

        // notify image consumer that the frame is done
        _consumer.imageComplete(ImageConsumer.SINGLEFRAMEDONE);
    }

    public void palette(int data[])
    {
        // update color model palette
        _model = model(_format,data);

        // check consumer reference
        if (_consumer==null) return;

        // set consumer palette model
        _consumer.setColorModel(_model);
    }

    public int width()
    {
        // get width
        return _width;
    }

    public int height()
    {
        // get height
        return _height;
    }

    public Format format()
    {
        // get format
        return _format;
    }

    public synchronized void addConsumer(ImageConsumer ic) 
    {
        // register image consumer
        _consumer = ic;

        // set image dimensions
        _consumer.setDimensions(_width,_height);

        // set image consumer hints for speed
        _consumer.setHints(ImageConsumer.TOPDOWNLEFTRIGHT|ImageConsumer.COMPLETESCANLINES|ImageConsumer.SINGLEPASS|ImageConsumer.SINGLEFRAME);

        // set image color model
        _consumer.setColorModel(_model);
    }

    public synchronized boolean isConsumer(ImageConsumer ic) 
    {
        // check if consumer is registered
        return true;
    }

    public synchronized void removeConsumer(ImageConsumer ic) 
    {
        // remove image consumer
    }

    public void startProduction(ImageConsumer ic) 
    {
        // add consumer
        addConsumer(ic);
    }

    public void requestTopDownLeftRightResend(ImageConsumer ic) 
    {
        // ignore resend request
    }

    private synchronized void paint()
    {
        // get component graphics object
        Graphics graphics = _component.getGraphics();

        // draw image to graphics context
        graphics.drawImage(_image,0,0,_width,_height,null);
    }

    private ColorModel model(Format format,int palette[])
    {
        // check format
        if (format.direct())
        {
            // create direct color model
            return new DirectColorModel(format.bits(),format.r(),format.g(),format.b(),format.a());
        }
        else
        {
            // create color arrays
            byte r[] = new byte[256];
            byte g[] = new byte[256];
            byte b[] = new byte[256];

            // setup color byte arrays
            for (int i=0; i<256; i++)
            {
                // unpack color integer to a,r,g,b
                r[i] = (byte) ( (palette[i] & 0x00FF0000) >> 16 );
                g[i] = (byte) ( (palette[i] & 0x0000FF00) >> 8  );
                b[i] = (byte) ( (palette[i] & 0x000000FF)       );
            }

            // create indexed color model
            return new IndexColorModel(format.bits(),256,r,g,b);
        }
    }

    // console data
    private int _width;
    private int _height;
    private Image _image;
    private Format _format;
    private ColorModel _model;
    private Component _component;
    private ImageConsumer _consumer;
}
