/*-------------------------------------------------------------------------------------------*\
++                                                                                           ++
++            Copyright (C) 1998, 1999 by Punk Productions Electronic Entertainment.         ++
++                                                                                           ++
++     Content: Code for loading 1-24bit PCXFiles into DirectDrawSurfaces                    ++
++ Day Created: 11-26-99		                                                             ++
++  Programmer: Nikolaus Brennig, (virtualnik@nol.at)									     ++
++     Version: 1.00                                                                         ++
++                                                                                           ++
\*-------------------------------------------------------------------------------------------*/

#undef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <windowsx.h>
#include <ddraw.h>
#include <stdio.h>
#include "DDLoadPCX.h"
#include <d3d.h>
#include "texture.h"
//---------------------------------------------------------------------------------------------
// The PCXHeader struct...
//---------------------------------------------------------------------------------------------
typedef struct _PcxHeader
{
	BYTE  Identifier;       /* Always 0x0A           */
    BYTE  Version;          /* Version Number        */
    BYTE  Encoding;         /* Encoding Format       */
    BYTE  BitsPerPixel;     /* Bits per Pixel        */

    WORD  XStart;           /* Left of Image         */
    WORD  YStart;           /* Top of Image          */
    WORD  XEnd;             /* Right of Image        */
    WORD  YEnd;             /* Bottom of Image       */
    WORD  HorzRes;          /* Horizontal Resolution */
    WORD  VertRes;          /* Vertical Resolution   */

    BYTE  PALETTE[48];      /* 16 Color EGA Palette  */
    BYTE  Reserved1;        /* Reserved (Always 0)   */
    BYTE  NumBitPlanes;     /* Number of Bit Planes  */
    WORD  BytesPerLine;     /* Bytes per Scanline    */
    WORD  PaletteType;      /* Palette Type          */
    WORD  HorzScreenSize;   /* Horizontal ScreenSize */
    WORD  VertScreenSize;   /* Vertical ScreenSize   */
    BYTE  Reserved2[54];    /* Reserved (Always 0)   */
} PCXHEAD;


//---------------------------------------------------------------------------------------------
// Loads a 1-24bit PCXfile...
//---------------------------------------------------------------------------------------------
LPDIRECTDRAWSURFACE7 DDLoadPCX( LPDIRECTDRAW7 lpDD, LPSTR Filename, int dx, int dy )
{
	LPDIRECTDRAWSURFACE7	lpDDSTemp;
	DDSURFACEDESC2			ddsd;
	BYTE					*destbuffer;
	HDC						hdc;
	PCXHEAD					pcx;
	FILE					*file;
	PBITMAPINFO				bmpi;
	BYTE					MyPal[768];
	INT						i;
	INT						W, H, Bits;
    INT						WidthBytes, bytes;
    INT						c, Row, k, l, n, t;
    BYTE					*line, *templine;
	BYTE					masktable[8]   = {	0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01  };
	BYTE					bittable[8]	   = {	0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80  };
    BYTE					egapalette[48] = {	0x00, 0x00, 0x0E, 0x00, 0x52, 0x07, 0x2C, 0x00,
    											0x0E, 0x00, 0x00, 0x00, 0xF8, 0x01, 0x2C, 0x00,
    											0x85, 0x0F, 0x42, 0x00, 0x00, 0x21, 0x00, 0x00,
    											0x00, 0x00, 0x6A, 0x24, 0x9B, 0x49, 0xA1, 0x5E,
    											0x90, 0x5E, 0x18, 0x5E, 0x84, 0x14, 0xD9, 0x95,
    											0xA0, 0x14, 0x12, 0x00, 0x06, 0x00, 0x68, 0x1F  };

	
	// Open the file...
	file = fopen( Filename, "rb" ); 
	if( file == NULL ) 
	{
		return 0;
	}

	// Get the Header...
	if( fread( &pcx, sizeof(char), sizeof(PCXHEAD), file ) != sizeof(PCXHEAD) ) 
	{
		fclose(file);
		return 0;
	}

	// ID Check...
	if( pcx.Identifier != 0x0a )
	{
		fclose(file);
		return 0;
	}

	// Compute the Metrics of the Image...
	W		= (pcx.XEnd - pcx.XStart + 1);
	H		= (pcx.YEnd - pcx.YStart + 1);
	Bits	= pcx.BitsPerPixel * pcx.NumBitPlanes;
    bytes	= pcx.BytesPerLine * pcx.NumBitPlanes;

	// Get the width in Bytes, depends on the bitdepth of the image...
	WidthBytes = GetWidthBytes( &Bits, W );

	// Load the palette...
	if( Bits <= 8 )
	{
		// Zero down...
		ZeroMemory( &MyPal, 768 );
		// Now check which palettetype is required...
		switch( Bits )
		{
			case 1: // Black & White
				memcpy(MyPal, "\000\000\000\377\377\377", 6); 
				break;

			case 3: // Fixed EGA Palette
				memcpy(MyPal, egapalette, 48);				
				break;

			case 4: // Modified EGA
				memcpy(MyPal, pcx.PALETTE, 48);				
				break;

			case 8: // VGA Palette
				fseek( file, -769L, SEEK_END );
	    		if( fgetc(file) == 12 ) 
					fread( MyPal, sizeof(*MyPal), 768, file );
				break;
		}
	}

	// Alloc memory... 
	destbuffer	= new BYTE[max(bytes, WidthBytes) * H];
	if( destbuffer == 0 )
	{
		fclose(file);
		return 0;
	}
	line		= new BYTE[max((short)W, bytes)];
	templine	= new BYTE[max((short)W, bytes)];

	// decompress the picture into our Buffer (Row by Row)...
	fseek(file, 128L, SEEK_SET);
	INT MyRow = (H-1) * WidthBytes;
	for( Row = 0; Row < H; ++Row ) 
	{
		// First decompress a single scanline...
		n = 0;
    	do 
		{
			if( ferror(file) || feof(file) || n >= bytes ) break;
	   		c = fgetc(file);
	   		if( (c & 0xc0) == 0xc0 )  
			{
    			i = c & 0x3f;
    			c = fgetc(file);
    			while(i--) 
				{
					if( ferror(file) || feof(file) || n >= bytes ) break;
					line[n++]=c;
				}
    		}
    		else 
			{
				if( ferror(file) || feof(file) || n >= bytes ) break;
    			line[n++]=c;
    		}
    	} 
		while(n < bytes);

		// If one plane only, simply copy, and go to next scanline in the loop...
        if( pcx.NumBitPlanes == 1 ) 
		{
			memcpy( destbuffer + MyRow, line, WidthBytes );
	   	}

		// Oh, more than one planes, could be 24bit, or 4bit...
        if( pcx.NumBitPlanes > 1 && pcx.NumBitPlanes <= 4 ) 
		{
            t = 0;
          	n = bytes / pcx.NumBitPlanes;
          	if( Bits == 24 ) 
			{
          	   	for(i = 0; i < n; ++i)
          			for(k = 2; k > -1; k--) templine[t++] = line[i + n * k];
          	}
          	else 
			{
          		memset(templine, 0, WidthBytes);
	          	for(i = 0; i < n; ++i) 
				{
	          		for(k = 0; k < 8; k += 2) 
					{
	          			for(l = 0; l < pcx.NumBitPlanes; ++l) 
						{
	          				if(line[i + l * n] & masktable[k])
								templine[t] |= bittable[l + 4];
	          				if(line[i + l * n] & masktable[k + 1])
	          					templine[t] |= bittable[l];
	          			}
	          		    t++;
	          		}
	          	}
            }
	        memcpy( destbuffer + MyRow, templine, WidthBytes );
        }

		// yo, we finished that scanline. Let's go to the next one...
   		MyRow -= WidthBytes;
	}

	// Bitmapstruct init...
    bmpi = (PBITMAPINFO) LocalAlloc(LPTR, sizeof(BITMAPINFOHEADER) + (256 * sizeof(RGBQUAD)));
	bmpi->bmiHeader.biSize				= sizeof(BITMAPINFOHEADER);
	bmpi->bmiHeader.biWidth				= W;
	bmpi->bmiHeader.biHeight			= H;
	bmpi->bmiHeader.biPlanes			= 1;
	bmpi->bmiHeader.biCompression       = BI_RGB; 
	bmpi->bmiHeader.biBitCount			= Bits;
	bmpi->bmiHeader.biSizeImage			= 0;
	bmpi->bmiHeader.biClrUsed			= 0; 
	bmpi->bmiHeader.biClrImportant		= 0; 

	// Now assign the palette to our Bitmap for drawing...
    if( Bits <= 8 )
	{
		for( int j = 0; j <256; j++ )
		{
			int tmp = j*3;
			bmpi->bmiColors[j].rgbRed      = MyPal[tmp+0];
			bmpi->bmiColors[j].rgbGreen    = MyPal[tmp+1];
			bmpi->bmiColors[j].rgbBlue     = MyPal[tmp+2];
			bmpi->bmiColors[j].rgbReserved = 0;
		}
	}

	// Does the caller want to scale the image at loadtime?
    int DestW = ((dx == 0) ? W : dx);
	int DestH = ((dy == 0) ? H : dy);

	// Now create the DirectDrawSurface...
    ZeroMemory(&ddsd, sizeof(ddsd));
    ddsd.dwSize			= sizeof(ddsd);
    ddsd.dwFlags		= DDSD_CAPS | DDSD_HEIGHT |DDSD_WIDTH;
    ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
    ddsd.dwWidth		= DestW;
    ddsd.dwHeight		= DestH;
    if( FAILED(lpDD->CreateSurface(&ddsd, &lpDDSTemp, NULL)) )
	{
		fclose( file );
		delete [] destbuffer;
		delete [] line;
		delete [] templine;
		LocalFree((PBITMAPINFO)bmpi);
		return 0;
	}

	// blt the stuff to the surface...
    if( SUCCEEDED(lpDDSTemp->GetDC(&hdc)) )
    {
		StretchDIBits( hdc, 0, 0, DestW, DestH, 0, 0, W, H, destbuffer, bmpi, DIB_RGB_COLORS, SRCCOPY );
		lpDDSTemp->ReleaseDC(hdc);
    }

	// Free the stuff...
	fclose( file );
	delete [] destbuffer;
    delete [] line;
	delete [] templine;
	LocalFree((PBITMAPINFO)bmpi);

	// return the surface now...
	return lpDDSTemp;
}


//---------------------------------------------------------------------------------------------
// Computes and return 0s the appropiate Width in Bytes and the possible bitdepth...
//---------------------------------------------------------------------------------------------
INT GetWidthBytes( int *Bits, int PixelWidth )
{
	INT WidthBytes = 0;

	// Compute our ImageWidth in Bytes, depends on the Bitdepth...
	if(*Bits == 1) WidthBytes = ((PixelWidth + 7) >> 3);
	if(*Bits > 1 && *Bits <= 4) 
	{
		WidthBytes = ((PixelWidth + 7) >> 3) << 2;
		*Bits = 4;
	}
	if(*Bits > 4 && *Bits <= 8) 
	{
		WidthBytes = PixelWidth;
		*Bits = 8;
	}
	if(*Bits > 8) 
	{
		WidthBytes = PixelWidth * 3;
		*Bits = 24;
	}
    if(WidthBytes & 0x003) WidthBytes = (WidthBytes | 3) + 1;

	return WidthBytes;
}

////////////////////////////////////////
// D3D TEXTURE loading utility ////////
// added by keus : 30/06/2000  ///////
/////////////////////////////////////

static HRESULT CALLBACK TextureSearchCallback16( DDPIXELFORMAT* pddpf,
                                               VOID* param )
{
    // Note: Return with DDENUMRET_OK to continue enumerating more formats.

    // Skip any funky modes
    if( pddpf->dwFlags & (DDPF_LUMINANCE|DDPF_BUMPLUMINANCE|DDPF_BUMPDUDV) )
        return DDENUMRET_OK;
    
    // Skip any FourCC formats
    if( pddpf->dwFourCC != 0 )
        return DDENUMRET_OK;

    // Skip alpha modes
    if( pddpf->dwFlags&DDPF_ALPHAPIXELS )
        return DDENUMRET_OK;

    // We only want 16-bit formats, so skip all others
    if( pddpf->dwRGBBitCount != 16 )
        return DDENUMRET_OK;

    // We found a good match. Copy the current pixel format to our output
    // parameter
    memcpy( (DDPIXELFORMAT*)param, pddpf, sizeof(DDPIXELFORMAT) );

    // Return with DDENUMRET_CANCEL to end enumeration.
    return DDENUMRET_CANCEL;
}


static LPDIRECTDRAWSURFACE7 CreateSurfaceTexture( LPDIRECT3DDEVICE7 pd3dDevice,
												DWORD dwWidth, 
												DWORD dwHeight)
{
    LPDIRECTDRAWSURFACE7 pddsTexture;
    HRESULT hr;

    // Get the device caps so we can check if the device has any constraints
    // when using textures
    D3DDEVICEDESC7 ddDesc;
    if( FAILED( pd3dDevice->GetCaps( &ddDesc ) ) )
        return NULL;

    // Get the bitmap structure (to extract width, height, and bpp)
//    BITMAP bm;
//  GetObject( hbm, sizeof(BITMAP), &bm );
//    DWORD dwWidth  = (DWORD)bm.bmWidth;
//    DWORD dwHeight = (DWORD)bm.bmHeight;

    // Setup the new surface desc for the texture. Note how we are using the
    // texture manage attribute, so Direct3D does alot of dirty work for us
    DDSURFACEDESC2 ddsd;
    ZeroMemory( &ddsd, sizeof(DDSURFACEDESC2) );
    ddsd.dwSize          = sizeof(DDSURFACEDESC2);
    ddsd.dwFlags         = DDSD_CAPS|DDSD_HEIGHT|DDSD_WIDTH|
                           DDSD_PIXELFORMAT|DDSD_TEXTURESTAGE;
    ddsd.ddsCaps.dwCaps  = DDSCAPS_TEXTURE;
    ddsd.dwWidth         = dwWidth;
    ddsd.dwHeight        = dwHeight;

    // Turn on texture management for hardware devices
    if( ddDesc.deviceGUID == IID_IDirect3DHALDevice )
        ddsd.ddsCaps.dwCaps2 = DDSCAPS2_TEXTUREMANAGE;
    else if( ddDesc.deviceGUID == IID_IDirect3DTnLHalDevice )
        ddsd.ddsCaps.dwCaps2 = DDSCAPS2_TEXTUREMANAGE;
    else
        ddsd.ddsCaps.dwCaps |= DDSCAPS_SYSTEMMEMORY;

    // Adjust width and height, if the driver requires it
    if( ddDesc.dpcTriCaps.dwTextureCaps & D3DPTEXTURECAPS_POW2 )
    {
        for( ddsd.dwWidth=1;  dwWidth>ddsd.dwWidth;   ddsd.dwWidth<<=1 );
        for( ddsd.dwHeight=1; dwHeight>ddsd.dwHeight; ddsd.dwHeight<<=1 );
    }
    if( ddDesc.dpcTriCaps.dwTextureCaps & D3DPTEXTURECAPS_SQUAREONLY )
    {
        if( ddsd.dwWidth > ddsd.dwHeight ) ddsd.dwHeight = ddsd.dwWidth;
        else                               ddsd.dwWidth  = ddsd.dwHeight;
    }

	// enumerate texture caps,first try 32bits texture
	// then 16bits if failed
	//pd3dDevice->EnumTextureFormats( TextureSearchCallback32, &ddsd.ddpfPixelFormat );
	//if( 0L == ddsd.ddpfPixelFormat.dwRGBBitCount )
	//{
		pd3dDevice->EnumTextureFormats( TextureSearchCallback16, &ddsd.ddpfPixelFormat );
		if( 0L == ddsd.ddpfPixelFormat.dwRGBBitCount )
			return NULL;
	//}

    // Get the device's render target, so we can then use the render target to
    // get a ptr to a DDraw object. We need the DirectDraw interface for
    // creating surfaces.
    LPDIRECTDRAWSURFACE7 pddsRender;
    LPDIRECTDRAW7        pDD;
    pd3dDevice->GetRenderTarget( &pddsRender );
    pddsRender->GetDDInterface( (VOID**)&pDD );
    pddsRender->Release();

    // Create a new surface for the texture
    if( FAILED( hr = pDD->CreateSurface( &ddsd, &pddsTexture, NULL ) ) )
    {
        pDD->Release();
        return NULL;
    }

    // Done with DDraw
    pDD->Release();

    // Now, copy the bitmap to the texture surface. To do this, we are creating
    // a DC for the bitmap and a DC for the surface, so we can use the BitBlt()
    // call to copy the actual bits.

    // Get a DC for the bitmap
    HDC hdcBitmap = CreateCompatibleDC( NULL );
    if( NULL == hdcBitmap )
    {
        pddsTexture->Release();
        return NULL;
    }
    /*SelectObject( hdcBitmap, hbm );

    // Get a DC for the surface
    HDC hdcTexture;
    if( SUCCEEDED( pddsTexture->GetDC( &hdcTexture ) ) )
    {
        // Copy the bitmap image to the surface.
        BitBlt( hdcTexture, 0, 0, bm.bmWidth, bm.bmHeight, hdcBitmap,
                0, 0, SRCCOPY );
        pddsTexture->ReleaseDC( hdcTexture );
    }*/
    DeleteDC( hdcBitmap );

    // Return the newly created texture
    return pddsTexture;
}


LPDIRECTDRAWSURFACE7 LoadPCXTexture( LPDIRECT3DDEVICE7 pd3dDevice, LPSTR Filename )
{
	LPDIRECTDRAWSURFACE7 lpSurfTemp = NULL;
	LPDIRECTDRAWSURFACE7 lpTexture  = NULL;
	LPDIRECTDRAWSURFACE7 pddsRender = NULL;
	LPDIRECTDRAW7        lpDD       = NULL;
	DWORD                dwWidth;
	DWORD                dwHeight;

    pd3dDevice->GetRenderTarget( &pddsRender );
    pddsRender->GetDDInterface( (VOID**)&lpDD );
    pddsRender->Release();
	lpSurfTemp = DDLoadPCX( lpDD, Filename,0 ,0 );
	lpDD->Release();

	DDSURFACEDESC2 ddsd;
    ZeroMemory( &ddsd, sizeof(DDSURFACEDESC2) );
    ddsd.dwSize = sizeof(DDSURFACEDESC2);
	lpSurfTemp->GetSurfaceDesc(&ddsd);

    D3DDEVICEDESC7 ddDesc;
    if( FAILED( pd3dDevice->GetCaps( &ddDesc ) ) )
        return NULL;
	// check texture size 
	if (ddsd.dwWidth > ddDesc.dwMaxTextureWidth )
		dwWidth = ddDesc.dwMaxTextureWidth;
	else dwWidth = ddsd.dwWidth;

	if (ddsd.dwHeight > ddDesc.dwMaxTextureHeight )
		dwHeight = ddDesc.dwMaxTextureHeight;
	else dwHeight = ddsd.dwHeight;

	// Create the D3D Texture
	lpTexture = CreateSurfaceTexture(pd3dDevice,dwWidth,dwHeight);

	lpTexture->Blt(NULL,lpSurfTemp,NULL,DDBLT_WAIT ,NULL);
       
	lpSurfTemp->Release();

	return lpTexture;
}