// This program calculates a convolution between a sound (*.WAV) and
// the two channels of a stereophonic impulse response. It is most commonly
// used as a brute force solution to produce reverberation.
// The calculation work is very heavy and you should expect long
// execution times even with a powerful processor. Please note that
// the calculations are done in RAM, using a very large buffer
// containing the whole input sound file. (64 megabytes hold
// approximately 10 minutes of monophonic sound).

// The sound and the impulse response inputs are in different formats.
// The output is temporarily written as floating-point values
// and normalized afterwards to integers.

#include <stdio.h>
#include <io.h>
#include <memory.h>
#include <math.h>


#define MAXSNDLEN 0x04000000
#define MAXIRLEN  0x00100000


short sb  [MAXSNDLEN];   // Sound buffer

long  iri [MAXIRLEN];    // Impulse response index (sound sample number)
long  irv [MAXIRLEN];    // Impulse response amplitude
char  irc [MAXIRLEN];    // Impulse response channel selector



int main (int argc, char *argv []) {

  double vl,         // Left channel sample value
         vr,         // Right channel sample value
         vmax;       // Highest sample value found
  FILE   *src,       // Input file
         *ir,        // Impulse response file
         *tgt;       // Output file
  long   i,
         j,
         srclen,     // Number of samples in the source file
         irlen,      // Number of samples in the impulse response file
         lv;         // A temporary value
  short  iv;         // A temporary value
  char   hdr [44];   // A WAV file header

  if (argc < 4) {
    printf ("Use: convolve <sourcefile> <irfile> <outfile>\n");
    return (1); }

  printf ("Loading source sound... "); fflush (stdout);
  src = fopen (argv [1], "rb");
  if (src == NULL) {
    printf ("Cannot open source file '%s'\n", argv [1]);
    return (2); }
  fread (hdr, 40, 1, src);
  fread (&srclen, sizeof (long), 1, src);
  srclen /= sizeof (short);
  for (i = 0 ; i < srclen ; i++) fread (sb + i, sizeof (short), 1, src);
  fclose (src);
  printf ("Done\n");

  printf ("Loading impulse response... "); fflush (stdout);
  ir = fopen (argv [2], "r");
  if (ir == NULL) {
    printf ("Cannot open impulse response file '%s'\n", argv [2]);
    return (3); }
  i = 0L;
  vmax = 0.0;
  while (1) {
    fscanf (ir, "%c %ld %ld\n", irc + i, iri + i, irv + i);
    if ((irc [i] | 0x20) == 'e') break;
    if (irlen < iri [i]) irlen = iri [i];
    i++; }
  fclose (ir);
  printf ("Done\n");

  tgt = fopen ("convolve.tmp", "wb");
  if (tgt == NULL) {
    printf ("Cannot open temporary output file\n");
    fclose (ir);
    fclose (src);
    return (4); }

  printf ("Calculating convolution\n");
  vmax = 0.0;
  for (i = 0 ; i < srclen  + irlen ; i++) {
    long  k, srcv, si;
    if (((srclen - i) & 0x000003ff) == 0L)
      printf ("%6ld kB to go\n",
              (unsigned long) (srclen + irlen - i) >> (unsigned long) 10L);
    vl = 0.0; vr = 0.0;
    for (j = 0 ; irc [j] != 'E' ; j++) {
      si = i - iri [j];
      if ((si < 0L) || (si >= srclen)) continue;
      srcv = (long) (sb [si]);
      if (irc [j] == 'L') vl += (double) (srcv * irv [j]);
      if (irc [j] == 'R') vr += (double) (srcv * irv [j]); }
    if (vmax < fabs (vl)) vmax = fabs (vl);
    if (vmax < fabs (vr)) vmax = fabs (vr);
    fwrite (&vl, sizeof (double), 1, tgt);
    fwrite (&vr, sizeof (double), 1, tgt); }

  fclose (tgt);

  src = fopen ("convolve.tmp", "rb");
  if (src == NULL) {
    printf ("Cannot open temporary input file\n");
    return (10); }

  tgt = fopen (argv [3], "wb");
  if (tgt == NULL) {
    printf ("Cannot open output file '%s'\n", argv [3]);
    fclose (src);
    return (11); }

  lv = (srclen + irlen) * 2L * sizeof (short) + 36L;
  memcpy (hdr + 4, &lv, sizeof (long));
  iv = 2;
  memcpy (hdr + 22, &iv, sizeof (short));
  memcpy (&lv, hdr + 28, sizeof (long));
  lv *= 2L;
  memcpy (hdr + 28, &lv, sizeof (long));
  iv = 4;
  memcpy (hdr + 32, &iv, sizeof (short));
  lv = (srclen + irlen) * 2L * sizeof (short);
  memcpy (hdr + 40, &lv, sizeof (long));

  fwrite (hdr, 44, 1, tgt);

  printf ("Normalizing from %lf\n", vmax);

  for (i = 0 ; i < 2L * (srclen + irlen) ; i++) {
    fread (&vl, sizeof (double), 1, src);
    iv = (short) (32767.0 * vl / vmax);
    fwrite (&iv, sizeof (short), 1, tgt); }

  fclose (tgt);
  fclose (src);

  remove ("convolve.tmp");

  return (0); }
