///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <ThCore.h>
#include <ThMemory.h>
#include <ThString.h>
#include <ThStream.h>
#include <ThChunk.h>

///////////////////////////////////////////////////////////////////////////////////////////////////

#pragma pack ( push, 1 )

struct ThChunk::Header
{
// data
	unsigned int										size;
	unsigned int										type;
	unsigned int										count;
	struct {
		unsigned int									name : 1;
	} flags;
};

#pragma pack ( pop )

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThChunk constructors ---------------------------------------------------------------------------

ThChunk::ThChunk(ThChunk* parent)
{
	if (parent)
	{
		// attached last to preserve order from file
		parent->m_children.attachLast(this);
	}

	m_type = THCHUNK_UNKNOWN;
	m_hash = 0;

	// NOTE: these operations are guaranteed.
	m_stream = CreateDynamicBlockStream();
	m_block  = dynamic_cast<IThDynamicBlock*>(m_stream.getObject());
}

ThChunk::~ThChunk(void)
{
}

// ThChunk methods --------------------------------------------------------------------------------

//! \todo Some kind of checksum thingy, or something, for validating input data.

/*!	Loads a chunk hierarchy from the specified stream.
 *	\param stream [in] The ThInputStream interface to load the hierarchy from.
 *	\return \c true if successful.
 */
bool ThChunk::load(IThInputStream* stream)
{
	release();

	Header header;

	// load header
	if (!stream->readItem(header))
	{
		release();
		return false;
	}

	// load name
	if (header.flags.name)
	{
		if (!stream->readString(m_name))
		{
			release();
			return false;
		}

		m_hash = m_name.hashNoCase();
	}

	// load data
	if (header.size)
	{
		m_block->allocate(header.size);

		const unsigned int size = stream->read(m_block->lock(), header.size);

		m_block->unlock();

		// fail on incomplete read, as it means the input data is corrupt.
		if (size < header.size)
		{
			release();
			return false;
		}
	}

	// load child chunks.
	for (unsigned int count = header.count;  count;  count--)
	{
		ThChunk* child = new ThChunk(this);

		if (!child->load(stream))
		{
			release();
			return false;
		}
	}

	return true;
}

/*!	Saves a chunk hierarchy to the specified stream.
 *	\param stream [in/out] The ThOutputStream interface to save the hierarchy to.
 *	\return \c true if successful, or \c false if an error occurred.
 */
bool ThChunk::save(IThOutputStream* stream)
{
	Header header;

	header.type  = m_type;
	header.size  = m_stream->getSize();
	header.count = m_children.getCount();

	if (m_name)
		header.flags.name = 1;
	else
		header.flags.name = 0;

	// save header.
	if (!stream->writeItem(header))
		return false;

	if (m_name)
	{
		// save name.
		if (!stream->writeString(m_name))
			return false;
	}

	// save data.
	if (header.size)
	{
		const unsigned int size = stream->write(m_block->lock(), header.size);

		m_block->unlock();

		if (size < header.size)
			return false;
	}

	// save child chunks.
	for (ThIterator<ThChunk> child(m_children);  child;  child.next())
	{
		if (!child->save(stream))
			return false;
	}

	return true;
}

/*!	Resets the chunk to it's original, empty state.
 *	All children and data are destroyed.
 */
void ThChunk::release(void)
{
	m_type = THCHUNK_UNKNOWN;
	m_name = NULL;
	m_hash = 0;

	m_block->release();

	m_children.release();
}

// ThChunk attributes -----------------------------------------------------------------------------

/*!	\return The type ID of this chunk.
 */
unsigned int ThChunk::getType(void) const
{
	return m_type;
}

/*!	Sets the type ID of this chunk.
 */
void ThChunk::setType(unsigned int type)
{
	m_type = type;
}

/*! \return The name of the chunk, or \c NULL if unnamed.
 */
const char* ThChunk::getName(void) const
{
	return m_name;
}

/*!	Sets the name of this chunk. Set to \c NULL to remove name.
 */
void ThChunk::setName(const char* name)
{
	m_name = name;
	m_hash = m_name.hashNoCase();
}

/*!	\return The hash of the name of this chunk, or zero if unnamed.
 */
unsigned int ThChunk::getHash(void) const
{
	return m_hash;
}

/*!	\return The first child to this chunk, or \c NULL if no children exist.
 */
ThChunk* ThChunk::getFirstChild(void)
{
	return m_children.getFirst();
}

/*! \return An IThDynamicBlock interface to the chunk's data.
 */
IThDynamicBlock* ThChunk::getBlock(void)
{
	return m_block;
}

/*! \return An IThStream interface to the chunk's data.
 */
IThStream* ThChunk::getStream(void)
{
	return m_stream;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
