///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <ThCore.h>
#include <ThMemory.h>
#include <ThString.h>
#include <ThStream.h>
#include <ThStorage.h>
#include <ThWindow.h>
#include <ThError.h>
#include <ThServer.h>
#include <ThDisplay.h>

#include "ThDisplay.h"

#include <png.h>

///////////////////////////////////////////////////////////////////////////////////////////////////

#define THDISPLAYWINDOWCLASSNAME "Theresa Display Window"

///////////////////////////////////////////////////////////////////////////////////////////////////

static void pngError(png_structp pngFile, png_const_charp error)
{
	Error->display("Display", error);
}

static void pngWarning(png_structp pngFile, png_const_charp warning)
{
	Error->write("Display", warning);
}

static void pngReadStream(png_structp pngFile, png_bytep data, png_size_t length)
{
	IThStream* stream = reinterpret_cast<IThStream*>(png_get_io_ptr(pngFile));

	stream->readItems(data, length);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThDisplay constructors ------------------------------------------------------------------------

IThDisplay::~IThDisplay(void)
{
}

// IThDisplay static methods ----------------------------------------------------------------------

IThDisplay* IThDisplay::createInstance(unsigned int width, unsigned int height)
{
	ThPtr<ThDisplay> display = new ThDisplay();

	if (!display->open(width, height))
		return NULL;

	return display.detach();
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThDisplayWindow constructors -------------------------------------------------------------------

ThDisplayWindow::ThDisplayWindow(void)
{
	m_width  = 0;
	m_height = 0;
}

// ThDisplayWindow methods ------------------------------------------------------------------------

bool ThDisplayWindow::create(unsigned int width, unsigned int height, bool fullscreen)
{
	WNDCLASSEX wcl;

	ZeroMemory(&wcl, sizeof(wcl));
	wcl.cbSize = sizeof(wcl);
	wcl.style         = CS_OWNDC;
	wcl.hCursor       = LoadCursor(NULL, IDC_ARROW);
	wcl.hbrBackground = (HBRUSH) COLOR_WINDOW + 1;
	wcl.lpszClassName = THDISPLAYWINDOWCLASSNAME;

	DWORD style   = WS_CLIPCHILDREN | WS_CLIPSIBLINGS;
	DWORD exStyle = 0;

	if (fullscreen)
	{
		style   |= WS_POPUP;
		exStyle |= WS_EX_TOPMOST;
	}
	else
	{
		style   |= WS_OVERLAPPEDWINDOW;
		exStyle |= WS_EX_APPWINDOW;
	}

	if (!ThWindow::create(wcl, style, exStyle))
		return false;

	RECT area;

	SetRect(&area, 0, 0, width, height);

	if (fullscreen)
	{
		ShowCursor(FALSE);
	}
	else
	{
		RECT workArea;

		SystemParametersInfo(SPI_GETWORKAREA, 0, &workArea, 0);
		OffsetRect(&area, workArea.left + (workArea.right - workArea.left - (int) width) / 2, workArea.top + (workArea.bottom - workArea.top - (int) height) / 2);
	}

	AdjustWindowRectEx(&area, style, FALSE, exStyle);
	MoveWindow(getHandle(), area.left, area.top, area.right - area.left, area.bottom - area.top, TRUE);

	return true;
}

// ThDisplayWindow attributes ---------------------------------------------------------------------

unsigned int ThDisplayWindow::getWidth(void)
{
	return m_width;
}

unsigned int ThDisplayWindow::getHeight(void)
{
	return m_height;
}

// ThDisplayWindow callbacks ----------------------------------------------------------------------

LRESULT ThDisplayWindow::message(UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message)
	{
		case WM_CLOSE:
		{
			Server->postMessage(THMSG_REQUEST_EXIT, THID_ANNOUNCE, THID_DISPLAY);

			break;
		}

		case WM_SYSCOMMAND:
		{
			switch (wParam)
			{
				case SC_SCREENSAVE:
				case SC_MONITORPOWER:
				{
					return 0;
				}
			}

			break;
		}

		case WM_KEYDOWN:
		{
			if (wParam == VK_ESCAPE)
				PostMessage(getHandle(), WM_CLOSE, 0, 0);

			break;
		}

		case WM_SIZE:
		{
			m_width  = LOWORD(lParam);
			m_height = HIWORD(lParam);

			glViewport(0, 0, m_width, m_height);

			break;
		}
	}

	return ThWindow::message(message, wParam, lParam);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThDisplay constructors -------------------------------------------------------------------------

ThDisplay::ThDisplay(void):
	ThServerObject(THID_DISPLAY)
{
	m_fullscreen = false;
	m_dc         = NULL;
	m_rc         = NULL;
}

ThDisplay::~ThDisplay(void)
{
	close();
}

// ThDisplay methods ------------------------------------------------------------------------------

bool ThDisplay::open(unsigned int width, unsigned int height)
{
	close();

	int result = MessageBox(NULL, "Show in fullscreen mode?", "Theresa", MB_YESNOCANCEL | MB_ICONQUESTION);

	if (result == IDCANCEL)
		return false;

	if (result == IDYES)
	{
		DEVMODE dm;

		ZeroMemory(&dm, sizeof(dm));
		dm.dmSize = sizeof(dm);
		dm.dmFields     = DM_PELSWIDTH | DM_PELSHEIGHT | DM_BITSPERPEL;
		dm.dmPelsWidth  = width;
		dm.dmPelsHeight = height;

		if (ChangeDisplaySettings(&dm, 0) != DISP_CHANGE_SUCCESSFUL)
			return false;

		m_fullscreen = true;
	}

	m_window = new ThDisplayWindow();

	if (!m_window->create(width, height, m_fullscreen))
	{
		THERROR(("Display", "Unable to create window."));
		return false;
	}

	setTitle("Untitled");
	m_window->show();

	m_dc = GetDC(m_window->getHandle());
	if (!m_dc)
	{
		THERROR(("Display", "Unable to retrieve window device context."));
		return false;
	}

	PIXELFORMATDESCRIPTOR pfd;

	ZeroMemory(&pfd, sizeof(pfd));
	pfd.nSize = sizeof(pfd);
	pfd.nVersion   = 1;
	pfd.dwFlags    = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
	pfd.iPixelType = PFD_TYPE_RGBA;
	pfd.cColorBits = 32;
	pfd.cDepthBits = 16;
	pfd.iLayerType = PFD_MAIN_PLANE;

	const int pixelFormat = ChoosePixelFormat(m_dc, &pfd);
	if (!pixelFormat)
	{
		THERROR(("Display", "Unable to choose pixel format."));
		return false;
	}

	if (!SetPixelFormat(m_dc, pixelFormat, &pfd))
	{
		THERROR(("Display", "Unable to set pixel format."));
		return false;
	}

	m_rc = wglCreateContext(m_dc);
	if (!m_rc)
	{
		THERROR(("Display", "Unable to create rendering context."));
		return false;
	}

	if (!wglMakeCurrent(m_dc, m_rc))
	{
		THERROR(("Display", "Unable to select rendering context."));
		return false;
	}

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

	return true;
}

void ThDisplay::close(void)
{
	if (m_fullscreen)
	{
		ChangeDisplaySettings(NULL, 0);
		m_fullscreen = false;
	}

	if (m_rc)
	{
		wglMakeCurrent(NULL, NULL);

		wglDeleteContext(m_rc);
		m_rc = NULL;
	}

	if (m_dc)
	{
		ReleaseDC(m_window->getHandle(), m_dc);
		m_dc = NULL;
	}

	m_window.release();
}

// ThDisplay interface methods --------------------------------------------------------------------

void ThDisplay::update(void)
{
	glFinish();

	if (unsigned int error = glGetError())
	{
		if (!Error->request("Display", (const char*) gluErrorString(error)))
			postMessage(THMSG_REQUEST_EXIT, THID_ANNOUNCE);
	}

	SwapBuffers(m_dc);
}

// ThDisplay interface texture methods ------------------------------------------------------------

bool ThDisplay::loadTexture(const char* fileName, bool mipmap)
{
	ThPtr<IThStream> file = Storage->openFile(fileName);
	if (!file)
		return false;

	return loadTexture(file, mipmap);
}

bool ThDisplay::loadTexture(IThInputStream* stream, bool mipmap)
{
	unsigned char header[8];

	stream->readItems(header, 8);

	if (png_sig_cmp(header, 0, 8))
		return false;

  png_structp pngFile = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, pngError, pngWarning);
	if (!pngFile)
		return false;

  png_set_read_fn(pngFile, stream, pngReadStream);

  png_infop pngInfo = png_create_info_struct(pngFile);
	if (!pngInfo)
	{
		png_destroy_read_struct(&pngFile, NULL, NULL);
		return false;
	}

  png_infop pngEndInfo = png_create_info_struct(pngFile);
	if (!pngEndInfo)
	{
		png_destroy_read_struct(&pngFile, &pngInfo, NULL);
		return false;
	}

  png_set_sig_bytes(pngFile, 8);

  png_read_png(pngFile, pngInfo, PNG_TRANSFORM_IDENTITY, NULL);

	ThByteBlock buffer;

	const unsigned int width  = png_get_image_width(pngFile, pngInfo);
	const unsigned int height = png_get_image_height(pngFile, pngInfo);
	const unsigned int size   = png_get_rowbytes(pngFile, pngInfo);

	buffer.allocate(height * size);

	png_bytepp rows;

	rows = png_get_rows(pngFile, pngInfo);

	for (unsigned int i = 0;  i < height;  i++)
		memcpy(buffer + i * size, rows[i], size);

	unsigned int format;
	unsigned int channels;

	switch (png_get_color_type(pngFile, pngInfo))
	{
		case PNG_COLOR_TYPE_GRAY:
			format = GL_LUMINANCE;
			channels = 1;
			break;
		case PNG_COLOR_TYPE_GRAY_ALPHA:
			format = GL_LUMINANCE_ALPHA;
			channels = 2;
			break;
		case PNG_COLOR_TYPE_RGB:
			format = GL_RGB;
			channels = 3;
			break;
		case PNG_COLOR_TYPE_RGB_ALPHA:
			format = GL_RGBA;
			channels = 4;
			break;
		default:
			return false;
	}

	unsigned int type;

	switch (png_get_bit_depth(pngFile, pngInfo))
	{
		case 8:
			type = GL_UNSIGNED_BYTE;
			break;
		case 16:
			type = GL_UNSIGNED_SHORT;
			break;
		default:
			return false;
	}

	png_destroy_read_struct(&pngFile, &pngInfo, &pngEndInfo);

	if (mipmap)
		gluBuild2DMipmaps(GL_TEXTURE_2D, channels, width, height, format, type, buffer);
	else
		glTexImage2D(GL_TEXTURE_2D, 0, channels, width, height, 0, format, type, buffer);

	if (const unsigned int error = glGetError())
	{
		if (!Error->request("Display", (const char*) gluErrorString(error)))
			postMessage(THMSG_REQUEST_EXIT, THID_ANNOUNCE);

		return false;
	}

	return true;
}

// ThDisplay interface attributes -----------------------------------------------------------------

unsigned int ThDisplay::getWidth(void)
{
	return m_window->getWidth();
}

unsigned int ThDisplay::getHeight(void)
{
	return m_window->getHeight();
}

const char* ThDisplay::getTitle(void)
{
	return m_title;
}

void ThDisplay::setTitle(const char* title)
{
	m_title = title;

	ThString caption;

	caption = title;
	caption.append(" - Theresa");

	m_window->setTitle(caption);
}

// ThDisplay callbacks ----------------------------------------------------------------------------

bool ThDisplay::recieve(ThMessage* message)
{
	return ThServerObject::recieve(message);
}

///////////////////////////////////////////////////////////////////////////////////////////////////
