///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <ThCore.h>
#include <ThMemory.h>
#include <ThServer.h>
#include <ThSystem.h>
#include <ThEffect.h>

#include "ThEffect.h"

///////////////////////////////////////////////////////////////////////////////////////////////////

ThSingleton<IThEffect> Effect;

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThEffectObject constructors --------------------------------------------------------------------

ThEffectObject::ThEffectObject(void):
	m_lifeTime(0.f),
	m_timeLeft(0.f)
{
}

ThEffectObject::~ThEffectObject(void)
{
}

// ThEffectObject callbacks -----------------------------------------------------------------------

bool ThEffectObject::update(float deltaTime)
{
	if (m_lifeTime)
	{
		m_timeLeft -= deltaTime;
		if (m_timeLeft <= 0.f)
			return false;
	}

	return true;
}

bool ThEffectObject::render(void)
{
	return true;
}

// ThEffectObject attributes ----------------------------------------------------------------------

float ThEffectObject::getLifeTime(void) const
{
	return m_lifeTime;
}

void ThEffectObject::setLifeTime(float lifeTime)
{
	m_lifeTime = lifeTime;
	m_timeLeft = lifeTime;
}

float ThEffectObject::getTimeLeft(void) const
{
	return m_timeLeft;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThEffect constructors -------------------------------------------------------------------------

IThEffect::~IThEffect(void)
{
}

// IThEffect static methods -----------------------------------------------------------------------

bool IThEffect::create(void)
{
	if (Effect)
		return true;

	ThPtr<ThEffect> object = new ThEffect();

	if (!object->open())
		return false;

	Effect.attach(object.detach());
	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThEffect constructors --------------------------------------------------------------------------

ThEffect::ThEffect(void):
	ThServerObject(THID_EFFECT)
{
}

ThEffect::~ThEffect(void)
{
	close();
}

// ThEffect methods -------------------------------------------------------------------------------

bool ThEffect::open(void)
{
	close();

	m_layers.attachLast(new Layer(THLAYER_BACKGROUND));
	m_layers.attachLast(new Layer(THLAYER_NORMAL));
	m_layers.attachLast(new Layer(THLAYER_FOREGROUND));
	m_layers.attachLast(new Layer(THLAYER_TOPMOST));

	if (!Server->registerObject(this))
		return false;

	return true;
}

void ThEffect::close(void)
{
	m_layers.release();
}

// ThEffect interface methods ---------------------------------------------------------------------

bool ThEffect::registerObject(ThEffectObject* object, unsigned int layerID)
{
	Layer* layer = findLayer(layerID);
	if (!layer)
		return false;

	layer->m_objects.attachFirst(object);
	return true;
}

// ThEffect methods -------------------------------------------------------------------------------

ThEffect::Layer* ThEffect::findLayer(unsigned int id)
{
	for (ThIterator<Layer> layer(m_layers);  layer;  layer.next())
	{
		if (layer->m_id == id)
			return layer;
	}

	return NULL;
}

// ThEffect callbacks -----------------------------------------------------------------------------

bool ThEffect::recieve(ThMessage* message)
{
	switch (message->getMessage())
	{
		case THMSG_UPDATE:
		{
			const float deltaTime = System->getDeltaTime();

			for (ThIterator<Layer> layer(m_layers);  layer;  layer.next())
			{
				for (ThIterator<ThEffectObject> object(layer->m_objects);  object; )
				{
					if (object->update(deltaTime))
						object.next();
					else
						object.release();
				}
			}

			break;
		}

		case THMSG_RENDER:
		{
			for (ThIterator<Layer> layer(m_layers);  layer;  layer.next())
			{
				for (ThIterator<ThEffectObject> object(layer->m_objects);  object; )
				{
					if (object->render())
						object.next();
					else
						object.release();
				}
			}

			break;
		}
	}

	return ThServerObject::recieve(message);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThEffect::Layer constructors -------------------------------------------------------------------

ThEffect::Layer::Layer(unsigned int id):
	m_id(id)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////
