///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <ThCore.h>
#include <ThMemory.h>
#include <ThString.h>
#include <ThStream.h>

#include "ThStream.h"

#include <cstring>
#include <cstdlib>

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThInputStream constructors --------------------------------------------------------------------

IThInputStream::~IThInputStream(void)
{
}

// IThInputStream helper methods ------------------------------------------------------------------

bool IThInputStream::readString(ThString& string)
{
	// cache position
	const unsigned int position = getPosition();

	unsigned int size = 0;

	// peek ahead to find string size

	for (;;)
	{
		char c;

		// fail and restore position on read failure before null character
		if (!readItem(c))
			return false;

		size++;

		// stop if null character found
		if (c == '\0')
			break;
	}

	// reset position
	if (!seek(position))
		return false;

	// create buffer to fit string
	if (string.getSize() < size)
		string.allocate(size);

	// fail and restore position on read failure
	if (!readItems(string.getData(), size))
		return false;

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThOutputStream constructors -------------------------------------------------------------------

IThOutputStream::~IThOutputStream(void)
{
}

// IThOutputStream helper methods -----------------------------------------------------------------

bool IThOutputStream::writeString(const char* string)
{
	const unsigned int size = ThString::length(string) + 1;

	if (!writeItems(string, size))
		return false;

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThStream constructors -------------------------------------------------------------------------

IThStream::~IThStream(void)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////

IThStream* CreateDynamicBlockStream(unsigned int size)
{
	return new ThDynamicBlockStream(size);
}

IThStream* CreateAggregateStream(IThStream* inner)
{
	return new ThAggregateStream(inner);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThDynamicBlockStream constructors --------------------------------------------------------------

ThDynamicBlockStream::ThDynamicBlockStream(unsigned int size):
	m_locks(0),
	m_position(0)
{
	m_data.allocate(size);
	m_size = m_data.getCount();
}

ThDynamicBlockStream::~ThDynamicBlockStream(void)
{
	THASSERT(m_locks == 0, "Dynamic must be completely unlocked before descruction.");
}

// ThDynamicBlockStream methods -------------------------------------------------------------------

unsigned int ThDynamicBlockStream::read(void* data, unsigned int size)
{
	if (isEOF())
		return 0;

	if (m_position + size >= m_size)
		size = m_size - m_position - 1;

	memcpy(data, m_data + m_position, size);

	m_position += size;
	return size;
}

unsigned int ThDynamicBlockStream::write(const void* data, unsigned int size)
{
	if (m_position + size >= m_data.getCount())
	{
		m_data.resize(m_position + size);
		m_size = m_data.getCount();
	}

	memcpy(m_data + m_position, data, size);

	m_position += size;
	return size;
}

unsigned int ThDynamicBlockStream::skip(unsigned int size)
{
	if (isEOF())
		return 0;

	if (m_position + size >= m_size)
		size = m_size - m_position - 1;

	m_position += size;
	return size;
}

bool ThDynamicBlockStream::seek(unsigned int position)
{
	m_position = position;
	return true;
}

bool ThDynamicBlockStream::truncate(void)
{
	m_size = m_position;
	return true;
}

void* ThDynamicBlockStream::lock(void)
{
	m_locks++;

	return m_data;
}

void ThDynamicBlockStream::unlock(void)
{
	if (m_locks)
		m_locks--;
}

void ThDynamicBlockStream::allocate(unsigned int size)
{
	THASSERT(m_locks == 0, "Cannot allocate dynamic block stream while locked.");

	m_data.allocate(size);
	m_size = m_data.getCount();

	m_position = 0;
}

void ThDynamicBlockStream::resize(unsigned int size)
{
	THASSERT(m_locks == 0, "Cannot resize dynamic block stream while locked.");

	if (m_data.getCount() > size)
		m_size = size;

	m_data.resize(size);
}

void ThDynamicBlockStream::release(void)
{
	THASSERT(m_locks == 0, "Cannot release dynamic block stream while locked.");

	m_data.release();
	m_size = 0;

	m_position = 0;
}

// ThDynamicBlockStream attributes ----------------------------------------------------------------

bool ThDynamicBlockStream::isEOF(void) const
{
	if (m_position < m_size)
		return false;

	return true;
}

unsigned int ThDynamicBlockStream::getSize(void) const
{
	return m_size;
}

unsigned int ThDynamicBlockStream::getPosition(void) const
{
	return m_position;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThAggregateStream constructors -----------------------------------------------------------------

ThAggregateStream::ThAggregateStream(IThStream* inner):
	m_inner(inner)
{
}

ThAggregateStream::~ThAggregateStream(void)
{
}

// ThAggregateStream methods ----------------------------------------------------------------------

unsigned int ThAggregateStream::read(void* data, unsigned int size)
{
	return m_inner->read(data, size);
}

unsigned int ThAggregateStream::write(const void* data, unsigned int size)
{
	return m_inner->write(data, size);
}

unsigned int ThAggregateStream::skip(unsigned int size)
{
	return m_inner->skip(size);
}

bool ThAggregateStream::seek(unsigned int position)
{
	return m_inner->seek(position);
}

bool ThAggregateStream::truncate(void)
{
	return m_inner->truncate();
}

// ThAggregateStream attributes -------------------------------------------------------------------

bool ThAggregateStream::isEOF(void) const
{
	return m_inner->isEOF();
}

unsigned int ThAggregateStream::getSize(void) const
{
	return m_inner->getSize();
}

unsigned int ThAggregateStream::getPosition(void) const
{
	return m_inner->getPosition();
}

///////////////////////////////////////////////////////////////////////////////////////////////////
