///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <ThCore.h>
#include <ThMemory.h>
#include <ThTimer.h>
#include <ThWindow.h>
#include <ThStream.h>
#include <ThStorage.h>
#include <ThServer.h>
#include <ThDisplay.h>
#include <ThMusic.h>
#include <ThSystem.h>

#include "ThSystem.h"

///////////////////////////////////////////////////////////////////////////////////////////////////

ThSingleton<IThSystem> System;

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThSystem constructors -------------------------------------------------------------------------

IThSystem::~IThSystem(void)
{
}

// IThSystem static methods -----------------------------------------------------------------------

bool IThSystem::create(void)
{
	if (System)
		return true;

	ThPtr<ThSystem> system = new ThSystem();

	if (!system->open())
		return false;

	System.attach(system.detach());
	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThSystem constructors --------------------------------------------------------------------------

ThSystem::ThSystem(void):
	ThServerObject(THID_SYSTEM)
{
	m_stopped  = false;
	m_currTime = 0.f;
	m_prevTime = 0.f;
}

ThSystem::~ThSystem(void)
{
	close();
}

// ThSystem methods -------------------------------------------------------------------------------

bool ThSystem::open(void)
{
	close();

	if (!Server->registerObject(this))
		return false;

	return true;
}

void ThSystem::close(void)
{
	closeMusic();
	closeDisplay();

	m_stopped  = false;
	m_currTime = 0.f;
	m_prevTime = 0.f;
}

// ThSystem interface methods ---------------------------------------------------------------------

bool ThSystem::start(void)
{
	if (m_timer.isStarted())
		return false;

	m_timer.start();

	sendMessage(THMSG_SYSTEM_START, THID_ANNOUNCE);

	while (!m_stopped)
	{
		if (!ThWindow::update(false))
			stop();

		m_prevTime = m_currTime;
		m_currTime = m_timer.getTime();

		const float deltaTime = m_currTime - m_prevTime;

		sendMessage(THMSG_UPDATE, THID_ANNOUNCE, &deltaTime);

		if (m_display)
		{
			sendMessage(THMSG_RENDER, THID_ANNOUNCE);

			m_display->update();
		}
	}

	sendMessage(THMSG_SYSTEM_STOP, THID_ANNOUNCE);

	m_timer.stop();

	m_stopped  = false;
	m_currTime = 0.f;
	m_prevTime = 0.f;

	return true;
}

void ThSystem::stop(void)
{
	m_stopped = true;
}

void ThSystem::pause(void)
{
	if (!m_timer.isStarted())
		return;

	sendMessage(THMSG_SYSTEM_PAUSE, THID_ANNOUNCE);

	if (m_music)
		m_music->pause();

	m_timer.pause();
}

void ThSystem::resume(void)
{
	if (!m_timer.isStarted())
		return;

	sendMessage(THMSG_SYSTEM_RESUME, THID_ANNOUNCE);

	if (m_music)
		m_music->resume();

	m_timer.resume();
}

// ThSystem interface display methods -------------------------------------------------------------

bool ThSystem::openDisplay(unsigned int width, unsigned int height)
{
	closeDisplay();

	m_display = IThDisplay::createInstance(width, height);
	if (!m_display)
		return false;

	sendMessage(THMSG_DISPLAY_OPEN, THID_ANNOUNCE);
	return true;
}

void ThSystem::closeDisplay(void)
{
	sendMessage(THMSG_DISPLAY_CLOSE, THID_ANNOUNCE);

	m_display.release();
}

// ThSystem interface music methods ---------------------------------------------------------------

bool ThSystem::openMusic(const char* fileName)
{
	closeMusic();

	ThPtr<IThStream> file = Storage->openFile(fileName);
	if (!file)
		return false;

	m_music = IThMusic::createInstance(file);
	if (!m_music)
		return false;

	sendMessage(THMSG_MUSIC_OPEN, THID_ANNOUNCE);
	return true;
}

void ThSystem::closeMusic(void)
{
	sendMessage(THMSG_MUSIC_CLOSE, THID_ANNOUNCE);

	m_music.release();
}

// ThSystem interface attributes ------------------------------------------------------------------

bool ThSystem::isStarted(void)
{
	return m_timer.isStarted();
}

bool ThSystem::isPaused(void)
{
	return m_timer.isPaused();
}

float ThSystem::getTotalTime(void)
{
	return m_currTime;
}

float ThSystem::getDeltaTime(void)
{
	return m_currTime - m_prevTime;
}

void ThSystem::setTotalTime(float time)
{
	if (m_music)
		m_music->setTime(time);

	m_timer.setTime(time);

	// NOTE: this will cause interresting effects if time < m_currTime

	m_prevTime = m_currTime;
	m_currTime = time;
}

IThDisplay* ThSystem::getDisplay(void)
{
	return m_display;
}

IThMusic* ThSystem::getMusic(void)
{
	return m_music;
}

// ThSystem callbacks -----------------------------------------------------------------------------

bool ThSystem::recieve(ThMessage* message)
{
	switch (message->getMessage())
	{
		case THMSG_REQUEST_EXIT:
		{
			stop();
			break;
		}
	}

	return ThServerObject::recieve(message);
}

///////////////////////////////////////////////////////////////////////////////////////////////////
