///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

// ThPtr constructors -----------------------------------------------------------------------------

/*! Default constructor.
 */
template <typename T>
inline ThPtr<T>::ThPtr(void):
	m_object(NULL)
{
}

/*! Constructor.
 *  \param object [in] The object to be owned by this pointer.
 */
template <typename T>
inline ThPtr<T>::ThPtr(T* object):
	m_object(NULL)
{
	operator = (object);
}

/*! NOP copy constructor.
 */
template <typename T>
inline ThPtr<T>::ThPtr(const ThPtr<T>& source):
	m_object(NULL)
{
	THASSERT(false, "This is not good!");
}

/*! Destructor.
 */
template <typename T>
inline ThPtr<T>::~ThPtr(void)
{
	release();
}

// ThPtr methods ----------------------------------------------------------------------------------

/*! Detaches (orphans) the object and returns it's address.
 *	\return The address of the previously owned object.
 *	\remarks Beware of the implications of calling this method! This transfers the responsibility of releasing memory to the caller. It should be used with extreme care.
 *	\remarks The main purpose of this method is to provide an automatic release mechanism for objects that failed to initialize properly, letting client code return without manual cleanup.
 */
template <typename T>
inline T* ThPtr<T>::detach(void)
{
	T* object;

	object = m_object;
	m_object = NULL;
	return object;
}

/*! Releases the allocated object.
 */
template <typename T>
inline void ThPtr<T>::release(void)
{
	if (m_object)
	{
		delete m_object;
		m_object = NULL;
	}
}

// ThPtr operators --------------------------------------------------------------------------------

template <typename T>
inline T* ThPtr<T>::operator -> (void)
{
	THASSERT(m_object != NULL, "This is not good.");

	return m_object;
}

template <typename T>
inline const T* ThPtr<T>::operator -> (void) const
{
	THASSERT(m_object != NULL, "This is not good.");

	return m_object;
}

template <typename T>
inline ThPtr<T>::operator T* (void)
{
	return m_object;
}

template <typename T>
inline ThPtr<T>::operator const T* (void) const
{
	return m_object;
}

template <typename T>
inline ThPtr<T>& ThPtr<T>::operator = (T* object)
{
	release();

	m_object = object;
	return *this;
}

template <typename T>
inline ThPtr<T>& ThPtr<T>::operator = (const ThPtr<T>& source)
{
	THASSERT(false, "This is not good!");

	return *this;
}

// ThPtr attributes -------------------------------------------------------------------------------

template <typename T>
inline T* ThPtr<T>::getObject(void)
{
	return m_object;
}

template <typename T>
inline const T* ThPtr<T>::getObject(void) const
{
	return m_object;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThRef constructors -----------------------------------------------------------------------------

template <typename T>
inline ThRef<T>::ThRef(void):
	m_object(NULL)
{
}

template <typename T>
inline ThRef<T>::ThRef(const ThRef<T>& source):
	m_object(NULL)
{
	operator = (source);
}

template <typename T>
inline ThRef<T>::~ThRef(void)
{
	release();
}

// ThRef methods ----------------------------------------------------------------------------------

template <typename T>
inline void ThRef<T>::release(void)
{
	if (m_object)
	{
		if (!static_cast<ThRefObject<T>*>(m_object)->release())
			delete m_object;

		m_object = NULL;
	}
}

// ThRef operators --------------------------------------------------------------------------------

template <typename T>
inline T* ThRef<T>::operator -> (void) const
{
	THASSERT(m_object != NULL, "This is not good.");

	return m_object;
}

template <typename T>
inline ThRef<T>::operator T* (void) const
{
	return m_object;
}

template <typename T>
inline ThRef<T>& ThRef<T>::operator = (T* object)
{
	if (object)
		static_cast<ThRefObject<T>*>(object)->reference();

	release();

	m_object = object;
	return *this;
}

template <typename T>
inline ThRef<T>& ThRef<T>::operator = (const ThRef<T>& source)
{
	return operator = (source.getObject());
}

// ThRef attributes -------------------------------------------------------------------------------

template <typename T>
inline T* ThRef<T>::getObject(void) const
{
	return m_object;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThRefObject constructors -----------------------------------------------------------------------

template <typename T>
inline ThRefObject<T>::ThRefObject(void):
	m_count(0)
{
}

template <typename T>
inline ThRefObject<T>::~ThRefObject(void)
{
	THASSERT(m_count == 0, "Reference object must be completely released before destruction.");
}

// ThRefObject methods ----------------------------------------------------------------------------

template <typename T>
inline void ThRefObject<T>::reference(void)
{
	m_count++;

	THASSERT(m_count > 0, "Reference count wrapping detected. Please insert coin to continue.");
}

template <typename T>
inline bool ThRefObject<T>::release(void)
{
	THASSERT(m_count > 0, "Reference object is already completely released.");

	m_count--;

	return m_count ? true : false;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThBlock constructors ---------------------------------------------------------------------------

/*! Default constructor.
 */
template <typename T>
inline ThBlock<T>::ThBlock(void):
	m_data(NULL),
	m_count(0)
{
}

/*! Creates a ThBlock object with the specified initial size.
 *	\param size [in] The requested initial size.
 */
template <typename T>
inline ThBlock<T>::ThBlock(unsigned int count):
	m_data(NULL),
	m_count(0)
{
	allocate(count);
}

/*! NOP copy operator. This is identical to the default constructor.
 */
template <typename T>
inline ThBlock<T>::ThBlock(const ThBlock<T>& source):
	m_data(NULL),
	m_count(0)
{
}

/*! Destroys a ThBlock object, releasing any allocated memory block.
 */
template <typename T>
inline ThBlock<T>::~ThBlock(void)
{
	release();
}

// ThBlock methods --------------------------------------------------------------------------------

/*!	Allocates a memory block of the specified size. Any previous memory block is destroyed.
 *	\param count [in] The number of items of type T to be allocated.
 */
template <typename T>
inline void ThBlock<T>::allocate(unsigned int count)
{
	release();

	if (count)
	{
		m_data  = new T [count];
		m_count = count;
	}
}

/*!	Resizes the memory block to the specified size. Any existing data is preserved, up to the size of the new block.
 *	\param count [in] The new number of items of type T to be allocated.
 *  \remarks If no memory block is allocated, this is identical to ThBlock::allocate.
 */
template <typename T>
inline void ThBlock<T>::resize(unsigned int count)
{
	if (m_data)
	{
		m_data  = (T*) ResizeBlock(m_data, count * sizeof(T));
		m_count = count;
	}
	else
		allocate(count);
}

/*! Releases any allocated memory block.
 */
template <typename T>
inline void ThBlock<T>::release(void)
{
	if (m_data)
	{
		delete [] m_data;
		m_data = NULL;

		m_count = 0;
	}
}

// ThBlock operators ------------------------------------------------------------------------------

/*! \return The address of the allocated memory block, or \c NULL if no block is allocated.
 */
template <typename T>
inline ThBlock<T>::operator T* (void)
{
	return m_data;
}

/*! \return The address of the allocated memory block, or \c NULL if no block is allocated.
 */
template <typename T>
inline ThBlock<T>::operator const T* (void) const
{
	return m_data;
}

/*! NOP assignment operator.
 */
template <typename T>
inline ThBlock<T>& ThBlock<T>::operator = (const ThBlock<T>& source)
{
	return *this;
}

// ThBlock attributes -----------------------------------------------------------------------------

/*! \return The address of the allocated memory block, or \c NULL if no block is allocated.
 */
template <typename T>
inline T* ThBlock<T>::getData(void)
{
	return m_data;
}

/*! \return The address of the allocated memory block, or \c NULL if no block is allocated.
 */
template <typename T>
inline const T* ThBlock<T>::getData(void) const
{
	return m_data;
}

/*! \return The size, n bytes, of the allocated block, or zero if no block is allocated.
 */
template <typename T>
inline unsigned int ThBlock<T>::getSize(void) const
{
	return m_count * sizeof(T);
}

/*! \return The number of items of type T in the memory block, or zero if no block is allocated.
 */
template <typename T>
inline unsigned int ThBlock<T>::getCount(void) const
{
	return m_count;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThListItem constructors ------------------------------------------------------------------------

template <typename T>
inline ThListItem<T>::ThListItem(void):
	m_prev(NULL),
	m_next(NULL),
	m_list(NULL)
{
}

template <typename T>
inline ThListItem<T>::ThListItem(const ThListItem<T>& source):
	m_prev(NULL),
	m_next(NULL),
	m_list(NULL)
{
}

template <typename T>
inline ThListItem<T>::~ThListItem(void)
{
	detach();
}

// ThListItem methods -----------------------------------------------------------------------------

template <typename T>
inline void ThListItem<T>::detach(void)
{
	if (m_list)
	{
		if (m_prev)
			static_cast<ThListItem<T>*>(m_prev)->m_next = m_next;
		else
			m_list->m_head = m_next;

		if (m_next)
			static_cast<ThListItem<T>*>(m_next)->m_prev = m_prev;

		m_list->m_count--;

		m_list = NULL;
		m_prev = NULL;
		m_next = NULL;
	}
}

// ThListItem operators ---------------------------------------------------------------------------

template <typename T>
inline ThListItem<T>& ThListItem<T>::operator = (const ThListItem<T>& source)
{
	return *this;
}

// ThListItem attributes --------------------------------------------------------------------------

template <typename T>
inline bool ThListItem<T>::isAttached(void) const
{
	return m_list != NULL;
}

template <typename T>
inline T* ThListItem<T>::getPrev(void)
{
	return m_prev;
}

template <typename T>
inline const T* ThListItem<T>::getPrev(void) const
{
	return m_prev;
}

template <typename T>
inline T* ThListItem<T>::getNext(void)
{
	return m_next;
}

template <typename T>
inline const T* ThListItem<T>::getNext(void) const
{
	return m_next;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThItemList constructors ------------------------------------------------------------------------

/*! Default constructor.
 */
template <typename T>
inline ThItemList<T>::ThItemList(void):
	m_head(NULL),
	m_count(0)
{
}

/*! NOP copy constructor.
 */
template <typename T>
inline ThItemList<T>::ThItemList(const ThItemList<T>& source):
	m_head(NULL),
	m_count(0)
{
}

/*! Destructor.
 *  \remarks This frees all items contained in the list at the time of destruction.
 */
template <typename T>
inline ThItemList<T>::~ThItemList(void)
{
	release();
}

// ThItemList methods -----------------------------------------------------------------------------

/*! Attaches the specified item as the first item in the list.
 *  \param item [in] The item to be attached.
 */
template <typename T>
inline void ThItemList<T>::attachFirst(T* item)
{
	static_cast<ThListItem<T>*>(item)->detach();
	
	static_cast<ThListItem<T>*>(item)->m_list = this;
	if (static_cast<ThListItem<T>*>(item)->m_next = m_head)
		static_cast<ThListItem<T>*>(m_head)->m_prev = item;

	m_head = item;
	m_count++;
}

/*! Attaches the specified item as the last item in the list.
 *  \param item [in] The item to be attached.
 */
template <typename T>
inline void ThItemList<T>::attachLast(T* item)
{
	static_cast<ThListItem<T>*>(item)->detach();
	
	static_cast<ThListItem<T>*>(item)->m_list = this;
	if (T* last = getLast())
	{
		static_cast<ThListItem<T>*>(last)->m_next = item;
		static_cast<ThListItem<T>*>(item)->m_prev = last;
	}
	else
		m_head = item;

	m_count++;
}

//! \todo Detaching before searching for item \c index just might once upon a blue moon create an Obi-Wan error. Create solid policy for such cases!
/*! Attaches the specified item at the specified index in the list, or last if the list is too small.
 *  \param item [in] The item to be attached.
 *  \param index [in] The index at which to attach the item.
 */
template <typename T>
inline void ThItemList<T>::attachAt(T* item, unsigned int index)
{
	static_cast<ThListItem<T>*>(item)->detach();
	
	if (index < m_count)
	{
		T* next = getAt(index);

		static_cast<ThListItem<T>*>(item)->m_list = this;
		static_cast<ThListItem<T>*>(item)->m_next = next;
		if (static_cast<ThListItem<T>*>(item)->m_prev = static_cast<ThListItem<T>*>(next)->m_prev)
			static_cast<ThListItem<T>*>(static_cast<ThListItem<T>*>(item)->m_prev)->m_next = item;
		else
			m_head = item;
		static_cast<ThListItem<T>*>(next)->m_prev = item;

		m_count++;
	}
	else
	{
		attachLast(item);
	}
}

/*! Starts an iteration from the first item in the list.
 *  \return An iterator which starts at the first item in the list.
 */
template <typename T>
inline ThIterator<T> ThItemList<T>::iterate(void)
{
	return ThIterator<T>(m_head);
}

/*! Starts an iteration from the first item in the list.
 *  \return An iterator which starts at the first item in the list.
 */
template <typename T>
inline ThConstIterator<T> ThItemList<T>::iterate(void) const
{
	return ThConstIterator<T>(m_head);
}

/*! Releases all items in the list.
 */
template <typename T>
inline void ThItemList<T>::release(void)
{
	while (m_head)
		delete m_head;
}

// ThItemList operators ---------------------------------------------------------------------------

/*! NOP assignment operator.
 */
template <typename T>
inline ThItemList<T>& ThItemList<T>::operator = (const ThItemList<T>& source)
{
	return *this;
}

// ThItemList attributes --------------------------------------------------------------------------

/*! \return The first item in the list, or NULL if the list is empty.
 */
template <typename T>
inline T* ThItemList<T>::getFirst(void)
{
	return m_head;
}

/*! \return The first item in the list, or NULL if the list is empty.
 */
template <typename T>
inline const T* ThItemList<T>::getFirst(void) const
{
	return m_head;
}

/*! \return The last item in the list, or NULL if the list is empty.
 */
template <typename T>
inline T* ThItemList<T>::getLast(void)
{
	T* item = m_head;

	while (item)
	{
		if (static_cast<ThListItem<T>*>(item)->m_next)
			item = static_cast<ThListItem<T>*>(item)->m_next;
		else
			return item;
	}

	return NULL;
}

/*! \return The last item in the list, or NULL if the list is empty.
 */
template <typename T>
inline const T* ThItemList<T>::getLast(void) const
{
	T* item = m_head;

	while (item)
	{
		if (static_cast<ThListItem<T>*>(item)->m_next)
			item = static_cast<ThListItem<T>*>(item)->m_next;
		else
			return item;
	}

	return NULL;
}

/*! \return The item at the specified index, or NULL if no item exists at that index.
 *  \param index [in] The index of the requested item.
 */
template <typename T>
inline T* ThItemList<T>::getAt(unsigned int index)
{
	unsigned int i = index;
	T* item = m_head;

	while (item)
	{
		if (i--)
			item = static_cast<ThListItem<T>*>(item)->m_next;
		else
			return item;
	}

	return NULL;
}

/*! \return The item at the specified index, or NULL if no item exists at that index.
 *  \param index [in] The index of the requested item.
 */
template <typename T>
inline const T* ThItemList<T>::getAt(unsigned int index) const
{
	unsigned int i = index;
	T* item = m_head;

	while (item)
	{
		if (i--)
			item = static_cast<ThListItem<T>*>(item)->m_next;
		else
			return item;
	}

	return NULL;
}

/*! \return The total size, in bytes, of all the items in the list.
 */
template <typename T>
inline unsigned int ThItemList<T>::getSize(void) const
{
	return m_count * sizeof(T);
}

/*! \return The number of items in the list.
 */
template <typename T>
inline unsigned int ThItemList<T>::getCount(void) const
{
	return m_count;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThHashItem constructors ------------------------------------------------------------------------

template <typename T>
inline ThHashItem<T>::ThHashItem(unsigned int hash):
	m_hash(hash)
{
}

template <typename T>
inline ThHashItem<T>::ThHashItem(const ThHashItem<T>& source):
	m_hash(0)
{
}

// ThHashItem operators ---------------------------------------------------------------------------

template <typename T>
inline ThHashItem<T>& ThHashItem<T>::operator = (const ThHashItem<T>& source)
{
	return *this;
}

// ThHashItem attributes --------------------------------------------------------------------------

template <typename T>
inline unsigned int ThHashItem<T>::getHash(void) const
{
	return m_hash;
}

template <typename T>
inline void ThHashItem<T>::setHash(unsigned int hash)
{
	m_hash = hash;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThHashList constructors ------------------------------------------------------------------------

template <typename T, unsigned int N>
inline ThHashList<T,N>::ThHashList(void)
{
	m_root.allocate(1 << N);
}

template <typename T, unsigned int N>
inline ThHashList<T,N>::ThHashList(const ThHashList<T,N>& source)
{
	m_root.allocate(1 << N);
}

// ThHashList methods -----------------------------------------------------------------------------

template <typename T, unsigned int N>
inline void ThHashList<T,N>::attach(T* item)
{
	unsigned int index = static_cast<ThHashItem<T>*>(item)->getHash() & (0xffffffff << N);

	static_cast<ThItemList<T>*>(m_root)[index].attachFirst(static_cast<ThHashItem<T>*>(item));
}

template <typename T, unsigned int N>
inline void ThHashList<T,N>::release(void)
{
	for (unsigned int i = 0;  i < (1 << N);  i++)
		static_cast<ThItemList<T>*>(m_root)[i].release();
}

// ThHashList operators ---------------------------------------------------------------------------

template <typename T, unsigned int N>
inline ThHashList<T>& ThHashList<T,N>::operator = (const ThHashList<T,N>& source)
{
	return *this;
}

// ThHashList attributes --------------------------------------------------------------------------

template <typename T, unsigned int N>
inline T* ThHashList<T,N>::getItem(unsigned int hash)
{
	unsigned int index = hash & (0xffffffff << N);

	for (ThIterator<T> item(m_root[index]);  item;  item.next())
	{
		if (static_cast<ThHashItem<T>*>(item.getItem())->getHash() == hash)
			return item;
	}

	return NULL;
}

template <typename T, unsigned int N>
inline const T* ThHashList<T,N>::getItem(unsigned int hash) const
{
	unsigned int index = hash & (0xffffffff << N);

	for (ThConstIterator<T> item(m_root[index]);  item;  item.next())
	{
		if (static_cast<const ThHashItem<T>*>(item.getItem())->getHash() == hash)
			return item;
	}

	return NULL;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThIterator constructors ------------------------------------------------------------------------

/*! Default constructor.
 */
template <typename T>
inline ThIterator<T>::ThIterator(void):
	m_item(NULL)
{
}

/*! Creates an iterator which starts at the specified item.
 * \param item [in] The item from which to start the iteration.
 */
template <typename T>
inline ThIterator<T>::ThIterator(T* item):
	m_item(item)
{
}

/*! Creates an iterator which starts with the first item in the specified list.
 * \param list [in] The list to be iterated.
 */
template <typename T>
inline ThIterator<T>::ThIterator(ThItemList<T>& list)
{
	m_item = list.getFirst();
}

// ThIterator methods -----------------------------------------------------------------------------

/*! Advances the iterator to the next item in the sequence.
 */
template <typename T>
inline T* ThIterator<T>::next(void)
{
	if (m_item)
		m_item = static_cast<ThListItem<T>*>(m_item)->getNext();

	return m_item;
}

/*! Detaches the object and returns it's address.
 *	\return The address of the previously referenced memory object.
 */
template <typename T>
inline T* ThIterator<T>::detach(void)
{
	T* item;

	item = m_item;
	m_item = NULL;
	return item;
}

/*! Releases the current item, and advances the iterator to the next item, if any.
 */
template <typename T>
inline void ThIterator<T>::release(void)
{
	if (m_item)
	{
		T* next = static_cast<ThListItem<T>*>(m_item)->getNext();

		delete m_item;
		m_item = next;
	}
}

// ThIterator operators ---------------------------------------------------------------------------

template <typename T>
inline T* ThIterator<T>::operator -> (void) const
{
	THASSERT(m_item != NULL, "This is not good.");

	return m_item;
}

template <typename T>
inline ThIterator<T>::operator T* (void) const
{
	return m_item;
}

template <typename T>
inline ThIterator<T>& ThIterator<T>::operator = (T* item)
{
	m_item = item;

	return *this;
}

// ThIterator attributes --------------------------------------------------------------------------

template <typename T>
inline T* ThIterator<T>::getItem(void)
{
	return m_item;
}

template <typename T>
inline const T* ThIterator<T>::getItem(void) const
{
	return m_item;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThConstIterator constructors -------------------------------------------------------------------

/*! Default constructor.
 */
template <typename T>
inline ThConstIterator<T>::ThConstIterator(void):
	m_item(NULL)
{
}

/*! Creates an iterator which starts at the specified item.
 * \param item [in] The item from which to start the iteration.
 */
template <typename T>
inline ThConstIterator<T>::ThConstIterator(const T* item):
	m_item(item)
{
}

/*! Creates an iterator which starts with the first item in the specified list.
 * \param list [in] The list which is to be iterated.
 */
template <typename T>
inline ThConstIterator<T>::ThConstIterator(const ThItemList<T>& list)
{
	m_item = list.getFirst();
}

// ThConstIterator methods ------------------------------------------------------------------------

/*! Advances the iterator to the next item in the sequence.
 */
template <typename T>
inline const T* ThConstIterator<T>::next(void)
{
	if (m_item)
		m_item = static_cast<const ThListItem<T>*>(m_item)->getNext();

	return m_item;
}

/*! Detaches the object and returns it's address.
 *	\return The address of the previously referenced memory object.
 */
template <typename T>
inline const T* ThConstIterator<T>::detach(void)
{
	const T* item;

	item = m_item;
	m_item = NULL;
	return item;
}

// ThConstIterator operators ----------------------------------------------------------------------

template <typename T>
inline const T* ThConstIterator<T>::operator -> (void) const
{
	THASSERT(m_item != NULL, "This is not good.");

	return m_item;
}

template <typename T>
inline ThConstIterator<T>::operator const T* (void) const
{
	return m_item;
}

template <typename T>
inline ThConstIterator<T>& ThConstIterator<T>::operator = (const T* item)
{
	m_item = item;

	return *this;
}

// ThConstIterator attributes ---------------------------------------------------------------------

template <typename T>
inline const T* ThConstIterator<T>::getItem(void) const
{
	return m_item;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThSingleton operators --------------------------------------------------------------------------

template <typename T>
inline T* ThSingleton<T>::operator -> (void)
{
	THASSERT(m_object != NULL, "This is not good.");

	return m_object;
}

template <typename T>
inline ThSingleton<T>::operator T* (void)
{
	return m_object;
}

// ThSingleton methods ----------------------------------------------------------------------------

/*! Attaches an object, transferring ownership of it to the container.
 */
template <typename T>
inline void ThSingleton<T>::attach(T* object)
{
	m_object = object;
}

/*! Destroys the currently attached object, if any.
 */
template <typename T>
inline void ThSingleton<T>::release(void)
{
	m_object.release();
}

///////////////////////////////////////////////////////////////////////////////////////////////////
