///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////
#ifndef SHARED_THSTREAM_H
#define SHARED_THSTREAM_H
///////////////////////////////////////////////////////////////////////////////////////////////////

/*! \interface IThInputStream
 *	The input stream interface. Represents a generic seekable input stream.
 */
/*! \fn unsigned int IThInputStream::read(void* data, unsigned int size)
 *	Reads data into the specified buffer.
 *	\param data [out] The buffer to write the data to.
 *	\param size [in] The number of bytes to read.
 *	\return The number of bytes actually read.
 */
/*! \fn bool IThInputStream::seek(unsigned int position)
 *	Sets the file pointer to the specified position.
 *	\param position [in] The position to set the file pointer to.
 *	\return \c true if sucessful, or \c false if an error ocurred.
 */
/*! \fn unsigned int IThInputStream::skip(unsigned int size)
 */
/*! \fn bool IThInputStream::readString(ThString& string)
 */
/*! \fn bool IThInputStream::readItem(T& item)
 */
/*! \fn bool IThInputStream::readItems(T* items, unsigned int count)
 */
/*! \fn bool IThInputStream::isEOF(void) const
 */
/*! \fn unsigned int IThInputStream::getSize(void) const
 */
/*! \fn unsigned int IThInputStream::getPosition(void) const
 */
interface IThInputStream
{
// constructors
	virtual														~IThInputStream(void);
// methods
	virtual unsigned int							read(void* data, unsigned int size) = 0;
	virtual bool											seek(unsigned int position) = 0;
	virtual unsigned int							skip(unsigned int size) = 0;
// helper methods
	virtual bool											readString(ThString& string);
// template helper methods
	template <typename T>
	inline bool												readItem(T& item)
		{ return read(&item, sizeof(T)) == sizeof(T); }
	template <typename T>
	inline bool												readItems(T* items, unsigned int count)
		{ return read(items, sizeof(T) * count) == sizeof(T) * count; }
// attributes
	virtual bool											isEOF(void) const = 0;
	virtual unsigned int							getSize(void) const = 0;
	virtual unsigned int							getPosition(void) const = 0;
};

//-------------------------------------------------------------------------------------------------

interface IThOutputStream
{
// constructors
	virtual														~IThOutputStream(void);
// methods
	virtual unsigned int							write(const void* data, unsigned int size) = 0;
	virtual bool											seek(unsigned int position) = 0;
	virtual bool											truncate(void) = 0;
// helper methods
	virtual bool											writeString(const char* string);
// template helper methods
	template <typename T>
	inline bool												writeItem(T& item)
		{ return write(&item, sizeof(T)) == sizeof(T); }
	template <typename T>
	inline bool												writeItems(T* items, unsigned int count)
		{ return write(items, sizeof(T) * count) == sizeof(T) * count; }
// attributes
	virtual unsigned int							getSize(void) const = 0;
	virtual unsigned int							getPosition(void) const = 0;
};

//-------------------------------------------------------------------------------------------------

interface IThStream : public IThInputStream, public IThOutputStream
{
// constructors
	virtual														~IThStream(void);
// methods
	virtual unsigned int							read(void* data, unsigned int size) = 0;
	virtual unsigned int							write(const void* data, unsigned int size) = 0;
	virtual bool											seek(unsigned int position) = 0;
	virtual unsigned int							skip(unsigned int size) = 0;
	virtual bool											truncate(void) = 0;
// attributes
	virtual bool											isEOF(void) const = 0;
	virtual unsigned int							getSize(void) const = 0;
	virtual unsigned int							getPosition(void) const = 0;
};

///////////////////////////////////////////////////////////////////////////////////////////////////

IThStream* CreateDynamicBlockStream(unsigned int size = 0);
IThStream* CreateAggregateStream(IThStream* inner);

///////////////////////////////////////////////////////////////////////////////////////////////////
#endif /* SHARED_THSTREAM_H */
///////////////////////////////////////////////////////////////////////////////////////////////////
