//==============================================================================
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//==============================================================================

//==============================================================================
// File: cTextureFont.cpp
// Project: Shooting Star
// Author: Jarmo Hekkanen <jarski@2ndpoint.fi>
// Copyrights (c) 2003 2ndPoint ry (www.2ndpoint.fi)
//------------------------------------------------------------------------------
// Revision history
//==============================================================================

//==============================================================================
// Includes
#include "cTextureFont.hpp"

#include <stdarg.h>
#include <stdio.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include "cTextureManager.hpp"
#include "Debug.hpp"
//------------------------------------------------------------------------------
// Namespaces
using namespace ShootingStar;
//==============================================================================

//! Constructor
cTextureFont::cTextureFont (void):
mTexture (0),
mListBase (0),
mWidth (0),
mHeight (0)
{
};

//! Constructor
cTextureFont::cTextureFont (string filename, int width, int height):
mListBase (0)
{
	Initialize (filename, width, height);
};

//! Destructor
cTextureFont::~cTextureFont (void)
{
	if ( mListBase != 0 )
		glDeleteLists (mListBase, 96);
};

void 
cTextureFont::Initialize (string filename, int width, int height)
{
	// Load texture
	int texWidth, texHeight;
	cTextureManager &texManager = cTextureManager::GetInstance ();
	mTexture = texManager.LoadTexture (filename);
	texManager.GetTextureSize (filename, texWidth, texHeight);
	
	mWidth = width;
	mHeight = height;
	
	MakeDisplayLists (texWidth, texHeight);
}

void 
cTextureFont::MakeDisplayLists (int texWidth, int texHeight)
{
	if ( mListBase != 0 )
		glDeleteLists (mListBase, 96);
	
	mListBase = glGenLists (96);
	
	float charWidth = float (mWidth) / float (texWidth);
	float charHeight = float (mHeight) / float (texHeight);
	int rowWidth = texWidth / mWidth;
	int charSlot = 0;
	
	glListBase (mListBase);
	for ( unsigned char i = 32; i < 128; i++ )
	{
		glNewList (mListBase + i - 32, GL_COMPILE);
		if ( i == 32 )
			glTranslatef (mWidth, 0.0f, 0.0f);
		else
		{
			float charX = float (charSlot % rowWidth) * charWidth;
			float charY = float (charSlot / rowWidth) * charHeight;
			charSlot++;
			glBegin (GL_QUADS);
				glTexCoord2f (charX, charY);
				glVertex2i (0, -mHeight);
				glTexCoord2f (charX, charY + charHeight);
				glVertex2i (0, 0);
				glTexCoord2f (charX + charWidth, charY + charHeight);
				glVertex2i (mWidth, 0);
				glTexCoord2f (charX + charWidth, charY);
				glVertex2i (mWidth, -mHeight);
			glEnd ();
			glTranslatef (mWidth, 0.0f, 0.0f);
		}
		glEndList ();
	}
	
	GLenum error = glGetError ();
	while ( error != GL_NO_ERROR )
	{
		dbgError () << "cTextureFont: OpenGL error: " << gluErrorString (error) << '\n';
		error = glGetError ();
	}
}

void 
cTextureFont::PrintString (string text)
{
	dbg::assertion (DBG_ASSERTION (mListBase != 0));
	
	if (text.empty ())
		return;

	glPushAttrib (GL_LIST_BIT|GL_ENABLE_BIT|GL_TEXTURE_BIT);
	glPushMatrix ();
	
	// Set texture
	glEnable (GL_TEXTURE_2D);
	glBindTexture (GL_TEXTURE_2D, mTexture);
	
	// Call lists
	glListBase (mListBase - 32);
	glCallLists (text.size (), GL_UNSIGNED_BYTE, text.c_str ());

	glPopMatrix ();
	glPopAttrib ();
}

void 
cTextureFont::Print (const char *pFormat, ...)
{
	dbg::assertion (DBG_ASSERTION (mListBase != 0));
	
	char buffer[255];
	
	va_list argumentList;
	va_start (argumentList, pFormat);

	// Format string to buffer
	vsnprintf (buffer, 255, pFormat, argumentList);
	
	va_end (argumentList);
	
	PrintString (buffer);
}

//! Print string with wave effect
void 
cTextureFont::PrintStringW (string text, float a, float deltaA)
{
	glPushMatrix ();

	float last = 0.0f, tmp;
	for ( unsigned int i = 0; i < text.size (); i++ )
	{
		// Wave
		tmp = sin (a + i * deltaA);
		glTranslatef (mWidth, (last - tmp) * mHeight, 0.0f);
		last = tmp;
	
		Print ("%c", text[i]);
	}
	
	glPopMatrix ();
}


//==============================================================================
// EOF
//==============================================================================
