
#include <Theresa.h>

#include <time.h>

#include "Demo.h"

bool DemoApp::open(void)
{
	if (!System->openMusic("data/music.xm"))
		return false;

	ThString name(20);

	for (unsigned int i = 0;  i < 13;  i++)
	{
		name.format("data/blob%u.png", i);

		if (!System->getDisplay()->createTexture(name, GL_RGBA, THFLAG(THTEX_MIPMAP)))
			return false;
	}

	for (unsigned int i = 0;  i < 7;  i++)
	{
		name.format("data/member%u.png", i);

		if (!System->getDisplay()->createTexture(name, GL_RGBA))
			return false;
			
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	}

	for (unsigned int i = 0;  i < 7;  i++)
	{
		name.format("data/face%u.png", i);

		if (!System->getDisplay()->createTexture(name, GL_RGBA))
			return false;
	}

	for (unsigned int i = 0;  i < 10;  i++)
	{
		name.format("data/greets%u.png", i);

		if (!System->getDisplay()->createTexture(name, GL_RGBA))
			return false;
			
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	}

	for (unsigned int i = 0;  i < 8;  i++)
	{
		name.format("data/symbol%u.png", i);

		if (!System->getDisplay()->createTexture(name, GL_RGBA))
			return false;
	}

	if (!System->getEngine()->loadResources("data/cubes.3ds"))
		return false;

	if (!System->getEngine()->loadResources("data/pommes.3ds"))
		return false;

	if (!System->getEngine()->loadResources("data/members.3ds"))
		return false;

	if (!System->getEngine()->loadResources("data/olives.3ds"))
		return false;

	if (!System->getEngine()->loadResources("data/font.3ds"))
		return false;

	ThEffectType* type;
	
	if (!System->getEffect()->registerType(type = new ThEffectTemplate<DemoSuper>("super")))
		return false;
		
	if (!type->registerAttribute("color", DemoSuper::COLOR))
		return false;
	
	if (!System->getEffect()->registerType(type = new ThEffectTemplate<DemoIntro>("intro")))
		return false;

	if (!type->registerAttribute("vibrate", DemoIntro::VIBRATE))
		return false;
	
	if (!System->getEffect()->registerType(type = new ThEffectTemplate<DemoOrgan>("organ")))
		return false;

	if (!type->registerAttribute("note", DemoOrgan::NOTE))
		return false;
	
	if (!System->getEffect()->registerType(type = new ThEffectTemplate<DemoPommes>("pommes")))
		return false;

	if (!type->registerAttribute("light", DemoPommes::LIGHT))
		return false;
	
	if (!System->getEffect()->registerType(type = new ThEffectTemplate<DemoCredits>("credits")))
		return false;

	if (!type->registerAttribute("member", DemoCredits::MEMBER))
		return false;
	
	if (!type->registerAttribute("explode", DemoCredits::EXPLODE))
		return false;
	
	if (!System->getEffect()->registerType(type = new ThEffectTemplate<DemoOlive>("olive")))
		return false;
	
	if (!type->registerAttribute("speed", DemoOlive::SPEED))
		return false;
	
	if (!type->registerAttribute("direction", DemoOlive::DIRECTION))
		return false;
	
	if (!System->getEffect()->registerType(type = new ThEffectTemplate<DemoHyper>("hyper")))
		return false;
	
	if (!type->registerAttribute("stop", DemoHyper::STOP))
		return false;
	
	if (!System->getEffect()->registerType(type = new ThEffectTemplate<DemoLens>("lens")))
		return false;
	
	if (!System->getEffect()->registerType(type = new ThEffectTemplate<DemoGreets>("greets")))
		return false;
	
	if (!System->getEffect()->createObject("super", "super"))
		return false;

	System->getEffect()->registerStopEvent((unsigned char) 0);
	
	System->getEffect()->registerNotifyEvent((unsigned char) 1, "super", "color", "0x000000");
	System->getEffect()->registerNotifyEvent((unsigned char) 2, "super", "color", "0x5e765f");
	System->getEffect()->registerNotifyEvent((unsigned char) 3, "super", "color", "0xe49d2e");
	System->getEffect()->registerNotifyEvent((unsigned char) 4, "super", "color", "0xdcdb92");
	System->getEffect()->registerNotifyEvent((unsigned char) 5, "super", "color", "0x90b35f");
	System->getEffect()->registerNotifyEvent((unsigned char) 6, "super", "color", "0x6692ea");
	System->getEffect()->registerNotifyEvent((unsigned char) 7, "super", "color", "0x9b82ca");
	System->getEffect()->registerNotifyEvent((unsigned char) 8, "super", "color", "0xef6bde");

	System->getEffect()->registerCreateEvent((unsigned char) 9, "hyper", "intro");
	System->getEffect()->registerDeleteEvent((unsigned char) 10, "hyper");
	System->getEffect()->registerNotifyEvent((unsigned char) 21, "hyper", "vibrate", "");

	System->getEffect()->registerCreateEvent((unsigned char) 11, "organ", "organ");
	System->getEffect()->registerDeleteEvent((unsigned char) 12, "organ");
	System->getEffect()->registerNotifyEvent((unsigned char) 13, "organ", "note", "0");
	System->getEffect()->registerNotifyEvent((unsigned char) 14, "organ", "note", "1");
	System->getEffect()->registerNotifyEvent((unsigned char) 15, "organ", "note", "2");
	System->getEffect()->registerNotifyEvent((unsigned char) 16, "organ", "note", "3");
	System->getEffect()->registerNotifyEvent((unsigned char) 17, "organ", "note", "4");
	System->getEffect()->registerNotifyEvent((unsigned char) 18, "organ", "note", "5");
	System->getEffect()->registerNotifyEvent((unsigned char) 19, "organ", "note", "6");
	System->getEffect()->registerNotifyEvent((unsigned char) 20, "organ", "note", "7");

	System->getEffect()->registerCreateEvent((unsigned char) 48, "pommes", "pommes");
	System->getEffect()->registerDeleteEvent((unsigned char) 22, "pommes");
	System->getEffect()->registerNotifyEvent((unsigned char) 38, "pommes", "light", "0");
	System->getEffect()->registerNotifyEvent((unsigned char) 39, "pommes", "light", "1");
	System->getEffect()->registerNotifyEvent((unsigned char) 40, "pommes", "light", "2");
	System->getEffect()->registerNotifyEvent((unsigned char) 41, "pommes", "light", "3");

	System->getEffect()->registerCreateEvent((unsigned char) 23, "credits", "credits");
	System->getEffect()->registerDeleteEvent((unsigned char) 24, "credits");
	System->getEffect()->registerNotifyEvent((unsigned char) 25, "credits", "member", "0");
	System->getEffect()->registerNotifyEvent((unsigned char) 26, "credits", "member", "1");
	System->getEffect()->registerNotifyEvent((unsigned char) 27, "credits", "member", "2");
	System->getEffect()->registerNotifyEvent((unsigned char) 28, "credits", "member", "3");
	System->getEffect()->registerNotifyEvent((unsigned char) 29, "credits", "member", "4");
	System->getEffect()->registerNotifyEvent((unsigned char) 30, "credits", "member", "5");
	System->getEffect()->registerNotifyEvent((unsigned char) 31, "credits", "member", "6");
	System->getEffect()->registerNotifyEvent((unsigned char) 50, "credits", "explode", "");

	System->getEffect()->registerCreateEvent((unsigned char) 32, "olive", "olive");
	System->getEffect()->registerDeleteEvent((unsigned char) 33, "olive");
	System->getEffect()->registerNotifyEvent((unsigned char) 34, "olive", "speed", "");
	System->getEffect()->registerNotifyEvent((unsigned char) 35, "olive", "direction", "");

	System->getEffect()->registerCreateEvent((unsigned char) 36, "font", "hyper");
	System->getEffect()->registerDeleteEvent((unsigned char) 37, "font");
	System->getEffect()->registerNotifyEvent((unsigned char) 49, "font", "stop", "");
	
	System->getEffect()->registerCreateEvent((unsigned char) 42, "lens", "lens");
	System->getEffect()->registerDeleteEvent((unsigned char) 43, "lens");
	
	System->getEffect()->registerCreateEvent((unsigned char) 44, "greets", "greets");
	System->getEffect()->registerDeleteEvent((unsigned char) 45, "greets");

	System->getEffect()->registerCreateEvent((unsigned char) 46, "burst", "burst");
	System->getEffect()->registerDeleteEvent((unsigned char) 47, "burst");

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(90.f, (float) System->getDisplay()->getContext()->getWidth() / (float) System->getDisplay()->getContext()->getHeight(), 1.f, 1000.f);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	return true;
}

bool DemoApp::update(float deltaTime)
{
	return ThApplication::update(deltaTime);
}

bool DemoApp::render(IThCanvas* target)
{
	target->apply();

	System->getDisplay()->renderObjects(target);

	target->update();

	return ThApplication::render(target);
}

bool DemoApp::receive(const IThMessage* message)
{
	return ThApplication::receive(message);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

DemoSuper::DemoSuper(ThEffectType* type, const char* name):
	ThEffectObject(type, name)
{
	m_oldColor.reset();
	m_newColor.reset();
	m_moment = 0.f;
}

bool DemoSuper::open(void)
{
	if (!System->getDisplay()->registerObject(this, THLAYER_CLEAR))
		return false;

	return true;
}

bool DemoSuper::update(float deltaTime)
{
	if (!ThEffectObject::update(deltaTime))
		return false;

	const float interval = 0.5f;
	const float elapsed  = getTimePast() - m_moment;

	if (elapsed < interval)
	{
		const float scale = elapsed / interval;
		
		m_curColor = m_oldColor * (1.f - scale) + m_newColor * scale;
	}
	else
		m_curColor = m_newColor;

	return true;
}

bool DemoSuper::render(IThCanvas* target)
{
	glClearColor(m_curColor.x, m_curColor.y, m_curColor.z, 0.f);
	
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	return ThEffectObject::render(target);
}

bool DemoSuper::notify(unsigned int attributeID, const char* value)
{
	switch (attributeID)
	{
		case COLOR:
		{
			m_oldColor = m_newColor;

			unsigned int color = ThString::convertToInt(value);
			m_newColor.set((color >> 16) / 255.f, ((color & 0xff00) >> 8) / 255.f, (color & 0xff) / 255.f);
			
			m_moment = getTimePast();
			return true;
		}
	}
	
	return ThEffectObject::notify(attributeID, value);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

DemoIntro::DemoIntro(ThEffectType* type, const char* name):
	ThEffectObject(type, name)
{
	m_vertices.allocate(16);
	m_vertices[0x0].set(-1.f,  1.f,  1.f,  1.f);
	m_vertices[0x1].set(-1.f, -1.f,  1.f,  1.f);
	m_vertices[0x2].set( 1.f, -1.f,  1.f,  1.f);
	m_vertices[0x3].set( 1.f,  1.f,  1.f,  1.f);
	m_vertices[0x4].set(-1.f,  1.f, -1.f,  1.f);
	m_vertices[0x5].set(-1.f, -1.f, -1.f,  1.f);
	m_vertices[0x6].set( 1.f, -1.f, -1.f,  1.f);
	m_vertices[0x7].set( 1.f,  1.f, -1.f,  1.f);
	m_vertices[0x8].set(-1.f,  1.f,  1.f, -1.f);
	m_vertices[0x9].set(-1.f, -1.f,  1.f, -1.f);
	m_vertices[0xA].set( 1.f, -1.f,  1.f, -1.f);
	m_vertices[0xB].set( 1.f,  1.f,  1.f, -1.f);
	m_vertices[0xC].set(-1.f,  1.f, -1.f, -1.f);
	m_vertices[0xD].set(-1.f, -1.f, -1.f, -1.f);
	m_vertices[0xE].set( 1.f, -1.f, -1.f, -1.f);
	m_vertices[0xF].set( 1.f,  1.f, -1.f, -1.f);

	m_moment = 0.f;
	m_active = false;
}

bool DemoIntro::open(void)
{
	if (!System->getDisplay()->registerObject(this, THLAYER_NORMAL))
		return false;

	return true;
}

bool DemoIntro::update(float deltaTime)
{
	return ThEffectObject::update(deltaTime);
}

bool DemoIntro::render(IThCanvas* target)
{
	glPushMatrix();
	glTranslatef(0.f, 0.f, -2.f);

	if (m_active)
	{
		const float interval = 1.f;
		const float elapsed = getTimePast() - m_moment;
		const float scale = 1.f - elapsed / interval;

		glTranslatef(sinf(elapsed * 100.f) * 0.1f * scale, cosf(elapsed * 100.f) * 0.1f * scale, 0.f);

		if (elapsed > interval)
			m_active = false;
	}

	const float angle = getTimePast() * 1.5f;

	ThMatrix4 transform;
	transform.reset();

	// produce 4D rotation transform
	{
		ThMatrix4 temp;

		temp.reset();
		temp.y.y = cosf(angle);
		temp.w.w = cosf(angle);
		temp.y.w = -sinf(angle);
		temp.w.y = sinf(angle);

		transform.concatenate(temp);

		temp.reset();
		temp.z.z = cosf(angle);
		temp.w.w = cosf(angle);
		temp.z.w = -sinf(angle);
		temp.w.z = sinf(angle);

		transform.concatenate(temp);

		temp.reset();
		temp.x.x = cosf(angle);
		temp.w.w = cosf(angle);
		temp.x.w = -sinf(angle);
		temp.w.x = sinf(angle);

		transform.concatenate(temp);
	}

	glPushAttrib(GL_LINE_BIT | GL_CURRENT_BIT);
	glLineWidth(target->getWidth() / 150.f);

	glColor3ub(128, 130, 208);

	glBegin(GL_LINE_LOOP);
	for (unsigned int i = 0;  i < 4;  i++)
		renderVertex(m_vertices[i], transform);
	glEnd();

	glBegin(GL_LINE_LOOP);
	for (unsigned int i = 4;  i < 8;  i++)
		renderVertex(m_vertices[i], transform);
	glEnd();

	glBegin(GL_LINES);
	for (unsigned int i = 0;  i < 4;  i++)
	{
		renderVertex(m_vertices[i + 0], transform);
		renderVertex(m_vertices[i + 4], transform);
	}
	glEnd();

	glBegin(GL_LINE_LOOP);
	for (unsigned int i = 8;  i < 12;  i++)
		renderVertex(m_vertices[i], transform);
	glEnd();

	glBegin(GL_LINE_LOOP);
	for (unsigned int i = 12;  i < 16;  i++)
		renderVertex(m_vertices[i], transform);
	glEnd();

	glBegin(GL_LINES);
	for (unsigned int i = 8;  i < 12;  i++)
	{
		renderVertex(m_vertices[i + 0], transform);
		renderVertex(m_vertices[i + 4], transform);
	}
	glEnd();

	glBegin(GL_LINES);
	for (unsigned int i = 0;  i < 8;  i++)
	{
		renderVertex(m_vertices[i + 0], transform);
		renderVertex(m_vertices[i + 8], transform);
	}
	glEnd();

	glPopAttrib();

	glPopMatrix();

	return ThEffectObject::render(target);
}

bool DemoIntro::notify(unsigned int attributeID, const char* value)
{
	switch (attributeID)
	{
		case VIBRATE:
		{
			m_moment = getTimePast();
			m_active = true;
			break;
		}
	}

	return ThEffectObject::notify(attributeID, value);
}

void DemoIntro::renderVertex(const ThVector4& vertex, const ThMatrix4& transform)
{
	ThVector4 result = vertex;

	transform.transformVector(result);

	if (result.w)
	{
		result.x = result.x / (result.w + 3.f);
		result.y = result.y / (result.w + 3.f);
		result.z = result.z / (result.w + 3.f);
	}
	else
		result.reset();

	glVertex3fv(result);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

DemoOrgan::DemoOrgan(ThEffectType* type, const char* name):
	ThEffectObject(type, name)
{
	for (unsigned int i = 0;  i < 6;  i++)
	{
		m_position[i].set((i % 3) * 2.f - 2.f, 0.5f, -6.f);
		m_speed[i].reset();
		m_accel[i].set(0.f, -60.f, 0.f);
	}
}

bool DemoOrgan::open(void)
{
	if (!System->getDisplay()->registerObject(this, THLAYER_NORMAL))
		return false;

	m_white = System->getEngine()->findMesh("whitecube");
	if (!m_white)
		return false;

	m_green = System->getEngine()->findMesh("greencube");
	if (!m_green)
		return false;

	m_light = System->getEngine()->createLight(getName());
	if (!m_light)
		return false;
		
	return true;
}

bool DemoOrgan::update(float deltaTime)
{
	for (unsigned int i = 0;  i < 6;  i++)
	{
		m_position[i] += m_speed[i] * deltaTime + m_accel[i] * deltaTime * deltaTime;
		m_speed[i]    += m_accel[i] * deltaTime;

		if (m_position[i].y < 0.5f)
			m_position[i].y = 0.5f;
	}

	return ThEffectObject::update(deltaTime);
}

bool DemoOrgan::render(IThCanvas* target)
{
	m_light->enable();

	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	gluPerspective(60.f, (float) System->getDisplay()->getContext()->getWidth() / (float) System->getDisplay()->getContext()->getHeight(), 1.f, 1000.f);
	glMatrixMode(GL_MODELVIEW);

	glPushMatrix();
	glRotatef(sinf(getTimePast()) * 3.f, 0.f, 1.f, 0.f);

	for (unsigned int i = 0;  i < 6;  i++)
	{
		glPushMatrix();
		glTranslatef(m_position[i].x, m_position[i].y * (i < 3 ? 1.f : -1.f), m_position[i].z);

		if (i < 3)
			m_white->render();
		else
			m_green->render();

		glPopMatrix();
	}

	glPopMatrix();

	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);

	m_light->disable();

	return ThEffectObject::render(target);
}

bool DemoOrgan::notify(unsigned int attributeID, const char* value)
{
	switch (attributeID)
	{
		case NOTE:
		{
			unsigned int index = ThString::convertToInt(value);

			m_speed[index].y = 12.f;
			return true;
		}
	}

	return ThEffectObject::notify(attributeID, value);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

DemoPommes::DemoPommes(ThEffectType* type, const char* name):
	ThEffectObject(type, name)
{
	for (unsigned int i = 0;  i < 4;  i++)
	{
		m_moments[i] = 0.f;
		m_active[i]  = false;
	}
	
	m_vector.set(randf(), randf(), randf());
	m_vector.normalize();
}

bool DemoPommes::open(void)
{
	if (!System->getDisplay()->registerObject(this, THLAYER_NORMAL))
		return false;
		
	m_mesh = System->getEngine()->findMesh("pommes");
	if (!m_mesh)
		return false;

	ThString name(10);

	for (unsigned int i = 0;  i < 4;  i++)
	{
		name.format("pommes%u", i);

		m_materials[i] = System->getEngine()->findMaterial(name);
		if (!m_materials[i])
			return false;
	}
	
	IThMaterial* material = System->getEngine()->findMaterial("pommesbas");
	if (!material)
		return false;

	m_color = material->getDiffuseColor();

	m_light = System->getEngine()->createLight(getName());
	if (!m_light)
		return false;

	return true;
}

bool DemoPommes::update(float deltaTime)
{
	if (!ThEffectObject::update(deltaTime))
		return false;

	for (unsigned int i = 0;  i < 4;  i++)
	{
		if (m_active[i])
		{
			const float interval = 1.f;
			const float elapsed  = getTimePast() - m_moments[i];

			if (elapsed > interval)
			{
				m_active[i] = false;
				m_materials[i]->setDiffuseColor(m_color);
			}
			else
			{
				const float scale = elapsed / interval;

				ThVector4 color(1.f, 1.f, 0.6f, 1.f);
				ThVector4 result = m_color * scale * scale + color * (1.f - scale * scale);

				m_materials[i]->setDiffuseColor(result);
			}
		}
	}

	return true;
}

bool DemoPommes::render(IThCanvas* target)
{
	m_light->enable();

	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	gluPerspective(90.f, (float) System->getDisplay()->getContext()->getWidth() / (float) System->getDisplay()->getContext()->getHeight(), 1.f, 1000.f);
	glMatrixMode(GL_MODELVIEW);

	glPushMatrix();
	glTranslatef(0.f, 0.f, -23.f);
	glRotatef(sinf((getTimePast() - m_moment) * 4.f) * 15.f, m_vector.x, m_vector.y, m_vector.z);

	m_mesh->render();

	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);

	glPopMatrix();

	m_light->disable();
	
	return ThEffectObject::render(target);
}

bool DemoPommes::notify(unsigned int attributeID, const char* value)
{
	switch (attributeID)
	{
		case LIGHT:
		{
			const unsigned int index = ThString::convertToInt(value);

			m_moments[index] = getTimePast();
			m_active[index]  = true;
			
			m_moment = getTimePast();
			m_vector.set(randf(), randf(), randf());
			m_vector.normalize();
			break;
		}
	}

	return ThEffectObject::notify(attributeID, value);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

DemoCredits::DemoCredits(ThEffectType* type, const char* name):
	ThEffectObject(type, name)
{
	m_member = 0;
	m_moment = 0.f;
	m_active = false;
	m_exploded = false;
	
	m_system.create(200);
	m_system.getEmitter()->m_position.set(3.f, 0.f, -8.5f);
	m_system.getEmitter()->m_vector.set(0.f, 0.f, 1.f);
	m_system.getEmitter()->m_angle.set(0.5f, -0.5f);
	m_system.getEmitter()->m_color.set(0, 1.f, 1.f);
	m_system.getEmitter()->m_color.set(1, 1.f, 1.f);
	m_system.getEmitter()->m_color.set(2, 1.f, 1.f);
	m_system.getEmitter()->m_color.set(3, 1.f, 1.f);
	m_system.getEmitter()->m_speed.set(15.f, 20.f);
	m_system.getEmitter()->m_time.set(1.f, 1.f);
	m_system.getEmitter()->m_size.set(0, 0.2f, 0.2f);
	m_system.getEmitter()->m_size.set(1, 0.3f, 0.3f);
	m_system.getEmitter()->m_ratio = 800.f;
	
	m_vectors.allocate(7);
	m_intervals.allocate(7);
	m_elapsed.allocate(7);
	
	for (unsigned int i = 0;  i < m_intervals.getCount();  i++)
	{
		m_intervals[i] = randf() + 0.5f;
		m_elapsed[i]   = 0.f;
		m_vectors[i].set(1.f, 0.f, 0.f);
	}
}

bool DemoCredits::open(void)
{
	if (!System->getDisplay()->registerObject(this, THLAYER_NORMAL))
		return false;

	m_meshes.allocate(7);
	m_textures.allocate(7);

	ThString name(20);

	for (unsigned int i = 0;  i < m_meshes.getCount();  i++)
	{
		name.format("member%u", i);

		m_meshes[i] = System->getEngine()->findMesh(name);
		if (!m_meshes[i])
			return false;
			
		name.format("data/member%u.png", i);
		
		m_textures[i] = System->getDisplay()->findTexture(name);
		if (!m_textures[i])
			return false;
	}

	IThMaterial* material = System->getEngine()->findMaterial("member");
	if (!material)
		return false;

	material->setShadeModel(GL_SMOOTH);
	
	m_lineMaterial = System->getEngine()->createMaterial("credits lines");
	if (!m_lineMaterial)
		return false;
		
	m_textMaterial = System->getEngine()->createMaterial("credits texts");
	if (!m_textMaterial)
		return false;
		
	m_textMaterial->setBlending(true);
	m_textMaterial->setSrcBlend(GL_SRC_ALPHA);
	m_textMaterial->setDstBlend(GL_ONE_MINUS_SRC_ALPHA);
	m_textMaterial->setCombineMode(GL_REPLACE);
	
	m_bombMaterial = System->getEngine()->createMaterial("credits bomb");
	if (!m_bombMaterial)
		return false;
		
	m_bombMaterial->setBlending(true);
	m_bombMaterial->setSrcBlend(GL_SRC_ALPHA);
	m_bombMaterial->setDstBlend(GL_ONE_MINUS_SRC_ALPHA);
		
	m_light = System->getEngine()->createLight(getName());
	if (!m_light)
		return false;

	m_light->setAmbientColor(ThVector4(0.4f, 0.4f, 0.4f, 0.f));
	m_light->setDiffuseColor(ThVector4(0.7f, 0.7f, 0.7f, 0.f));

	return true;
}

bool DemoCredits::update(float deltaTime)
{
	for (unsigned int i = 0;  i < 7;  i++)
	{
		m_elapsed[i] += deltaTime;
		if (m_elapsed[i] > m_intervals[i])
		{
			m_vectors[i].set(randf(), randf() + randf(), randf());
			m_vectors[i].normalize();
			
			m_intervals[i] = randf() + 0.5f;
			m_elapsed[i] = 0.f;
		}
	}

	if (getTimePast() - m_moment > 1.5f)
		m_active = false;
	
	if (m_exploded)
	{
		m_system.update(deltaTime);
		if (m_system.getTime() > 1.f)
			m_system.stop();
	}
		
	return ThEffectObject::update(deltaTime);
}

bool DemoCredits::render(IThCanvas* target)
{
	m_light->enable();

	for (unsigned int i = 0;  i < 7;  i++)
	{
		if (m_exploded && i == 4)
			continue;
			
		glPushMatrix();
		glTranslatef(i * 3.f - 9.f, 0.f, -8.5f);
		
		const float scale = m_elapsed[i] / m_intervals[i];
		
		glRotatef(sinf(scale * (float) M_PI) * 10.f, m_vectors[i].x, m_vectors[i].y, m_vectors[i].z);

		m_meshes[i]->render();

		glPopMatrix();
	}
	
	m_light->disable();

	if (m_exploded)
	{
		m_bombMaterial->apply();
		m_system.render(target);
	}

	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	gluOrtho2D(0.f, 10.f, 10.f, 0.f);
	glMatrixMode(GL_MODELVIEW);

	glPushAttrib(GL_CURRENT_BIT | GL_ENABLE_BIT);
	glDisable(GL_CULL_FACE);
	glDisable(GL_LIGHTING);
	glLineWidth(target->getWidth() / 150.f);

	if (m_active)
	{
		const float elapsed = getTimePast() - m_moment;
		const float scale   = (m_member < 4 ? 1.f : -1.f);
		const bool blink    = ((unsigned int) (elapsed * 10.f) & 1) ? true : false;

		float base = 1.f + (8.f / 6.f) * m_member;
		
		m_lineMaterial->apply();

		if (elapsed > 0.5f || blink)
		{
			glColor3f(0.f, 0.f, 0.f);
			
			glBegin(GL_LINE_STRIP);

			glVertex2f(base, 6.5f);
			glVertex2f(base + 1.f * scale, 8.f);
			glVertex2f(base + 4.f * scale, 8.f);

			glEnd();
		}

		if (m_member < 4)
			base = base + 1.f;
		else
			base = base - 4.f;
		
		ThSprite2 sprite;

		if (elapsed > 1.f || (elapsed > 0.5f && blink))
		{
			sprite.m_color.set(0.f, 0.f, 0.f, 0.f);
			sprite.m_size.set(0.2f, 0.3f);
			sprite.m_position.set(base + sprite.m_size.x / 2.f, 8.2f + sprite.m_size.y / 2.f);
			
			sprite.render();
		}
		
		if (elapsed > 1.f)
		{
			ThString name(20);
			name.format("data/member%u.png", m_member);
			
			m_textMaterial->setTextureName(name);
			m_textMaterial->apply();
			
			sprite.m_size.set(3.f, 1.5f);
			sprite.m_position.set(base + sprite.m_size.x / 2.f + 0.4f, 8.2f + sprite.m_size.y / 2.f);
			
			sprite.render();
		}
	}
	
	glPopAttrib();

	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);

	return ThEffectObject::render(target);
}

bool DemoCredits::notify(unsigned int attributeID, const char* value)
{
	switch (attributeID)
	{
		case MEMBER:
		{
			m_moment = getTimePast();
			m_member = ThString::convertToInt(value);
			m_active = true;
			return true;
		}
		
		case EXPLODE:
		{
			m_system.start();
			m_exploded = true;
			break;
		}
	}

	return ThEffectObject::notify(attributeID, value);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

DemoOlive::DemoOlive(ThEffectType* type, const char* name):
	ThEffectObject(type, name)
{
	m_speed = 200.f;
	m_vector.set(1.f, 1.f, 1.f);
	m_vector.normalize();
}

bool DemoOlive::open(void)
{
	if (!System->getDisplay()->registerObject(this, THLAYER_NORMAL))
		return false;

	m_mesh = System->getEngine()->findMesh("olive");
	if (!m_mesh)
		return false;
		
	IThMaterial* material;
	
	material = System->getEngine()->findMaterial("olive");
	if (!material)
		return false;
		
	material->setShadeModel(GL_SMOOTH);

	m_light = System->getEngine()->createLight(getName());
	if (!m_light)
		return false;

	return true;
}

bool DemoOlive::update(float deltaTime)
{
	return ThEffectObject::update(deltaTime);
}

bool DemoOlive::render(IThCanvas* target)
{
	m_light->enable();

	glPushMatrix();
	glTranslatef(0.f, 0.f, -12.f);
	glRotatef(getTimePast() * m_speed, m_vector.x, m_vector.y, m_vector.z);

	m_mesh->render();

	glPopMatrix();

	m_light->disable();

	return ThEffectObject::render(target);
}

bool DemoOlive::notify(unsigned int attributeID, const char* value)
{
	switch (attributeID)
	{
		case SPEED:
		{
			m_speed = randf() * 300.f + 200.f;
			break;
		}
		
		case DIRECTION:
		{
			m_vector.set(randf() - 0.5f, randf() - 0.5f, randf() - 0.5f);
			m_vector.normalize();
			break;
		}
	}
	
	return ThEffectObject::notify(attributeID, value);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

DemoHyper::DemoHyper(ThEffectType* type, const char* name):
	ThEffectObject(type, name)
{
	m_angles.allocate(9);

	for (unsigned int i = 0;  i < 9;  i++)
		m_angles[i] = randf() * 360.f;

	m_stopped = false;
}

bool DemoHyper::open(void)
{
	if (!System->getDisplay()->registerObject(this, THLAYER_NORMAL))
		return false;

	m_meshes.allocate(9);

	ThString name(10);

	for (unsigned int i = 0;  i < m_meshes.getCount();  i++)
	{
		name.format("font%u", i);

		m_meshes[i] = System->getEngine()->findMesh(name);
		if (!m_meshes[i])
			return false;
	}

	m_light = System->getEngine()->createLight(getName());
	if (!m_light)
		return false;

	return true;
}

bool DemoHyper::update(float deltaTime)
{
	const float speed = 100.f;

	for (unsigned int i = 0;  i < 9;  i++)
	{
		if (m_stopped)
		{
			if (m_angles[i])
			{
				m_angles[i] = m_angles[i] + deltaTime * speed;

				if (m_angles[i] > 360.f)
					m_angles[i] = 0.f;
			}
		}
		else
			m_angles[i] = fmodf(m_angles[i] + deltaTime * speed, 360.f);
	}

	return ThEffectObject::update(deltaTime);
}

bool DemoHyper::render(IThCanvas* target)
{
	m_light->enable();

	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	gluPerspective(60.f, (float) System->getDisplay()->getContext()->getWidth() / (float) System->getDisplay()->getContext()->getHeight(), 1.f, 1000.f);
	glMatrixMode(GL_MODELVIEW);

	for (unsigned int i = 0;  i < 9;  i++)
	{
		glPushMatrix();
		if (i < 5)
			glTranslatef(i * 4.f - 8.f, 3.f, -15.f);
		else
			glTranslatef((i - 5) * 4.f - 6.f, -3.f, -15.f);
		glRotatef(m_angles[i], 0.f, 1.f, 0.f);

		m_meshes[i]->render();

		glPopMatrix();
	}

	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);

	m_light->disable();

	return ThEffectObject::render(target);
}

bool DemoHyper::notify(unsigned int attributeID, const char* value)
{
	switch (attributeID)
	{
		case STOP:
		{
			m_stopped = true;
			break;
		}
	}

	return ThEffectObject::notify(attributeID, value);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

DemoLens::DemoLens(ThEffectType* type, const char* name):
	ThEffectObject(type, name)
{
	m_positions.allocate(8);

	for (unsigned int i = 0;  i < 8;  i++)
		m_positions[i].set(1.25f + randf() * 1.5f, randf() * 1.1f - 0.05f);
}

bool DemoLens::open(void)
{
	if (!System->getDisplay()->registerObject(this, THLAYER_NORMAL))
		return false;

	m_texture = System->getDisplay()->createCanvasTexture(64, 64, "lens");
	if (!m_texture)
		return false;

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);

	m_canvas = System->getDisplay()->createTextureCanvas(m_texture);
	if (!m_canvas)
		return false;

	m_faceMaterial = System->getEngine()->createMaterial("lens face");
	if (!m_faceMaterial)
		return false;

	m_faceMaterial->setBlending(true);
	m_faceMaterial->setSrcBlend(GL_SRC_ALPHA);
	m_faceMaterial->setDstBlend(GL_ONE_MINUS_SRC_ALPHA);
	m_faceMaterial->setCombineMode(GL_REPLACE);

	m_symbolMaterial = System->getEngine()->createMaterial("lens symbol");
	if (!m_symbolMaterial)
		return false;

	m_symbolMaterial->setBlending(true);
	m_symbolMaterial->setSrcBlend(GL_SRC_ALPHA);
	m_symbolMaterial->setDstBlend(GL_ONE_MINUS_SRC_ALPHA);
	m_symbolMaterial->setCombineMode(GL_REPLACE);

	m_renderMaterial = System->getEngine()->createMaterial("lens render");
	if (!m_renderMaterial)
		return false;

	m_renderMaterial->setTextureName("lens");
	m_renderMaterial->setCombineMode(GL_DECAL);

	m_symbols.allocate(8);
	
	ThString name(20);
	
	for (unsigned int i = 0;  i < m_symbols.getCount();  i++)
	{
		name.format("data/symbol%u.png", i);
		
		m_symbols[i] = System->getDisplay()->findTexture(name);
		if (!m_symbols[i])
			return false;
	}
	
	m_faces.allocate(7);
	
	for (unsigned int i = 0;  i < m_faces.getCount();  i++)
	{
		name.format("data/face%u.png", i);
		
		m_faces[i] = System->getDisplay()->findTexture(name);
		if (!m_faces[i])
			return false;
	}

	return true;
}

bool DemoLens::update(float deltaTime)
{
	for (unsigned int i = 0;  i < 8;  i++)
	{
		m_positions[i].x -= deltaTime * 0.5f;
		if (m_positions[i].x < -0.25f)
			m_positions[i].set(1.25f, randf() * 1.1f - 0.05f);
	}

	return ThEffectObject::update(deltaTime);
}

bool DemoLens::render(IThCanvas* target)
{
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	gluOrtho2D(0.f, 1.f, 0.f, 1.f);
	glMatrixMode(GL_MODELVIEW);

	m_canvas->apply();

	render();

	m_canvas->update();

	target->apply();

	render();

	m_renderMaterial->apply();

	ThSprite2 sprite;
	sprite.m_area.set(0.1f, 0.1f, 0.8f, 0.8f);
	sprite.m_position.set(0.5f, 0.5f);
	sprite.m_size.set(0.8f, 0.8f);

	sprite.render();
	
	glPushAttrib(GL_SCISSOR_BIT);
	glEnable(GL_SCISSOR_TEST);
	glScissor((unsigned int) (0.1f * target->getWidth()), (unsigned int) (0.1f * target->getHeight()), (unsigned int) (0.8f * target->getWidth()), (unsigned int) (0.8f * target->getHeight()));
	
	const float time = (getTimePast() * 0.8f);
	const unsigned int index = (unsigned int) time % 7;
	const float progress = time - (float) (unsigned int) time;
	
	const float ratio = (3.f / 4.f) * (1.f + ((float) m_faces[index]->getWidth() / (float) m_faces[index]->getHeight() - 1.f) * 0.5f);
	
	sprite.m_area.set(0.f, 0.f, 1.f, -1.f);
	sprite.m_position.set(0.7f, sinf(progress * (float) M_PI) * 0.75f - 0.4f);
	sprite.m_size.set(0.5f * ratio, 0.5f);
	
	m_faceMaterial->setTextureName(m_faces[index]->getName());
	m_faceMaterial->apply();
	
	sprite.render();
	
	glPopAttrib();
	
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);

	return ThEffectObject::render(target);
}

bool DemoLens::notify(unsigned int attributeID, const char* value)
{
	return ThEffectObject::notify(attributeID, value);
}

void DemoLens::render(void)
{
	glClear(GL_COLOR_BUFFER_BIT);

	ThSprite2 sprite;
	sprite.m_size.set(0.25f, 0.25f);
	sprite.m_area.set(0.f, 0.f, 1.f, -1.f);

	for (unsigned int i = 0;  i < m_symbols.getCount();  i++)
	{
		sprite.m_position = m_positions[i];

		m_symbolMaterial->setTextureName(m_symbols[i]->getName());
		m_symbolMaterial->apply();

		sprite.render();
	}
}

///////////////////////////////////////////////////////////////////////////////////////////////////

DemoGreets::DemoGreets(ThEffectType* type, const char* name):
	ThEffectObject(type, name)
{
	m_offset.set(0.f, 1.f);
	m_size.set(0.f, (float) M_PI);
	m_index = 0;

	m_points.allocate(200);

	for (unsigned int i = 1;  i < m_points.getCount();  i++)
	{
		m_points[i].position = randf() * (10.f + (float) M_PI) - (float) M_PI * 0.5f;
		m_points[i].offset   = m_offset.generate();
		m_points[i].size     = m_size.generate();
		m_points[i].index    = i * 12 / 200;
	}

	m_points[0].position = - (float) M_PI * 0.5f;
	m_points[0].offset   = 0.f;
	m_points[0].size     = (float) M_PI;
	m_points[0].index    = 12;
}

bool DemoGreets::open(void)
{
	if (!System->getDisplay()->registerObject(this, THLAYER_NORMAL))
		return false;

	m_textures.allocate(13);

	ThString name(20);

	for (unsigned int i = 0;  i < m_textures.getCount();  i++)
	{
		name.format("data/blob%u.png", i);

		m_textures[i] = System->getDisplay()->findTexture(name);
		if (!m_textures[i])
			return false;
	}

	m_texts.allocate(10);

	for (unsigned int i = 0;  i < m_texts.getCount();  i++)
	{
		name.format("data/greets%u.png", i);

		m_texts[i] = System->getDisplay()->findTexture(name);
		if (!m_texts[i])
			return false;
	}

	m_material = System->getEngine()->createMaterial(getName());
	if (!m_material)
		return false;

	m_material->setBlending(true);
	m_material->setSrcBlend(GL_SRC_ALPHA);
	m_material->setDstBlend(GL_ONE_MINUS_SRC_ALPHA);
	m_material->setCombineMode(GL_REPLACE);
	m_material->setTextureName(m_textures[m_points[0].index]->getName());
	
	return true;
}

bool DemoGreets::update(float deltaTime)
{
	for (unsigned int i = 0;  i < m_points.getCount();  i++)
	{
		if (i)
			m_points[i].position += deltaTime * 3.f;
		else
			m_points[i].position += deltaTime * 6.f;

		if (m_points[i].position - (float) M_PI * 0.5f > 10.f)
		{
			m_points[i].position = -(float) M_PI * 0.5f;
			if (!i)
				m_index = (m_index + 1) % m_texts.getCount();
		}
	}

	return ThEffectObject::update(deltaTime);
}

bool DemoGreets::render(IThCanvas* target)
{
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	gluOrtho2D(0.f, 10.f, 0.f, 10.f);
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();

	m_material->apply();

	const float height = 2.f;

	for (unsigned int i = 0;  i < m_points.getCount();  i++)
	{
		Point& point = m_points[i];

		if (i)
		{
			if (point.index != m_points[i - 1].index)
				m_textures[point.index]->apply();
		}

		const float offset = fabsf(sinf(m_points[i].size + getTimePast() * 3) + 0.1f) * 0.5f;

		glBegin(GL_QUADS);

		ThVector2 center(point.position, sinf(point.position + getTimePast() + point.offset) * height + height);

		glTexCoord2f(0.f, 0.f);
		glVertex2f(center.x - offset, center.y - offset);

		glTexCoord2f(1.f, 0.f);
		glVertex2f(center.x + offset, center.y - offset);

		glTexCoord2f(1.f, 1.f);
		glVertex2f(center.x + offset, center.y + offset);

		glTexCoord2f(0.f, 1.f);
		glVertex2f(center.x - offset, center.y + offset);

		glEnd();

		if (!i)
		{
			glPushAttrib(GL_CURRENT_BIT);
			glLineWidth(target->getWidth() / 150.f);
			glColor4f(0.f, 0.f, 0.f, 1.f);
			glBegin(GL_LINES);
			glVertex2f(center.x, center.y);
			glVertex2f(center.x, center.y + 3.f);
			glEnd();
			glPopAttrib();

			ThSprite2 sprite;
			sprite.m_area.set(0.f, 0.f, 1.f, -1.f);
			sprite.m_position.set(center.x, center.y + 5.f);
			sprite.m_size.set(3.f, 2.f);

			m_texts[m_index]->apply();
			sprite.render();
		}
	}

	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();

	return ThEffectObject::render(target);
}

bool DemoGreets::notify(unsigned int attributeID, const char* value)
{
	return ThEffectObject::notify(attributeID, value);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

int main(int argc, char* argv[])
{
	if (!initializeTheresa(argc, argv))
		return -1;

	int result = start();
	
	shutdownTheresa();
	return result;
}

int start(void)
{
	srand(time(NULL));
	
	if (!System->openDisplay())
		return -1;
		
	System->getDisplay()->getContext()->setTitle("We own your language");

	if (!System->openEffect())
		return -1;

	if (!System->openEngine())
		return -1;
		
	ThPtr<DemoApp> app = new DemoApp();
	
	if (!app->open())
		return -1;

	System->getMusic()->start();
	
	System->start();
	
	return 0;
}
