///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////
#ifndef SHARED_THENGINE_H
#define SHARED_THENGINE_H
///////////////////////////////////////////////////////////////////////////////////////////////////

class ThVertex
{
public:
// data
	ThVector2													m_texCoord;
	ThVector3													m_normal;
	ThVector3													m_vertex;
};

//-------------------------------------------------------------------------------------------------

class ThTriangle
{
public:
// data
	unsigned int											m_indices[3];
	ThVector3													m_normal;
	ThVector3													m_center;
	IThMaterial*											m_material;
};

//-------------------------------------------------------------------------------------------------

class ThGeometry
{
public:
// data
	ThBlock<unsigned int>							m_indices;
	ThBlock<ThTriangle*>							m_triangles;
	IThMaterial*											m_material;
	unsigned int											m_mode;
};

//-------------------------------------------------------------------------------------------------

class ThMeshData
{
public:
// data
	ThBlock<ThVertex>									m_vertices;
	ThBlock<ThTriangle>								m_triangles;
	ThBlock<ThTriangle*>							m_transparent;
	ThBlock<ThGeometry>								m_geometries;
	ThSphere													m_sphere;
};

//-------------------------------------------------------------------------------------------------

class ThLightData
{
public:
// constructors
																		ThLightData(void);
// methods
	void															reset(void);
// constants
	enum { DIRECTIONAL, POSITIONAL };
// data
	ThVector4													m_ambient;
	ThVector4													m_diffuse;
	ThVector4													m_specular;
	ThVector3													m_position;
	ThVector3													m_direction;
	unsigned int											m_type;
};

//-------------------------------------------------------------------------------------------------

class ThMaterialData
{
public:
// constructors
																		ThMaterialData(void);
// methods
	void															reset(void);
// data
	bool															m_culling;
	bool															m_lighting;
	bool															m_blending;
	bool															m_depthTesting;
	bool															m_depthWriting;
	ThVector4													m_ambientColor;
	ThVector4													m_diffuseColor;
	ThVector4													m_specularColor;
	unsigned int											m_shadeModel;
	unsigned int											m_depthFunction;
	unsigned int											m_srcFactor;
	unsigned int											m_dstFactor;
	unsigned int											m_culledFace;
	unsigned int											m_combineMode;
	ThVector4													m_combineColor;
	unsigned int											m_minFilter;
	unsigned int											m_magFilter;
	bool															m_coordGen[4];
	unsigned int											m_genModes[4];
	ThVector4													m_objPlanes[4];
	ThVector4													m_eyePlanes[4];
	ThString													m_textureName;
	unsigned int											m_textureHash;
};

//-------------------------------------------------------------------------------------------------

class ThEngineNode : public ThListItem<ThEngineNode>
{
public:
// constructors
																		ThEngineNode(void);
	virtual														~ThEngineNode(void);
// methods
	void															attachChild(ThEngineNode* child);
// attributes
	ThEngineNode*											getParent(void);
	ThEngineNode*											getFirstChild(void);
	const ThTransform&								getLocalTransform(void) const;
	void															setLocalTransform(const ThTransform& transform);
	const ThTransform&								getWorldTransform(void);
	const ThSphere&										getNodeBound(void);
	const ThSphere&										getTreeBound(void);
private:
// methods
	void															invalidateWorld(void);
	bool															updateWorld(void);
	void															updateBound(void);
// data
	bool															m_invalidWorld;
	bool															m_invalidBound;
	ThEngineNode*											m_parent;
	ThList<ThEngineNode>							m_children;
	ThTransform												m_local;
	ThTransform												m_world;
	ThSphere													m_nodeBound;
	ThSphere													m_treeBound;
};

///////////////////////////////////////////////////////////////////////////////////////////////////

class IThMesh
{
public:
// constructors
	virtual														~IThMesh(void);
// methods
	virtual void											render(void) = 0;
// attributes
	virtual ThVertex*									getVertex(unsigned int index) = 0;
	virtual unsigned int							getVertexCount(void) const = 0;
	virtual ThTriangle*								getTriangle(unsigned int index) = 0;
	virtual unsigned int							getTriangleCount(void) const = 0;
	virtual ThGeometry*								getGeometry(unsigned int index) = 0;
	virtual unsigned int							getGeometryCount(void) const = 0;
	virtual const char*								getName(void) const = 0;
	virtual unsigned int							getHash(void) const = 0;
	virtual ThMeshData&								getData(void) = 0;
};

//-------------------------------------------------------------------------------------------------

class IThLight
{
public:
// constructors
	virtual 													~IThLight(void);
// methods
	virtual bool											enable(void) = 0;
	virtual void											disable(void) = 0;
// attributes
	virtual bool											isEnabled(void) const = 0;
	virtual const ThVector4&					getAmbientColor(void) const = 0;
	virtual void											setAmbientColor(const ThVector4& color) = 0;
	virtual const ThVector4&					getDiffuseColor(void) const = 0;
	virtual void											setDiffuseColor(const ThVector4& color) = 0;
	virtual const ThVector4&					getSpecularColor(void) const = 0;
	virtual void											setSpecularColor(const ThVector4& color) = 0;
	virtual const ThVector3&					getPosition(void) const = 0;
	virtual void											setPosition(const ThVector3& position) = 0;
	virtual const ThVector3&					getDirection(void) const = 0;
	virtual void											setDirection(const ThVector3& direction) = 0;
	virtual unsigned int							getType(void) const = 0;
	virtual void											setType(unsigned int type) = 0;
	virtual const char*								getName(void) const = 0;
	virtual unsigned int							getHash(void) const = 0;
	virtual ThLightData&							getData(void) = 0;
};

//-------------------------------------------------------------------------------------------------

class IThMaterial
{
public:
// constructors
	virtual 													~IThMaterial(void);
// methods
	virtual void											apply(void) = 0;
// attributes
	virtual bool											isCulling(void) = 0;
	virtual bool											isLighting(void) = 0;
	virtual bool											isBlending(void) = 0;
	virtual bool											isDepthTesting(void) = 0;
	virtual bool											isDepthWriting(void) = 0;
	virtual void											setCulling(bool enabled) = 0;
	virtual void											setLighting(bool enabled) = 0;
	virtual void											setBlending(bool enabled) = 0;
	virtual unsigned int							getSrcBlend(void) = 0;
	virtual void											setSrcBlend(unsigned int factor) = 0;
	virtual unsigned int							getDstBlend(void) = 0;
	virtual void											setDstBlend(unsigned int factor) = 0;
	virtual void											setDepthTesting(bool enabled) = 0;
	virtual void											setDepthWriting(bool enabled) = 0;
	virtual unsigned int							getDepthFunc(void) = 0;
	virtual void											setDepthFunc(unsigned int func) = 0;
	virtual const ThVector4&					getAmbientColor(void) = 0;
	virtual void											setAmbientColor(const ThVector4& color) = 0;
	virtual const ThVector4&					getDiffuseColor(void) = 0;
	virtual void											setDiffuseColor(const ThVector4& color) = 0;
	virtual const ThVector4&					getSpecularColor(void) = 0;
	virtual void											setSpecularColor(const ThVector4& color) = 0;
	virtual unsigned int							getShadeModel(void) = 0;
	virtual void											setShadeModel(unsigned int mode) = 0;
	virtual unsigned int							getCulledFace(void) = 0;
	virtual void											setCulledFace(unsigned int face) = 0;
	virtual unsigned int							getCombineMode(void) = 0;
	virtual void											setCombineMode(unsigned int mode) = 0;
	virtual const ThVector4&					getCombineColor(void) = 0;
	virtual void											setCombineColor(const ThVector4& color) = 0;
	virtual bool											getCoordGen(unsigned int axis) = 0;
	virtual void											setCoordGen(unsigned int axis, bool enabled) = 0;
	virtual unsigned int							getCoordGenMode(unsigned int axis) = 0;
	virtual void											setCoordGenMode(unsigned int axis, unsigned int mode) = 0;
	virtual const ThVector4&					getCoordObjPlane(unsigned int axis) = 0;
	virtual void											setCoordObjPlane(unsigned int axis, const ThVector4& plane) = 0;
	virtual const ThVector4&					getCoordEyePlane(unsigned int axis) = 0;
	virtual void											setCoordEyePlane(unsigned int axis, const ThVector4& plane) = 0;
	virtual const char*								getTextureName(void) = 0;
	virtual void											setTextureName(const char* name) = 0;
	virtual const char*								getName(void) const = 0;
	virtual unsigned int							getHash(void) const = 0;
	virtual ThMaterialData&						getData(void) = 0;
};

//-------------------------------------------------------------------------------------------------

class IThRenderQueue
{
public:
// constructors
	virtual														~IThRenderQueue(void);
// methods
	virtual void											prepare(void) = 0;
	virtual void											render(void) = 0;
	virtual void											reset(void) = 0;
// object methods
	virtual void											addMesh(IThMesh* mesh, const ThMatrix4& transform) = 0;
	virtual void											addGeometry(const ThGeometry* geometry, const ThMatrix4& transform) = 0;
	virtual void											addTriangle(const ThTriangle* triangle, const ThMatrix4& transform) = 0;
	virtual void											addLight(IThLight* light, const ThMatrix4& transform) = 0;
};

//-------------------------------------------------------------------------------------------------

class IThEngine
{
public:
// constructors
	virtual														~IThEngine(void);
// methods
	virtual bool											loadResources(const char* fileName) = 0;
	virtual bool											loadResources(IThStream* stream) = 0;
// object methods
	virtual IThMesh*									createMesh(const char* name) = 0;
	virtual IThMesh*									findMesh(const char* name) = 0;
	virtual IThLight*									createLight(const char* name) = 0;
	virtual IThLight*									findLight(const char* name) = 0;
	virtual IThMaterial*							createMaterial(const char* name) = 0;
	virtual IThMaterial*							findMaterial(const char* name) = 0;
	virtual IThRenderQueue*						createRenderQueue(void) = 0;
// attributes
	virtual IThDisplay*								getDisplay(void) = 0;
	virtual ThEngineNode*							getRootNode(void) = 0;
// static methods
	static IThEngine*									createInstance(IThDisplay* display);
};

///////////////////////////////////////////////////////////////////////////////////////////////////
#endif /*SHARED_THENGINE_H*/
///////////////////////////////////////////////////////////////////////////////////////////////////
