///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////
#ifndef SHARED_THMEMORY_H
#define SHARED_THMEMORY_H
///////////////////////////////////////////////////////////////////////////////////////////////////

template <typename T>
class ThPtr
{
public:
// constructors
	inline														ThPtr(void);
	inline														ThPtr(T* object);
	inline														ThPtr(const ThPtr<T>& source);
	inline														~ThPtr(void);
// methods
	inline T*													detach(void);
	inline void												release(void);
// operators
	inline T*													operator -> (void);
	inline const T*										operator -> (void) const;
	inline														operator T* (void);
	inline														operator const T* (void) const;
	inline ThPtr<T>&									operator = (T* object);
	inline ThPtr<T>&									operator = (const ThPtr<T>& source);
// attributes
	inline T*													getObject(void);
	inline const T*										getObject(void) const;
private:
// data
	T*																m_object;
};

//-------------------------------------------------------------------------------------------------

template <typename T>
class ThRef
{
public:
// constructors
	inline														ThRef(void);
	inline														ThRef(const ThRef<T>& source);
	inline														~ThRef(void);
// methods
	inline void												release(void);
// operators
	inline T*													operator -> (void) const;
	inline														operator T* (void) const;
	inline ThRef<T>&									operator = (T* object);
	inline ThRef<T>&									operator = (const ThRef<T>& source);
// attributes
	inline T*													getObject(void) const;
private:
// data
	T*																m_object;
};

//-------------------------------------------------------------------------------------------------

template <typename T>
class ThRefObject
{
// friends
	friend class ThRef<T>;
public:
// constructors
	inline														ThRefObject(void);
	inline														~ThRefObject(void);
protected:
// methods
	inline void												reference(void);
	inline bool												release(void);
private:
// data
	unsigned int											m_count;
};

//-------------------------------------------------------------------------------------------------

template <typename T>
class ThBlock
{
public:
// constructors
	inline														ThBlock(void);
	inline explicit										ThBlock(unsigned int count);
	inline														ThBlock(const ThBlock<T>& source);
	inline														~ThBlock(void);
// methods
	inline void												allocate(unsigned int count);
	inline void												resize(unsigned int count);
	inline void												release(void);
// operators
	inline														operator T* (void);
	inline														operator const T* (void) const;
	inline ThBlock<T>&								operator = (const ThBlock<T>& source);
// attributes
	inline T*													getData(void);
	inline const T*										getData(void) const;
	inline unsigned int								getSize(void) const;
	inline unsigned int								getCount(void) const;
private:
// data
	T*																m_data;
	unsigned int											m_count;
};

//-------------------------------------------------------------------------------------------------

typedef ThBlock<char> ThByteBlock;

//-------------------------------------------------------------------------------------------------

template <typename T>
class ThListItem
{
// friends
	friend class ThList<T>;
public:
// constructors
	inline														ThListItem(void);
	inline														ThListItem(const ThListItem<T>& source);
	inline														~ThListItem(void);
// methods
	inline void												detach(void);
// operators
	inline ThListItem<T>&							operator = (const ThListItem<T>& source);
// attributes
	inline bool												isAttached(void) const;
	inline T*													getPrev(void);
	inline const T*										getPrev(void) const;
	inline T*													getNext(void);
	inline const T*										getNext(void) const;
private:
// data
	T*																m_prev;
	T*																m_next;
	ThList<T>*												m_list;
};

//-------------------------------------------------------------------------------------------------

template <typename T>
class ThList
{
// friends
	friend class ThListItem<T>;
public:
// constructors
	inline														ThList(void);
	inline														ThList(const ThList<T>& source);
	inline														~ThList(void);
// methods
	inline void												attachFirst(T* item);
	inline void												attachLast(T* item);
	inline void												attachAt(T* item, unsigned int index);
	inline ThIterator<T>							iterate(void);
	inline ThConstIterator<T>					iterate(void) const;
	inline void												release(void);
// operators
	inline ThList<T>&									operator = (const ThList<T>& source);
// attributes
	inline T*													getFirst(void);
	inline const T*										getFirst(void) const;
	inline T*													getLast(void);
	inline const T*										getLast(void) const;
	inline T*													getAt(unsigned int index);
	inline const T*										getAt(unsigned int index) const;
	inline unsigned int								getSize(void) const;
	inline unsigned int								getCount(void) const;
private:
// data
	T*																m_head;
	unsigned int											m_count;
};

//-------------------------------------------------------------------------------------------------

template <typename T>
class ThHashItem : public ThListItem<T>
{
public:
// constructors
	inline explicit										ThHashItem(unsigned int hash = 0);
	inline														ThHashItem(const ThHashItem<T>& source);
// operators
	inline ThHashItem<T>&							operator = (const ThHashItem<T>& source);
// attributes
	inline unsigned int								getHash(void) const;
	inline void												setHash(unsigned int hash);
private:
// data
	unsigned int											m_hash;
};

//-------------------------------------------------------------------------------------------------

template <typename T, unsigned int N = 10>
class ThHashList
{
public:
// constructors
	inline														ThHashList(void);
	inline														ThHashList(const ThHashList<T,N>& source);
// methods
	inline void												attach(T* item);
	inline void												release(void);
// operators
	inline ThHashList<T,N>&						operator = (const ThHashList<T,N>& source);
// attributes
	inline T*													getItem(unsigned int hash);
	inline const T*										getItem(unsigned int hash) const;
	inline ThList<T>*									getList(unsigned int index);
	inline const ThList<T>*						getList(unsigned int index) const;
	inline unsigned int								getListCount(void) const;
private:
// data
	ThBlock<ThList<T> >								m_root;
};

//-------------------------------------------------------------------------------------------------

template <typename T>
class ThIterator
{
public:
// constructors
	inline														ThIterator(void);
	inline														ThIterator(T* item);
	inline														ThIterator(ThList<T>& list);
// methods
	inline T*													next(void);
	inline T*													detach(void);
	inline void												release(void);
// operators
	inline T*													operator -> (void) const;
	inline														operator T* (void) const;
	inline ThIterator<T>&							operator = (T* item);
// attributes
	inline T*													getItem(void);
	inline const T*										getItem(void) const;
private:
// data
	T*																m_item;
};

//-------------------------------------------------------------------------------------------------

template <typename T>
class ThConstIterator
{
public:
// constructors
	inline														ThConstIterator(void);
	inline														ThConstIterator(const T* item);
	inline														ThConstIterator(const ThList<T>& list);
// methods
	inline const T*										next(void);
	inline const T*										detach(void);
// operators
	inline const T*										operator -> (void) const;
	inline														operator const T* (void) const;
	inline ThConstIterator<T>&				operator = (const T* item);
// attributes
	inline const T*										getItem(void) const;
private:
// data
	const T*													m_item;
};

//-------------------------------------------------------------------------------------------------

template <typename T>
class ThSingleton
{
public:
// operators
	inline T*													operator -> (void);
	inline														operator T* (void);
// methods
	inline void												release(void);
	inline void												attach(T* object);
private:
// data
	ThPtr<T>													m_object;
};

///////////////////////////////////////////////////////////////////////////////////////////////////

/*! \class IThStaticBlock
 *	Encapsulates a static memory block with lock semantics.
 */
/*! void* IThStaticBlock::lock(void)
 *	Locks the memory block and returns a pointer to it.
 *	\return A pointer to the locked memory block, or \c NULL if an error ocurred.
 */
/*! void IThStaticBlock::unlock(void)
 *	Unlocks the memory block, invalidating the pointer returned by the corresponding lock.
 */
/*! unsigned int IThStaticBlock::getSize(void) const
 *	\return The size, in bytes, of the memory block.
 */
/*! IThStaticBlock* IThStaticBlock::createInstance(unsigned int size)
 *	Creates and returns an instance of a static memory block object with the specified size.
 *	\param size The size of the memory block.
 *	\return The newly created static memory block object.
 */
/*! IThStaticBlock* IThStaticBlock::createInstance(void* data, unsigned int size)
 *	Creates and returns an instance of a static memory block object on top of the specified memory block.
 *	\param data The memory block to create the object on.
 *	\param size The size of the specified memory block.
 *	\return The newly created static memory block object.
 */
class IThStaticBlock
{
public:
// constructors
	virtual														~IThStaticBlock(void);
// methods
	virtual void*											lock(void) = 0;
	virtual void											unlock(void) = 0;
// attributes
	virtual unsigned int							getSize(void) const = 0;
// static methods
	static IThStaticBlock*						createInstance(unsigned int size);
	static IThStaticBlock*						createInstance(void* data, unsigned int size);
};

//-------------------------------------------------------------------------------------------------

class IThDynamicBlock : public IThStaticBlock
{
public:
// constructors
	virtual														~IThDynamicBlock(void);
// methods
	virtual void											allocate(unsigned int size) = 0;
	virtual void											resize(unsigned int size) = 0;
	virtual void											release(void) = 0;
// static methods
	static IThDynamicBlock*						createInstance(unsigned int size);
};

///////////////////////////////////////////////////////////////////////////////////////////////////

#include "ThMemory.hpp"

///////////////////////////////////////////////////////////////////////////////////////////////////
#endif /* SHARED_THMEMORY_H */
///////////////////////////////////////////////////////////////////////////////////////////////////
