///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <shared/ThCore.h>
#include <shared/ThMemory.h>
#include <shared/ThString.h>
#include <shared/ThError.h>
#include <shared/ThServer.h>
#include <shared/ThContext.h>

#include <theresa/AGL/ThContext.h>

///////////////////////////////////////////////////////////////////////////////////////////////////

bool enumerateResolutions(ThList<ThResolution>& resolutions)
{
	resolutions.release();
	
	// TODO: implement resolution enumeration!
	
	resolutions.attachFirst(new ThResolution(640, 480, 16, false));
	resolutions.attachFirst(new ThResolution(800, 600, 16, false));
	resolutions.attachFirst(new ThResolution(640, 480, 32, false));
	resolutions.attachFirst(new ThResolution(800, 600, 32, false));
	
	return true;
}

bool requestContextMode(ThContextMode& contextMode)
{
	// TODO: request user selection!

	contextMode.set(ThResolution(THERESA_DEFAULT_WIDTH, THERESA_DEFAULT_HEIGHT, THERESA_DEFAULT_DEPTH, THERESA_DEFAULT_WINDOWED), 16, 0, 32);

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThContext constructors -------------------------------------------------------------------------

ThContext::ThContext(void):
	ThServerObject(THID_CONTEXT)
{
	m_window  = NULL;
	m_handler = NULL;
	m_context = NULL;
	
	m_mode.m_resolution.m_windowed = true;
}

ThContext::~ThContext(void)
{
	close();
}

// ThContext methods ------------------------------------------------------------------------------

bool ThContext::open(const ThContextMode& mode, unsigned int listenerID)
{
	close();
	
	// load resources from nib
	{
		IBNibRef nib;

#if THERESA_MACOSX_FRAMEWORK
		CFBundleRef bundle = CFBundleGetBundleWithIdentifier(CFSTR("org.hypercube.theresa"));
		if (!bundle)
		{
			Error->display("Context", "Unable to retrieve the Theresa framework bundle.");
			return false;
		}

		if (CreateNibReferenceWithCFBundle(bundle, CFSTR("theresa"), &nib) != noErr)
		{
			Error->display("Context", "Unable to retrieve Nib-file from the Theresa framework bundle.");
			return false;
		}
#else
		if (CreateNibReference(CFSTR("theresa"), &nib) != noErr)
		{
			Error->display("Context", "Unable to retrieve the main bundle Nib-file.");
			return false;
		}
#endif

		if (mode.m_resolution.m_windowed)
		{
			if (CreateWindowFromNib(nib, CFSTR("MainWindow"), &m_window) != noErr)
			{
				Error->display("Context", "Unable to create window.");
				return false;
			}
		}
	
		DisposeNibReference(nib);
		nib = NULL;
	}
	
	if (mode.m_resolution.m_windowed)
	{
		EventTypeSpec events[] = {
			{ kEventClassWindow, kEventWindowResizeCompleted }, 
			{ kEventClassWindow, kEventWindowClose },
			{ kEventClassKeyboard, kEventRawKeyDown } };
		
		m_handler = NewEventHandlerUPP(eventCallback);
		
		if (InstallWindowEventHandler(m_window, m_handler, sizeof(events) / sizeof(EventTypeSpec), events, this, NULL) != noErr)
		{
			Error->display("Context", "Unable to install window event handler.");
			return false;
		}
	
		SizeWindow(m_window, mode.m_resolution.m_width, mode.m_resolution.m_height, true);
		ShowWindow(m_window);
		SetPortWindowPort(m_window);
	}
	
	// create rendering context
	{
		// TODO: enumerate all active devices!
	
		GDHandle device = DMGetFirstScreenDevice(true);
	
		GLint attributes[512];
		unsigned int attributeIndex = 0;
		
		attributes[attributeIndex++] = AGL_RGBA;
		attributes[attributeIndex++] = AGL_DOUBLEBUFFER;
		attributes[attributeIndex++] = AGL_ACCELERATED;
		attributes[attributeIndex++] = AGL_PIXEL_SIZE;
		attributes[attributeIndex++] = mode.m_resolution.m_depth;
		
		if (!mode.m_resolution.m_windowed)
			attributes[attributeIndex++] = AGL_FULLSCREEN;
		
		if (mode.m_depthBits)
		{
			attributes[attributeIndex++] = AGL_DEPTH_SIZE;
			attributes[attributeIndex++] = mode.m_depthBits;
		}
		
		if (mode.m_stencilBits)
		{
			attributes[attributeIndex++] = AGL_STENCIL_SIZE;
			attributes[attributeIndex++] = mode.m_stencilBits;
		}
		
		attributes[attributeIndex++] = AGL_NO_RECOVERY;
		attributes[attributeIndex] = AGL_NONE;
		
		AGLPixelFormat format = aglChoosePixelFormat(&device, 1, attributes);
		if (!format)
		{
			Error->display("Context", "Unable to select OpenGL pixel format.%s%s", THERESA_NEWLINE, aglErrorString(aglGetError()));
			return false;
		}
	
		m_context = aglCreateContext(format, NULL);
	
		aglDestroyPixelFormat(format);
		format = NULL;
	
		if (!m_context)
		{
			Error->display("Context", "Unable to create OpenGL rendering context.%s%s", THERESA_NEWLINE, aglErrorString(aglGetError()));
			return false;
		}
	}

	// set context drawable
	{
		if (!mode.m_resolution.m_windowed)
		{
			aglSetFullScreen(m_context, mode.m_resolution.m_width, mode.m_resolution.m_height, 0, 0);
			if (!aglGetDrawable(m_context))
			{
				Error->display("Context", "Unable to set fullscreen drawable for OpenGL context.%s%s", THERESA_NEWLINE, aglErrorString(aglGetError()));
				return false;
			}

			HideCursor();
		}
		else
			aglSetDrawable(m_context, GetWindowPort(m_window));
	}
	
	aglSetCurrentContext(m_context);
	
	// reset matrices
	{
		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();

		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();
	}

	// list extensions
	{
		if (const char* extensions = (const char*) glGetString(GL_EXTENSIONS))
		{
			ThString extension;

			while (extensions = extension.copyToken(extensions))
				m_extensions.attachFirst(new ThStringItem(extension));
		}
	}

	m_mode       = mode;
	m_listenerID = listenerID;

	setTitle("");

	return true;
}

void ThContext::close(void)
{
	if (m_context)
	{
		if (!m_mode.m_resolution.m_windowed)
			ShowCursor();

		aglSetCurrentContext(NULL);
		aglSetDrawable(m_context, NULL);
		aglDestroyContext(m_context);
		m_context = NULL;
	}
	
	if (m_window)
	{
		DisposeWindow(m_window);
		m_window = NULL;
	}
	
	if (m_handler)
	{
		DisposeEventHandlerUPP(m_handler);
		m_handler = NULL;
	}
	
	m_mode.reset();
	m_mode.m_resolution.m_windowed = true;
}

// ThContext interface methods --------------------------------------------------------------------

void ThContext::apply(void)
{
	aglSetCurrentContext(m_context);
}

void ThContext::update(void)
{
	glFinish();

	if (unsigned int error = glGetError())
	{
		Error->display("Context", "OpenGL reported an error during update: %s", gluErrorString(error));
		Server->postMessage(THMSG_REQUEST_EXIT, THID_ANNOUNCE, THID_CONTEXT);
	}

	aglSwapBuffers(m_context);
}

// ThContext interface extension methods ----------------------------------------------------------

bool ThContext::findExtension(const char* name) const
{
	if (m_extensions.findNoCase(name))
		return true;

	return false;
}

ThEntryPoint ThContext::findEntryPoint(const char* name) const
{
	// TODO: retrieve extension entry point!

	return (ThEntryPoint) NULL;
}

// ThContext interface attributes -----------------------------------------------------------------

const char* ThContext::getTitle(void) const
{
	return m_title;
}

void ThContext::setTitle(const char* title)
{
	m_title = title;

	ThString caption;

	caption = title;
	if (!caption.isEmpty())
		caption.append(" - ");

	caption.append("Theresa");
	
	if (CFStringRef string = CFStringCreateWithCString(NULL, caption, CFStringGetSystemEncoding()))
	{
		SetWindowTitleWithCFString(m_window, string);
		
		CFRelease(string);
		string = NULL;
	}
}

unsigned int ThContext::getWidth(void) const
{
	return m_mode.m_resolution.m_width;
}

unsigned int ThContext::getHeight(void) const
{
	return m_mode.m_resolution.m_height;
}

const ThContextMode& ThContext::getMode(void) const
{
	return m_mode;
}

// ThContext callbacks ----------------------------------------------------------------------------

bool ThContext::receive(const IThMessage* message)
{
	return ThServerObject::receive(message);
}

// ThContext static methods -----------------------------------------------------------------------

OSStatus ThContext::eventCallback(EventHandlerCallRef handler, EventRef event, void* data)
{
	ThContext* object = reinterpret_cast<ThContext*>(data);
	
	switch (GetEventKind(event))
	{
		case kEventWindowClose:
		{
			object->sendMessage(THMSG_REQUEST_EXIT, THID_ANNOUNCE);
			
			return noErr;
		}
	}
	
	OSStatus result = CallNextEventHandler(handler, event);
		
	switch (GetEventKind(event))
	{
		case kEventWindowResizeCompleted:
		{
			aglUpdateContext(object->m_context);

			WindowRef window;
			
			GetEventParameter(event, kEventParamDirectObject, typeWindowRef, NULL, sizeof(WindowRef), NULL, &window);
			
			Rect rect;
			
			GetWindowPortBounds(window, &rect);
			
			object->m_mode.m_resolution.m_width  = rect.right;
			object->m_mode.m_resolution.m_height = rect.bottom;
			
			glViewport(0, 0, rect.right, rect.bottom);
			
			if (object->m_listenerID != THID_INVALID)
				object->sendMessage(THMSG_CONTEXT_RESIZE, object->m_listenerID);
				
			break;
		}
		
		case kEventParamKeyCode:
		{
			unsigned int keyCode;
			
			GetEventParameter(event, kEventParamKeyCode, typeUInt32, NULL, sizeof(unsigned int), NULL, &keyCode);

			// TODO: I don't know...
			break;
		}
	}

	return result;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
