///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <shared/ThCore.h>
#include <shared/ThMemory.h>
#include <shared/ThString.h>
#include <shared/ThError.h>

#include <Carbon/Carbon.h>

#include <unistd.h>
#include <sys/param.h>

///////////////////////////////////////////////////////////////////////////////////////////////////

static EventHandlerUPP s_handler = NULL;

///////////////////////////////////////////////////////////////////////////////////////////////////

static OSStatus eventCallback(EventHandlerCallRef handler, EventRef event, void* data);

///////////////////////////////////////////////////////////////////////////////////////////////////

bool initializePlatform(int argc, char* argv[])
{
	InitCursor();
	
	// set current directory
	{
		CFBundleRef bundle = CFBundleGetMainBundle();
		if (!bundle)
		{
			displayErrorMessage("System", "Unable to retreive main bundle.");
			return false;
		}
		
		CFURLRef url = CFBundleCopyBundleURL(bundle);
		if (!url)
		{
			displayErrorMessage("System", "Unable to retreive main bundle URL.");
			return false;
		}
		
		CFStringRef path = CFURLCopyFileSystemPath(url, kCFURLPOSIXPathStyle);
		if (!path)
		{
			CFRelease(url);
			displayErrorMessage("System", "Unable to retreive main bundle path.");
			return false;
		}
		
		ThString buffer(MAXPATHLEN);

		// convert CFString to ThString
		{		
			const bool result = CFStringGetCString(path, buffer, buffer.getSize(), CFStringGetSystemEncoding());
			
			CFRelease(url);
			CFRelease(path);
			
			if (!result)
			{
				displayErrorMessage("System", "Unable to retreive main bundle path.");
				return false;
			}
		}

		// completely unneccessary safety-hack
		if (const unsigned int length = buffer.length())
		{
			char* p = buffer + length - 1;
			if (*p == '/')
				*p = '\0';
		}

		// remove bundle from path
		if (char* p = buffer.reverseFind('/'))
			*p = '\0';
		
		chdir(buffer);
	}

	// load menu bar from resource file
	{
		IBNibRef nib;
	
#if THERESA_MACOSX_FRAMEWORK
		CFBundleRef bundle = CFBundleGetBundleWithIdentifier(CFSTR("org.hypercube.theresa"));
		if (!bundle)
		{
			displayErrorMessage("System", "Unable to retreive the Theresa framwork bundle.");
			return false;
		}
	
		if (CreateNibReferenceWithCFBundle(bundle, CFSTR("theresa"), &nib) != noErr)
		{
			displayErrorMessage("System", "Unable to retreive Nib-file from the Theresa framework bundle.");
			return false;
		}
#else
		if (CreateNibReference(CFSTR("theresa"), &nib) != noErr)
		{
			displayErrorMessage("System", "Unable to retreive main bundle Nib-file.");
			return false;
		}
#endif
	
		if (SetMenuBarFromNib(nib, CFSTR("MenuBar")) != noErr)
		{
			displayErrorMessage("System", "Unable to set menu bar from Nib-file.");
			return false;
		}
	
		DisposeNibReference(nib);
		nib = NULL;
	}
	
	// attach about box event handler
	{
		EventTypeSpec events[] = {
			{ kEventClassCommand, kEventProcessCommand } };
		
		s_handler = NewEventHandlerUPP(eventCallback);
		
		if (InstallApplicationEventHandler(s_handler, sizeof(events) / sizeof(EventTypeSpec), events, NULL, NULL) != noErr)
		{
			Error->display("System", "Unable to install application event handler.");
			return false;
		}
	}
	
	return true;
}

void shutdownPlatform(void)
{
}

static OSStatus eventCallback(EventHandlerCallRef handler, EventRef event, void* data)
{
	OSStatus result = CallNextEventHandler(handler, event);
		
	switch (GetEventKind(event))
	{
		case kEventProcessCommand:
		{
			HICommand command;
			GetEventParameter(event, kEventParamDirectObject, typeHICommand, NULL, sizeof(HICommand), NULL, &command);
			
			switch (command.commandID)
			{
				case 'abou':
				{
					IBNibRef nib = NULL;
					OSStatus result = noErr;

#if THERESA_MACOSX_FRAMEWORK
					CFBundleRef bundle = CFBundleGetBundleWithIdentifier(CFSTR("org.hypercube.theresa"));
					if (!bundle)
					{
						Error->display("Context", "Unable to open Theresa's resource bundle. Your application bundle may be damaged or incorrectly built.");
						return false;
					}
			
					result = CreateNibReferenceWithCFBundle(bundle, CFSTR("theresa"), &nib);
#else
					result = CreateNibReference(CFSTR("theresa"), &nib);
#endif

					if (result == noErr)
					{
						WindowRef dialog;
	
						if (CreateWindowFromNib(nib, CFSTR("AboutDialog"), &dialog) == noErr)
						{
							ShowWindow(dialog);
							SelectWindow(dialog);
						}
					}
					else
						Error->display("Context", "Unable to create about dialog from resource bundle.");

					if (nib)
						DisposeNibReference(nib);

					break;
				}
			}
		}
	}
	
	return result;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
