///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <shared/ThCore.h>
#include <shared/ThMemory.h>
#include <shared/ThString.h>
#include <shared/ThError.h>

#if HAVE_STDIO_H
	#include <stdio.h>
#endif

#if HAVE_STDARG_H
	#include <stdarg.h>
#endif

#include <theresa/ThError.h>

///////////////////////////////////////////////////////////////////////////////////////////////////

ThSingleton<IThError> Error;

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThError constructors --------------------------------------------------------------------------

IThError::~IThError(void)
{
}

// IThError static methods ------------------------------------------------------------------------

bool IThError::create(void)
{
	if (Error)
		return true;

	ThPtr<ThError> error = new ThError();

	if (!error->open())
		return false;

	Error.attach(error.detach());
	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThError constructors ---------------------------------------------------------------------------

ThError::ThError(void)
{
	m_file  = NULL;
	m_count = 0;
}

ThError::~ThError(void)
{
	close();
}

// ThError methods --------------------------------------------------------------------------------

bool ThError::open(void)
{
	close();

	// create log file
	{
		m_path = tempnam(NULL, "the");
		if (!m_path)
			return false;

		m_file = fopen(m_path, "wb");
		if (!m_file)
			return false;
	}

	return true;
}

void ThError::close(void)
{
	if (m_file)
	{
		fclose(m_file);
		m_file = NULL;
		
		if (!m_count)
			remove(m_path);
	}
	
	m_path.release();
	m_count = 0;
}

// ThError interface methods ----------------------------------------------------------------------

void ThError::write(const char* name, const char* format, ...)
{
	char buffer[THERESA_BUFFER_SIZE];

	THSAFEFORMAT(buffer, sizeof(buffer), format);

	char message[THERESA_BUFFER_SIZE];

	ThString::formatS(message, sizeof(message), "Module: %s%s%s%s", name, THERESA_NEWLINE, buffer, THERESA_NEWLINE);

#ifdef THERESA_DEBUG_BUILD
	writeDebugMessage(message);
#endif

	fwrite(message, 1, ThString::length(message), m_file);
	fflush(m_file);
	
	m_count++;
}

void ThError::display(const char* name, const char* format, ...)
{
	char description[THERESA_BUFFER_SIZE];

	THSAFEFORMAT(description, sizeof(description), format);

	write(name, "%s", description);

	char message[THERESA_BUFFER_SIZE];

	ThString::formatS(message, sizeof(message), "A runtime error has occurred in module %s.", name);
	
	displayErrorMessage(message, description);
}

///////////////////////////////////////////////////////////////////////////////////////////////////
