///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <shared/ThCore.h>
#include <shared/ThMemory.h>

#include <theresa/ThMemory.h>

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThStaticBlock constructors --------------------------------------------------------------------

IThStaticBlock::~IThStaticBlock(void)
{
}

// IThStaticBlock static methods ------------------------------------------------------------------

IThStaticBlock* IThStaticBlock::createInstance(unsigned int size)
{
	return new ThStaticBlock(NULL, size);
}

IThStaticBlock* IThStaticBlock::createInstance(void* data, unsigned int size)
{
	return new ThStaticBlock(data, size);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThDynamicBlock constructors -------------------------------------------------------------------

IThDynamicBlock::~IThDynamicBlock(void)
{
}

// IThDynamicBlock static methods -----------------------------------------------------------------

IThDynamicBlock* IThDynamicBlock::createInstance(unsigned int size)
{
	return new ThDynamicBlock(size);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThStaticBlock constructors ---------------------------------------------------------------------

ThStaticBlock::ThStaticBlock(void* data, unsigned int size):
	m_locks(0)
{
	if (data)
	{
		m_data = reinterpret_cast<char*>(data);
		m_size = size;

		m_attached = true;
	}
	else
	{
		m_data = new char [size];
		m_size = size;

		m_attached = false;
	}
}

ThStaticBlock::~ThStaticBlock(void)
{
	THASSERT(m_locks == 0, "Static block must be completely unlocked before descruction.");

	if (m_data)
	{
		if (!m_attached)
		{
			delete [] m_data;
			m_data = NULL;
		}

		m_size = 0;
	}
}

// ThStaticBlock methods --------------------------------------------------------------------------

void* ThStaticBlock::lock(void)
{
	m_locks++;

	return m_data;
}

void ThStaticBlock::unlock(void)
{
	THASSERT(m_locks > 0, "Cannot unlock non-locked static block.");

	if (m_locks)
		m_locks--;
}

// ThStaticBlock attributes -----------------------------------------------------------------------

unsigned int ThStaticBlock::getSize(void) const
{
	return m_size;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThDynamicBlock constructors --------------------------------------------------------------------

ThDynamicBlock::ThDynamicBlock(unsigned int size):
	m_locks(0)
{
	m_data.allocate(size);
}

ThDynamicBlock::~ThDynamicBlock(void)
{
	THASSERT(m_locks == 0, "Dynamic block must be completely unlocked before descruction.");
}

// ThDynamicBlock methods -------------------------------------------------------------------------

void* ThDynamicBlock::lock(void)
{
	m_locks++;

	return m_data;
}

void ThDynamicBlock::unlock(void)
{
	THASSERT(m_locks > 0, "Cannot unlock non-locked dynamic block.");

	if (m_locks)
		m_locks--;
}

void ThDynamicBlock::allocate(unsigned int size)
{
	THASSERT(m_locks == 0, "Cannot allocate dynamic block while locked.");

	m_data.allocate(size);
}

void ThDynamicBlock::resize(unsigned int size)
{
	THASSERT(m_locks == 0, "Cannot resize dynamic block while locked.");

	m_data.resize(size);
}

void ThDynamicBlock::release(void)
{
	THASSERT(m_locks == 0, "Cannot release dynamic block while locked.");

	m_data.release();
}

// ThDynamicBlock attributes ----------------------------------------------------------------------

unsigned int ThDynamicBlock::getSize(void) const
{
	return m_data.getSize();
}

///////////////////////////////////////////////////////////////////////////////////////////////////
