///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <shared/ThCore.h>
#include <shared/ThMemory.h>
#include <shared/ThTimer.h>
#include <shared/ThString.h>
#include <shared/ThError.h>
#include <shared/ThStream.h>
#include <shared/ThServer.h>
#include <shared/ThMusic.h>

#include <fmod.h>

#include <theresa/ThMusic.h>

///////////////////////////////////////////////////////////////////////////////////////////////////

#define THMUSIC_WARNING THERESA_NEWLINE THERESA_NEWLINE "If you choose to continue, music playback will be disabled."

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThMusic constructors --------------------------------------------------------------------------

IThMusic::~IThMusic(void)
{
}

// IThMusic static methods ------------------------------------------------------------------------

IThMusic* IThMusic::createInstance(IThStream* source)
{
	ThPtr<ThMusic> music = new ThMusic();

	if (!music->open(source))
		return false;

	return music.detach();
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThMusic constructors ---------------------------------------------------------------------------

ThMusic::ThMusic(void):
	ThServerObject(THID_MUSIC)
{
	m_started = false;
	m_paused  = false;
	m_module  = NULL;
	m_stream  = NULL;
	m_channel = 0;
}

ThMusic::~ThMusic(void)
{
	close();
}

// ThMusic methods --------------------------------------------------------------------------------

bool ThMusic::open(IThStream* source)
{
	close();
	
	m_mutex = IThMutex::createInstance();
	if (!m_mutex)
	{
		Error->display("Music", "Unable to create synchronization mutex.");
		return false;
	}

	// initialize library
	{
		if (FSOUND_GetVersion() != FMOD_VERSION)
			Error->display("Music", "Mismatched version of FMOD detected during initialization. Proceed at your own risk.");

		if (!FSOUND_Init(44100, 32, FSOUND_INIT_GLOBALFOCUS))
		{
			Error->display("Music", "Unable to initialize FMOD. %s", THMUSIC_WARNING);
			return false;
		}
	}

	// load music file
	{
		m_buffer.allocate(source->getSize());

		if (!source->readItems(m_buffer.getData(), m_buffer.getCount()))
		{
			Error->display("Music", "Unable to read data from music file %s.", THMUSIC_WARNING);
			return false;
		}

		m_module = FMUSIC_LoadSongMemory(m_buffer, m_buffer.getSize());
		if (m_module)
		{
			if (!FMUSIC_SetUserData(m_module, (unsigned int) this))
			{
				Error->display("Music", "Unable to register object for module. %s", THMUSIC_WARNING);
				return false;
			}

			if (!FMUSIC_SetZxxCallback(m_module, synchCallback))
			{
				Error->display("Music", "Unable to set music channel synchronizer. %s", THMUSIC_WARNING);
				return false;
			}
		}
		else
		{
			m_stream = FSOUND_Stream_OpenFile(m_buffer, FSOUND_LOADMEMORY, m_buffer.getSize());
			if (!m_stream)
			{
				Error->display("Music", "Unable to load music file as any known format." THMUSIC_WARNING);
				return false;
			}
		}
	}

	return true;
}

void ThMusic::close(void)
{
	stop();

	m_started = false;
	m_paused  = false;

	if (m_stream)
	{
		FSOUND_Stream_Close(m_stream);
		m_stream = NULL;
	}

	if (m_module)
	{
		FMUSIC_FreeSong(m_module);
		m_module = NULL;
	}

	m_buffer.release();

	FSOUND_Close();
	
	m_mutex.release();
}

// ThMusic interface methods ----------------------------------------------------------------------

void ThMusic::start(void)
{
	stop();

	if (m_module)
		FMUSIC_PlaySong(m_module);
	else if (m_stream)
		m_channel = FSOUND_Stream_Play(FSOUND_FREE, m_stream);
	else
		m_timer.start();

	m_started = true;
}

void ThMusic::stop(void)
{
	if (m_started)
	{
		if (m_module)
			FMUSIC_StopSong(m_module);
		else if (m_stream)
		{
			FSOUND_Stream_Stop(m_stream);
			m_channel = 0;
		}
		else
			m_timer.stop();

		m_started = false;
	}
}

void ThMusic::pause(void)
{
	if (m_paused)
		return;

	if (m_module)
		FMUSIC_SetPaused(m_module, true);
	else if (m_stream)
		FSOUND_SetPaused(m_channel, true);
	else
		m_timer.pause();

	m_paused = true;
}

void ThMusic::resume(void)
{
	if (!m_paused)
		return;

	if (m_module)
		FMUSIC_SetPaused(m_module, false);
	else if (m_stream)
		FSOUND_SetPaused(m_channel, false);
	else
		m_timer.resume();

	m_paused = false;
}

// ThMusic interface attributes -------------------------------------------------------------------

bool ThMusic::isStarted(void) const
{
	return m_started;
}

bool ThMusic::isPaused(void) const
{
	return m_paused;
}

float ThMusic::getTime(void) const
{
	if (m_module)
		return (float) FMUSIC_GetTime(m_module) / 1000.f;
	else if (m_stream)
		return (float) FSOUND_Stream_GetTime(m_stream) / 1000.f;

	return m_timer.getTime();
}

bool ThMusic::setTime(float time)
{
	if (m_module)
	{
		// TODO: implement!
		return false;
	}
	else if (m_stream)
	{
		if (!FSOUND_Stream_SetTime(m_stream, (int) (time * 1000.f)))
			return false;
	}
	else
		m_timer.setTime(time);

	return true;
}

// ThMusic callbacks ------------------------------------------------------------------------------

bool ThMusic::receive(const IThMessage* message)
{
	switch (message->getMessage())
	{
		case THMSG_LOCAL_UPDATE:
		{
			ThList<Signal> signals;

			if (m_mutex->request())
			{
				while (Signal* signal = m_signals.getFirst())
					signals.attachFirst(signal);

				m_mutex->release();
			}

			for (ThIterator<Signal> signal(signals);  signal;  signal.release())
				sendMessage(THMSG_MUSIC_SYNCH, THID_ANNOUNCE, &signal->m_value, sizeof(signal->m_value));
			
			break;
		}
	}
	
	return ThServerObject::receive(message);
}

// ThMusic static methods	-------------------------------------------------------------------------

void ThMusic::synchCallback(FMUSIC_MODULE* module, unsigned char param)
{
	ThMusic* object = reinterpret_cast<ThMusic*>(FMUSIC_GetUserData(module));
	
	if (object->m_mutex->request())
	{
		object->m_signals.attachLast(new Signal(param));

		object->m_mutex->release();
	}
}

///////////////////////////////////////////////////////////////////////////////////////////////////

ThMusic::Signal::Signal(unsigned char value):
	m_value(value)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////
