///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <shared/ThCore.h>
#include <shared/ThMemory.h>
#include <shared/ThString.h>
#include <shared/ThError.h>
#include <shared/ThServer.h>
#include <shared/ThContext.h>

#include <theresa/WGL/ThDialog.h>
#include <theresa/WGL/ThContext.h>
#include <theresa/WGL/resource.h>

///////////////////////////////////////////////////////////////////////////////////////////////////

bool enumerateResolutions(ThList<ThResolution>& resolutions)
{
	DEVMODE dm;

	unsigned int index = 0;

	resolutions.release();

	while (EnumDisplaySettings(NULL, index++, &dm))
	{
		if ((dm.dmBitsPerPel < THERESA_MINIMUM_DEPTH) || (dm.dmBitsPerPel != 16 && dm.dmBitsPerPel != 32))
			continue;

		if (dm.dmPelsWidth < THERESA_MINIMUM_WIDTH || dm.dmPelsHeight < THERESA_MINIMUM_HEIGHT)
			continue;

		if (findExactResolution(resolutions, dm.dmPelsWidth, dm.dmPelsHeight, dm.dmBitsPerPel))
			continue;

		resolutions.attachFirst(new ThResolution(dm.dmPelsWidth, dm.dmPelsHeight, dm.dmBitsPerPel, false));
	}

	if (!resolutions.getCount())
		return false;

	return true;
}

bool requestContextMode(ThContextMode& contextMode)
{
	ThList<ThResolution> resolutions;

	if (!enumerateResolutions(resolutions))
		return false;

	ThPtr<ThContextDialog> dialog = new ThContextDialog(resolutions);

	if (dialog->display() != IDOK)
		return false;

	contextMode.set(dialog->getResolution(), 16, 0, 32);
	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThContextDialog constructors -------------------------------------------------------------------

ThContextDialog::ThContextDialog(const ThList<ThResolution>& resolutions):
	ThDialog(IDD_CONTEXT_DIALOG),
	m_resolutions(resolutions)
{
}

ThContextDialog::~ThContextDialog(void)
{
}

// ThContextDialog attributes ---------------------------------------------------------------------

const ThResolution& ThContextDialog::getResolution(void) const
{
	return m_resolution;
}

// ThContextDialog callbacks ----------------------------------------------------------------------

BOOL ThContextDialog::initialize(HWND control)
{
	char buffer[THERESA_BUFFER_SIZE];

	HWND comboModes = getControl(IDC_CONTEXT_COMBOMODES);

	for (ThConstIterator<ThResolution> resolution(m_resolutions);  resolution;  resolution.next())
	{
		ThString::formatS(buffer, sizeof(buffer), "%u by %u, %u bits color", resolution->m_width, resolution->m_height, resolution->m_depth);

		ComboBox_AddString(comboModes, buffer);
	}

	ComboBox_SetCurSel(comboModes, 0);

	ThString::formatS(buffer, sizeof(buffer), "%u", THERESA_DEFAULT_WIDTH);
	Edit_SetText(getControl(IDC_CONTEXT_EDITWIDTH), buffer);

	ThString::formatS(buffer, sizeof(buffer), "%u", THERESA_DEFAULT_HEIGHT);
	Edit_SetText(getControl(IDC_CONTEXT_EDITHEIGHT), buffer);

	Button_SetCheck(getControl(IDC_CONTEXT_RADIOFULLSCREEN), TRUE);

	// workaround for buggy OS
	SendMessage(getHandle(), WM_COMMAND, MAKEWPARAM(IDC_CONTEXT_RADIOFULLSCREEN, BN_PUSHED), 0);

	return ThDialog::initialize(control);
}

BOOL ThContextDialog::command(HWND control, WORD identifier, BOOL accel)
{
	switch (identifier)
	{
		case IDOK:
		{
			if (Button_GetCheck(getControl(IDC_CONTEXT_RADIOWINDOWED)) == BST_CHECKED)
			{
				HWND editWidth = getControl(IDC_CONTEXT_EDITWIDTH);

				char buffer[THERESA_BUFFER_SIZE];

				if (!Edit_GetText(editWidth, buffer, sizeof(buffer)))
				{
					MessageBox(getHandle(), "You must specify the width of the window.", "Theresa", MB_OK | MB_ICONWARNING);

					SetFocus(editWidth);
					return TRUE;
				}

				const unsigned int width = ThString::convertToInt(buffer);
				if (width < THERESA_MINIMUM_WIDTH)
				{
					ThString::formatS(buffer, sizeof(buffer), "The window width needs to be at least %u.", THERESA_MINIMUM_WIDTH);
					MessageBox(getHandle(), buffer, "Theresa", MB_OK | MB_ICONWARNING);

					ThString::formatS(buffer, sizeof(buffer), "%u", THERESA_MINIMUM_WIDTH);
					Edit_SetText(editWidth, buffer);
					Edit_SetSel(editWidth, 0, -1);
					SetFocus(editWidth);
					return TRUE;
				}

				HWND editHeight = getControl(IDC_CONTEXT_EDITHEIGHT);

				if (!Edit_GetText(editHeight, buffer, sizeof(buffer)))
				{
					MessageBox(getHandle(), "You must specify the height of the window.", "Theresa", MB_OK | MB_ICONWARNING);

					SetFocus(editHeight);
					return TRUE;
				}

				const unsigned int height = ThString::convertToInt(buffer);
				if (height < THERESA_MINIMUM_HEIGHT)
				{
					ThString::formatS(buffer, sizeof(buffer), "The window height needs to be at least %u.", THERESA_MINIMUM_HEIGHT);
					MessageBox(getHandle(), buffer, "Theresa", MB_OK | MB_ICONWARNING);

					ThString::formatS(buffer, sizeof(buffer), "%u", THERESA_MINIMUM_HEIGHT);
					Edit_SetText(editHeight, buffer);
					Edit_SetSel(editHeight, 0, -1);
					SetFocus(editHeight);
					return TRUE;
				}

				m_resolution.set(width, height, THERESA_DEFAULT_DEPTH, true);
			}
			else
			{
				int index = ComboBox_GetCurSel(getControl(IDC_CONTEXT_COMBOMODES));
				if (index == CB_ERR)
				{
					MessageBox(getHandle(), "Invalid error.", "Theresa", MB_OK | MB_ICONERROR);

					return TRUE;
				}

				m_resolution = *m_resolutions.getAt(index);
			}

			break;
		}

		case IDCANCEL:
		{
			break;
		}

		case IDC_CONTEXT_RADIOFULLSCREEN:
		{
			EnableWindow(getControl(IDC_CONTEXT_COMBOMODES), TRUE);

			EnableWindow(getControl(IDC_CONTEXT_EDITWIDTH), FALSE);
			EnableWindow(getControl(IDC_CONTEXT_EDITHEIGHT), FALSE);

			SetFocus(getControl(IDC_CONTEXT_COMBOMODES));
			break;
		}

		case IDC_CONTEXT_RADIOWINDOWED:
		{
			EnableWindow(getControl(IDC_CONTEXT_COMBOMODES), FALSE);

			EnableWindow(getControl(IDC_CONTEXT_EDITWIDTH), TRUE);
			EnableWindow(getControl(IDC_CONTEXT_EDITHEIGHT), TRUE);

			Edit_SetSel(getControl(IDC_CONTEXT_EDITWIDTH), 0, -1);
			SetFocus(getControl(IDC_CONTEXT_EDITWIDTH));
			break;
		}

		case IDC_DISPLAY_EDITWIDTH:
		{
			if (accel == EN_CHANGE)
			{
				char buffer[THERESA_BUFFER_SIZE];

				if (!Edit_GetText(getControl(IDC_CONTEXT_EDITWIDTH), buffer, sizeof(buffer)))
					break;

				const unsigned int width = ThString::convertToInt(buffer);
				if (width < THERESA_MINIMUM_WIDTH)
					break;

				ThString::formatS(buffer, sizeof(buffer), "%u", (unsigned int) (width * 0.75f));

				Edit_SetText(getControl(IDC_CONTEXT_EDITHEIGHT), buffer);
			}

			break;
		}
	}

	return ThDialog::command(control, identifier, accel);
}

BOOL ThContextDialog::message(UINT message, WPARAM wParam, LPARAM lParam)
{
	return ThDialog::message(message, wParam, lParam);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThContext constructors -------------------------------------------------------------------------

ThContext::ThContext(void):
	ThServerObject(THID_CONTEXT)
{
	m_class  = 0;
	m_window = NULL;
	m_dc     = NULL;
	m_rc     = NULL;

	m_mode.m_resolution.m_windowed = true;
}

ThContext::~ThContext(void)
{
	close();
}

// ThContext methods ------------------------------------------------------------------------------

bool ThContext::open(const ThContextMode& mode, unsigned int listenerID)
{
	close();

	if (!mode.m_resolution.m_windowed)
	{
		// set display mode

		DEVMODE dm;

		ZeroMemory(&dm, sizeof(dm));
		dm.dmSize = sizeof(dm);
		dm.dmFields     = DM_PELSWIDTH | DM_PELSHEIGHT | DM_BITSPERPEL;
		dm.dmPelsWidth  = mode.m_resolution.m_width;
		dm.dmPelsHeight = mode.m_resolution.m_height;
		dm.dmBitsPerPel = mode.m_resolution.m_depth;

		if (ChangeDisplaySettings(&dm, 0) != DISP_CHANGE_SUCCESSFUL)
		{
			close();

			Error->display("Context", "Unable to enter specified OpenGL-capable resolution (%u by %u, %u bits color).", mode.m_resolution.m_width, mode.m_resolution.m_height, mode.m_resolution.m_depth);
			return false;
		}

		m_mode.m_resolution.m_windowed = false;
	}

	// setup display window
	{
		// create window
		{
			WNDCLASSEX wcl;

			ZeroMemory(&wcl, sizeof(wcl));
			wcl.cbSize = sizeof(wcl);
			wcl.style         = CS_OWNDC;
			wcl.hCursor       = LoadCursor(NULL, IDC_ARROW);
			wcl.hbrBackground = (HBRUSH) COLOR_WINDOW + 1;
			wcl.lpszClassName = THCONTEXT_WINDOWCLASSNAME;
			wcl.lpfnWndProc   = windowCallback;
			wcl.hInstance     = GetModuleHandle(NULL);
			wcl.hIcon         = LoadIcon(GetModuleHandle(NULL), MAKEINTRESOURCE(IDI_HYPERCUBE));

			m_class = RegisterClassEx(&wcl);
			if (!m_class)
			{
				close();

				Error->display("Context", "Unable to register window class.");
				return false;
			}

			DWORD style   = WS_CLIPCHILDREN | WS_CLIPSIBLINGS;
			DWORD exStyle = 0;

			if (mode.m_resolution.m_windowed)
			{
				style   |= WS_OVERLAPPEDWINDOW;
				exStyle |= WS_EX_APPWINDOW;
			}
			else
			{
				style   |= WS_BORDER | WS_POPUP;
				exStyle |= WS_EX_TOPMOST | WS_EX_APPWINDOW;
			}

			RECT area;
			SetRect(&area, 0, 0, mode.m_resolution.m_width, mode.m_resolution.m_height);

			if (mode.m_resolution.m_windowed)
			{
				RECT workArea;
				SystemParametersInfo(SPI_GETWORKAREA, 0, &workArea, 0);

				OffsetRect(&area, workArea.left + (workArea.right - workArea.left - (int) mode.m_resolution.m_width) / 2, workArea.top + (workArea.bottom - workArea.top - (int) mode.m_resolution.m_height) / 2);
			}

			AdjustWindowRectEx(&area, style, FALSE, exStyle);

			m_window = CreateWindowEx(exStyle, wcl.lpszClassName, "", style, area.left, area.top, area.right - area.left, area.bottom - area.top, NULL, NULL, GetModuleHandle(NULL), (LPVOID) this);
			if (!m_window)
			{
				close();

				Error->display("Context", "Unable to create window.");
				return false;
			}

			ShowWindow(m_window, SW_SHOW);

			BringWindowToTop(m_window);
			SetForegroundWindow(m_window);

			UpdateWindow(m_window);
		}

		if (!mode.m_resolution.m_windowed)
			ShowCursor(FALSE);
	}

	m_dc = GetDC(m_window);
	if (!m_dc)
	{
		close();

		Error->display("Context", "Unable to retrieve the window device context.");
		return false;
	}

	// select pixel format
	{
		PIXELFORMATDESCRIPTOR pfd;

		ZeroMemory(&pfd, sizeof(pfd));
		pfd.nSize = sizeof(pfd);
		pfd.nVersion     = 1;
		pfd.dwFlags      = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
		pfd.iPixelType   = PFD_TYPE_RGBA;
		pfd.cColorBits   = mode.m_resolution.m_depth;
		pfd.cDepthBits   = mode.m_depthBits;
		pfd.cStencilBits = mode.m_stencilBits;
		pfd.iLayerType   = PFD_MAIN_PLANE;

		if (mode.m_depthBits)
			pfd.dwFlags |= PFD_DEPTH_DONTCARE;

		const int pixelFormat = ChoosePixelFormat(m_dc, &pfd);
		if (!pixelFormat)
		{
			close();

			Error->display("Context", "Unable to choose OpenGL pixel format.");
			return false;
		}

		if (!SetPixelFormat(m_dc, pixelFormat, &pfd))
		{
			close();

			Error->display("Context", "Unable to set OpenGL pixel format.");
			return false;
		}

		DescribePixelFormat(m_dc, pixelFormat, sizeof(pfd), &pfd);

		if ((pfd.dwFlags & PFD_GENERIC_FORMAT) && !(pfd.dwFlags & PFD_GENERIC_ACCELERATED))
		{
			if (MessageBox(m_window, "Unable to accelerate selected OpenGL pixel format. Continue anyway?", "Theresa", MB_YESNO | MB_ICONERROR) == IDNO)
			{
				close();

				Error->display("Context", "Unable to accelerate selected OpenGL pixel format.");
				return false;
			}
		}
	}

	// create rendering context
	{
		m_rc = wglCreateContext(m_dc);
		if (!m_rc)
		{
			close();

			Error->display("Context", "Unable to create OpenGL rendering context.");
			return false;
		}

		if (!wglMakeCurrent(m_dc, m_rc))
		{
			close();

			Error->display("Context", "Unable to apply OpenGL rendering context.");
			return false;
		}

		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();

		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();
	}

	// list extensions
	{
		if (const char* extensions = (const char*) glGetString(GL_EXTENSIONS))
		{
			ThString extension;

			while (extensions = extension.copyToken(extensions))
				m_extensions.attachFirst(new ThStringItem(extension));
		}
	}

	m_mode       = mode;
	m_listenerID = listenerID;

	setTitle("");

	return true;
}

void ThContext::close(void)
{
	if (!m_mode.m_resolution.m_windowed)
		ChangeDisplaySettings(NULL, 0);

	m_mode.reset();
	m_mode.m_resolution.m_windowed = true;

	if (m_rc)
	{
		wglMakeCurrent(NULL, NULL);

		wglDeleteContext(m_rc);
		m_rc = NULL;
	}

	if (m_dc)
	{
		ReleaseDC(m_window, m_dc);
		m_dc = NULL;
	}

	if (m_window)
	{
		DestroyWindow(m_window);
		m_window = NULL;
	}

	if (m_class)
	{
		UnregisterClass(MAKEINTATOM(m_class), GetModuleHandle(NULL));
		m_class = 0;
	}
}

// ThContext interface methods --------------------------------------------------------------------

void ThContext::apply(void)
{
	if (!wglMakeCurrent(m_dc, m_rc))
	{
		Error->display("Context", "Unable to apply OpenGL rendering context.");
		postMessage(THMSG_REQUEST_EXIT, THID_ANNOUNCE);
	}
}

void ThContext::update(void)
{
	glFinish();

	if (unsigned int error = glGetError())
	{
		Error->display("Context", "OpenGL reported an error during update.%s%s", THERESA_NEWLINE, gluErrorString(error));
		postMessage(THMSG_REQUEST_EXIT, THID_ANNOUNCE);
	}

	SwapBuffers(m_dc);
}

// ThContext interface extension methods ----------------------------------------------------------

bool ThContext::findExtension(const char* name) const
{
	if (m_extensions.findNoCase(name))
		return true;

	return false;
}

ThEntryPoint ThContext::findEntryPoint(const char* name) const
{
	return (ThEntryPoint) wglGetProcAddress(name);
}

// ThContext interface attributes -----------------------------------------------------------------

const char* ThContext::getTitle(void) const
{
	return m_title;
}

void ThContext::setTitle(const char* title)
{
	m_title = title;

	ThString caption;

	caption = title;
	if (!caption.isEmpty())
		caption.append(" - ");

	caption.append("Theresa");

	SetWindowText(m_window, caption);
}

unsigned int ThContext::getWidth(void) const
{
	return m_mode.m_resolution.m_width;
}

unsigned int ThContext::getHeight(void) const
{
	return m_mode.m_resolution.m_height;
}

const ThContextMode& ThContext::getMode(void) const
{
	return m_mode;
}

// ThContext callbacks ----------------------------------------------------------------------------

bool ThContext::receive(const IThMessage* message)
{
	return ThServerObject::receive(message);
}

LRESULT ThContext::message(UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message)
	{
		case WM_CLOSE:
		{
			// suggest application exit
			postMessage(THMSG_REQUEST_EXIT, THID_ANNOUNCE);
			break;
		}

		case WM_ACTIVATEAPP:
		{
			if (m_listenerID != THID_INVALID)
			{
				if (wParam)
					sendMessage(THMSG_CONTEXT_ACTIVATE, m_listenerID);
				else
					sendMessage(THMSG_CONTEXT_DEACTIVATE, m_listenerID);
			}

			break;
		}

		case WM_SYSCOMMAND:
		{
			switch (wParam)
			{
				case SC_SCREENSAVE:
				case SC_MONITORPOWER:
				{
					return 0;
				}
			}

			break;
		}

		case WM_KEYDOWN:
		{
			if (wParam == VK_ESCAPE)
				PostMessage(m_window, WM_CLOSE, 0, 0);

			break;
		}

		case WM_GETMINMAXINFO:
		{
			if (m_mode.m_resolution.m_windowed)
			{
				RECT area;

				SetRect(&area, 0, 0, THERESA_MINIMUM_WIDTH, THERESA_MINIMUM_HEIGHT);
				AdjustWindowRectEx(&area, GetWindowLong(m_window, GWL_STYLE), FALSE, GetWindowLong(m_window, GWL_EXSTYLE));

				MINMAXINFO* mmi = (MINMAXINFO*) lParam;

				mmi->ptMinTrackSize.x = area.right - area.left;
				mmi->ptMinTrackSize.y = area.bottom - area.top;
			}

			break;
		}

		case WM_SIZE:
		{
			m_mode.m_resolution.m_width  = LOWORD(lParam);
			m_mode.m_resolution.m_height = HIWORD(lParam);

			glViewport(0, 0, m_mode.m_resolution.m_width, m_mode.m_resolution.m_height);

			if (m_listenerID != THID_INVALID)
				sendMessage(THMSG_CONTEXT_RESIZE, m_listenerID);

			break;
		}
	}

	return DefWindowProc(m_window, message, wParam, lParam);
}

// ThContext static methods -----------------------------------------------------------------------

LRESULT CALLBACK ThContext::windowCallback(HWND window, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message)
	{
		case WM_CREATE:
		{
			CREATESTRUCT* cs = (CREATESTRUCT*) lParam;

			SetWindowLong(window, GWL_USERDATA, (LONG) cs->lpCreateParams);
			break;
		}

		default:
		{
			if (ThContext* object = (ThContext*) GetWindowLong(window, GWL_USERDATA))
				return object->message(message, wParam, lParam);

			break;
		}
	}

	return DefWindowProc(window, message, wParam, lParam);
}

///////////////////////////////////////////////////////////////////////////////////////////////////
