///////////////////////////////////////////////////////////////////////
// Super Mario Lngben Bros.
// Copyright (c) 2004 Camilla Drefvenborg <elmindreda@home.se>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any
// damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any
// purpose, including commercial applications, and to alter it and
// redistribute it freely, subject to the following restrictions:
//
//  1. The origin of this software must not be misrepresented; you
//     must not claim that you wrote the original software. If you use
//     this software in a product, an acknowledgment in the product
//     documentation would be appreciated but is not required.
//
//  2. Altered source versions must be plainly marked as such, and
//     must not be misrepresented as being the original software.
//
//  3. This notice may not be removed or altered from any source
//     distribution.
//
///////////////////////////////////////////////////////////////////////

#include <Theresa.h>

#include "SML.h"

#include <ctype.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>

#define GAME_LENGTH 58

inline int maximum(int a, int b)
{
  if (a > b)
    return a;
  else
    return b;
}

GameApp::GameApp(void)
{
}

bool GameApp::open(void)
{
  if (!System->getDisplay())
  {
    if (!System->openDisplay())
      return false;
    
    System->getDisplay()->getContext()->setTitle("Super Mario Lngben Bros.");
  }
  
  if (!System->openInput())
    return false;
  
  if (!System->getSound())
  {
    if (!System->openSound())
      return false;
  }
  
  if (!System->getEngine())
  {
    if (!System->openEngine())
      return false;
  }
  
  if (!System->getEffect())
  {
    if (!System->openEffect())
      return false;
  }
  
  m_mapping = System->getInput()->createMapping("main");
  if (!m_mapping)
    return false;
  
  ThInputAction* action;
  
  action = m_mapping->createAction(RIGHT, ThInputAction::BUTTON, THKEY_RIGHT);
  if (!action)
    return false;
  
  action = m_mapping->createAction(LEFT, ThInputAction::BUTTON, THKEY_LEFT);
  if (!action)
    return false;
  
  action = m_mapping->createAction(JUMP, ThInputAction::BUTTON, THKEY('X')/*_LCONTROL*/);
  if (!action)
    return false;
  
  action = m_mapping->createAction(TURBO, ThInputAction::BUTTON, THKEY('Z')/*_LALT*/);
  if (!action)
    return false;
  
  IThTexture* texture;
  IThMaterial* material;

  // intro material
  {
    texture = System->getDisplay()->findTexture("intro");
    if (!texture)
      return false;
    
    texture->apply();
    
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    
    material = System->getEngine()->createMaterial(texture->getName());
    if (!material)
      return false;
    
    material->setTextureName(texture->getName());
  }
  
  // level material
  {
    texture = System->getDisplay()->findTexture("level");
    if (!texture)
      return false;
    
    m_center = (4.f / 3.f) / ((float) texture->getWidth() / (float) texture->getHeight());
    
    texture->apply();
    
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    
    material = System->getEngine()->createMaterial(texture->getName());
    if (!material)
      return false;
    
    material->setTextureName(texture->getName());
  }

  // font
  {
    texture = System->getDisplay()->findTexture("font");
    if (!texture)
      return false;
    
    texture->apply();
    
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    
    material = System->getEngine()->createMaterial(texture->getName());
    if (!material)
      return false;
    
    material->setBlending(true);
    material->setSrcBlend(GL_SRC_ALPHA);
    material->setDstBlend(GL_ONE_MINUS_SRC_ALPHA);
    material->setTextureName(texture->getName());
  }
  
  // player
  {
    texture = System->getDisplay()->findTexture("player");
    if (!texture)
      return false;
    
    texture->apply();
    
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    
    material = System->getEngine()->createMaterial(texture->getName());
    if (!material)
      return false;
    
    material->setBlending(true);
    material->setSrcBlend(GL_SRC_ALPHA);
    material->setDstBlend(GL_ONE_MINUS_SRC_ALPHA);
    material->setTextureName(texture->getName());
  }
  
  // gomba
  {
    texture = System->getDisplay()->findTexture("gomba");
    if (!texture)
      return false;
    
    texture->apply();
    
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    
    material = System->getEngine()->createMaterial(texture->getName());
    if (!material)
      return false;
    
    material->setBlending(true);
    material->setSrcBlend(GL_SRC_ALPHA);
    material->setDstBlend(GL_ONE_MINUS_SRC_ALPHA);
    material->setTextureName(texture->getName());
  }

  // boxes
  {
    for (unsigned int i = 0;  i < 3;  i++)
    {
      ThString name(5);
      
      name.format("box%u", i + 1);
      
      texture = System->getDisplay()->findTexture(name);
      if (!texture)
        return false;
      
      texture->apply();
      
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    }
    
    material = System->getEngine()->createMaterial("boxes");
    if (!material)
      return false;
    
    material->setBlending(true);
    material->setSrcBlend(GL_SRC_ALPHA);
    material->setDstBlend(GL_ONE_MINUS_SRC_ALPHA);
  }
  
  // coins
  {
    for (unsigned int i = 0;  i < 3;  i++)
    {
      ThString name(6);
      
      name.format("coin%u", i + 1);
      
      texture = System->getDisplay()->findTexture(name);
      if (!texture)
        return false;
      
      texture->apply();
      
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    }
    
    material = System->getEngine()->createMaterial("coins");
    if (!material)
      return false;

    material->setBlending(true);
    material->setSrcBlend(GL_SRC_ALPHA);
    material->setDstBlend(GL_ONE_MINUS_SRC_ALPHA);
  }
  
  if (!subscribeMessage(THMSG_INPUT_KEYDOWN))
    return false;
  
  if (!subscribeMessage(THMSG_INPUT_KEYUP))
    return false;
  
  IThSample* sample;
  
  sample = System->getSound()->findSample("intro");
  if (!sample)
    return false;
  
  sample = System->getSound()->findSample("jump");
  if (!sample)
    return false;
  
  sample = System->getSound()->findSample("music");
  if (!sample)
    return false;
  
  sample = System->getSound()->findSample("gameover");
  if (!sample)
    return false;
  
  m_mode = INTRO;
  m_lives = 3;
  
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  
  return true;
}

bool GameApp::update(float deltaTime)
{
  // JAVIVALAMA
  
  switch (m_mode)
  {
    case INTRO:
    {
      if (m_timer.isStarted())
      {
        if (m_timer.getTime() > 8.f)
        {
          m_timer.stop();
          m_mode = WORLD;
        }
      }
      else
      {
        System->getSound()->findSample("intro")->start();
        
        m_timer.start();
      }
      
      break;
    }
      
    case WORLD:
    {
      if (m_timer.isStarted())
      {
        if (m_timer.getTime() > 3.f)
        {
          m_timer.stop();
          
          m_mode = GAME;
          m_scroll = 0.f;
          m_position.set(0.2f, 0.f);
          m_velocity.reset();
        }
      }
      else
        m_timer.start();        
      
      break;
    }
      
    case GAME:
    {
      if (m_timer.isStarted())
      {
        if (m_timer.getTime() > (float) GAME_LENGTH)
        {
          m_timer.stop();
          
          System->getSound()->findSample("music")->stop();
          
          m_velocity.set(0.f, 1.f);
          m_mode = DEATH;
          
          return update(deltaTime);
        }
      }
      else
      {
        System->getSound()->findSample("music")->start();
        m_timer.start();
      }
      
      if (m_mapping->findAction(RIGHT)->isActive())
      {
        m_velocity.x += deltaTime * 2.f;
        if (m_velocity.x > 1.f)
          m_velocity.x = 1.f;
      }
      
      if (m_mapping->findAction(LEFT)->isActive())
      {
        m_velocity.x -= deltaTime * 2.f;
        if (m_velocity.x < -1.f)
          m_velocity.x = -1.f;
      }
      
      if (m_mapping->findAction(JUMP)->isActive())
      {
        if (m_position.y == 0.f)
        {
          System->getSound()->findSample("jump")->start();
          
          m_velocity.y = 1.f;
        }
      }
            
      // physics
      m_position += m_velocity * deltaTime;
      m_velocity.y -= deltaTime * 3.f;
      
      // gomba
      float quux = sinf(m_timer.getTime());
      quux = quux * quux * quux;
      m_gomba = quux * 3.f;
      
      // check boundaries
      
      if (m_position.x < m_scroll)
      {
        m_position.x = m_scroll;
        m_velocity.x = 0.f;
      }
      else if (m_position.x > m_scroll + 0.5f)
      {
        m_scroll = m_position.x - 0.5f;
        if (m_scroll > 0.2f)
        {
          m_scroll = 0.2f;
          m_position.x = m_scroll + 0.5f;
          m_velocity.x = 0.f;
        }
      }
      
      if (m_position.y < 0.f)
        m_position.y = 0.f;
      
      if (m_position.y == 0.f)
        m_velocity.y = 0.f;
      
      if (m_mapping->findAction(TURBO)->isActive())
        m_velocity.x -= m_velocity.x * deltaTime * 3.f;
      else
        m_velocity.x -= m_velocity.x * deltaTime * 5.f;
      
      break;
    }
      
    case DEATH:
    {
      if (m_timer.isStarted())
      {
        if (m_timer.getTime() > 3.f)
        {
          m_timer.stop();
          
          m_lives--;
          
          if (m_lives)
            m_mode = WORLD;
          else
            m_mode = OVER;
          
          return update(deltaTime);
        }
      }
      else
      {
        System->getSound()->findSample("death")->start();
        m_timer.start();
      }
      
      // physics
      m_position += m_velocity * deltaTime;
      m_velocity.y -= deltaTime * 3.f;
      
      float quux = sinf(m_timer.getTime() + GAME_LENGTH);
      quux = quux * quux * quux;
      m_gomba = quux * 3.f;
      break;
    }

    case OVER:
    {
      if (m_timer.isStarted())
      {
        if (m_timer.getTime() > 4.f)
        {
          m_timer.stop();
          
          m_lives = 3;
          m_mode = CREDITS;
        }
      }
      else
      {
        System->getSound()->findSample("gameover")->start();
        m_timer.start();
      }
      
      break;
    }
      
    case GREETS:
    {
      if (m_timer.isStarted())
      {
        if (m_timer.getTime() > 7.f)
        {
          m_timer.stop();
          m_mode = WORLD;
        }
      }
      else
        m_timer.start();
      
      break;
    }
      
    case CREDITS:
    {
      if (m_timer.isStarted())
      {
        if (m_timer.getTime() > 7.f)
        {
          m_timer.stop();
          m_mode = GREETS;
        }
      }
      else
        m_timer.start();
      
      break;
    }
  }
  
  return true;
}

bool GameApp::render(IThCanvas* target)
{
  target->apply();
  
  const unsigned int indices[] = { 1, 2, 3, 2, 1 };
  
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  gluOrtho2D(0.f, 1.f, 1.f, 0.f);
  glMatrixMode(GL_MODELVIEW);
  
  switch (m_mode)
  {
    case INTRO:
    {
      ThSprite2 intro;
      
      System->getEngine()->findMaterial("intro")->apply();
      
      intro.render();
      break;
    }
      
    case WORLD:
    {
      glClearColor(0.f, 0.f, 0.f, 0.f);
      glClear(GL_COLOR_BUFFER_BIT);
      
      print(ThVector2(18.f, 14.f), "WORLD 1-1");
      print(ThVector2(22.f, 18.f), "* %2u", m_lives);
      
      // render player
      {
        ThSprite2 player;
        
        IThTexture* texture = System->getDisplay()->findTexture("player");
        
        System->getEngine()->findMaterial(texture->getName())->apply();
        
        player.m_size.x = 0.1f / 4.f * 3.f;
        player.m_size.y = 0.1f * (float) texture->getHeight() / (float) texture->getWidth();
        player.m_position.set(0.4f, 0.55f);
        
        player.render();
      }
      
      break;
    }
      
    case GAME:
    case DEATH:
    {
      // render level
      {
        ThSprite2 level;
        
        System->getEngine()->findMaterial("level")->apply();
        
        level.m_area.m_size.x = m_center;
        level.m_area.m_pos.x = m_scroll * m_center;
        level.render();
      }
      
      // render gomba
      {
        ThVector2 position(22.f, 14.f);
        
        IThTexture* texture = System->getDisplay()->findTexture("gomba");
        IThMaterial* material = System->getEngine()->findMaterial(texture->getName());
        
        ThSprite2 gomba;
        
        gomba.m_size.x = 1.f / 16.f * 3.f / 4.f;
        gomba.m_size.y = 1.f / 16.f;
        gomba.m_position = tileToScreen(ThVector2(22.f + m_gomba, 14.f));
        
        if ((unsigned int) (Server->getTime() * 3.f) & 1)
          gomba.m_size.x = -gomba.m_size.x;
        
        material->apply();
        
        gomba.render();
      }
      
      // render boxes
      {
        ThVector2 positions[] = { ThVector2(17.f, 11.f), ThVector2(22.f, 11.f), ThVector2(23.f, 7.f), ThVector2(24.f, 11.f) };
        
        ThSprite2 box;
        
        IThMaterial* material = System->getEngine()->findMaterial("boxes");
        
        box.m_size.x = 1.f / 16.f * 3.f / 4.f;
        box.m_size.y = 1.f / 16.f;
        
        for (unsigned int i = 0;  i < sizeof(positions) / sizeof(ThVector2);  i++)
        {
          ThString name(5);
          name.format("box%u", indices[(unsigned int) (Server->getTime() * 6.f) % (sizeof(indices) / sizeof(unsigned int))]);
          
          IThTexture* texture = System->getDisplay()->findTexture(name);
          
          material->setTextureName(texture->getName());
          material->apply();
          
          box.m_position = tileToScreen(positions[i]);
          box.m_area.m_pos.x = -0.25f / (float) texture->getWidth();
          box.m_area.m_pos.y = -0.25f / (float) texture->getHeight();
          
          box.render();
        }
      }
      
      // render player
      {
        ThSprite2 player;
        
        IThTexture* texture = System->getDisplay()->findTexture("player");
        
        System->getEngine()->findMaterial(texture->getName())->apply();
        
        player.m_size.x = 0.2f / 4.f * 3.f;
        player.m_size.y = 0.2f * (float) texture->getHeight() / (float) texture->getWidth();
        player.m_position.x = m_position.x - m_scroll;
        player.m_position.y = 0.873f - player.m_size.y / 2.f - m_position.y;
        
        player.render();
      }

      // render time
      {
        int time = 0;
        
        if (m_mode == GAME)
          time = maximum(0, GAME_LENGTH - (int) m_timer.getTime());
        
        print(ThVector2(36.f, 4.f), "%03i", time);
      }
      break;
    }
      
    case OVER:
    {
      glClearColor(0.f, 0.f, 0.f, 0.f);
      glClear(GL_COLOR_BUFFER_BIT);
      
      print(ThVector2(18.f, 17.f), "GAME OVER");      
      break;
    }
      
    case CREDITS:
    {
      glClearColor(0.f, 0.f, 0.f, 0.f);
      glClear(GL_COLOR_BUFFER_BIT);
      
      print(ThVector2(5.f, 10.f), "Super Mario Lngben Bros by HyperCube");
      print(ThVector2(5.f, 13.f), "code     - elmindreda");
      print(ThVector2(5.f, 15.f), "sound    - retro Gazman");
      print(ThVector2(5.f, 17.f), "graphics - elmindreda retro Gazman");
      break;
    }
      
    case GREETS:
    {
      glClearColor(0.f, 0.f, 0.f, 0.f);
      glClear(GL_COLOR_BUFFER_BIT);
      
      print(ThVector2(4.f, 10.f), "elmindreda greets");
      print(ThVector2(6.f, 12.f), "#opengl  #macscene  Illuminati");
      print(ThVector2(6.f, 14.f), "Simonize  Woodgrove  Scarab Corpse");
      print(ThVector2(4.f, 17.f), "retro greets");
      print(ThVector2(6.f, 19.f), "Insider  Jannike Bjrling  U77");
      print(ThVector2(6.f, 21.f), "Gazmans Mamma");
      print(ThVector2(4.f, 24.f), "Gazman greets");
      print(ThVector2(6.f, 26.f), "Truckflatan 3000  Kanuppen");
      break;
    }
  } 
  
  switch (m_mode)
  {
    case WORLD:
    case GAME:
    case DEATH:
    case OVER:
    {
      print(ThVector2(6.f, 3.f), "LNGBEN            WORLD     TIME");
      print(ThVector2(6.f, 4.f), "000000     *00      1-1");

      IThMaterial* material = System->getEngine()->findMaterial("coins");
      
      ThString name(6);
      name.format("coin%u", indices[(unsigned int) (Server->getTime() * 6.f) % (sizeof(indices) / sizeof(unsigned int))]);
      
      material->setTextureName(name);
      material->apply();
      
      ThSprite2 coin;
      
      coin.m_size.set(1.f / 32.f * 3.f / 4.f, 1.f / 32.f);
      coin.m_position.set((16.f - 0.5f) / (m_center * 64.f), (4.f - 0.5f) / 32.f);
      coin.render();
      
      break;
    }
  }
  
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);
  
  target->update();
  
  return true;
}

bool GameApp::receive(const IThMessage* message)
{
  switch (message->getID())
  {
    case THMSG_INPUT_KEYDOWN:
    {
      const unsigned int key = *reinterpret_cast<const unsigned int*>(message->getData());
      
      switch (key)
      {
        case THKEY_ESCAPE:
        {
          Server->stop();
          break;
        }
      }
      
      break;
    }
  }
  
  return ThApplication::receive(message);
}

void GameApp::print(const ThVector2& position, const char* format, ...)
{
  char buffer[THERESA_BUFFER_SIZE];
  
  THSAFEFORMAT(buffer, sizeof(buffer), format);
  
  char* table = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ -*#";
  
  const float length = (float) strlen(table);
  float x = position.x;
  float y = position.y;
  
  System->getEngine()->findMaterial("font")->apply();
  
  ThSprite2 character;
  
  character.m_area.m_size.set(1.f / length, 1.f);
  character.m_size.set(1.f / 32.f * 3.f / 4.f, 1.f / 32.f);

  for (char* c = buffer;  *c != '\0';  c++)
  {
    if (const char* p = strchr(table, toupper(*c)))
    {
      const unsigned int index = p - table;
      
      character.m_area.m_pos.set((float) index * 1.f / length, 0.f);
      character.m_position.set((x - 0.5f) / (m_center * 64.f), (y - 0.5f) / 32.f);
      
      x += 1.f;
      
      character.render();
    }
  }
}
  
ThVector2 GameApp::tileToScreen(const ThVector2& position)
{
  return ThVector2((position.x - 0.5f) / (m_center * 32.f) - m_scroll, (position.y - 0.5f) / 16.f);
}

int main(int argc, char** argv)
{
  if (ThInitialize(argc, argv))
  {
    if (System->readScript("Game.effect"))
    {
      ThPtr<GameApp> app = new GameApp();
      
      if (app->open())
        Server->start();
      
      app.release();
    }
    
    ThShutdown();
  }
  
  return 0;
}
