                                 //-----------------------------------------------------------------------------
//           Name: dx9_hlsl_fx_simple.fx
//         Author: Kevin Harris
//  Last Modified: 03/13/03
//    Description: This effect file demonstrates how to write vertex and pixel
//                 shaders using Direct3D's High-Level Shading Language.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Effect File Variables
//-----------------------------------------------------------------------------

float4x4 World	: WORLD; // This matrix will be loaded by the application
float4x4 View	: VIEW; // This matrix will be loaded by the application
float4x4 Proj	: PROJECTION; // This matrix will be loaded by the application

float4x4		ViewProj;
float4x4		WVP; // This matrix will be loaded by the application
float4x4		InvWVP; // This matrix will be loaded by the application
float4x4		InvWorld; // This matrix will be loaded by the application

float			fTime;
float4			eyeVec;		// This camera position will be loaded by the application
float4			lightVec;	 // This light will be loaded by the application
texture			BaseTexture;
texture			SphereMapTexture;



// light intensity
float4 I_a = { 0.3f, 0.3f, 0.3f, 1.0f };    // ambient
float4 I_d = { 1.0f, 1.0f, 1.0f, 1.0f };    // diffuse
float4 I_s = { 1.0f, 1.0f, 1.0f, 1.0f };    // specular


// material reflectivity
float4 k_a : MATERIALAMBIENT = { 0.1f, 0.1f, 0.1f, 1.0f };	// ambient
float4 k_d : MATERIALDIFFUSE = { 1.0f, 1.0f, 1.0f, 1.0f };	// diffuse
float4 k_s : MATERIALSPECULAR= { 1.0f, 1.0f, 1.0f, 1.0f };	// specular
int    n   : MATERIALPOWER = 32;				// power




sampler SamplerBase = sampler_state
{
    Texture	= (BaseTexture);
    MipFilter	= NONE;//LINEAR;
    MinFilter	= LINEAR;
    MagFilter	= LINEAR;
    AddressU	= WRAP;
    AddressV	= WRAP;
};

sampler SamplerSphereMap = sampler_state
{
    Texture	= (SphereMapTexture);
    MipFilter	= NONE;//LINEAR;
    MinFilter	= LINEAR;
    MagFilter	= LINEAR;
    AddressU	= WRAP;
    AddressV	= WRAP;
};

//-----------------------------------------------------------------------------
// Vertex Definitions
//-----------------------------------------------------------------------------

// Our sample application will send vertices 
// down the pipeline laid-out like this...

struct APP_OUTPUT
{
	float3 position	: POSITION;
	float3 normal	: NORMAL;
	float2 texture0 : TEXCOORD0;
};

// Once the vertex shader is finished, it will 
// pass the vertices on the pixel shader like this...

struct VS_OUTPUT
{
	float4 hposition : POSITION;
	float4 diffuse	 : COLOR0;
	float4 specular	 : COLOR1;
	float2 texture0  : TEXCOORD0;
	float2 texture1  : TEXCOORD1;
};

// And finally, the pixel shader will send a single 
// color value to the frame buffer like this...

struct PS_OUTPUT
{
	float4 color : COLOR;
};

//-----------------------------------------------------------------------------
// Simple Vertex Shader
//-----------------------------------------------------------------------------

VS_OUTPUT PhongIllumVS( APP_OUTPUT IN )
{
    VS_OUTPUT OUT = (VS_OUTPUT) 0;

	// temp position holder
	float4 tempPos = float4(IN.position, 1);

	// transform normal to world space
	float3 N = normalize( mul(IN.normal, (float3x3)World) );	// Normal in world space

	// Transform to world-space
	float3 worldPos = mul( IN.position, (float4x3)World );		// vertex (world space)

	float3 V = normalize( eyeVec - worldPos );			// view vector (world space)

	//float3 L = normalize( mul(lightVec, (float4x3)World) );	// light (world space)
	float3 L = normalize( lightVec.xyz );				// light vector (world space)
	//float3 L = normalize( lightDir );				// light vector (world space)

	float diff = saturate(dot( N, L ));				// N dot L

	float3 R = normalize(2 * diff * N - L);				// reflection vector (world space)
	float3 H = normalize( L + V );					// half angle vector (world space)

	float4 specular = pow((dot(R, V)), n);				// R.V^n
	//float specular = saturate(dot( N, H ));
	//specular = pow( specular, n );


	OUT.hposition = mul( tempPos, WVP );

	//OUT.diffuse  = k_a + k_d * diff;
	//OUT.specular = specular;//(k_s * specular );

	OUT.diffuse = I_a * k_a + I_d * k_d * diff;			// diffuse + ambient (metal)
	OUT.specular = I_s * k_s * pow(max(0, dot(R, V)), n);		// specular
 
	OUT.texture0 = IN.texture0;
	OUT.texture1.x = (R.x + 1.0f) * 0.5f;
	OUT.texture1.y = (R.y + 1.0f) * 0.5f;


	return OUT;
}

//-----------------------------------------------------------------------------
// Simple Pixel Shader
//-----------------------------------------------------------------------------

PS_OUTPUT PhongIllumPS( VS_OUTPUT IN )
{
	PS_OUTPUT OUT = (PS_OUTPUT) 0;

	// base texture + diffuse + specular
//	OUT.color = tex2D( SamplerBase, IN.texture0 ) * IN.diffuse + IN.specular;

	// (texture0 * texture1) * diffuse + specular
	float4 diff = tex2D( SamplerBase, IN.texture0 );
	float4 refl = tex2D( SamplerSphereMap, IN.texture1 );
	OUT.color = (diff * refl) * IN.diffuse + IN.specular;

	// diffuse + specular
	//OUT.color = IN.diffuse + IN.specular;

	return OUT;
}


//-----------------------------------------------------------------------------
// Environment mapping VS
//-----------------------------------------------------------------------------

VS_OUTPUT EnvMapVS( APP_OUTPUT IN )
{
    VS_OUTPUT OUT = (VS_OUTPUT) 0;

	// temp holders
	float4 tempPos = float4(IN.position, 1);

	// Do bulge effect (displace vectors)
//	float3 dispDir = float3( IN.normal.x, IN.normal.y, IN.normal.z );
//	float dispValue = 4.0f * sin(tempPos.y + fTime * 4);

	// Displace vertex by some value
//	tempPos.xyz += dispDir.xyz * dispValue;


	float3 P = mul( tempPos, (float4x3)World );			// Vertex in world space
	float3 N = mul( IN.normal, (float3x3)World );			// Normal in world space

	float3 E = eyeVec.xyz;						// Camera position in world space

	float3 V = normalize( E - P );					// View vector in world space

	//float d = dot(V, N) * 2.0f;
	//float3 R = normalize(d * N - V);				// Reflection vector in world space
	float3 R = reflect( V, N );


	// Transform vertex 
	OUT.hposition = mul( tempPos, WVP );

	OUT.diffuse = float4( 1.0f, 0.95f, 0.95f, 1.0f ); 
	OUT.specular = float4( 0.0f, 0.0f, 0.0f, 1.0f ); 

	//OUT.texture0.x = ((N.x * 0.5f) + 0.5f);//*0.5f;
	//OUT.texture0.y = ((N.y * 0.5f) + 0.5f);//*0.5f;
	OUT.texture0.x = (R.x + 1.0f) * 0.5f;
	OUT.texture0.y = (R.y + 1.0f) * 0.5f;

	//OUT.texture0.x = ((R.x * 0.5f) + 0.5f);
	//OUT.texture0.y = ((R.y * 0.5f) + 0.5f);

	OUT.texture1 = IN.texture0;
	//OUT.texture0.x = (N.y * 0.5f) + 0.5f;
	//OUT.texture0.y = (N.z * 0.5f) - 0.5f;

	return OUT;
}

//-----------------------------------------------------------------------------
// EnvMap Pixel Shader
//-----------------------------------------------------------------------------

PS_OUTPUT EnvMapPS( VS_OUTPUT IN )
{
	PS_OUTPUT OUT = (PS_OUTPUT) 0;

	float4 negFlag = float4( 1, 1, 1, 1 );

	// texture0 * diffuse * texture1 + specular
	//float4 diff = tex2D( SamplerBase, IN.texture0 );
	float4 refl = tex2D( SamplerSphereMap, IN.texture0 );
	float4 tempColor = refl * IN.diffuse + IN.specular;

	// texture0 * diffuse + specular
	//float4 tempColor = tex2D( SamplerSphereMap, IN.texture0 ) * IN.diffuse + IN.specular;

	OUT.color = tempColor;


	return OUT;
}




//-----------------------------------------------------------------------------
// Simple Effect (1 technique with 1 pass)
//-----------------------------------------------------------------------------

technique TechniquePhongModel
{
    pass Pass0
    {
		ZEnable			= TRUE;
		ZWriteEnable		= TRUE;
		Lighting		= TRUE;
		CullMode		= CCW;

		// enable alpha blending
		AlphaBlendEnable	= TRUE;
		SrcBlend		= SRCALPHA;
		DestBlend		= INVSRCALPHA;

		// set up texture stage states to use the diffuse color
		ColorOp[0]		= MODULATE;//SelectArg1;
		ColorArg1[0]		= TEXTURE;
		ColorArg2[0]		= DIFFUSE;
		AlphaOp[0]		= MODULATE;

		ColorOp[1]		= DISABLE;
		AlphaOp[1]		= DISABLE;


		Sampler[0]		= (SamplerBase);	// Needed by pixel shader
		Sampler[1]		= (SamplerSphereMap);	// Needed by pixel shader


		VertexShader = compile vs_1_1 PhongIllumVS();
		PixelShader  = compile ps_1_3 PhongIllumPS();
    }
}


technique TechniqueEnvMap
{
    pass Pass0
    {
		ZEnable			= TRUE;
		ZWriteEnable		= TRUE;
		Lighting		= FALSE;
		FogEnable		= FALSE;
		CullMode		= CCW;

		// enable alpha blending
		AlphaBlendEnable	= FALSE;
		//SrcBlend		= SRCALPHA;
		//DestBlend		= INVSRCALPHA;

		// set up texture stage states to use the diffuse color
		ColorOp[0]		= SelectArg1;
		ColorArg1[0]		= TEXTURE;
		ColorArg2[0]		= DIFFUSE;
		//AlphaOp[0]		= MODULATE;

		//ColorOp[1]		= MODULATE;
		//ColorArg1[1]		= TEXTURE;
		//ColorArg1[2]		= CURRENT;


		Sampler[0]		= (SamplerSphereMap);	// Needed by pixel shader
		Sampler[1]		= (SamplerBase);	// Needed by pixel shader


		VertexShader		= compile vs_1_1 EnvMapVS();
		PixelShader		= compile ps_1_3 EnvMapPS();
    }
}